swc.constructors.Phonebook = swc.BaseCollection.extend({

    model: 'PhonebookContact',

    url: {
        read: "/sysbus/Phonebook:getAllContacts"
    },

    /**
     * Name of request parameter ID which is stored in the localStorage
     */
    requestParamNameID: 'telephony:phonebook:edit-contactID',

    validation: {
        checkAllowableName: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            // Ascii is defined as the characters in the range of 000-177 (octal), therefore non-printing characters \000-\037
            if (/[\000-\037]+/.test(validationData[validateKey])) {
                validationMessages.push('allowed characters ASCI non-printable');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        checkEmpty: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            if (_.isEmpty(validationData[validateKey])) {
                validationMessages.push('must not be empty');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        checkAllowableNumber: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            if (!/^[\d\b#*]*$/.test(validationData[validateKey])) {
                validationMessages.push('phone number contain not allowed characters');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        firstName:  function(data) { return this.checkAllowableName(data, 'firstName'); },
        lastName:   function(data) { return this.checkAllowableName(data, 'lastName'); },
        homeNumber: function(data) { return this.checkAllowableNumber(data, 'homeNumber'); },
        cellNumber: function(data) { return this.checkAllowableNumber(data, 'cellNumber'); },
        workNumber: function(data) { return this.checkAllowableNumber(data, 'workNumber'); }
    },

    comparator: function(model) {
        return $.trim(model.get('formattedName')).toLowerCase();
    },

    /**
     * Get phonebook
     * @returns json
     * {
     *  "status":[
     *      {
     *          "key":"5",
     *          "lastRevision":"1970-01-01T01:54:48Z",
     *          "uniqueID":"urn:uuid:153a6f06-bd93-4b2b-97e9-9b2da9d27745",
     *          "name":"N:Gogo;Pogo;",
     *          "formattedName":"Gogo Pogo",
     *          "birthDay":"0001-01-01T00:00:00Z",
     *          "title":"",
     *          "organisation":"",
     *          "url":"",
     *          "ringtone":"",
     *          "tags":"",
     *          "addresses":[],
     *          "labels":[],
     *          "telephoneNumbers":[
     *              {
     *                  "key":"1",
     *                  "name":"67890",
     *                  "type":"WORK",
     *                  "preferred":false
     *              },{
     *                  "key":"2",
     *                  "name":"12345",
     *                  "type":"CELL",
     *                  "preferred":false
     *              },{
     *                  "key":"3",
     *                  "name":"12345",
     *                  "type":"HOME",
     *                  "preferred":false
     *              }
     *          ],
     *          "emails":[],
     *          "photos":[]
     *      }
     *  ]
     *}
     */
    apiToJSON: function(response) {
        var telephoneNumbers = [],
            names,
            parsed = [];

        _.each(response.status, function(contact) {
            // sort telephone numbers
            telephoneNumbers = [
                _.where(contact.telephoneNumbers, {'type': 'HOME'})[0],
                _.where(contact.telephoneNumbers, {'type': 'CELL'})[0],
                _.where(contact.telephoneNumbers, {'type': 'WORK'})[0]
            ];
            names = contact.name.substr(2).split(';');

            telephoneNumbers = _.compact(telephoneNumbers);

            parsed.push({
                'formattedName': contact.formattedName,
                'name': contact.name,
                'firstName': names[1] || '',
                'lastName': names[0] || '',
                'telephoneNumbers': telephoneNumbers,
                'id': contact.uniqueID,
                'modified': false,
                'deleted': false
            });
        });

        return parsed;
    },

    deleteAllContacts: function() {
        var deletedContacts = this.where({deleted: true}),
            self = this,
            deferred = new $.Deferred(),
            modificationArray = [];

        /**
         * TODO: create deletion in general way via changedModels({ onlyDeleted: true  })
         * and model.sync('delete') methods (see save() method in the
         * swc.constructors.StorageSettingsDatasharingView as example)
         */
        _.each(deletedContacts, function(model){
            modificationArray.push(model.destroy());
        });

        $.when.apply(this, modificationArray)
            .done(function() {
                deferred.resolve();
            }).
            fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }

});