/**
 * Collection for connected storage devices and internal storage.
 * usage:
 * swc.models.storageDevices = new swc.constructors.StorageDevices();
 * swc.models.storageDevices.getDevices();
 *
 * every model contains device data.
 * device.get('files') - collection of devices' files
 */
swc.constructors.StorageDevices = Backbone.Collection.extend({

    initialize: function() {
        this.temporaryDevices = [];
        this.isRefreshable = true;
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        // Reset collection before sync process:
        this.reset();

        $.when(self.getDevices(fromListener))
            .done(function (response) {
                deferred.resolve();
            })
            .fail(function () {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Comparator for collection with USB devices
     * Sorts devices by 'label'
     *
     * @param {Backbone Model}
     * @returns {string}
     */
    comparator: function(device) {
        return device.get('label').toUpperCase();
    },

    /**
     * Get connected storage devices and internal storage data
     *
     * Devices response:
     * [
     *    {
     *       id : "4C532000021016111590", // Static ID of storage device: uniquely identifies the device.
     *       type: "USB"                  // Either "SD" or "USB"
     *       label: "Transcend 32Gb",     // User-friendly name of the device
     *       fsType: "NTFS",              // Detected file system type e.g. NTFS, FAT32, etc.
     *       capacity: 4194304,           // Capacity in KiB
     *       freeSpace: 2046122,          // Free space in KiB
     *       isSync : true,               // Is synchronized - either true or false.
     *       syncState : "COMPLETE",      // One of: "COMPLETE", "ERROR", "IN_PROGRESS", null
     *       syncError : 7                // Error ID, that can be translated to the meaningful message (either number, or null).
     *    },
     *    ...
     * ]
     *
     * @localParam storage - List of devices with the following attributes
     *
     *  {
     *      device-loaded : 'completed'
     *      devices : Array of devices params
     *  }
     *
     * @localParam fileManager - List of files for each device
     * @returns {deferred.then()}
     */
    getDevices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            storage = new swc.constructors.CentralStorage(),
            fileManager = null; // this list is populated after device is loaded

        $.when(storage.getUSBDevices(fromListener))
            .done(function() {
                var deviceList = storage.models[0].get('devices'),
                    devicesCount = deviceList.length, // Number of devices
                    devicesFilled = 0;                // number of devices with successful files request

                // If no devices in storage collection, it's not possible to retrieve files for them
                if (deviceList.length === 0) {
                    return deferred.resolve();
                }

                _.each(deviceList, function(deviceParams) {
                    var device = new Backbone.Model();
                    var deviceSyncState;

                    // Set device params
                    device.set(deviceParams);

                    // request for files for every separate device.
                    fileManager = new swc.constructors.FileManager([], { deviceID: device.get('id') });

                    $.when(fileManager.sync('read', {isListener: fromListener}))
                        .done(function() {
                            // Add files params
                            device.set('files', fileManager);

                            // Check if all files on the device are synced:
                            // If device have not any content, we take it's sync status from syncState
                            // progress - sync is turned off
                            // complete - sync is turned on
                            if (fileManager.length === 0) {
                                device.set('isSync', device.get('syncState') === 'complete');
                            } else {
                                device.set('isSync', !fileManager.findWhere({ isSync: false }));
                            }

                            self.temporaryDevices.push(device);

                            devicesFilled++;

                            if (devicesFilled === devicesCount) {
                                if (self.isRefreshable) {
                                    self.reset(self.temporaryDevices);
                                }

                                self.temporaryDevices = [];
                                deferred.resolve();
                            }
                        });
                });
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Sends changed sync statuses of files and devices to server
     *
     * @returns deferred
     */
    setSyncStatus: function() {
        var sendArray = [],
            deferred = new $.Deferred();

        /**
         * make the right json structure for sending - all arrays should contain stringified objects
         */
        this.each(function(model){
            var deviceObj = {},
                supportedUsbDevices = _.keys(swc.settings.application.get('supportedUsbDevices'));

            deviceObj.deviceId = model.get('id');
            deviceObj.sync = model.get('isSync');

            // Fill the files
            deviceObj.content = [];
            model.get('files').each(function(model){
                var fileObj = {};
                fileObj.path = '/'+model.get('name');
                fileObj.sync = model.get('isSync');
                fileObj.virtual = model.get('virtual');
                deviceObj.content.push(JSON.stringify(fileObj));
            });

            if ($.inArray(model.get('type').toLowerCase(), supportedUsbDevices) !== -1) {
                sendArray.push(JSON.stringify(deviceObj));
            }
        });

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                "method":"syncFSResources",
                "parameters":{
                    "resources": sendArray
                }
            },
            
            success: function(response) {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    unmountDevice: function(deviceId) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method":"unmount",
                "parameters":{
                    "id": deviceId
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    renameDevice: function(deviceId) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method":"rename",
                "parameters":{
                    "id": deviceId,
                    "label": this.get(deviceId).get('label')
                }
            },

            success: function(response) {
                deferred.resolve();
            },
            
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    renameDevices: function() {
        var deferred = new $.Deferred(),
            self = this,
            requestsArray= [];
        
        self.each(function(model){
            requestsArray.push(self.renameDevice(model.id));
        });
        
        $.when.apply(this, requestsArray).done(function(){
            deferred.resolve();
        });

        return deferred.promise();
    }

});
