// cloud services list
swc.constructors.CloudAccountCollection = swc.BaseCollection.extend({

    model: 'CloudAccount',

    url: {
        read: "/ws"
    },

    ajaxSettings: {
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json',
        requestType: {
            read: 'post'
        }
    },

    ajaxParameters:  JSON.stringify({
        method: 'getCloudSyncInfo',
        service: 'com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup',
        parameters: {}
    }),

    validation: {

        usbUID: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    usbUID: { elementName: 'usbUID' }
                }, data),
                validationMessages = [];

            if (validationData.usbUID === "no-devices-connected") {
                validationMessages.push('no connected devices');
            }
            
            if (validationData.usbUID === "choose-device") {
                validationMessages.push('no device selected');
            }
            
            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        folderName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    usbUID: { elementName: 'usbUID' },
                    folderName: { elementName: 'folderName' }
                }, data),
                validationMessages = [],
                deferred;
            
            // Don't show error about folder name if device was not selected yet
            if (validationData.usbUID === "choose-device") {
                return {
                    status: true,
                    messages: validationMessages
                };
            }

            validationData.folderName = $.trim(validationData.folderName);

            if (!validationData.folderName.length) {
                validationMessages.push('can not be empty');

                return {
                    status: false,
                    messages: validationMessages
                };
            }

            deferred = new $.Deferred();

            $.when(swc.models.CloudServices.validateFolder(validationData.usbUID, validationData.folderName))
                .done(function (status) {
                    deferred.resolve();
                })
                .fail(function (status) {
                    if (!_.isUndefined(status)) {
                        switch (status.failure) {
                            case "SYNC_FOLDER_ALREADY_EXISTS":
                                validationMessages.push('already exists');
                                break;

                            case "SYNC_FOLDER_INVALID":
                            case "BAD_REQUEST":
                            case "INTERNAL_ERROR":
                                validationMessages.push('unable to create folder');
                                break;

                            default:
                                validationMessages.push('unable to create folder');
                                break;
                        }
                    } else {
                        validationMessages.push('unable to create folder');
                    }

                    deferred.reject({
                            status: false,
                            messages: validationMessages
                        });
                });

            return deferred.promise();
        }
    },

    /**
     * Get login url to cloud service from <default> cloud service
     * @param cloudServiceID {String}
     * @returns {*}
     */
    getLoginURL: function(cloudServiceID) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "getServiceAuthUrl",
                parameters: {
                    data: JSON.stringify({
                            serviceId: cloudServiceID,
                            domain: location.host
                        })
                }
            },

            success: function(response) {
                var parsedResponse = JSON.parse(response.status),
                    url;
                if (parsedResponse.data && parsedResponse.data.authUrl) {
                    url = parsedResponse.data.authUrl;
                    deferred.resolve(url);
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Create folder on NAS devices, while connecting to cloud account
     * @param partitionUID {String} - Usb device id
     * @param folderName {String} - user defined folder
     */
    createFolder: function(partitionUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            // There is some specific in AP backend for this call
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "syncFSResources",
                parameters: {
                    resources: [
                        JSON.stringify({
                            deviceId: partitionUID,
                            content: [
                                JSON.stringify({
                                    path: folderName,
                                    action: 'CREATE'
                                })
                            ]
                        })
                    ]
                }
            },

            success: function(response) {
                var responseParsed = JSON.parse(response.status),
                    status = responseParsed.status,
                    error = responseParsed.error || "";

                switch (status) {
                    case 409:
                        error = "already exists";
                        break;

                    case 422:
                        error = "invalid name";
                        break;
                }

                if (_.isEmpty(error)) {
                    deferred.resolve();
                } else {
                    deferred.reject({ message: error });
                }
            }
        });

        return deferred.promise();

    },
    
    /**
     * Movee folder on NAS devices, while connecting to cloud account
     * @param partitionUID {String} - Usb device id
     * @param folderName {String} - user defined folder
     */
    moveFolder: function(model, partitionUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            // There is some specific in AP backend for this call
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "moveSyncFolder",
                parameters: {
                    data: JSON.stringify({
                        oldFolder: {
                            partitionUUID: model.get('deviceId'),
                            folderName: model.get('folderName')
                        },
                        newFolder: {
                            partitionUUID: partitionUID,
                            folderName: folderName
                        }
                    })
                }
            },

            success: function(response) {
                var responseParsed = JSON.parse(response.status),
                    status = responseParsed.status,
                    error = responseParsed.error || "";

                switch (status) {
                    case 409:
                        error = "already exists";
                        break;

                    case 422:
                        error = "invalid name";
                        break;
                }

                if (_.isEmpty(error)) {
                    deferred.resolve();
                } else {
                    deferred.reject({ message: error });
                }
            }
        });

        return deferred.promise();

    },
    
    // toJSON: function () {
    //     return {
    //         method: 'getCloudSyncInfo',
    //         service: 'com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup'
    //     };
    // },

    apiToJSON: function(response) {
        var respStatus = {},
            respData = {},
            accounts = [],
            result = [];

        try {
            // try JSON.parse
            respStatus = JSON.parse(response.status);
            if (respStatus && respStatus.data) {
                accounts = respStatus.data.syncServices; // TODO: old API is used temporarily
            }
        } catch (e) {
        }

        _.each(accounts, function(cloudAccount, key) {
            var deviceId = '',
                deviceLabel = '[Unknown Device]';

            if (cloudAccount.authStatusCode !== 503) {
                if (cloudAccount.partitionUUID) {
                    deviceLabel = cloudAccount.partitionLabel;
                    deviceId = cloudAccount.partitionUUID;
                }

                result.push({
                    id: cloudAccount.id,
                    cloudServiceName: getTranslationStrings(cloudAccount.id === 'dropbox' ? 'Dropbox' : 'Google drive'),
                    deviceLabel: deviceLabel,
                    deviceId: deviceId,
                    accountName: cloudAccount.accountName,
                    status: cloudAccount.syncState ? cloudAccount.syncState : 'DISABLED',
                    error: cloudAccount.error || '',
                    inaccessibleFiles: cloudAccount.inaccessibleFiles,
                    // they return space in megabytes, but we operate with Kilobytes, thus have to multiply by 1024
                    totalSpace: cloudAccount.totalSpace * 1024,
                    usedSpace: cloudAccount.usedSpace * 1024,
                    lastSyncDate: cloudAccount.lastSyncDate,
                    folderName: cloudAccount.localFolderPath
                });
            }
        });

        return result;
    },

    /**
     * Handle oauth callback.
     * This method called when AP requested OAuthCallback handler after first phase of cloud login procedure
     * 
     * @param oauthCallbackParams {Object|Map} list of parameters send to WebUI by AP during first step of sign-in phase
     */
    handleOauthCallback: function (oauthCallbackParams) {
        var deferred = new $.Deferred();
        
        swc.models.Rest.sendRequest({
            url: '/ws',
            
            timeout: 60000,

            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "handleAuthCallback",
                parameters: { authParams: oauthCallbackParams }
            },

            success: function (response) {
                var parsedResponse = JSON.parse(response.status);
                if (parsedResponse.status !== 200) {
                    deferred.reject();
                } else {
                    deferred.resolve(parsedResponse);
                }
            },
            
            error: function(xhr, status, error) {
                deferred.reject();
            }
        });
        
        return deferred.promise();
    }
});
