swc.constructors.Schedulers = Backbone.Collection.extend({

    emptySchedulers: [],

    defaultScheduler: {
        schedule:  {
            "monday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "tuesday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "wednesday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "thursday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "friday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "saturday":[
                {
                    "begin":108,
                    "end":414
                }
            ],
            "sunday":[
                {
                    "begin":108,
                    "end":414
                }
            ]
        }
    },

    comparator: function(model) {
        var priority = 3;

        if (model.get('status')) {
            priority -= 1;
        }

        if (model.get('active')) {
            priority -= 2;
        }

        return priority;
    },

    sync: function(fromListener) {
        var self = this, deferred = new $.Deferred();

        $.when(self.getAPSchedule(fromListener), self.getWiFiSchedule(fromListener))
            .done(function() {
                deferred.resolve();
            });

        return deferred.promise();
    },

    /**
     * invert weekly scheduler - in our web UI we need 'enabled' ranges, but REST needs 'disabled' ranges
     * @param schedule
     */
    invertSchedule: function (schedule) {
        var sortedSchedule = _.sortBy(schedule, function(obj) {
            return obj.begin;
        });

        var startPoint = 0, hops = 0, invertedSchedule = [];
        _.each(sortedSchedule, function(obj) {
            var begin = 0, end = 0;
            begin = startPoint;
            end = obj.begin;

            startPoint = obj.end;
            hops++;
            if ((end-begin)>60) {
                invertedSchedule.push({begin: begin, end: end});
            }
            if (hops === sortedSchedule.length) {
                begin = startPoint;
                end = 604800;
                if ((end-begin)>60) {
                    invertedSchedule.push({begin: begin, end: end});
                }
            }
        });

        if (schedule.length === 0) {
            invertedSchedule = [{'begin': 0, 'end': 604800 }];
        }

        return invertedSchedule;
    },

    /**
     * separate scheduler range to 7 parts by days
     * @param schedule
     */
    separateSchedule: function (schedule, scaleWidth) {

        var sortedSchedule = _.sortBy(schedule, function(obj) {
            return obj.begin;
        });

        var convertVar = 86400/scaleWidth;

        var hops = 0, separatedSchedule = {
            "monday":[],
            "tuesday":[],
            "wednesday":[],
            "thursday":[],
            "friday":[],
            "saturday":[],
            "sunday":[]
        };

        var getDayFromHop = function(hop) {
            var day = 'monday';

            switch (hop) {
            case 0:
                day = "monday";
                break;
            case 1:
                day = "tuesday";
                break;
            case 2:
                day = "wednesday";
                break;
            case 3:
                day = "thursday";
                break;
            case 4:
                day = "friday";
                break;
            case 5:
                day = "saturday";
                break;
            case 6:
                day = "sunday";
                break;
            }

            return day;
        };

        $.each(separatedSchedule,function(currentDay, dayArray) {
            _.each(sortedSchedule, function(obj) {
                var begin, end;
                var secondsInDayBegin = getDaySeconds(obj.begin),
                    secondsInDayEnd = getDaySeconds(obj.end);

                if (getDayFromHop(secondsInDayBegin.hops) === currentDay) {
                    begin = secondsInDayBegin.seconds;
                    if (getDayFromHop(secondsInDayEnd.hops) !== currentDay) {
                        end = 86400;
                    }
                }
                if (getDayFromHop(secondsInDayEnd.hops) === currentDay) {
                    end = secondsInDayEnd.seconds;
                    if (getDayFromHop(secondsInDayBegin.hops) !== currentDay) {
                        begin = 0;
                    }
                }
                if (secondsInDayBegin.hops < hops && secondsInDayEnd.hops > hops) {
                    begin = 0;
                    end = 86400;
                }
                begin = Math.round(begin/convertVar);
                end = Math.round(end/convertVar);

                if (end && end > begin && Math.abs(begin-end) >= (scaleWidth/48)) {
                    dayArray.push({begin: begin, end: end});
                }


            });
            hops++;
        });

        return separatedSchedule;
    },

    /**
     * Split schedule from object into array for sending to server
     * @param schedule
     */
    splitSchedule: function (schedule, scaleWidth) {
        var splittedSchedule = [],
            hops = 0,
            convertVar = 86400/scaleWidth;

        $.each(schedule, function(day, array) {
            _.each(array, function(obj) {
                var range = {},
                    begin = obj.begin * convertVar,
                    end = obj.end * convertVar;

                range.begin = hops * 86400 + begin;
                range.end = hops * 86400 + end;

                if (range.end) {
                    splittedSchedule.push(range);
                }
            });
            hops++;
        });

        splittedSchedule.sort(function(obj1, obj2) {
            return obj1.begin - obj2.begin;
        });

        _.each(splittedSchedule, function(obj, index) {
            if (splittedSchedule[index + 1]) {
                if (obj.end === splittedSchedule[index+1].begin) {
                    var range = {
                        begin: obj.begin,
                        end: splittedSchedule[index+1].end
                    };
                    splittedSchedule[index]=range;
                    splittedSchedule.splice(index+1,1);
                }
            }
        });

        splittedSchedule = this.invertSchedule(splittedSchedule);

        _.each(splittedSchedule, function(obj, index) {
            obj.state = "Disable";
        });

        return splittedSchedule;
    },

    /**
     * Set parameters of chosen range to default settings
     * @param id
     */
    makeDefault: function(id) {
        var model = this.get(id);

        if (model) {
            model.set({
                'pixelSchedule': JSON.parse(JSON.stringify(this.defaultScheduler.schedule)),
                'enable': true
            }, {silent:true} );
            model.set({'schedule': this.splitSchedule(model.get('pixelSchedule'), 432)}, {silent:true});
        }
    },
    /**
     * Add scheduler model
     * @param data - Object, should contain schedule(array), enable(boolean)
     * @param type - string, type of scheduler. Can be 'wifi' or 'ap'
     */
    addScheduleModel: function(data, type) {
        var self = this,
            model = this.get(type);

        if (!model) {
            model = new Backbone.Model();
            self.add(model);

            model.on('change:schedule', function() {
                var invertedRanges = self.invertSchedule(model.get('schedule'));
                model.set({'pixelSchedule': self.separateSchedule(invertedRanges, 432)}, {silent:true});
            });

            model.on('change:pixelSchedule', function() {
                model.set({'schedule': self.splitSchedule(model.get('pixelSchedule'), 432)}, {silent:true} );
                if (model.id === 'ap' || model.id === 'wifi') {
                    if (!self.invertSchedule(model.get('schedule'))[0]) {
                        self.emptySchedulers.push(model.id);
                    } else {
                        self.emptySchedulers = _.without(self.emptySchedulers, model.id);
                    }
                }
            });
        }

        model.set('override', "");
        model.set('enable', data.enable);
        model.set('active', data.active);
        model.set('id', type);

        //We send data.schedule = false when scheduler is fresh, so we will use default scheduler
        if (data.schedule === false) {
            model.set('pixelSchedule', JSON.parse(JSON.stringify(this.defaultScheduler.schedule)));
        } else {
            /**
             * array with scheduler data got from server
             * it has the format of seconds from 0 (0.00 of monday)
             * to 604800 (23.59 of sunday)
             */
            model.set('schedule', data.schedule);
        }

    },

    getDevicesSchedulers: function() {
        var self = this,
            deferred = new $.Deferred();

        var fillArray = function() {
            var array = [];

            swc.models.NetworkDevices.each(function(model) {
                if (model.get('mac')) {
                    var id = model.get('mac');
                    array.push(self.getDeviceScheduler(id, model.get('status')));
                }
            });

            $.when.apply(this, array).done(function() {
                deferred.resolve();
            });
        };

        var checkDevices = function() {
            if (!swc.models.NetworkDevices || !swc.models.NetworkDevices.find(function(model) {
                return model.get('mac');
            })) {
                setTimeout(function() { checkDevices(); }, 200);
            } else {
                fillArray();
            }
        };

        checkDevices();

        return deferred.promise();
    },

    getDeviceScheduler: function(id, isActive) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Scheduler:getSchedule',
            method: 'POST',
            contentType: 'application/x-sah-ws-1-call+json',
            data: {
                "parameters": {
                    "type": "ToD",
                    "ID": id
                }
            },

            success: function(resp) {

                var dataForModel,
                    data,
                    response = resp.result;

                if (response.status) {
                    data = response.data.scheduleInfo;
                    dataForModel = {
                        override: "",
                        schedule: data.schedule || [],
                        enable: data.enable,
                        active: isActive,
                        "ID": id
                    };
                } else {
                    /**
                     * response.status=false responses when device is new and fresh.
                     * so for the first time we send empty data to turn scheduler on
                     */
                    dataForModel = {
                        override: "",
                        enable: false,
                        schedule: false,
                        active: isActive,
                        "ID": id
                    };
                }

                if (self.get(id)) {
                    self.remove(self.get(id));
                }

                self.addScheduleModel(dataForModel, id);
                deferred.resolve();
            },

            error:function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setDevicesSchedulers: function() {
        var self = this,
            deferred = new $.Deferred(),
            array = [];

        this.each(function(model) {
            if (model.id !== 'ap' && model.id !== 'wifi') {
                array.push(self.addSchedule(model.id));
            }
        });

        $.when.apply(this, array)
            .done(function(){
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    getWiFiSchedule: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Scheduler:getCompleteSchedules',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "type": "WLAN"
                }
            },

            success: function(response) {
                var schedulerStatus = response.status,
                    schedulerObj = {},
                    dataForModel = {};

                if (schedulerStatus && response.data.scheduleInfo[0]) {
                    // In old APs we dont have universal "WiFiSchedule" and will use wl0
                    schedulerObj = _.where(response.data.scheduleInfo, {"ID":"wl0"})[0];

                    dataForModel = {
                        override: "",
                        enable: schedulerObj.enable,
                        schedule: schedulerObj.schedule
                    };

                } else {
                    /**
                     * response.status=false responses when device is new and fresh.
                     * so for the first time we send empty data to turn scheduler on
                     */
                    dataForModel = {
                        override: "Enabled",
                        enable: false,
                        schedule: false
                    };
                }

                self.addScheduleModel(dataForModel, "wifi");
                deferred.resolve();
            },

            error:function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getAPSchedule: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            request = {
                "parameters":{
                    type: "AP"
                }
            };

        swc.models.Rest.sendRequest({
            url: '/sysbus/Scheduler:getCompleteSchedules',
            method: 'POST',
            contentType: 'application/x-sah-ws-1-call+json',
            fromListener: fromListener,

            data: request,

            success: function(resp) {
                var response = resp.result,
                    dataForModel,
                    data;

                if (response.status && response.data.scheduleInfo[0]) {
                    data = _.where(response.data.scheduleInfo, {"ID":"APSchedule"})[0];
                    dataForModel = {
                        override: "",
                        schedule: data.schedule || [],
                        enable: data.enable
                    };
                } else {
                    /**
                     * response.status=false responses when device is new and fresh.
                     * so for the first time we save empty data to turn scheduler on
                     */
                    dataForModel = {
                        override: "Enabled",
                        enable: false,
                        schedule: false
                    };
                }

                self.addScheduleModel(dataForModel, 'ap');
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    addSchedule: function(type) {
        var self = this,
            deferred = new $.Deferred(),
            model = self.get(type),
            schedule = model.get('schedule'),
            requestDataType,
            requestDataId;

        if (type === "wifi") {
            requestDataType = "WLAN";
            requestDataId = "wl0";
        } else if (type === "ap") {
            requestDataType = "AP";
            requestDataId = "APSchedule";

        } else {
            requestDataType = "ToD";
            requestDataId = type;
        }

        // Send default scheduler if our scheduler is empty
        if ((type === "wifi" || type === "ap") && (schedule[0] && schedule[0].begin === 0 && schedule[0].end === 604800)) {
            self.emptySchedulers = _.without(self.emptySchedulers, type);
            self.makeDefault(type);
            model.set('enable', false);
            schedule = model.get('schedule');
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/Scheduler:addSchedule',

            data: {
                "parameters": {
                    "type": requestDataType,
                    "info": {
                        "ID": requestDataId,
                        "enable": model.get('enable'),
                        "base": "Weekly",
                        "def": "Enable",
                        "override": "",
                        "schedule": schedule
                    }
                }
            },

            success: function(response) {
                // TODO: put this error handler in the common place
                // in this case response can be successful and
                // contain error message. That is why it should be
                //  checked on it here
                if (_.isUndefined(response.errors)) {
                    deferred.resolve(response);
                } else {
                    deferred.reject(response);
                }
            },

            error: function(xhr) {
                deferred.reject(xhr);
            }
        });

        return deferred.promise();
    }

});
