swc.constructors.CentralStorage = Backbone.Collection.extend({

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        // Delete deprecated models from collection
        this.reset();

        // Load new models to the collection
        $.when(self.getGlobalStatus(fromListener))
            .done(function() {
                $.when(self.getMediaServerStatus(fromListener), self.getUSBDevices(fromListener))
                    .done(function() {
                        self.trigger('central-storage-update');
                        deferred.resolve();
                    });
            });

        return deferred.promise();
    },

    getGlobalStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:getMode',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                // Create new model:
                model.set('id', 'status');

                // Check if data exists:
                if (!response.data) {
                    model.set('status', false);
                } else {
                    model.set('status', response.data.mode === "ON");
                }

                // Add new model to the collection:
                self.add(model);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setGlobalStatus: function(state) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:setMode',
            data: {
                "parameters": {
                    "mode": state ? "ON" : "OFF"
                }
            },

            success: function(response) {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getMediaServerStatus: function(fromListener) {
        var model = this.where({ id: 'status' })[0],
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,

            data: {
                "service": "com.swisscom.stargate/ws/dlna.com.swisscom.stargate.dlna",
                "method": "getMediaStreamingState",
                "parameters":{}
            },

            success: function(response) {
                var data;

                // Check if AP has successfully started
                if (response.status.indexOf('status') === -1) {
                    data = false;
                } else {
                    data = JSON.parse(response.status).data;
                }

                // Update status model with media streaming server value:
                model.set('mediaServerStatus', data["mediaStreamingEnabled"]);

                deferred.resolve();
            },

            error: function(response) {
                // Update status model with media streaming server value:
                model.set('mediaServerStatus', false);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * Get device icon by its type
     *
     * @param deviceType {String} usb-disk | sd-card
     * @param supported {Boolean} If this device is supported by InternetBox
     * @return {String}
     */
    getIcon: function(deviceType, supported) {
        var icon;

        // if the storage device is not SD-card(mouse, keyboard etc)
        // then it's considered as usb-disk
        if (deviceType === 'sd-card') {
            icon = 'sd-card';
        } else {
            icon = 'usb-disk';
        }

        if (supported !== true) {
            icon += '-undefined';
        }

        return icon;
    },

    /**
     * Set cloud backup state
     *
     * @param responseSyncState
     * @returns {String}
     */
    getSyncState: function(responseSyncState) {
        var map = {
            'IN_PROGRESS': 'progress',
            'UPLOAD_IN_PROGRESS': 'progress',
            'DELETE_IN_PROGRESS': 'progress',
            'COMPLETE': 'complete',
            'ERROR': 'error'
        };

        if (_.isUndefined(map[responseSyncState])) {
            return 'not_sync';
        }

        return map[responseSyncState];
    },

    /**
     * Comparator for collection with USB devices
     * Sorts devices by 'label'
     *
     * @param {Backbone Model}
     * @returns {string}
     */
    comparatorUSBDevices: function(device) {
        return device.get('label').toUpperCase();
    },

    /**
     * Get list of USB devices
     *
     * @param fromListener {Boolean}
     * @returns {Promise}
     */
    getUSBDevices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            usbModel = new swc.constructors.SimpleModel(),
            usbDevices = new Backbone.Collection();

        // set method of sorting for device list
        usbDevices.comparator = this.comparatorUSBDevices;

        // Set model id:
        usbModel.set('id', 'usb-devices');
        usbModel.set('device-loaded', 'in-progress');

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,

            data: {
                "service": "com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method": "list",
                "parameters": {}
            },

            success: function(response) {
                var devicesData;

                // Check if AP has successfully started
                if (response.status.indexOf('status') === -1) {
                    devicesData = { data: [] };
                } else {
                    devicesData = JSON.parse(response.status);
                }

                if (devicesData.data.length) {
                    $.each(devicesData.data, function(key, device) {
                        var model = new swc.constructors.SimpleModel();

                        model.set({
                            id: device.id,
                            capacity: swc.Utils.getBytesWithUnit(device.capacity * 1024),
                            freeSpace: swc.Utils.getBytesWithUnit(device.freeSpace * 1024),
                            fsType: device.fsType,
                            isSync: device.isSync || false,
                            label: device.label,
                            syncErrorCode: device.syncErrorCode,
                            syncState: self.getSyncState(device.syncState),
                            type: self.getIcon(device.type, device.supported)
                        });

                        // Add current device to usb devices collection:
                        usbDevices.add(model);
                    });
                }

                // Add new devices collection to data storage:
                usbModel.set("devices", usbDevices.toJSON());
                usbModel.set('device-loaded', 'completed');

                // Add new usb model to the collection:
                self.add(usbModel);

                deferred.resolve();
            },

            error: function(xhr, status, error) {

                // Add new devices collection to data storage:
                usbModel.set("devices", []);
                usbModel.set('device-loaded', 'none');

                // Add new devices collection to data storage:
                self.add(usbModel);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    // FIXME: Create corresponding method in Storage model and call it here
    removeUSBDevice: function(id) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method": "unmount",
                "parameters": {
                    "id": id
                }
            },

            success: function(response) {
                deferred.resolve();
            },
            
            error: function (xhr, status, error) {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get necessary parameter from one of the storage models:
     * @param parameter {String}
     * @param component {String}
     */
    getParameter: function(parameter, component) {
        var search = this.where({ id: component });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return '';
        }
    },

    /**
    * Returns true if the Central Storage is ON, false otherwise
    */
    isEnabled: function () {
        var centralStorageInfo = this.get('status');

        return centralStorageInfo ? centralStorageInfo.get('status') : false;
    },

    /**
     * Returns the list of all supported usb devices and sd-cards
     * in dropdown-ready format.
     */
    getDeviceDropdownOptions: function () {
        var usbDevices = this.get('usb-devices'),
            devices = _.isEmpty(usbDevices) ? [] : usbDevices.get('devices'),
            supportedUsbDevices = _.keys(swc.settings.application.get('supportedUsbDevices')),
            options = [];

        // Get all usb disks to the options list
        _.each(devices, function(device, key) {

            // Don't show "undefined-storage" - unsupported device type
            if ($.inArray(device.type.toLowerCase(), supportedUsbDevices) !== -1) {
                options.push({
                    name: device.label,
                    value: device.id
                });
            }
        });

        if (_.isEmpty(options)) {
            options = [
                {
                    name: getTranslationStrings("No Connected Devices"),
                    value: "no-devices-connected",
                    isDefault: true
                }
            ];
        } else {
            options = _.sortBy(options, function (option) {
                return option.name.toLowerCase();
            });

            options.unshift({
                name: getTranslationStrings("Choose a device"),
                value: "choose-device",
                isDefault: true
            });
        }

        return options;
    },

    /**
     * Helper that returns oldValue if there is  an options element with value===oldValue
     * and returns the id of the first element otherwise.
     * Is used to process getDeviceDropdownOptions method results.
     *
     * @param options {Object}
     * @param dropdownDataValue {String} - value saved in dropdown `data` attribute
     */
    getDefaultDeviceOption: function (options, dropdownDataValue) {
        var defaultOptions = [ "no-devices-connected", "choose-device" ],
            defaultDevice = defaultOptions[0];

        // Set device to previously selected, or show "choose a device" option
        if (options.length >= 2) {
            if (!_.isEmpty(dropdownDataValue) && _.indexOf(dropdownDataValue, defaultOptions) === -1) {
                _.each(options, function (obj, key) {
                    if (obj.value === dropdownDataValue) {
                        defaultDevice = obj.value;
                    }
                });
            } else {
                defaultDevice = defaultOptions[1];
            }
        }

        return defaultDevice;
    }
});
