swc.constructors.PortForwarding = swc.BaseCollection.extend({

    model: 'PortForwardingRule',

    url: {
        read: "/sysbus/Firewall:getPortForwarding"
    },

    comparator: function(item) {
        return parseInt(item.get('ports').entry, 10);
    },

    // TODO :: when normal validation mechanism will be done -> refactor this
    validation: {

        deviceIP: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    deviceIP: { elementName: 'deviceIP' }
                }, data),
                validationMessages = [];

            if (validationData.deviceIP === "0.0.0.0") {
                validationMessages.push('device must be selected');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        serviceName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'serviceName' },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [];

            // Skip validation for predefined mode
            if (validationData.mode !== "predefined") {
                var searchModel = swc.models.PortForwarding.findWhere({ name: validationData.name });

                if (!$.trim(validationData.name)) {
                    validationMessages.push('must not be empty');
                }

                if (validationData.name.length < 1 || validationData.name.length > 40) {
                    validationMessages.push('must be from 1 to 40 characters');
                }

                // Ascii is defined as the characters in the range of 000-177 (octal),
                // therefore non-printing characters \000-\037
                if (/[\000-\037]/.test(validationData.name)) {
                    validationMessages.push('allowed characters ASCI non-printable');
                }

                if (!_.isUndefined(searchModel)) {
                    // Skip check if model is editing
                    if (searchModel.get('id') !== validationData.id) {
                        validationMessages.push('already existed rule names are not allowed');
                    }
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        predefinedServiceName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'predefinedServiceName' }
                }, data),
                validationMessages = [];

            // Skip validation for predefined mode
            if (validationData.mode !== "custom") {
                var searchModel = swc.models.PortForwarding.findWhere({ name: validationData.name });

                if (!_.isUndefined(searchModel)) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        entryPort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "entry-ports-mode" },
                    from: { elementName: "entryPort", elementClass: "from" },
                    to: { elementName: "entryPort", elementClass: "to" },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [],
                portsRange = [];

            // Convert to integer values:
            validationData.from = +validationData.from;
            validationData.to = +validationData.to;
            
            if (validationData.from < 1 || validationData.from > 65535) {
                validationMessages.push('allowed range 1 to 65535');
            }

            if (/\D+/.test(validationData.from)) {
                validationMessages.push('allowed characters 0-9');
            }

            // Skip validation for predefined mode
            if (validationData.mode === "range") {

                if (!$.trim(validationData.to)) {
                    validationMessages.push('must not be empty');
                }

                if (/\D+/.test(validationData.to)) {
                    validationMessages.push('allowed characters 0-9');
                }

                if (validationData.to < 1 || validationData.to > 65535) {
                    validationMessages.push('allowed range 1 to 65535');
                }

                if (validationData.to <= validationData.from) {
                    validationMessages.push('range option first value MUST be less then second value');
                }

                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }

            // Check for TR-069 blocked ports
            if (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [1024, 1048])) {
                validationMessages.push('ports 1024-1048 are not allowed because this ports reserved for system use');
            }

            // Check ports crossing conflict:
            _.each(swc.models.PortForwarding.models, function(model, key) {
                if (!swc.models.PortForwarding.validation.helpers.
                    portsNotInRange(portsRange, model.get('ports').entry)) {

                    // Skip check when editing model
                    if (model.get('id') !== validationData.id) {
                        validationMessages.push('already used ports are not allowed');
                    }
                }
            });

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        destinationPort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "destination-ports-mode" },
                    from: { elementName: "destinationPort", elementClass: "from" },
                    to: { elementName: "destinationPort", elementClass: "to" },
                    deviceIP: { elementName: 'deviceIP' },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [],
                portsRange = [];

            // Convert to integer values:
            validationData.from = +validationData.from;
            validationData.to = +validationData.to;

            if (/\D+/.test(validationData.from)) {
                validationMessages.push('allowed characters 0-9');
            }

            if (validationData.from < 1 || validationData.from > 65535) {
                validationMessages.push('allowed range 1 to 65535');
            }

            if (validationData.mode === "range") {
                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }

            // Check for TR-069 blocked ports
            if (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [1024, 1048])) {
                validationMessages.push('ports 1024-1048 are not allowed because this ports reserved for system use');
            }

            // Check ports crossing conflict for current device:
            _.each(swc.models.PortForwarding.where({ 'deviceIP': validationData.deviceIP}), function(model, key) {
                if (!swc.models.PortForwarding.validation.helpers.
                    portsNotInRange(portsRange, model.get('ports').destination)) {

                    // Skip check when editing model
                    if (model.get('id') !== validationData.id) {
                        validationMessages.push('already used ports for same device are not allowed');
                    }
                }
            });

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        helpers: {

            /**
             * Check if ports are not crossing between each other
             *
             * @description
             *
             *  [1, 5] and [3, 6] are conflicting, because they are in same range
             *
             *  [1, 5] and [6, 7] are not conflicting
             *
             * @param ports {Array} length (1 || 2) Port to validate for
             * @param range {Array} length (1 || 2) Ports to validate against
             *
             * @returns {boolean}
             */
            portsNotInRange: function(ports, range){
                // Case 1 [] && []
                if (ports.length === 1 && range.length === 1) {
                    if (parseInt(ports[0], 10) !== parseInt(range[0], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 2 [][] && []
                if (ports.length === 2 && range.length === 1) {
                    if (parseInt(range[0], 10) < parseInt(ports[0], 10) ||
                        parseInt(range[0], 10) > parseInt(ports[1], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 3 [] && [][]
                if (ports.length === 1 && range.length === 2) {
                    if (parseInt(ports[0], 10) < parseInt(range[0], 10) ||
                        parseInt(ports[0], 10) > parseInt(range[1], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 4 [][] && [][]
                if (ports.length === 2 && range.length === 2) {
                    if (
                        (parseInt(ports[0], 10) < parseInt(range[0], 10) &&
                            parseInt(ports[1], 10) < parseInt(range[0], 10)) ||
                            (parseInt(range[0], 10) < parseInt(ports[0], 10) &&
                                parseInt(range[1], 10) < parseInt(ports[0], 10))
                        ) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        }
    },

    /**
     * List of predefined service which is used to allow user easily add rules using this templates
     *
     * @param predefinedServices {Map}
     */
    predefinedServices: {
        "FTP":    { port: "21",   protocol: "6"  },
        "SSH":    { port: "22",   protocol: "6"  },
        "Telnet": { port: "23",   protocol: "6"  },
        "SMTP":   { port: "25",   protocol: "6"  },
        "TFTP":   { port: "69",   protocol: "17" },
        "HTTP":   { port: "80",   protocol: "6"  },
        "HTTPS":  { port: "443",  protocol: "6"  },
        "POP3":   { port: "110",  protocol: "6"  },
        "SNMP":   { port: "161",  protocol: "17" },
        "PPTP":   { port: "1723", protocol: "6"  }
    },

    apiToJSON: function(response) {
        var rules = response.status,
            rulesSourcesSkip = [ "cwmp" ], // Disabled rules sources (acs.. etc.)
            rulesParsed = [];
        
        _.each(rules, function(rule, key) {
            if (_.indexOf(rulesSourcesSkip, rule.Origin) === -1) {
                rulesParsed.push({
                    id: rule.Id,
                    isUPnP: key.indexOf('upnp') !== -1,
                    status: rule.Enable,
                    name: rule.Description,
                    deviceIP: rule.DestinationIPAddress,
                    protocol: rule.Protocol,
                    origin: rule.Origin,
                    sourceInterface: rule.SourceInterface,
                    ports: {
                        entry: rule.ExternalPort.split('-'),
                        destination: rule.InternalPort.split('-')
                    }
                });
            }
        });
        
        return rulesParsed;
    }

});
