swc.constructors.NetworkDevices = Backbone.Collection.extend({

    /**
     * Sort ASC by IPv4 address.
     * Disconnected devices should be displayed last
     */
    comparator: function(device1, device2) {
        // If no IP address - show device as last
         if (device1.get('address').IPv4 && !device2.get('address').IPv4) {
            return -1;
        }

        // All devices with IP address provided should be sorted ASC
        if (device1.get('address') && device2.get('address')) {
            if (!_.isEmpty(device1.get('address').IPv4) && !_.isEmpty(device2.get('address').IPv4)) {
                var device1_ipComponents = device1.get('address').IPv4.split("."),
                    device2_ipComponents = device2.get('address').IPv4.split(".");
                return _.reduce(device1_ipComponents, function (memo, nextVal) {
                    return memo * 255 + parseInt(nextVal, 10);
                }) < _.reduce(device2_ipComponents, function (memo, nextVal) {
                    return memo * 255 + parseInt(nextVal, 10);
                }) ? -1 : 1;
            } else {
                return 1;
            }
        }

        return 0;
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        if (!swc.models['Wireless']) {
            swc.models['Wireless'] = new swc.constructors['Wireless']();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/Hosts:getDevices',
            method: 'POST',
            contentType: 'application/x-sah-event-1-call+json',
            fromListener: fromListener,
            data: {
                "parameters":{}
            },
            success: function(response) {
                $.when(swc.models.Wireless.sync(true)).done(function() {
                    var disabledDevices = [
                        'internetbox-nas'
                    ];

                    // Remove old list of devices and fill collection with relevant devices list:
                    if (response.result && response.result.status) {
                        self.reset();

                        $.each(response.result.status, function(key, device) {
                            if ($.inArray(device.hostName, disabledDevices) === -1) {
                                self.add(self.createDevice(device));
                            }
                        });
                    }

                    // Trigger event to notify - devices update finished
                    self.trigger('devices-loaded', true);

                    deferred.resolve();
                });
            }
        });

        return deferred.promise();
    },

    /**
     * Set type of device from data param and check if this type
     * should be mapped according to map "devicesMapping" in the
     * application.json file
     *
     * This map was added according the following tickets:
     *      - SA-2134
     *      - SA-2514
     *
     * @param data {Object} Response from request "/sysbus/Hosts:getDevices"
     * @returns {String}
     */
    getDeviceType: function(data) {
        var devicesMapping = swc.settings.application.get('devicesMapping'),
            type = '';

        if (!data) {
            return type;
        }

        type = data.deviceType.toLowerCase();

        if (devicesMapping[type]) {
            type = devicesMapping[type];
        }

        return type;
    },

    /**
     * Get icon of device by deviceType param from hashes
     * "supportedWiredDevices" and "supportedWirelessDevices"
     * in the application.json file
     *
     * @param interfaceType {String} wired | wireless
     * @param deviceType {String} Type of device as is from the field "deviceType" in the response
     */
    getDeviceIcon: function(interfaceType, deviceType) {
        var supportedWired = swc.settings.application.get('supportedWiredDevices'),
            supportedWireless = swc.settings.application.get('supportedWirelessDevices'),
            icon = '';

        if (interfaceType === 'wired') {
            if (!supportedWired[deviceType]) {
                icon = supportedWired["unrecognized"].deviceIcon;
            } else {
                icon = supportedWired[deviceType].deviceIcon;
            }
        }

        if (interfaceType === 'wireless') {
            if (!supportedWireless[deviceType]) {
                icon = supportedWireless["unrecognized"].deviceIcon;
            } else {
                icon = supportedWireless[deviceType].deviceIcon;
            }
        }

        return icon;
    },

    /**
     * Get interface of access point based on the
     * field "layer2Interface" in the response
     * of the request "/sysbus/Hosts:getDevices"
     *
     * @param wifiApLayer {String}
     */
    getInterfaceAP: function(wifiApLayer) {
        var map = { 'wl0': '2.4GHz', 'wl1': '5GHz', 'wl0.1': 'Guest-2.4GHz', 'wl1.1': 'Guest-5GHz' };

        return !_.isUndefined(map[wifiApLayer]) ? map[wifiApLayer] : '';
    },

    /**
     * Parse data in the response of the request "/sysbus/Hosts:getDevices"
     * and set appropriate fields to new device model
     *
     * @param data {Object} Response data
     * @return {Backbone.Model} Device model
     */
    createDevice: function(data) {
        var model = new swc.constructors.SimpleModel(),
            status = false,
            statusClass='disconnected',
            addressIPv4 = '', addressIPv6 = [],
            addressObjIPv4 = {}, addressObjIPv6 = [],
            interfaceType = '', interfaceSSID = '',
            wifiOperatingChannelBandwidth = getTranslationStrings('Auto'),
            wifiOperatingStandards = '',
            gatewayPort = '', icon = '',
            type, wifiApLayer, interfaceAP;

        // Check if device is valid
        if (!data) {
            return model;
        }

        type = this.getDeviceType(data);
        wifiApLayer = data.layer2Interface;

        // Define interface type
        if (data.interfaceType === "Ethernet") {
            interfaceType = "wired";
            icon = this.getDeviceIcon('wired', type);
        } else {
            interfaceType = "wireless";
            icon = this.getDeviceIcon('wireless', type);
            interfaceAP = this.getInterfaceAP(wifiApLayer);
            interfaceSSID = swc.models.Wireless.getParameter("name", interfaceAP);
            wifiOperatingChannelBandwidth = swc.models.Wireless.getParameter("OperatingChannelBandwidth", interfaceAP);
            wifiOperatingStandards = swc.models.Wireless.getParameter("OperatingStandards", interfaceAP);
        }
        
        // Define IP Address:
        $.each(data.addresses, function(key, ipAddrItem){
            if ( ipAddrItem.family === 'IPv4') {
                addressIPv4 = ipAddrItem.ipAddress;
                addressObjIPv4 = ipAddrItem;
            } else {
                addressIPv6.push(ipAddrItem.ipAddress);
                addressObjIPv6.push(ipAddrItem);
            }
        });

        // Define device status:
        if ((addressIPv4.length || addressIPv6.length) && data.active) {
            status = true;
        }

        if (status) {
            statusClass = 'connected';
        }

        if (data.interfaceType === "Ethernet" && /eth\d*/.test(wifiApLayer)){
            gatewayPort = parseInt(wifiApLayer.replace('eth',''), 10) + 1;
        }

        // Base data:
        model.set('mac', data.physAddress);
        model.set('name', data.hostName);
        model.set('dnsName', data.dnsName);
        model.set('type', type);
        model.set('icon', icon);
        model.set('status', status);
        model.set('statusClass', statusClass);

        // IP Address:
        model.set('address', {
            'IPv4': addressIPv4,
            'IPv6': addressIPv6
        });
        model.set('addressObjects', {
            'IPv4': addressObjIPv4,
            'IPv6': addressObjIPv6
        });

        // Interface type:
        model.set('interfaceType', interfaceType);
        model.set('interfaceSSID', interfaceSSID);
        model.set('interfaceAP', interfaceAP);
        model.set('gatewayPort', gatewayPort);
        
        // Some information about WiFi connection
        model.set('wifiOperatingChannelBandwidth', wifiOperatingChannelBandwidth);
        model.set('wifiOperatingStandards', wifiOperatingStandards);

        // Last connection and change:
        model.set('lastChange', data.lastChange);
        model.set('lastConnection', data.lastConnection);

        return model;
    },

    getDevice: function(MacAddress) {
        var search = this.where({ mac: MacAddress });

        if (search.length) {
            return search[0].attributes;
        } else {
            return false;
        }
    },

    getDeviceIP: function(MacAddress) {
        var device = this.getDevice(MacAddress),
            ip = '';

        if (!device) {
            return ip;
        }

        if (device.address['IPv4']) {
            ip = device.address['IPv4'];
        } else {
            if (device.address['IPv6'].length) {
                ip = device.address['IPv6'][0];
            }
        }

        return ip;
    },

    getDeviceMac: function(IPAddress) {
        var devicesAll = this.getConnectedDevices(),
            devices = devicesAll['wired'].concat(devicesAll['wireless']),
            mac = '';

        $.each(devices, function(key, device) {
            if (device.address['IPv4'] === IPAddress) {
                mac = device.mac;
            } else {
                if (device.address['IPv6'].length) {
                    $.each(device.address['IPv6'], function(ipKey, ipValue) {
                        if (ipValue === IPAddress) {
                            mac = device.mac;
                        }
                    });
                }
            }
        });

        return mac;
    },

    getDeviceByIP: function(IPAddress) {
        var device;

        _.each(this.models, function(model, key) {
            if (!_.isEmpty(model.get('address')['IPv4']) && model.get('address')['IPv4'] === IPAddress) {
                device = model;
            } else {
                if (!_.isEmpty(model.get('address')['IPv6'])) {
                    _.each(model.get('address')['IPv6'], function(IPv6, key) {
                        if (IPv6 === IPAddress) {
                            device = model;
                        }
                    });
                }
            }
        });

        return device;
    },

    getDevicesList: function() {
        var devices = [],
            connected = [],
            disconnected = [];

        // Sort devices to connected and disconnected
        $.each(this.models, function(key, model) {
            var device = model.attributes;

            if (!device.status) {
                if (disconnected.length < 10) {
                    disconnected.push(device);
                }
            } else {
                connected.push(device);
            }
        });

        return devices.concat(connected).concat(disconnected);
    },

    getConnectedDevices: function() {
        var devices = {
            'wired': [],
            'wireless': []
        };

        // Sort devices to connected and disconnected
        $.each(this.models, function(key, model) {
            var device = model.attributes;

            if (device.status) {
                devices[device.interfaceType].push(device);
            }
        });

        return devices;
    },

    getDisconnectedDevices: function() {
        var devices = {
            'wired': [],
            'wireless': []
        };

        // Sort devices to connected and disconnected
        $.each(this.models, function(key, model) {
            var device = model.attributes;

            if (device.status) {
                devices[device.interfaceType].push(device);
            }
        });

        return devices;
    },

    countWiredConnectedDevices: function() {
        return this.where({ status: true, interfaceType: 'wired' }).length;
    },

    removeDevice: function(mac) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Hosts:delHost',
            method: 'POST',
            contentType: 'application/x-sah-event-1-call+json',
            data: {
                "parameters":{
                    "physAddress": mac
                }
            },
            success: function(response) {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    customizeDevice: function(data) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Hosts:setDevice',
            data: {
                "parameters":{
                    "device": {
                        "physAddress": data.deviceMac,
                        "hostName": data.deviceName,
                        "deviceType": data.deviceType
                    }
                }
            },

            success: function(response) {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getDeviceDropdownOptions: function(params) {
        var devices = swc.models.NetworkDevices.filter(function(model) {
                    if (!params || !params.filter) {
                        return true;
                    } else if (params.filter === 'status') {
                        return !!model.get('status');
                    } else if (params.filter === 'DHCPLeases') {
                        return !swc.models.DHCPLeases.get(model.get("mac"));
                    }
                }),
            options = _.map(devices, function(model) {
                var ipAddress = model.get('status') ? model.get('address').IPv4 : "";

                if (ipAddress && ipAddress.length > 15) { // Truncate IPv6 last 3 groups
                    ipAddress = ipAddress.match(/.*((?::[a-z0-9]*){3})/)[1];
                }

                var option = {
                    "name": model.get("name"),
                    "value": model.get("mac"),
                    "additionalLabel": ipAddress
                };

                return option;
            });

        return options;
    }
});
