swc.constructors.Network = Backbone.Collection.extend({

    validation: {

        /** Provides valid interface for view.pageValidation */
        validateIPAddress: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');
            
            return this.validateIPAddressDecimal(Address, Netmask, DHCPMin, DHCPMax);
        },

        validateNetmask: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');
            
            return this.validateNetmaskDecimal(Address, Netmask, DHCPMin, DHCPMax);
        },

        validateDHCPSettings: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');
            
            return this.validateDHCPSettingsDecimal(Address, Netmask, DHCPMin, DHCPMax);
        },

        validateDMZ: function(data) {
            var dmzAddress = _.findWhere(data, {parameterName: 'DMZIPAddress'}),
                validationStatus = {
                    status: true,
                    messages: []
                };

            if (dmzAddress === false) {
                return validationStatus;
            }

            if (($.inArray('disabled', dmzAddress.parameterClasses) ===-1) && !dmzAddress.parameterValue) {
                validationStatus.status = false;
                validationStatus.messages.push("global");
            }
            
            return validationStatus;
        },

        // Validates decimal values of IP Address and other fields
        validateIPAddressDecimal: function(Address, Netmask, DHCPMin, DHCPMax) {
            var validationStatus = {
                status: true,
                messages: []
            };
            
            if (Address === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Invalid IPv4 Address Format");
                return validationStatus;
            }

            if (!(Address > 3232235520 && Address < 3232301055)/* 192.168.0.0 - 192.168.255.255*/ &&
                !(Address > 167772160 && Address < 184549375)/* 10.0.0.0 - 10.255.255.255 */ &&
                !(Address > 2886729728 && Address < 2887778303)/* 172.16.0.0 - 172.32.255.255 */) {
                validationStatus.status = false;
                validationStatus.messages.push("Non-private subnet");
                //'You have choosen an invalid network. Only private Networks according RFC 1918 are allowed'
                return validationStatus;
            }

            //We need Netmask to proceed validation
            if (Netmask === false) {
                return validationStatus;
            }
            
            var netmaskValidation = this.validateNetmaskSubnetDecimal(Address, Netmask);
            if (netmaskValidation.status === false) {
                return netmaskValidation;
            }

            //We need DHCP to validate further
            if (DHCPMin === false || DHCPMax === false) {
                return validationStatus;
            }
            var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
            if (validDHCPSubnet.status === false) {
                return validDHCPSubnet;
            }

            return validationStatus;
        },

        // Validate decimal value of Netmask and other fields if provided
        validateNetmaskDecimal: function(Address, Netmask, DHCPMin, DHCPMax) {
            var validationStatus = {
                status: true,
                messages: []
            };
            
            if (Netmask === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Invalid Netmask format");
                return validationStatus;
            }

            var netmaskString = Netmask.toString(2);
            if (netmaskString.match(/01/) || !netmaskString.match(/[0]{2,}$/ig)) { //Has raising sequence 01 or less than 2 zeroes at end
                validationStatus.status = false;
                validationStatus.messages.push("Invalid Netmask format");
                return validationStatus;
            }

            var netmaskValidation = this.validateNetmaskSubnetDecimal(Address, Netmask);
            if (netmaskValidation.status === false) {
                return netmaskValidation;
            }

            //We need DHCP to validate further
            if (DHCPMin === false || DHCPMax === false) {
                return validationStatus;
            }
            var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
            if (validDHCPSubnet.status === false) {
                return validDHCPSubnet;
            }

            return validationStatus;
        },

        // Validates that DHCP settings fit subnet and are correct
        validateDHCPSettingsDecimal: function(Address, Netmask, DHCPMin, DHCPMax) {
            var validationStatus = {
                status: true,
                messages: []
            };

            if (DHCPMin === false || DHCPMax === false) {
                validationStatus.status = false;
                validationStatus.messages.push('Invalid IPv4 Address Format');
                return validationStatus;
            }

            // We can proceed only if Address and Netmask are specified
            if (Address === false || Netmask === false) {
                return validationStatus;
            }

            var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
            if (validDHCPSubnet.status === false) {
                return validDHCPSubnet;
            }

            // Check if DHCP Settings range is correct
            if (DHCPMax - DHCPMin < 1) {
                validationStatus.status = false;
                validationStatus.messages.push("Incorrect DHCP range");
            }

            return validationStatus;
        },

        // Check if netmask and address combination is ok for private subnet
        // return bool validation status with message
        validateNetmaskSubnetDecimal: function(Address, Netmask) {
            var validationStatus = {
                    status: true,
                    messages: []
                },
                validNets  = ['0xC0A80000', /*r192*/   '0xAC100000', /*r172*/ '0x0A000000' /*r10*/ ],
                validMasks = ['0xFFFF0000', /*mask16*/ '0xFFF00000', /*mask20*/ '0xFF000000' /*mask24*/],
                usedNetGuestWifi = this.ipStringToDecimal('10.128.0.0'),
                usedNetAPNP = this.ipStringToDecimal('10.129.0.0'),
                usedNetNetmask = this.ipStringToDecimal('255.255.0.0');

            var Subnet = (Address & Netmask) >>> 0;
            var Broadcast = (Address | ~Netmask) >>> 0;

            validNets = _.map(validNets, function(val) { return parseInt(val, 16); });
            validMasks = _.map(validMasks, function(val) { return parseInt(val, 16); });

            var netValid = false;
            
            $.each(validNets, function(i, net) {
                if ((Address & validMasks[i]) >>> 0 === validNets[i]) {
                    netValid = true;
                    
                    // The subnet mask can also be more specific than the default one
                    if ((Netmask & validMasks[i]) >>> 0  !== validMasks[i]) {
                        validationStatus.status = false;
                        validationStatus.messages.push("The subnetmask does not fit to the ip range");
                    }
                }
            });

            if (!netValid) {
                validationStatus.status = false;
                validationStatus.messages.push("Non-private subnet");
            }

            if (Address === Broadcast) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match broadcast");
                return validationStatus;
            }

            if (Address === Subnet) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match subnet");
                return validationStatus;
            }

            // Validate that network doesn't match guest wifi or AP-NP network
            var maxNetmask = (Netmask & usedNetNetmask) >>> 0;
            
            if ((Address & maxNetmask) >>> 0 === usedNetGuestWifi) {
                validationStatus.status = false;
                validationStatus.messages.push("Subnet 10.128.0.0/16 is used for guest wlan");
                return validationStatus;
            }

            if ((Address & maxNetmask) >>> 0 === usedNetAPNP) {
                validationStatus.status = false;
                validationStatus.messages.push("Subnet 10.129.0.0/16 is used for internal purposes");
                return validationStatus;
            }

            return validationStatus;
        },


        // We need to check if DHCP subnet is ok, if one of [Address, Netmask, DHCP] have changed
        validateDHCPSubnetDecimal: function(DHCPMin, DHCPMax, Address, Netmask) {
            var validationStatus = {
                status: true,
                messages: []
            };

            var Subnet = (Address & Netmask) >>> 0;
            var Broadcast = (Address | ~Netmask) >>> 0;

            // Check if DHCP Subnet match to IP Subnet
            if ( (DHCPMin & Netmask) >>> 0 !== Subnet || (DHCPMax & Netmask) >>> 0 !== Subnet ) {
                validationStatus.status = false;
                validationStatus.messages.push("DHCP Subnet doesnt match");
                return validationStatus;
            }

            // Validate if IP Address matches part is out of range:
            if (DHCPMax >= Address && Address >= DHCPMin) {
                validationStatus.status = false;
                validationStatus.messages.push("IPv4 Address Within DHCP Range");

                return validationStatus;
            }

            // Check if DHCP Settings range is correct
            if (DHCPMax === Broadcast || DHCPMin === Subnet) {
                validationStatus.status = false;
                validationStatus.messages.push("Incorrect DHCP range");

                return validationStatus;
            }

            return validationStatus;
        },

        // data is [{parameterName: key, parameterValue: value, parameterClasses: []}] array
        // returns false if text inserted into field is not a valid IP address
        // returns decimal value of IP address if everything is ok
        getIPField: function(data, key, className) {
            var val = _.where(data, {parameterName: key});

            if (className) {
                val = _.filter(val, function(element) {
                    return $.inArray(className, element.parameterClasses) !== -1;
                });
            }

            if (val.length > 0) {
                if ($.inArray('disabled', val[0].parameterClasses) !== -1) {
                    return false;
                }

                var ipString = val[0].parameterValue;
                return this.ipStringToDecimal(ipString);
            }
            
            return false;
        },

        /** Replaces subnet address part:
        *
        *   1) It splits previousAddress into Network (192.168.1.) and Host(.25) parts using Netmask.
        *   2) Replaces Network part with one from Address (10.0.0.). Address Host part is gained using Netmask.
        *   3) Result is constructed from Address network part, and previousAddress Host part
        *
        *   This is used to auto-replace some Static IP / DHCP Range to new subnet, as result of Address/Netmask field change.
        *
        *   Example:  self.replaceSubnet(192.168.1.25, 10.0.0.1, 255.255.255.0)
        *   Result: 10.0.0.25
        */
        replaceSubnet: function(previousAddressStr, AddressStr, NetmaskStr) {
            var Address = this.ipStringToDecimal(AddressStr),
                Netmask = this.ipStringToDecimal(NetmaskStr),
                previousAddress = this.ipStringToDecimal(previousAddressStr);

            if (Address && Netmask && previousAddress) {
                previousAddress = ((previousAddress & ~Netmask) + (Address & Netmask)) >>>0;
                return this.decimalToIpString(previousAddress);
            } else {
                return false;
            }
        },

        // Converts 192.168.1.1 to decimal representation
        ipStringToDecimal: function(ipString) {
            if (!_.isString(ipString) || !ipString.match(/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)) { //Not valid format
                return false;
            }
            
            var ip = ipString.split("."),
                ipNumber = 0,
                invalid = false;
            
            $.each(ip, function(i, v) {
                var octet = parseInt(v, 10);

                if (octet > 255 || octet < 0) { // Check if octet number is not exceeding valid value 
                    invalid = true;
                }
                ipNumber += octet * Math.pow(2, 8*(3-i));
            });

            if (invalid) {
                return false;
            }

            return ipNumber;
        },

        // Converts decimal to string (192.168.1.1) representation
        decimalToIpString: function(ipDecimal) {
            var result = "",
                octet = 0,
                left = 0,
                pow = Math.pow(2,8);

            left = ipDecimal;

            for (var i=0; i<4; i++) {
                octet = left % pow;
                left = Math.floor(left / pow);
                result = "." + octet.toString(10) + result;
            }

            return result.substr(1);
        },

        // Checks if address is withing required subnet
        isAddressWithinSubnet: function(CheckIP, Address, Netmask) {
            if (Address===false || Netmask===false) {
                return false;
            }
            return (Address & Netmask >>> 0) === (CheckIP & Netmask >>> 0);
        },

        /** Calculates allowed host range string for subnet
        *   @param  int   123451231      Ip address in decimal form that is contained in subnet
        *   @param  int   3423525325     Ip netmask in decimal form to identify subnet
        *   @return string               Range string '192.168.1.1 - 192.168.1.126' if Address/Netmask are valid, bool false if not
        */
        getAllowedHostRange: function(Address, Netmask) {
            var MinAddr, MaxAddr;

            if (Address!==false && Netmask!==false) {
                MinAddr = ((Address & Netmask) >>> 0) + 1;
                MaxAddr = ((Address & Netmask >>> 0) - 1 + ~Netmask) >>> 0;

                return this.decimalToIpString(MinAddr) + " - " +this.decimalToIpString(MaxAddr);
            }
            
            return false;
        }
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        this.reset();

        $.when(this.getIPSettings(fromListener), this.getNetworkStatus(fromListener), this.loadDMZSettings(fromListener)).done(function() {
            self.trigger('change');
            deferred.resolve();
        });

        return deferred.promise();
    },

    getIPSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:getLANIP',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                if (response.data) {
                    var data = response.data;

                    // Set model id
                    model.set('id', 'ip-settings');

                    // Set model data
                    model.set('LocalIPAddress', data.Address);
                    model.set('Netmask', data.Netmask);
                    model.set('DHCPStatus', data.DHCPEnable);
                    model.set('DHCPStart', data.DHCPMinAddress);
                    model.set('DHCPFinish', data.DHCPMaxAddress);

                    // Add model to current collection
                    self.add(model);

                    deferred.resolve();
                } else {
                    deferred.resolve();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getNetworkStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:getWANStatus',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    moreInfo = [];

                if (response.data) {
                    var data = response.data;

                    // Set model id
                    model.set('id', 'status');

                    // Set model data
                    model.set('ConnectionType', data.LinkType);
                    model.set('ConnectionState', data.ConnectionState);
                    model.set('ConnectionStatus', data.LinkState !== "down" && data.IPAddress.length !== 0);
                    model.set('MacAddress', data.MACAddress);
                    model.set('DeviceName', 'Internet Box');
                    model.set('Protocol', data.Protocol);
                    model.set('LastConnection', 'none');
                    model.set('RemoteIPAddress', data.IPAddress);
                    model.set('DNSServers', data.DNSServers ? data.DNSServers.split(',') : []);
                    model.set('IPv6Address', data.IPv6Address);
                    model.set('StorageIPAddress', 'unknown');

                    // Add model to current collection
                    self.add(model);

                    moreInfo.push(self.getStorageIP(model, fromListener));
                    if (!model.get('ConnectionStatus')) {
                        moreInfo.push(self.getLastConnection(model, fromListener));
                    }
                    $.when.apply(null, moreInfo).then(function () {
                        deferred.resolve();
                    });

                } else {
                    deferred.resolve();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getLastConnection: function (model, fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/data:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                if (response.status) {
                    var data = response.status,
                        last;
                    if (model.get('ConnectionType') === 'ethernet' ) {
                        last = data.eth;
                    }
                    else {
                        last = data.dsl;
                    }
                    last = _.values(last);
                    if (last.length > 0 && last[0].LastChange) {
                        last = (new Date()).getTime() - last[0].LastChange * 1000;
                        model.set('LastConnection', last);

                    }
                }
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });
        return deferred.promise();
    },

    getStorageIP: function (model, fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/LAN',
            method: "GET",
            fromListener: fromListener,

            success: function(response) {
                if (response.parameters) {
                    var data = response.parameters;
                    _.each(data, function (val) {
                        if (val.name === 'IP') {
                            model.set('StorageIPAddress', val.value);
                        }
                    });
                }
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });
        return deferred.promise();
    },

    saveIPSettings: function(data) {
        var self = this,
            address, netmask, dhcpStatus, dhcpMin, dhcpMax,
            refreshUrl = swc.settings.application.get('url') + '#network/settings/ip-settings',
            addressOld = this.get('ip-settings').get('LocalIPAddress'),
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "Address") {
                address = value.parameterValue;
            }

            if (value.parameterName === "Netmask") {
                netmask = value.parameterValue;
            }

            if (value.parameterName === "DHCPEnable") {
                dhcpStatus = value.parameterValue;
            }

            if (value.parameterName === "DHCPRange") {
                if ($.inArray('range-from', value.parameterClasses) !== -1) {
                    dhcpMin = value.parameterValue;
                } else {
                    dhcpMax = value.parameterValue;
                }
            }
        });

        if (!dhcpMin) { //NP requires DHCPMin and DHCPMax even if DHCPStatus=false
            dhcpMin = self.get('ip-settings').get('DHCPStart');
        }

        if (!dhcpMax) {
            dhcpMax = self.get('ip-settings').get('DHCPFinish');
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:setLANIP',
            timeout: 2000,
            data: {
                parameters: {
                    Address: address,
                    Netmask: netmask,
                    DHCPEnable: dhcpStatus,
                    DHCPMinAddress: dhcpMin,
                    DHCPMaxAddress: dhcpMax
                }
            },

            success: function(response) {
                if (response.errors && response.errors.length > 0) {
                    deferred.reject(response.errors[0].description);
                } else {
                    // in case of success the return "{status: null}"
                    deferred.resolve();
                }
            },

            error: function(xhr, ajaxOptions, thrownError) {
                if (ajaxOptions === "timeout") {
                    if (addressOld !== address) {
                        document.location.href = refreshUrl;
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            }
        });

        return deferred.promise();
    },

    loadDMZSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:getDMZ',
            fromListener: fromListener,
            data: {
                parameters: {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    data = {};

                model.set('id', 'dmz');

                // Check if DMS is empty
                if (!response.status.webui) {
                    model.set('deviceIP', '');
                } else {
                    model.set('deviceIP',
                        response.status.webui.DestinationIPAddress // Bullshit
                    );
                }

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveDMZSettings: function(data) {
        var self = this,
            deferred = new $.Deferred(),
            enabled = false,
            assotiatedDeviceMAC,
            assotiatedDeviceIP = '';

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "DMZEnable") {
                enabled = value.parameterValue;
            }

            if (value.parameterName === "DMZIPAddress") {
                assotiatedDeviceMAC = value.parameterValue;
            }
        });

        // Check for critical situation:
        if (enabled && !assotiatedDeviceMAC) {
            return deferred.resolve();
        }

        // Get device ip address:
        assotiatedDeviceIP = swc.models.NetworkDevices.getDeviceIP(assotiatedDeviceMAC);

        if (!enabled) {
            $.when(this.disableDMZ()).done(function (){
                deferred.resolve();
            });
        } else {
            $.when(this.enableDMZ(assotiatedDeviceIP)).done(function (){
                deferred.resolve();
            });
        }

        return deferred.promise();
    },

    enableDMZ: function(deviceIP) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:setDMZ',
            data: {
                parameters: {
                    destinationIPAddress: deviceIP,
                    enable: true,
                    id: "webui",
                    sourceInterface: "data"
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    disableDMZ: function() {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:deleteDMZ',
            data: {
                parameters: {
                    id: "webui"
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getParameter: function(parameter, storage) {
        var search = this.where({ id: storage });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return false;
        }
    }

});
