swc.constructors.FirewallRules = Backbone.Collection.extend({

    /**
     * List of predefined rules:
     * @param predefined {Array}
     */
    predefined: [
        { "name": "Internet Key Exchange", "port": [ "500" ], "protocol": "17" },
        { "name": "Kerberos", "port": [ "88" ], "protocol": "6,17" },
        { "name": "SUN RPC",  "port": [ "111" ], "protocol": "6" },
        { "name": "Microsoft RPC", "port": [ "135" ], "protocol": "6" },
        { "name": "NETBIOS-SSN", "port": [ "139" ], "protocol": "6" },
        { "name": "Microsoft-DS", "port": [ "445" ], "protocol": "6" },
        { "name": "Remote Login", "port": [ "513" ], "protocol": "6" },
        { "name": "Remote Shell", "port": [ "514" ], "protocol": "6" },
        { "name": "AFP", "port": [ "548" ], "protocol": "6" },
        { "name": "Internet Printing protocol", "port": [ "631" ], "protocol": "6" },
        { "name": "SSDP (Port 9000)", "port": [ "1900" ], "protocol": "17" },
        { "name": "SSDP (Port 2869)", "port": [ "2869" ], "protocol": "6" },
        { "name": "UPnP Discovery", "port": [ "3702" ], "protocol": "17" },
        { "name": "Multicast DNS", "port": [ "5353" ], "protocol": "17" },
        { "name": "LLMNR", "port": [ "5355" ], "protocol": "17" },
        { "name": "SSH", "port": [ "22" ], "protocol": "6" },
        { "name": "Telnet", "port": [ "23" ], "protocol": "6" },
        { "name": "Http", "port": [ "80" ], "protocol": "6" },
        { "name": "Remote Desktop protocol", "port": [ "3389" ], "protocol": "6" },
        { "name": "VNC", "port": [ "5900" ], "protocol": "6" }
    ],

    /**
     * List of validation methods:
     */
    validation: {

        RuleName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    policy: { elementName: 'policy' },
                    name: { elementName: 'RuleName' },
                    id: { elementName: 'RuleKey' }
                }, data),
                predefinedRules = swc.models.FirewallRules.predefined,
                validationStatus = true,
                validationMessages = [],
                searchModel;


            // Skip validation for predefined mode
            if (validationData.mode !== "predefined") {

                if (!$.trim(validationData.name)) {
                    validationMessages.push('must not be empty');
                }

                if (validationData.name.length < 1 || validationData.name.length > 40) {
                    validationMessages.push('must be from 1 to 40 characters');
                }

                // Ascii is defined as the characters in the range of 000-177 (octal),
                // therefore non-printing characters \000-\037
                if (/[\000-\037]/.test(validationData.name)) {
                    validationMessages.push('allowed characters ASCII non-printable');
                }

                // Search for same rules only within one chain target (inbound / outbound)
                if (validationData.policy.split('_')[1] === 'inbound') {
                    searchModel = swc.models.FirewallRules.
                        findWhere({ name: validationData.name, chain: 'Custom_V6In' });
                } else {
                    searchModel = swc.models.FirewallRules.
                        findWhere({ name: validationData.name, chain: 'Custom_V6Out' });
                }

                // Check for custom rules:
                if (!_.isUndefined(searchModel)) {

                    // Skip check if model is editing
                    if (searchModel.get('id') !== validationData.id) {
                        validationMessages.push('already existed rule names are not allowed');
                    }
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        PredefinedRuleName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'PredefinedRuleName' },
                    policy: { elementName: 'policy' }
                }, data),
                predefinedRules = swc.models.FirewallRules.predefined,
                validationStatus = true,
                validationMessages = [],
                searchModel;

            // Skip validation for predefined mode
            if (validationData.mode !== "custom") {

                // Search for same rules only within one chain target (inbound / outbound)
                if (validationData.policy.split('_')[1] === 'inbound') {
                    searchModel = swc.models.FirewallRules.
                        findWhere({name: validationData.name, chain: 'Custom_V6In'});
                } else {
                    searchModel = swc.models.FirewallRules.
                        findWhere({name: validationData.name, chain: 'Custom_V6Out'});
                }

                // Check if rule matched:
                if (!_.isUndefined(searchModel)) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        RulePort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "ports-mode" },
                    from: { elementName: "RulePort", elementClass: "from" },
                    to: { elementName: "RulePort", elementClass: "to" },
                    id: { elementName: 'RuleKey' },
                    policy: { elementName: 'policy' }
                }, data),
                validationMessages = [],
                searchModels = [],
                portsRange = [];

            if (!$.trim(validationData.from)) {
                validationMessages.push('must not be empty');
            }

            if (/\D/.test(validationData.from)) {
                validationMessages.push('allowed characters 0-9');
            }

            // Convert to integer values:
            validationData.from = +validationData.from;
            validationData.to = +validationData.to;

            if (validationData.from < 1 || validationData.from > 65535) {
                validationMessages.push('allowed range 1 to 65535');
            }

            if (validationData.mode === "range") {

                if (!$.trim(validationData.to)) {
                    validationMessages.push('must not be empty');
                }

                if (/\D/.test(validationData.to)) {
                    validationMessages.push('allowed characters 0-9');
                }

                if (validationData.to < 1 || validationData.to > 65535) {
                    validationMessages.push('allowed range 1 to 65535');
                }

                if (validationData.to <= validationData.from) {
                    validationMessages.push('range option first value MUST be less then second value');
                }

                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }

            // Check for TR-069 blocked ports
            if (!swc.models.FirewallRules.validation.helpers.portsNotInRange(portsRange, [ 1024, 1048 ])) {
                validationMessages.push('ports 1024-1048 are not allowed because this ports reserved for system use');
            }

            // Search for same rules only within one chain target (inbound / outbound)
            if (validationData.policy.split('_')[1] === 'inbound') {
                searchModels = swc.models.FirewallRules.where({ chain: 'Custom_V6In' });
            } else {
                searchModels = swc.models.FirewallRules.where({ chain: 'Custom_V6Out' });
            }

            // Check ports crossing conflict for custom rules:
            _.each(searchModels, function(rule, key) {
                var rulePorts = rule.get('port'),
                    rulePortsNotCrossing = swc.models.FirewallRules.validation.helpers.
                        portsNotInRange(portsRange, rulePorts);

                if (!rulePortsNotCrossing && (rule.get('id') !== validationData.id)) {
                    validationMessages.push('already used ports are not allowed');
                }
            });

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        helpers: {
            /**
             * Check for ports not conflicting
             *
             * @param ports [1-2]
             * @param range [1-2]
             *
             * @returns {boolean}
             */
            portsNotInRange: function(ports, range) {
                // Case 1 [] && []
                if (ports.length === 1 && range.length === 1) {
                    if (parseInt(ports[0], 10) !== parseInt(range[0], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 2 [][] && []
                if (ports.length === 2 && range.length === 1) {
                    if (parseInt(range[0], 10) < parseInt(ports[0], 10) ||
                        parseInt(range[0], 10) > parseInt(ports[1], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 3 [] && [][]
                if (ports.length === 1 && range.length === 2) {
                    if (parseInt(ports[0], 10) < parseInt(range[0], 10) ||
                        parseInt(ports[0], 10) > parseInt(range[1], 10)) {
                        return true;
                    } else {
                        return false;
                    }
                }

                // Case 4 [][] && [][]
                if (ports.length === 2 && range.length === 2) {
                    if (
                        (parseInt(ports[0], 10) < parseInt(range[0], 10) &&
                            parseInt(ports[1], 10) < parseInt(range[0], 10)) ||
                            (parseInt(range[0], 10) < parseInt(ports[0], 10) &&
                                parseInt(range[1], 10) < parseInt(ports[0], 10))
                        ) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        }
    },

    /**
     * All rules sorted by port value ASC.
     *
     * @param rule1 {Object -> Backbone.Model}
     * @param rule2 {Object -> Backbone.Model}
     *
     * @returns {number}
     */
    comparator: function(rule1, rule2) {
        return parseInt(rule1.get("port"), 10) - parseInt(rule2.get("port"), 10);
    },

    /**
     * Get all rules from the server and compare them to the collection
     * @param fromListener {Boolean}
     * @returns {*}
     */
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        $.when(this.fetchInbound(fromListener), this.fetchOutbound(fromListener))
            .done(function(inbound, outbound) {
                self.parseRules(inbound.status, outbound.status);

                deferred.resolve();
            });

        return deferred.promise();
    },

    /**
     * Request inbound rules:
     * @param fromListener {Boolean}
     * @returns {*}
     */
    fetchInbound: function(fromListener) {
        return this.requestRules({ parameters: { chain: "Custom_V6In" }}, fromListener);
    },

    /**
     * Request outbound rules:
     * @param fromListener {Boolean}
     * @returns {*}
     */
    fetchOutbound: function(fromListener) {
        return this.requestRules({ parameters: { chain: "Custom_V6Out" }}, fromListener);
    },

    /**
     * Send request to the server to get list of rules
     * @param params {Object}
     * @param fromListener {Boolean}
     * @returns {*}
     */
    requestRules: function(params, fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:getCustomRule',
            data: params,
            fromListener: fromListener,

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Create normalized JSON basing on server response
     * @param inbound {Array of Objects}
     * @param outbound {Array of Objects}
     */
    parseRules: function(inbound, outbound) {
        var parsedRules = [];

        // Parse inbound rules:
        _.each(inbound, function(rule, key) {
            parsedRules.push({
                id: key,
                status: rule.Enable,
                name: rule.Description,
                port: rule.DestinationPort.split('-'),
                policy: rule.Target + '_inbound',
                protocol: rule.Protocol,
                chain: 'Custom_V6In'
            });
        });

        // Parse outbound rules:
        _.each(outbound, function(rule, key) {
            parsedRules.push({
                id: key,
                status: rule.Enable,
                name: rule.Description,
                port: rule.DestinationPort.split('-'),
                policy: rule.Target + '_outbound',
                protocol: rule.Protocol,
                chain: 'Custom_V6Out'
            });
        });

        // Update collection data:
        this._reset();
        this.set(parsedRules);
    },

    /**
     * Save changed rules (edit / delete) to the server
     * @param rulesToChangeState {Array of objects}
     * @param rulesToDelete {Array}
     * @returns {*}
     */
    updateRulesList: function(rulesToChangeState, rulesToDelete) {
        var self = this,
            deferred = new $.Deferred(),
            deferredActions = [];

        // Prepare list of actions for changed rules:
        _.each(rulesToChangeState, function(rule, key) {
            var ruleModel = self.findWhere({ id: key }),
                ruleModelNew = {};

            if (!_.isUndefined(ruleModel)) {
                deferredActions.push(
                    self.changeRule(ruleModel.toJSON(), _.extend(ruleModel.toJSON(), { status: rule.value }))
                );
            }

        });

        // Prepare list of actions for deleted rules:
        _.each(rulesToDelete, function(rule, key) {
            var ruleModel = self.findWhere({ id: rule });

            if (!_.isUndefined(ruleModel)) {
                deferredActions.push(self.removeRule(ruleModel.toJSON()));
            }
        });

        $.when.apply(null, deferredActions)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Send request to create rule on the server:
     * @param ruleData {Object}
     * @returns {*}
     */
    createRule: function(ruleData) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:setCustomRule',
            data: {
                parameters: {
                    id: swc.Utils.generateId(),
                    description: ruleData.name,
                    enable: ruleData.status,
                    protocol: ruleData.protocol,
                    ipversion: 6,
                    destinationPort: ruleData.port.join('-'),
                    persistent: true,
                    action: ruleData.policy.split('_')[0],
                    chain: ruleData.chain
                }
            },

            success: function(response) {
                swc.models.Rest.sendRequest({
                    url: '/sysbus/Firewall:commit',
                    data: {
                        parameters: {}
                    },

                    success: function(response) {
                        deferred.resolve();
                    }
                });
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Change rule on the server (1. delete, 2. create new)
     * @param previousRule {Object}
     * @param newRule {Object}
     * @returns {*}
     */
    changeRule: function(previousRule, newRule) {
        var self = this,
            deferred = new $.Deferred();

        $.when(this.removeRule(previousRule)).done(function() {
            $.when(self.createRule(newRule)).done(function() {
                deferred.resolve();
            });
        });

        return deferred.promise();
    },

    /**
     * Delete rule from the server:
     * @param ruleData {Object}
     * @returns {*}
     */
    removeRule: function(ruleData) {
        var deferred = new $.Deferred(),
            self = this;

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:deleteCustomRule',
            data: {
                parameters: {
                    id: ruleData.id,
                    chain: ruleData.chain
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }

});
