module.exports = function(grunt) {

    // Project configuration.
    grunt.initConfig({
        
        // NOTICE:
        // This value should be updated manually on start of each sprint.
        // It is a Jira, PM related feature, no more :-)
        // Moreover, looks like value will be changed by DevOps engineer by use of post-build script
        sprint: 17,

        pkg: grunt.file.readJSON('package.json'),

        libFiles: [
            "static/vendor/javascript/jquery/jquery.js",
            "static/vendor/javascript/jquery/jquery.cookie.js",
            "static/vendor/javascript/jquery/jquery.tmpl.js",
            "static/vendor/javascript/jquery/jquery.placeholder.js",

            "static/vendor/javascript/third-party/json2.js",

            "static/vendor/javascript/backbone/underscore.js",
            "static/vendor/javascript/backbone/backbone.js",

            "static/vendor/javascript/bootstrap/modals.js",
            "static/vendor/javascript/bootstrap/popovers.js",

            "static/vendor/javascript/moments/moment.min.js",
            "static/vendor/javascript/moments/langs-selected.min.js"
        ],

        appFiles: [
            "js/core.js",

            "static/vendor/javascript/application/**/*.js",

            "js/core/1.0/**/*.js",
            "js/core/2.0/**/*.js",

            "js/modules/**/*.js",

            "js/application/**/*.js"
        ],

        clean: {
            "pre-build": ["build/", "junit-report/"],
            "post-build": ["build/static/translations/locales.xlsx", "build/static/vendor"]
        },

        concat: {
            javascript: {
                options: {
                    separator: ';',
                    stripBanners: true,
                    banner: '\n/*! <%= pkg.name %> - v<%= pkg.version %> - ' +
                        '<%= grunt.template.today("yyyy-mm-dd hh-mm-ss") %> */\n\n'
                },

                files: {
                    "build/application/lib.min.js": [ "<%= libFiles %>" ],

                    "build/application/app.min.js": [ "<%= appFiles %>" ]
                }
            },

            templates: {
                src: [ "js/**/*.html" ],
                dest: "build/application/tpl.min.html"
            }
        },

        jshint: {
            options: {

                /**
                 * http://www.jshint.com/docs/options/#curly
                 *
                 * This option requires you to always put curly braces around blocks in loops and conditionals.
                 * JavaScript allows you to omit curly braces when the block consists of only one statement,
                 * for example:
                 *
                 *      while (day)
                 *          shuffle();
                 *
                 * However, in some circumstances, it can lead to bugs (you'd think that sleep() is a part of the
                 * loop while in reality it is not):
                 *
                 *      while (day)
                 *          shuffle();
                 *          sleep();
                 *
                 */
                curly: true,

                /**
                 * http://www.jshint.com/docs/options/#eqeqeq
                 *
                 * This options prohibits the use of == and != in favor of === and !==. The former try to coerce values
                 * before comparing them which can lead to some unexpected results. The latter don't do any coercion so
                 * they are generally safer. If you would like to learn more about type coercion in JavaScript,
                 * we recommend Truth, Equality and JavaScript by Angus Croll.
                 *
                 */
                eqeqeq: true,

                /**
                 * http://www.jshint.com/docs/options/#boss
                 *
                 * This option suppresses warnings about the use of assignments in cases where comparisons are expected.
                 * More often than not, code like if (a = 10) {} is a typo. However, it can be useful in cases like
                 * this one:
                 *
                 *      for (var i = 0, person; person = people[i]; i++) {}
                 *
                 * You can silence this error on a per-use basis by surrounding the assignment with parenthesis, 
                 * such as:
                 *
                 *      for (var i = 0, person; (person = people[i]); i++) {}
                 *
                 */
                boss: true,

                /**
                 * http://www.jshint.com/docs/options/#evil
                 *
                 * This option suppresses warnings about the use of eval. The use of eval is discouraged because it can
                 * make your code vulnerable to various injection attacks and it makes it hard for JavaScript
                 * interpreter to do certain optimizations.
                 *
                 */
                evil: true,

                /**
                 * http://www.jshint.com/docs/options/#lastsemic
                 *
                 * This option suppresses warnings about missing semicolons, but only when the semicolon is omitted for
                 * the last statement in a one-line block:
                 *
                 *      var name = (function() { return 'Anton' }());
                 *
                 * This is a very niche use case that is useful only when you use automatic JavaScript code generators.
                 *
                 */
                lastsemic: true,

                /**
                 * http://www.jshint.com/docs/options/#loopfunc
                 *
                 * This option suppresses warnings about functions inside of loops. Defining functions inside of loops
                 * can lead to bugs such as this one:
                 *
                 */
                loopfunc: true,

                /**
                 * http://www.jshint.com/docs/options/#passfail
                 *
                 * This option makes JSHint stop on the first error or warning.
                 *
                 */
                // TODO passfail: true,

                /**
                 * http://www.jshint.com/docs/options/#trailing
                 *
                 * This option makes it an error to leave a trailing whitespace in your code. Trailing whitespaces can
                 * be source of nasty bugs with multi-line strings in JavaScript:
                 *
                 */
                trailing: true,

                /**
                 * http://www.jshint.com/docs/options/#immed
                 *
                 * This option prohibits the use of immediate function invocations without wrapping them in parentheses.
                 * Wrapping parentheses assists readers of your code in understanding that the expression is the result
                 * of a function, and not the function itself.
                 *
                 */
                immed: true,

                /**
                 * http://www.jshint.com/docs/options/#latedef
                 *
                 * This option prohibits the use of a variable before it was defined. JavaScript has function scope only
                 * and, in addition to that, all variables are always moved—or hoisted— to the top of the function. This
                 * behavior can lead to some very nasty bugs and that's why it is safer to always use variable only 
                 * after they have been explicitly defined.
                 *
                 * Setting this option to "nofunc" will allow function declarations to be ignored.
                 * For more in-depth understanding of scoping and hoisting in JavaScript, read JavaScript Scoping and
                 * Hoisting by Ben Cherry.
                 *
                 */
                latedef: true,

                /**
                 * http://www.jshint.com/docs/options/#newcap
                 *
                 * This option requires you to capitalize names of constructor functions. Capitalizing functions that
                 * are intended to be used with new operator is just a convention that helps programmers to visually
                 * distinguish constructor functions from other types of functions to help spot mistakes when using this.
                 *
                 * Not doing so won't break your code in any browsers or environments but it will be a bit harder to
                 * figure out—by reading the code—if the function was supposed to be used with or without new. And this
                 * is important because when the function that was intended to be used with new is used without it, this
                 * will point to the global object instead of a new object.
                 *
                 */
                newcap: true,

                /**
                 * http://www.jshint.com/docs/options/#noarg
                 *
                 * This option prohibits the use of arguments.caller and arguments.callee. Both .caller and .callee make
                 * quite a few optimizations impossible so they were deprecated in future versions of JavaScript. In
                 * fact, ECMAScript 5 forbids the use of arguments.callee in strict mode.
                 *
                 */
                noarg: true,

                /**
                 * http://www.jshint.com/docs/options/#sub
                 *
                 * This option suppresses warnings about using [] notation when it can be expressed in dot notation:
                 *
                 *      person['name'] vs. person.name.
                 *
                 */
                sub: true,

                /**
                 * http://www.jshint.com/docs/options/#undef
                 *
                 * This option prohibits the use of explicitly undeclared variables. This option is very useful for
                 * spotting leaking and mistyped variables.
                 *
                 * If your variable is defined in another file, 
                 * you can use {'global'} directive to tell JSHint about it.
                 *
                 */
                undef: true,

                /**
                 * http://www.jshint.com/docs/options/#eqnull
                 *
                 * This option suppresses warnings about == null comparisons. Such comparisons are often useful when
                 * you want to check if a variable is null or undefined.
                 *
                 */
                eqnull: true,

                /**
                 * http://www.jshint.com/docs/options/#browser
                 *
                 * This option defines globals exposed by modern browsers: all the way from good old document and
                 * navigator to the HTML5 FileReader and other new developments in the browser world.
                 *
                 * Note: This option doesn't expose variables like alert or console. 
                 * See option devel for more information.
                 *
                 */
                browser: true,

                /**
                 * http://www.jshint.com/docs/options/#es3
                 * 
                 * This option tells JSHint that your code needs to adhere to ECMAScript 3 specification. 
                 * Use this option if you need your program to be executable in older browsers—such as 
                 * Internet Explorer 6/7/8/9—and other legacy JavaScript environments.
                 */
                es3: true,

                globals: {
                    jQuery: false,
                    $: false,
                    _: false,
                    Backbone: false,
                    moment: false,
                    head: false,
                    module: false,
                    console: false,
                    alert: false,
                    swc: true,
                    SWCElements: true,

                    // FIXME: Move all such globals into swc.Utils namespace or convenient Model/View class
                    getParameter: false,
                    getObjectLength: false,
                    getLinerLength: false,
                    getCharacterByKey: false,
                    getDaySeconds: false,
                    getTranslationStrings: false,
                    getTranslationStringsjQuery: false,
                    getDeviceID: false,
                    getApplicationLocale: false,
                    detectMobileBrowser: false
                }
            },

            files: [
                "js/**/*.js"
            ]
        },

        less: {
            development: {
                files: {
                    "build/application/app.min.css": [ "styles/styles.less" ]
                }
            },

            production: {
                options: {
                    yuicompress: true
                },

                files: {
                    "build/application/app.min.css": [ "styles/styles.less" ]
                }
            }
        },

        uglify: {
            options: {
                compress: {
                    global_defs: {
                        "DEBUG": true
                    }
                },

                banner: '/*! <%= pkg.name %>-<%= pkg.version %> <%= grunt.template.today("yyyy-mm-dd") %> */\n'
            },

            testing: {
                options: {
                    compress: {
                        global_defs: {
                            "DEBUG": false // clean-up debug code
                        }
                    }
                },

                files: {
                    'build/application/app.min.js': [ "<%= appFiles %>" ],
                    'build/application/lib.min.js': [ "<%= libFiles %>" ]
                }
            },

            production: {
                options: {
                    compress: {
                        global_defs: {
                            "DEBUG": false // clean-up debug code
                        },
                        dead_code: true
                    }
                },

                files: {
                    'build/application/app.min.js': [ "<%= appFiles %>" ],
                    'build/application/lib.min.js': [ "<%= libFiles %>" ]
                }
            }
        },

        copy: {
            main: {
                files: [
                    { src: "index.html", dest: "build/index.html" },
                    { src: "captivePortal.html", dest: "build/captivePortal.html" },
                    { src: "oauthCallback.html", dest: "build/oauthCallback.html" },
                    { src: "homeAuthentificationRemote.html", dest: "build/homeAuthentificationRemote.html" },
                    { src: "static/**/*", dest: "build/" },
                    { src: "styles/browsers/ie8.css", dest: "build/application/ie8.css" },
                    { src: "static/vendor/javascript/jquery/ie8.fix.js", dest: "build/application/ie8.fix.js" }
                ]
            }
        },

        jasmine: {
            unit: {
                src: [
                    "<%= libFiles %>",
                    "<%= appFiles %>"
                ],
                options: {
                    host : 'http://127.0.0.1:8000/',

                    vendor: [
                        "<%= libFiles %>"
                    ],

                    specs: 'tests/unit/spec/*Spec.js',
                    helpers: 'tests/unit/spec/*Helper.js',

                    junit: {
                        consolidate: true,
                        path: "junit-report"
                    }
                }
            }
        },

        // for unit tests
        "karma": {
            options: {
                runnerPort: 9999,
                browsers: ['Chrome'], // FIXME: Find out why tests are return ERROR in Firefox
                autoWatch: false,
//                frameworks: ['jasmine'],
                frameworks: ['mocha', 'sinon-chai', 'chai-as-promised' ], // 'expect'], - 'expect' - to test in IE8
                reporters: ['progress', 'junit'],

                junitReporter: {
                    // will be resolved to basePath (in the same way as files/exclude patterns)
                    outputFile: 'junit-report/test-results.xml'
                },

                client: {
                    mocha: {
                        ui: 'bdd'
                    }
                },
                
//                logLevel: "LOG_WARN",

                files: [
                    // Application vendors files
                    "static/vendor/javascript/jquery/jquery.js",
                    "static/vendor/javascript/jquery/jquery.cookie.js",
                    "static/vendor/javascript/jquery/jquery.tmpl.js",
                    "static/vendor/javascript/jquery/jquery.placeholder.js",

                    "static/vendor/javascript/third-party/json2.js",

                    "static/vendor/javascript/backbone/underscore.js",
                    "static/vendor/javascript/backbone/backbone.js",

                    "static/vendor/javascript/bootstrap/modals.js",
                    "static/vendor/javascript/bootstrap/popovers.js",

                    "static/vendor/javascript/moments/moment.min.js",
                    "static/vendor/javascript/moments/langs-selected.min.js",

                    // fixtures + matchers for jasmine
//                    "static/vendor/javascript/third-party/jasmine-jquery.js",
                    "static/vendor/javascript/third-party/fixtures.js",
                    // settings for fixtures lib
                    "static/vendor/javascript/third-party/fixtures-config.js",

                    // Application core files
                    "js/core.js",

                    "static/vendor/javascript/application/**/*.js",

                    "js/core/1.0/**/*.js",
                    "js/core/2.0/**/*.js",

                    "js/modules/**/*.js",

                    "js/application/**/*.js",

                    // simple patterns to load the needed testfiles 'test/fixtures/**/*.html',
                    'tests/unit/spec/**/*Spec.js',
                    // JSON fixtures for unit tests
                    {
                        pattern: 'tests/fixtures/json/**/*.json',
                        included: false,
                        served: true,
                        watched: false
                    }
                ]
            },

            continuous: {
                options: {
                    singleRun: true,
                    browsers: [ 'PhantomJS' ]
                }
            },

            development: {
                options: {
                    runnerPort: 9999,
                    singleRun: true,
                    browsers: [ 'PhantomJS' ]
                }
            }
        }
    });

    // This task needed to embed concatenated templates into index.html file
    grunt.registerTask('postBuildProcess', function() {
        var pkg = grunt.file.readJSON('package.json');

        var indexFileContent = grunt.file.read('build/index.html');
        var stickyPagesFileContent = grunt.file.read('build/captivePortal.html');
        var cssFileContent = grunt.file.read('build/application/app.min.css');
        var settings = grunt.file.readJSON('build/static/settings/application.json');

        // Update production settings
        settings.build.build = grunt.option("build.number");
        settings.build.version = grunt.option("build.version") || pkg.version;
        settings.build.created = grunt.template.today("yyyy-mm-dd HH:mm");
        settings.build.sprint = grunt.config.get("sprint");

        // Update application version in Index.html
        grunt.file.write('build/index.html', indexFileContent.replace(/APP_VERSION/g, settings.build.version));

        // Update application version in captivePortal.html
        grunt.file.write('build/captivePortal.html', stickyPagesFileContent.replace(/APP_VERSION/g, settings.build.version));
        
        // Update application version in concatenated css file
        grunt.file.write('build/application/app.min.css', cssFileContent.replace(/APP_VERSION/g, settings.build.version));

        grunt.file.write('build/static/settings/application.json', JSON.stringify(settings));
    });

    // Load plugins that provide configured task.
    grunt.loadNpmTasks('grunt-contrib-copy');
    grunt.loadNpmTasks('grunt-contrib-jshint');
    grunt.loadNpmTasks('grunt-contrib-handlebars');
    grunt.loadNpmTasks('grunt-contrib-less');
    grunt.loadNpmTasks('grunt-contrib-concat');
    grunt.loadNpmTasks('grunt-contrib-uglify');
    grunt.loadNpmTasks('grunt-contrib-clean');
    grunt.loadNpmTasks('grunt-contrib-jasmine');
    grunt.loadNpmTasks('grunt-contrib-watch');
    grunt.loadNpmTasks('grunt-karma');
    
    grunt.registerTask('pre-commit', ['jshint', 'karma:development']);

    grunt.registerTask('build-dev', [
        'clean:pre-build', 'jshint', 'less:development', 'concat', 'copy', 'postBuildProcess', 'clean:post-build'
    ]);
    
    // So far TEST build consists of the same tasks as production
    grunt.registerTask('build-test', [
        'clean:pre-build', 'jshint', 'less:production', 'concat', 'uglify:testing', 'karma:continuous',
        'copy', 'postBuildProcess', 'clean:post-build'
    ]);

    grunt.registerTask('build-prod', [
        'clean:pre-build', 'jshint', 'less:production', 'concat', 'uglify:production', 'karma:continuous',
        'copy', 'postBuildProcess', 'clean:post-build'
    ]);

};
