/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.property;

import javax.ide.spi.ProviderNotFoundException;
import javax.ide.Service;

/**
 * The <code>PropertyPageRegistry</code> is the registry of information 
 * describing what extension specific property pages have been added to an IDE.
 * <p>
 * Extensions register property page information in the extension 
 * manifest using the property-pages-hook tag. The property pages hook 
 * allows extension to declare property pages for the IDE and project 
 * preferences. Extension use the property-page object-class tag to specify
 * the object class for which the pages are declared. For example, if the 
 * pages are for a project, extension should specify javax.ide.model.Project as 
 * the object class. If the pages are for the IDE, extension should specify
 * javax.ide.IDE as the object class. Pages registered for the project and 
 * IDE are shown when the project and IDE preference dialogs are displayed.
 * <p>
 * Extensions can also use this registry  * to associate pages with any object 
 * class. Doing so makes them responsiblefor providing the graphical user 
 * interface to display these pages.
 * <p>
 * Propety pages are associated with an object class. In general, extension 
 * writers can define pages for the IDE and/or a project.
 */
public abstract class PropertyPageRegistry extends Service
{
  /**
   * Constant identifying the project class name.
   */
  public static final String PROJECT_CLASS_NAME = "javax.ide.model.Project";
  
  /**
   * Constant identifying the IDE class name.
   */
  public static final String IDE_CLASS_NAME = "javax.ide.IDE";
  
  /**
   * Get the pages associated with the specified <code>objectClassName</code>. 
   *
   * @param objectClassName The object class whose properties will be edited by
   * the returned property pages.
   *
   * @return The property pages associated with the specified 
   * <code>objectClassName</code>.
   */
  public abstract PropertyPage[] getPropertyPages( String objectClassName );
  
  /**
   * Get the project property pages. This method just calls the 
   * {@link #getPropertyPages} method passing as argument 
   * {@link #PROJECT_CLASS_NAME}.
   * 
   * @return An array of property pages associated with a project. Must not 
   *    return null, may return an empty array.
   */
  public final PropertyPage[] getProjectPropertyPages()
  {
    return getPropertyPages( PROJECT_CLASS_NAME );
  }
  
  /**
   * Get the IDE property pages. This method just calls the 
   * {@link #getPropertyPages} method passing as argument {@link #IDE_CLASS_NAME}.
   * 
   * @return An array of property pages associated with the IDE.
   */  
  public final PropertyPage[] getIDEPropertyPages()
  {
    return getPropertyPages( IDE_CLASS_NAME );
  }
  
  /**
   * Get the property page registry implementation for this IDE.
   * 
   * @return the property page registry implementation registered with this
   *    IDE.
   */
  public static PropertyPageRegistry getPropertyPageRegistry()
  {
    try
    {
      return (PropertyPageRegistry) getService( PropertyPageRegistry.class );
    }
    catch ( ProviderNotFoundException lnfe )
    {
      lnfe.printStackTrace();
      throw new IllegalStateException( "no property page registry" );
    }
  }
}
