/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.property;

import java.util.prefs.Preferences;
import javax.ide.command.Context;
import javax.ide.view.GUIPanel;

/**
 * The <code>PropertyPage</code> interface lets client introduce new property 
 * pages for setting IDE and project preferences.
 */
public abstract class PropertyPage
{
  /**
   * Get the root graphical user interface component. This component 
   * will be hosted by the IDE.
   * 
   * @return the gui for this page. Must not be null.
   */
  public abstract GUIPanel  getGUI();

  /**
   * Method called when this page is selected by the user. Implementors 
   * are responsible for initializing the GUI with the data provided by 
   * the <code>preferences</code> object.
   *
   * @param preferences The data used to initialize the GUI.
   *
   * @param context The current {@link Context}.
   */
  public void onEntry( Preferences preferences, Context context )
  {
    
  }

  /**
   * Method called when users leave this page for another page
   * and when they press the dialog's OK button.  Implementors 
   * are responsible for getting the information entered by the 
   * user and updating the <code>preferences</code> object.<p>
   * 
   * It's the responsibility of an implementing IDE to take care of ensuring
   * that preference changes are only persisted if the user confirms the
   * changes (e.g. by clicking the OK button vs. the Cancel button in a 
   * dialog).
   *
   * @param preferences The data used to initialize the GUI.
   * @param context The current {@link Context}.
   *
   * @exception InvalidPropertyException This exception is thrown when the 
   * user tries to leave the page and the information entered does not 
   * validate.
   */
  public void onExit( Preferences preferences, 
                      Context context ) throws InvalidPropertyException
  {
    
  }
  
  /**
   * Get the property keys this property page uses in the preferences object.
   * This method is provided so that IDEs can potentially perform optimizations
   * (e.g. avoid modifying a physical project file until real changes are made)
   * 
   * @return the property keys used by this page.
   */
  public String[] getKeys()
  {
    return new String[ 0 ];
  }
}
