/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.model.spi;

import java.util.ArrayList;
import java.util.Collection;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.ide.model.Element;
import javax.ide.spi.LookupProvider;
import javax.ide.spi.ProviderNotFoundException;

/**
 * A model apapter factory is responsible for creating the best possible 
 * adapter for a given element. The adapter is responsible for managing the 
 * communication between JSR model objects and IDE specific model objects.
 */
public abstract class ModelAdapterFactory
{
  private static ModelAdapterFactory _impl;

  /**
   * Creates an instance of an adapter that will delegate requests to IDE 
   * specific model objects.
   * 
   * @param element the element that the newly created adapter adapts.
   * @return an adapter instance.
   */
  public abstract ElementImpl getImpl( Element element );

  /**
   * Get the ModelAdpaterFactory implementation for this IDE.
   * 
   * @return the DocumentFactory implementation for this IDE.
   */
  public static ModelAdapterFactory getModelAdapterFactory()
  {
    try
    {
      if ( _impl == null )
      {
        Collection impls = LookupProvider.lookupAll(
          Thread.currentThread().getContextClassLoader(), 
          ModelAdapterFactory.class
        );
        if ( impls.size() == 1 )
        {
          _impl = (ModelAdapterFactory) impls.iterator().next();
        }
        else
        {
          List copy = new ArrayList( impls );
          Collections.reverse( copy );
          _impl = new ChainedFactories( copy );
        }
      }
    }
    catch ( ProviderNotFoundException pnfe )
    {
      throw new IllegalStateException( "No model adapter factory." );
    }
    
    return _impl;
  }
  
  
  private static final class ChainedFactories extends ModelAdapterFactory
  {
    private final List _factories;
    
    ChainedFactories( List factories )
    {
      _factories = factories;
    }
  
    public ElementImpl getImpl( Element element )
    {
      for ( Iterator i = _factories.iterator(); i.hasNext(); )
      {
        ModelAdapterFactory f = (ModelAdapterFactory) i.next();
        ElementImpl result = f.getImpl( element );
        if ( result != null )
        {
          return result;
        }
      }
      return null;
    }
  }
  
}
