/*
 * @(#)LiteralExpressionT.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * An expression representing a lexer literal. <p/>
 *
 * Class literals and this literals, though called "literal", are not
 * here considered to be literal expressions. As a result of this
 * definition, the value of all LiteralExpressionTs can be determined
 * at parse-time whereas the value of class literals and this literals
 * can only be determined at compile-time.
 *
 * @author Andy Yu
 * */
public interface LiteralExpressionT
  extends ExpressionT
{
  // ----------------------------------------------------------------------

  /**
   * @return The kind of literal this is. Valid values are LiteralKind.*.
   */
  public LiteralKind getLiteralKind();

  /**
   * In this version, this class is 1.4 compatible. In a later version,
   * it will be redone as an enum. <p/>
   */
  public static final class LiteralKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int VALUE_INT = 0;
    private static final int VALUE_LONG = 1;
    private static final int VALUE_FLOAT = 2;
    private static final int VALUE_DOUBLE = 3;
    private static final int VALUE_BOOLEAN = 4;
    private static final int VALUE_CHAR = 5;
    private static final int VALUE_STRING = 6;
    private static final int VALUE_NULL = 7;

    /** Integer literal. */
    public static final LiteralKind INT =
      new LiteralKind( VALUE_INT, "INT" );

    /** Long literal. */
    public static final LiteralKind LONG =
      new LiteralKind( VALUE_LONG, "LONG" );

    /** Float literal. */
    public static final LiteralKind FLOAT =
      new LiteralKind( VALUE_FLOAT, "FLOAT" );

    /** Double literal. */
    public static final LiteralKind DOUBLE =
      new LiteralKind( VALUE_DOUBLE, "DOUBLE" );

    /** Boolean literal. */
    public static final LiteralKind BOOLEAN =
      new LiteralKind( VALUE_BOOLEAN, "BOOLEAN" );

    /** Char literal. */
    public static final LiteralKind CHAR =
      new LiteralKind( VALUE_CHAR, "CHAR" );

    /** String literal. */
    public static final LiteralKind STRING =
      new LiteralKind( VALUE_STRING, "STRING" );

    /** Null literal. */
    public static final LiteralKind NULL =
      new LiteralKind( VALUE_NULL, "NULL" );


    // ----------------------------------------------------------------------

    private final int ordinal;
    
    private final String name;

    private LiteralKind(int ordinal, String name)
    {
      this.ordinal = ordinal;
      this.name = name;
      
      values.put( name, this );
    }

    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(LiteralKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof LiteralKind)
      {
        final LiteralKind tk = (LiteralKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return LiteralKind.class;
    }


    // ----------------------------------------------------------------------

    public static LiteralKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static LiteralKind valueOf(Class ignored, String name)
    {
      return (LiteralKind) values.get(name);
    }

    public static LiteralKind[] values()
    {
      final Collection entries = values.values();
      return (LiteralKind[])entries.toArray(new LiteralKind[entries.size()]);
    }


    // ----------------------------------------------------------------------
  }
}
