/*
 * @(#)ForStatementT.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * A for statement. JLS3 14.14. <p/>
 *
 * @author Andy Yu
 * */
public interface ForStatementT
  extends CompoundStatementT
{
  // ----------------------------------------------------------------------

  /**
   * @return Valid values are ForKind.*.
   */
  public ForKind getForKind();

  /**
   * In a standard for loop with variable declarations, this is the
   * local variable declaration itself.
   *
   * @return The local variable declaration. Null if none.
   */
  public LocalVariableDeclT getForVariableDeclaration();

  /**
   * Attempts to set the variable declaration on this for statement.
   * 
   * @throws UnsupportedOperationException if this for loop does not allow
   *                                       a variable declaration.
   */
  public void setForVariableDeclaration(LocalVariableDeclT decl);

  /**
   * In a standard for loop with variable declarations, these are the
   * variables declared in the initialization declaration. If there is
   * a variable declaration, then this is equivalent to calling
   * getForVariableDeclaration().getVariables().
   *
   * @return The array of variable declarations. <p/>
   *
   * List of LocalVariableTs.
   */
  public List getForVariables();

  /**
   * In a standard for loop with no variable declarations, this is the
   * list wrapping the initialization terms. Unless you need the
   * actual ListExpression, you should probably use
   * getForInitializations().
   *
   * @return The ListExpression wrapping the initializations. Null if
   * none.
   */
  public ListExpressionT getForInitializationList();

  /**
   * In a standard for loop with no variable declarations, these are
   * the initialization terms. If there are initialization terms, this
   * is equivalent to calling getForInitializationList().getOperands().
   *
   * @return The array of initialization terms. <p/>
   *
   * List of ExpressionTs.
   */
  public List getForInitializations();

  /**
   * In a standard for loop, this is the condition expression.
   * Syntax is "for( initialization; conditional; update) stmt".
   * @return The conditional term. Null if there is none.
   */
  public ExpressionT getForConditional();

  /**
   * Attempts to set the conditional expression on this for statement.
   * 
   * @throws UnsupportedOperationException if this is an enhanced for loop.
   */
  public void setForConditional(ExpressionT e);

  /**
   * In a standard for loop, this is the list wrapping the update
   * terms.  Syntax is "for( initialization; conditional; update)
   * stmt". Unless you need the actual ListExpression, you should
   * probably use getForUpdates().
   *
   * @return The ListExpression wrapping the updates. Null if none.
   */
  public ListExpressionT getForUpdateList();

  /**
   * In a standard for loop, these are the update terms.  Syntax is
   * "for( initialization; conditional; update) stmt". If there are
   * update terms, this is equivalent to calling
   * getForUpdateList().getOperands().
   *
   * @return The array of update terms. <p/>
   *
   * List of ExpressionTs.
   */
  public List getForUpdates();

  /**
   * In an enhanced for loop, this is the collection expression.
   * Syntax is "for ( type name: collection ) stmt".
   * @return The expression for the collection term.
   */
  public ExpressionT getForCollection();

  /**
   * Attempts to set the collection expression on this for statement.
   * 
   * @throws UnsupportedOperationException if this is not an enhanced for
   *                                       loop.
   */
  public void setForCollection(ExpressionT e);


  // ----------------------------------------------------------------------

  /**
   * In this version, this class is 1.4 compatible. In a later version,
   * it will be redone as an enum. <p/>
   */
  public static final class ForKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int VALUE_EXPRESSION = 0;
    private static final int VALUE_VARIABLE_D = 1;
    private static final int VALUE_ENHANCED = 2;

    /** Regular for loop that does not declare variables. */
    public static final ForKind EXPRESSION =
      new ForKind( VALUE_EXPRESSION, "EXPRESSION" );

    /** Regular loop that declares variables. */
    public static final ForKind VARIABLE_D =
      new ForKind( VALUE_VARIABLE_D, "VARIABLE_D" );

    /** Enhanced for loop. */
    public static final ForKind ENHANCED =
      new ForKind( VALUE_ENHANCED, "ENHANCED" );


    // ----------------------------------------------------------------------

    private final int ordinal;
    
    private final String name;

    private ForKind(int ordinal, String name)
    {
      this.ordinal = ordinal;
      this.name = name;
      
      values.put( name, this );
    }

    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(ForKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof ForKind)
      {
        final ForKind tk = (ForKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return ForKind.class;
    }


    // ----------------------------------------------------------------------

    public static ForKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static ForKind valueOf(Class ignored, String name)
    {
      return (ForKind) values.get(name);
    }

    public static ForKind[] values()
    {
      final Collection entries = values.values();
      return (ForKind[])entries.toArray(new ForKind[entries.size()]);
    }


    // ----------------------------------------------------------------------
  }
}
