/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.I18NCharVisitor;
import javax.ide.extension.I18NStringVisitor;
import javax.ide.extension.IconVisitor;
import javax.ide.util.IconDescription;

/**
 * Handler for menu in a menu bar.
 */
class MenuVisitor extends PositionableVisitor
{
  static final ElementName MENU = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "menu" );

  private final static ElementName LABEL = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "label" );
  private final static ElementName MNEMONIC = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "mnemonic" );
  private final static ElementName TOOLTIP = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "tooltip" );
  private final static ElementName ICONPATH = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "iconpath" );

  private static final String KEY_DEFAULT_SECTION = "defaultsection";

  private ElementVisitor _sectionVisitor; // = new SectionVisitor();


  protected void positionable(ElementStartContext context, String id,
    String before, String after, Float weight)
  {
    MenuBar menuBar =
      (MenuBar) context.getScopeData().get( MenusVisitor.KEY_MENU_BAR );
    Menu menu = menuBar.getMenu( id );

    if ( menu == null )
    {
      menu = new Menu( id );
      menuBar.addMenu( menu );
    }

    processMenu( context, menu, before, after, weight );
  }

  protected final void processMenu( ElementStartContext context, final Menu menu,
    String before, String after, Float weight )
  {
    // Log a warning if any of the attributes has already been set
    if ( weight != null )
    {
      if ( menu.getWeight() != null )
      {
        log( context, Level.WARNING,
             "The " + menu.getID() + " menu already has a defined weight of " + menu.getWeight() );
      }
      else
      {
        menu.setWeight( weight );
      }
    }
    else if ( before != null )
    {
      if ( menu.getBefore() != null )
      {
        log( context, Level.WARNING,
             "The " + menu.getID() + " menu already has a defined before section of " +
             menu.getBefore() );
      }
      else
      {
        menu.setBefore( before );
      }
      log( context, Level.WARNING,
           "The weight attribute should be used in place of before or after attributes." );
    }
    else if ( after != null )
    {
      if ( menu.getAfter() != null )
      {
        log( context, Level.WARNING,
             "The " + menu.getID() + " menu already has a defined 'after' section of " +
             menu.getAfter() );
      }
      else
      {
        menu.setAfter( after );
      }
      log( context, Level.WARNING,
           "The 'weight' attribute should be used in place of 'before' or 'after' attributes." );
    }

    if ( _sectionVisitor == null )
    {
      _sectionVisitor = new SectionVisitor();
    }

    context.getScopeData().put( MenuHook.KEY_SECTION_CONTAINER, menu );
    context.registerChildVisitor( SectionVisitor.SECTION, _sectionVisitor );

    final String defaultSection = context.getAttributeValue( KEY_DEFAULT_SECTION );
    if ( defaultSection != null )
    {
      if ( menu.getDefaultSection() != null )
      {
        // Log a warning if the default section has already been set
        log( context, Level.WARNING,
             "The " + menu.getID() + " menu already has a defined defaultsection of " +
             menu.getDefaultSection() );
      }
      else
      {
        menu.setDefaultSection( Float.valueOf( defaultSection ) );
      }
    }

    context.registerChildVisitor( LABEL, new I18NStringVisitor()
    {
      protected void string( ElementContext context, String string )
        {
          menu.setLabel( string );
        }
      } );

    context.registerChildVisitor( MNEMONIC, new I18NCharVisitor()
    {
      protected void characterValue( ElementContext context, char c )
        {
          menu.setMnemonic( c );
        }
      } );

    context.registerChildVisitor( TOOLTIP, new I18NStringVisitor()
    {
      protected void string( ElementContext context, String string )
        {
          menu.setTooltip( string );
        }
      } );

    context.registerChildVisitor( ICONPATH, new IconVisitor()
    {
      protected void icon( ElementContext context, IconDescription icon )
        {
          menu.setIcon( icon );
        }
      } );
  }

}
