/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.ide.command.Controller;
import javax.ide.menu.IDEAction;
import javax.ide.menu.ItemFactory;
import javax.ide.util.MetaClass;

/**
 * The menu model retrieved from extension manifests.
 */
public final class MenuModel 
{
  private final Map _actions = new HashMap();
  private final Map _pullDownMenus = new HashMap();
  private final Map _popupMenus = new HashMap();
  private final Map _toolbars = new HashMap();
  private final Map _actionControllers = new HashMap();
  private final Set /*<MetaClassFactory>*/ _itemFactories = new HashSet();

  void addAction( IDEAction action )
  {
    _actions.put( action.getID(), action );
  }
  
  boolean isUniqueActionId( String id )
  {
    return !_actions.containsKey( id );
  }

  Toolbar findOrCreateToolbar( String id )
  {
    Toolbar toolbar =  (Toolbar) _toolbars.get( id );
    if ( toolbar == null )
    {
      toolbar = new Toolbar( id );
      _toolbars.put( id, toolbar );
    }
    return toolbar;
  }
  
  MenuBar findOrCreatePullDownMenu( String id )
  {
    MenuBar mc =  (MenuBar) _pullDownMenus.get( id );
    if ( mc == null )
    {
      mc = new MenuBar( id );
      _pullDownMenus.put( id, mc );
    }
    return mc;
  }
  
  PopupMenu findOrCreatePopupMenu( String id )
  {
    PopupMenu mc =  (PopupMenu) _popupMenus.get( id );
    if ( mc == null )
    {
      mc = new PopupMenu( id );
      _popupMenus.put( id, mc );
    }
    return mc;
  }
  
  void addItemFactory( MetaClass itemFactory )
  {
    _itemFactories.add( new MetaClassWrapper( itemFactory ) );
  }
  
  /**
   * Get a set of ids of all actions that have registered controllers.
   * The action ids in this collection may be a superset of the set of all
   * registered action ids.
   * 
   * @return a set of action ids for which there are controllers (or
   *    overridden controllers).
   */
  public Set /*<String>*/ getActionsWithControllers()
  {
    return _actionControllers.keySet();
  }
  
  /**
   * Get all declaratively registered controllers for the specified action
   * id.
   * 
   * @param actionId an action id.
   * @return a list of controllers.
   */
  public List /*<Controller>*/ getControllers( String actionId )
  {
    List list = (List) _actionControllers.get( actionId );
    if ( list == null )
    {
      return Collections.EMPTY_LIST;
    }
    return Collections.unmodifiableList( list );
  }
  
  void addController( String actionId, Controller controller )
  {
    List controllers = (List) _actionControllers.get( actionId );
    if ( controllers == null )
    {
      controllers = new ArrayList();
      _actionControllers.put( actionId, controllers );
    }
    controllers.add( controller );
  }

  /**
   * Get all actions in the model.
   * 
   * @return the map of actions. Keys are action ids, values are IDEAction
   *    instances.
   */
  public Map /*<String,IDEAction> */ getActions()
  {
    return _actions;
  }
  
  /**
   * Get all menu bars in the model
   * 
   * @return the map of menu bars. Keys are menu bar ids, values are 
   *    MenuBar instances.
   */
  public Map /*<String,MenuBar> */ getMenuBars()
  {
    return _pullDownMenus;
  }

  /**
   * Get all pull down menus in the model.
   * 
   * @return the map of menus. Keys are menu ids, values are MenuItemContainer
   *    instances.
   */
  public Map /*<String,PopupMenu> */ getPopupMenus()
  {
    return _popupMenus;
  }
  
  
  /**
   * Get all toolbars in the model.
   * 
   * @return the map of toolbars. Keys are toolbar ids, values are Toolbar
   *    instances.
   */
  public Map /*<String,Toolbar>*/ getToolbars()
  {
    return _toolbars;
  }
  
  /**
   * Gets the MetaClass objects for all registered ItemFactory objects.
   * This avoids class loading, therefore extension loading.
   * @see #getItemFactory(String)
   */
  public Collection<MetaClass> getItemFactoryClasses()
  {
    Set factories;
    synchronized ( this )
    {
      factories = new HashSet( _itemFactories );
    }
    
    Set<MetaClass> classes = new HashSet<MetaClass>();
    for ( Iterator i = factories.iterator(); i.hasNext(); )
    {
      MetaClassWrapper mcw = ( MetaClassWrapper ) i.next();
      classes.add(mcw._metaClass);
    }
     return classes;
  }
  
  /**
   * This method will return an instance of the ItemFactory class
   * with the supplied class name, or null if none was registered.
   * @param itemFactoryClass  the name of a registered ItemFactory
   */
  public ItemFactory getItemFactory(String itemFactoryClass)
  {
    if (itemFactoryClass == null || itemFactoryClass.isEmpty())
    {
      return null;
    }
    
    Set factories;
    synchronized ( this )
    {
      factories = new HashSet( _itemFactories );
    }
    
    ItemFactory factory = null;
    
    for ( Iterator i = factories.iterator(); i.hasNext(); )
    {
      MetaClassWrapper mcw = (MetaClassWrapper) i.next();
      if ( itemFactoryClass.equals( mcw._metaClass.getClassName() ) )
      {
        try
        {
          Object o = mcw.getInstance();
          if (o instanceof ItemFactory)
          {
            factory = (ItemFactory) o;
            break;
          }
        }
        catch ( Exception e )
        {
          e.printStackTrace();
          factory = null;
        }
      }
    }

    return factory;
  }
  
  /**
   * @deprecated Calling this method may cause extension loading. 
   * Use {@link #getItemFactoryClasses()} and {@link #getItemFactory(String)}
   * to ensure extensions are not loaded unexpectedly.
   */
  @Deprecated
  public Collection /*<ItemFactory>*/ getItemFactories()
  {
    Set factories;
    synchronized ( this )
    {
      factories = new HashSet( _itemFactories );
    }
    
    Set classes = new HashSet();
    for ( Iterator i = factories.iterator(); i.hasNext(); )
    {
      MetaClassWrapper mcw = ( MetaClassWrapper ) i.next();
      try
      {
        classes.add( mcw.getInstance() );
      }
      catch ( Exception e )
      {
        e.printStackTrace();
      }
    }

    return Collections.unmodifiableCollection( classes );
  }
  
  /**
   * This is generally useful, should move elsewhere.
   */
  private class MetaClassWrapper
  {
    private MetaClass _metaClass;
    private Object _instance;
    
    MetaClassWrapper( MetaClass mc )
    {
      _metaClass = mc;
    }
    
    public Object getInstance()
      throws ClassNotFoundException, IllegalAccessException, InstantiationException
    {
      if (_instance == null)
      {
        _instance = _metaClass.newInstance();
      }
      return _instance;
    }
  }
}
