/* Copyright (c) 2006, 2011, Oracle and/or its affiliates. 
All rights reserved. */
package javax.ide.extension.spi;

/**
 * Expands macros.<p>
 * 
 * Create a subclass of this class which provides an implementation of the
 * {@link #getMacroValue(String)} method.
 * 
 *  @author  bduff   
 *  @since   11.0
 */
public abstract class MacroExpander
{
  
  public static boolean containsMacro(String s) {
    int startBracketIndex = s.indexOf("${");
    int endBracketIndex = s.lastIndexOf("}");
    return startBracketIndex != -1 && startBracketIndex < endBracketIndex;
  }
  
  public static String stripOffBrackets(String s) {
    int startBracketIndex = s.indexOf("${");
    int endBracketIndex = s.lastIndexOf("}");
    if (startBracketIndex != -1 && startBracketIndex < endBracketIndex) 
    {
      return s.substring(startBracketIndex + 2, endBracketIndex);
    }
    else 
    {
      return s;
    }
  }
  
  /**
   * Substitute all macros in the specified string. 
   * 
   * @param s a string in which to substitute macros. If null is specified,
   *    then null will be returned.
   * @return the substituted string, or null if <tt>s</tt> is null.
   */
  public final String substituteMacros( String s )
  {
    if ( s == null )
    {
      return null;
    }

    StringBuilder result = new StringBuilder();

    int start = 0;
    int pos;
    while ( (pos = s.indexOf( '$', start )) != -1 )
    {
      result.append( s.substring( start, pos ) );
      if ( pos == s.length() - 1)
      {
        break;
      }
      if ( s.charAt( pos + 1 ) == '{' )
      {
        int fin = s.indexOf( '}', pos+1 );
        if ( fin == -1 )
        {
          // OK we have an unterminated macro string.
          result.append( "$" );
          start = pos + 1;
          continue;
        }
        String macro = s.substring( pos+2, fin );
        String value = getMacroValue( macro );
        if ( value == null )
        {
          result.append( "${" + macro + "}" );
        }
        else
        {
          result.append( value );
        }
        start = fin+1;
      }
      else
      {
       result.append( s.substring( pos, pos+1 ) );
       start = pos+1;
      }
    }
    if ( start < s.length() )
    {
      result.append( s.substring( start ) );
    }

    return result.toString();
  }
  
  /**
   * Get the value of the specified macro. 
   * 
   * @param macroText the text of the macro. If the unexpanded string contains
   *    "This is a ${test}", then the macroText of the first macro in this
   *    String is "test". Null will never be passed.
   * @return the value of the macro <tt>macroText</tt>, or <tt>null</tt> if
   *    the specified macro is unrecognized.
   */
  protected abstract String getMacroValue( String macroText );
}
