/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import javax.ide.extension.Extension;
import javax.ide.net.URIFactory;
import javax.ide.net.VirtualFileSystem;

/**
 * A default implementation of ExtensionSource which loads an extension 
 * manifest from the META-INF/extension.xml entry of a JAR file.
 */
public class JARExtensionSource implements ExtensionSource
{
  private final static String MANIFEST_ENTRY = "META-INF/extension.xml";
  private final static String MANIFEST_ALT_ENTRY = "meta-inf/extension.xml";
  private final URI _jarURI;
  private URI _manifestURI;

  public JARExtensionSource( URI jarURI )
  {
    _jarURI = jarURI;
  }



  /**
   * Get the URI of this JAR file.
   * 
   * @return the URI of this jar file. 
   */
  public final URI getURI()
  {
    return _jarURI;
  }

  public final URI getClasspathEntry()
  {
    return _jarURI;
  }

  /**
   * Get the URI of the manifest file within the extension source.
   * 
   * @return the URI of the manifest file.
   */
  public URI getManifestURI()
  {
    if ( _manifestURI == null )
    {
      _manifestURI = URIFactory.newJarURI( _jarURI, MANIFEST_ENTRY );
      if ( !VirtualFileSystem.getVirtualFileSystem().exists( _manifestURI ) )
      {
        _manifestURI = URIFactory.newJarURI( _jarURI, MANIFEST_ALT_ENTRY );
      }
    }
    return _manifestURI;
  }
  
  /**
   * Resolve a relative path from the manifest file. For JAR sources, the
   * path may either be within the JAR (if the path starts with a /), 
   * or relative to the location of the jar file otherwise. 
   * 
   * @param extension the extension being processed.
   * @param path a relative path within the JAR.
   * @return the absolute URI of the referenced resource.
   */
  public URI resolvePath( Extension extension, String path )
  {
    // First check if it's already an absolute file name.
    File f = new File( path );
    if ( f.isAbsolute() && f.exists() )
    {
      return URIFactory.newFileURI( f );
    }
  
    // First see whether it's resolvable within the jar file.
    if ( path.length() >= 2 && path.charAt( 0 ) == '/' )
    {
      return URIFactory.newJarURI( _jarURI, path.substring( 1 ) );
    }
  
    URI parentOfJar = 
      VirtualFileSystem.getVirtualFileSystem().getParent( _jarURI );
    
    // 04-AUG-09 kmchorto sometimes extensions have URI's
    // of the form oracle/jdeveloper/jdev/../jdev which causes problems 
    // later on, so naromalize the URI.
    
    URI parentOfJarNorm = parentOfJar.normalize();

    return URIFactory.newURI( parentOfJarNorm, path );
  }
  
  /**
   * Get the name of this source. This will be used to present the source
   * in human readable messages.<p>
   * 
   * This implementation returns the result of calling 
   * VirtualFileSytem.toDisplayString() on the jar URI.
   * 
   * @return the name of this source.
   */  
  public String getName()
  {
    return VirtualFileSystem.getVirtualFileSystem().toDisplayString(
      _jarURI
    );
  }
  
  
  public InputStream getInputStream()
    throws IOException
  {
    return VirtualFileSystem.getVirtualFileSystem().openInputStream( 
      getManifestURI() );
  }

  public String toString ()
  {
    return getName();
  }
  
}
