package javax.ide.extension.spi;

import java.net.URI;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import javax.ide.extension.ElementContext;

import javax.ide.extension.ElementVisitor;

import javax.ide.extension.Extension;
import javax.ide.net.VirtualFileSystem;

import org.xml.sax.Locator;

/**
 * Extension log records hold positional information about the location where
 * log messages occurred while processing the extension manifest.
 */
public final class ExtensionLogRecord extends LogRecord
{
  private LocationAdapter _location;

  /**
   * A parse-method-independent constructor for extension log records.
   * 
   * @param locationAdapter
   * @param level
   * @param msg
   */
  public ExtensionLogRecord( LocationAdapter locationAdapter, Level level, String msg )
  {
    super( level, msg );
    _location = locationAdapter.copyMe();
  }

  /**
   * @deprecated use parse-method-independent constructor
   * @param locator
   * @param level
   * @param msg
   */
  public ExtensionLogRecord( Locator locator, Level level, String msg )
  {
    super( level, msg );
    initLocationAdapterAndParameters(locator);
  }
  
  private void initLocationAdapterAndParameters(Object locator) {
    _location = XMLParsingUtils.copyAndCastToLocationAdapter( locator );

    // For backwards compatibility (early versions of the API used the
    // parameters array to store the locator)
    if ( _location != null )
    {
      setParameters( new Object[] { _location } );
    }
  }

  /**
   * Creates a log record.
   * 
   * @param context the current procesing context.
   * @param level the message log level.
   * @param msg the message to log.
   */
  public ExtensionLogRecord( ElementContext context, Level level, String msg )
  {
    super( level, msg );
    initLocationAdapterAndParameters( context.getScopeData().get( ElementVisitor.KEY_LOCATOR ) );
  }
  
  /**
   * Creates a log record. 
   * 
   * @param level the message log level
   * @param msg the message
   * @param extension the extension 
   * @param lineNumber the line number within the extension manifest.
   * @since 2.0
   */
  public ExtensionLogRecord( Level level, String msg, Extension extension, 
    int lineNumber )
  {
    super(level, msg);
    initLocationAdapterAndParameters( extensionToLocator( extension, lineNumber ) );
  }
  
  /**
   * Creates a log record. 
   * 
   * @param level the message log level
   * @param msg the message
   * @param extension the extension 
   * @param lineNumber the line number within the extension manifest.
   * @param thrown the exception that was thrown.
   * @since 2.0
   */
  public ExtensionLogRecord( Level level, String msg, Extension extension, 
    int lineNumber, Throwable thrown )
  {
    this(level, msg, extension, lineNumber);
    setThrown( thrown );
  }
  
  private static Locator extensionToLocator( final Extension extension, 
    final int lineNumber )
  {
    return new Locator() 
    {

      public String getPublicId()
      {
        return null;
      }

      public String getSystemId()
      {
        if ( extension instanceof DefaultExtension )
        {
          ExtensionSource source =  ((DefaultExtension)extension).getSource();
          if ( source == null ) return "";
          URI uri = source.getManifestURI();
          if (uri != null)
          {
            return VirtualFileSystem.getVirtualFileSystem().getPlatformPathName(uri);
          }
        }
        return "";
      }

      public int getLineNumber()
      {
        return lineNumber;
      }

      public int getColumnNumber()
      {
        return 0;
      }
    };
  }
  
  /**
   * Gets the locator.
   * 
   * @deprecated call {@code ExtensionLogRecord#getLocationAdapter()} instead.
   * @return the locator.
   */
  public Locator getLocator()
  {
    return _location;
  }
  
  /**
   * Gets the location adapter. Clients should use this method to write parse-method-independent code. 
   * 
   * @return the location adapter
   */
  public LocationAdapter getLocationAdapter()
  {
    return _location;
  }
}
