package javax.ide.extension;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.ide.extension.spi.DefaultElementContext;
import javax.ide.extension.spi.DeferredElementVisitorHelper;

/**
 * MultiDeferredElementVisitorHook is similar to DeferredElementVisitorHook
 * except that it supports attaching multiple ElementVisitors.  A client
 * of this API specifies the maximum number of visitors that can be 
 * attached.  After that many visitors are attached, the recorded data
 * will be released, and an exception will be thrown if someone attempts
 * to attach another visitor.
 * 
 * If MultiDeferredElementVisitorHook is registered as the hook handler
 * for an element, it will record the XML data for all instances 
 * of that element encountered during extension.xml processing.
 * 
 * At a later time, when that data is needed, the client would
 * retrieve the MultiDeferredElementVisitorHook instance from the
 * ExtensionRegistry and call attachElementVisitor() passing
 * an ElementVisitor implementation.  Before attachElementVisitor() 
 * returns, all of the recorded XML data will be replayed to the
 * attached ElementVisitor instance visitor methods (start/end).
 * 
 * MultiDeferredElementVisitor attempts to replicate the original
 * parsing context (such as scope data and locators), such that 
 * the attached ElementVisitor would have access to the same
 * information it would have if it had been registered as the
 * hook handler.
 * 
 * Once an ElementVisitor is attached, if additional instances
 * of the element are encountered during extension.xml processing,
 * the attached ElementVisitor's start/end method will be called.
 * 
 */
public class MultiDeferredElementVisitorHook extends ParameterizedExtensionHook
{
  public MultiDeferredElementVisitorHook()
  {
    super();
  }
  
  public void setHookHandlerParameters(Map<String,String> parameterMap)
  {
    String maxVisitorsParam = parameterMap.get(_MAX_VISITOR_HOOK_HANDLER_PARAM);
    if (maxVisitorsParam != null)
    {
      try
      {
        int intValue = Integer.parseInt(maxVisitorsParam);
        if (intValue > 1)
        {
          setMaximumNumberOfVisitors(intValue);
        }
      }
      catch (Exception ignored)
      {
      }
    }
  }
  
  public void setMaximumNumberOfVisitors(int maxVisitors)
  {
    synchronized(_attachmentLock)
    {
      if (maxVisitors == _maxVisitors)
      {
        return;
      }
      
      //don't let it be set lower
      if (_maxVisitors > maxVisitors)
      {
        throw new IllegalArgumentException("Illegal argument: maxVisitors is already set to " + _maxVisitors + " it cannot be set lower to " + maxVisitors);
      }
      
      //if max already reached, don't let it set higher (since we already released the data)
      if (_attachedElementVisitorList.size() == _maxVisitors)
      {
        throw new IllegalStateException("Illegal attempt to setMaximumNumberOfVisitors after maximum has already been reached");
      }
      
      _maxVisitors = maxVisitors;
    }
  }
  
  public int getMaximumNumberOfVisitors()
  {
    return _maxVisitors;
  }

  @Override
  public void start(ElementStartContext context)
  {
    synchronized(_attachmentLock)
    {
      //Sadly, we can't just pass this context directly to the attached visitors
      //start methods.  That is because one context is for one visitor, since the
      //visitor will mutate the context.  Because we are multiplexing this data
      //to multiple visitors, we must always record the info, and then in the
      //end method, we will play it back separately to each attached visitor
      _helper.recordTopLevelElementStart(context);
    }
  }

  @Override
  public void end(ElementEndContext context)
  {
    synchronized(_attachmentLock)
    {
      _helper.recordTopLevelElementEnd(context);
      
      for (ElementVisitor visitor : _attachedElementVisitorList)
      {
        ExtensionRegistry extensionRegistry = ExtensionRegistry.getExtensionRegistry();
        DefaultElementContext freshContext = (DefaultElementContext) extensionRegistry.createInitialContext();
        _helper.visitNewlyRecordedData(freshContext, visitor);
      }
      
      if (_attachedElementVisitorList.size() == _maxVisitors)
      {
        _helper.clearRecordedData();
      }
    }
  }
  
  
  /**
   * When the XML data associated with the hook handler instance needs
   * to be processed, call this method passing an ElementVisitor
   * implementation.  Before attachElementVisitor() returns, 
   * all of the recorded XML data will be replayed to the
   * given ElementVisitor instance visitor methods (start/end).
   * 
   * Once this method has been called, any additional XML data
   * processed by the DeferredElementVisitorHookHandler will
   * be immediately passed through to the attached ElementVisitor
   * with no buffering.
   * 
   * It is illegal to call this method more than once passing in
   * different ElementVisitor instances.  
   * 
   * @param elementVisitor
   */
  public void attachElementVisitor(ElementVisitor elementVisitor)
  { 
    if (elementVisitor == null)
    {
      throw new IllegalArgumentException("attachElementVisitor passed a null element visitor"); //NOTRANS
    }
    
    synchronized(_attachmentLock)
    {
      if (_attachedElementVisitorList.contains(elementVisitor))
      {
        return;
      }
      
      if (_attachedElementVisitorList.size() == _maxVisitors)
      {
        throw new IllegalStateException("attachElementVisitor called after maximum number of visitors have already been attached");
      }
      
      if (elementVisitor instanceof ExtensionHook)
      {
        if (getProvider() != null)
        {
          ((ExtensionHook) elementVisitor).setProvider(getProvider());
        }
        if (getSchemaLocation() != null)
        {
          ((ExtensionHook) elementVisitor).setSchemaLocation(getSchemaLocation());
        }
      }

      _attachedElementVisitorList.add(elementVisitor);

      ExtensionRegistry extensionRegistry = ExtensionRegistry.getExtensionRegistry();
      DefaultElementContext context = (DefaultElementContext) extensionRegistry.createInitialContext();
      _helper.visitRecordedData(context, elementVisitor);
      
      if (_attachedElementVisitorList.size() == _maxVisitors)
      {
        _helper.clearRecordedData();
      }
    }
  }
  
  public int getNumberOfAttachedVisitors()
  {
    return _attachedElementVisitorList.size();
  }
  
  public List<ElementVisitor> getAttachedVisitors()
  {
    List<ElementVisitor> copy = new ArrayList<ElementVisitor>(_attachedElementVisitorList);
    return copy;
  }
  
  private static final String _MAX_VISITOR_HOOK_HANDLER_PARAM = "max-visitors";
  private final DeferredElementVisitorHelper _helper = new DeferredElementVisitorHelper();
  private List<ElementVisitor> _attachedElementVisitorList = new CopyOnWriteArrayList<ElementVisitor>();
  private Object _attachmentLock = new Object();
  private volatile int _maxVisitors = 1;
}