/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

/**
 * Represents a qualified element name in XML. A qualified element name
 * consists of a namespace URI (which may be null) and a local name.
 */
public final class ElementName 
{
  private final String _namespaceURI;
  private final String _localName;

  /**
   * Constructs an ElementName with the specified namespace URI and local
   * name.
   * 
   * @param namespaceURI the namespace uri of the qualified name. May be null.
   * @param localName the local name of the qualified name. Must not be null.
   */
  public ElementName( String namespaceURI, String localName )
  {
    if ( localName == null )
    {
      throw new NullPointerException( "null localName" );
    }
    _namespaceURI = namespaceURI;
    _localName = localName;
  }
  
  /**
   * Get the namespace uri of this element name.
   * 
   * @return the namespace uri of this element name.
   */
  public String getNamespaceURI()
  {
    return _namespaceURI;
  }
  
  /**
   * The local name of this element name.
   * 
   * @return the local name of this element name.
   */
  public String getLocalName()
  {
    return _localName;
  }
  
  /**
   * Get a string representation of this element name.
   * 
   * @return a string representation of this element name.
   */
  public String toString()
  {
    return ElementName.class.getName() + "[namespaceURI="+
      String.valueOf( _namespaceURI ) + ", localName="+
      _localName + "]";
  }
  
  /**
   * Get a hashcode for this object.
   * 
   * @return a hash code.
   */
  public int hashCode()
  {
    int hash = 42;
    if ( _namespaceURI != null )
    {
      hash = 37 * hash + _namespaceURI.hashCode();
    }
    hash = 37 * hash + _localName.hashCode();
    
    return hash;
  }
  
  /**
   * Get whether this element name equals some other object.
   * 
   * @param o another object to compare to this one.
   * @return true if the objects represent the same element name.
   */
  public boolean equals( Object o ) 
  {
    if ( o == this )
    {
      return true;
    }
    if ( !(o instanceof ElementName) )
    {
      return false;
    }
    ElementName that = (ElementName) o;
    return _localName.equals( that._localName ) &&
      ( ( _namespaceURI == null && that._namespaceURI == null ) ||
        ( _namespaceURI != null && _namespaceURI.equals( that._namespaceURI )));
  }
  
  
}
