package javax.ide.extension;

import javax.ide.extension.spi.DefaultElementContext;
import javax.ide.extension.spi.DeferredElementVisitorHelper;

/**
 * DeferredElementVisitorHook is a specialized hook handler class
 * that helps clients with existing custom hook handlers convert
 * to a lazier, pull based model.
 * 
 * If DeferredElementVisitorHook is registered as the hook handler
 * for an element, it will record the XML data for all instances 
 * of that element encountered during extension.xml processing.
 * 
 * At a later time, when that data is needed, the client would
 * retrieve the DeferredElementVisitorHook instance from the
 * ExtensionRegistry and call attachElementVisitor() passing
 * an ElementVisitor implementation.  Before attachElementVisitor() 
 * returns, all of the recorded XML data will be replayed to the
 * attached ElementVisitor instance visitor methods (start/end).
 * 
 * DeferredElementVisitor attempts to replicate the original
 * parsing context (such as scope data and locators), such that 
 * the attached ElementVisitor would have access to the same
 * information it would have if it had been registered as the
 * hook handler.
 * 
 * Once an ElementVisitor is attached, if additional instances
 * of the element are encountered during extension.xml processing,
 * they will be directly passed to the attached ElementVisitor's
 * start/end method with no buffering.
 * 
 */
public class DeferredElementVisitorHook extends ExtensionHook
{
  public DeferredElementVisitorHook()
  {
    super();
  }

  @Override
  public void start(ElementStartContext context)
  {
    synchronized(_attachmentLock)
    {
      if (_attachedElementVisitor != null)
      {
        _attachedElementVisitor.start(context);
      }
      else
      {
        _helper.recordTopLevelElementStart(context);
      }
    }
  }

  @Override
  public void end(ElementEndContext context)
  {
    synchronized(_attachmentLock)
    {
      if (_attachedElementVisitor != null)
      {
        _attachedElementVisitor.end(context);
      }
      else
      {
        _helper.recordTopLevelElementEnd(context);
      }
    }
  }
  
  
  /**
   * When the XML data associated with the hook handler instance needs
   * to be processed, call this method passing an ElementVisitor
   * implementation.  Before attachElementVisitor() returns, 
   * all of the recorded XML data will be replayed to the
   * given ElementVisitor instance visitor methods (start/end).
   * 
   * Once this method has been called, any additional XML data
   * processed by the DeferredElementVisitorHookHandler will
   * be immediately passed through to the attached ElementVisitor
   * with no buffering.
   * 
   * It is illegal to call this method more than once passing in
   * different ElementVisitor instances.  
   * 
   * @param elementVisitor
   */
  public void attachElementVisitor(ElementVisitor elementVisitor)
  { 
    if (elementVisitor == null)
    {
      throw new IllegalArgumentException("attachElementVisitor passed a null element visitor"); //NOTRANS
    }
    
    synchronized(_attachmentLock)
    {
      if (_attachedElementVisitor != null)
      {
        if (_attachedElementVisitor == elementVisitor)
        {
          return;
        }
        
        throw new IllegalStateException("attachElementVisitor called a second time with a different element visitor instance");
      }
      
      if (elementVisitor instanceof ExtensionHook)
      {
        if (getProvider() != null)
        {
          ((ExtensionHook) elementVisitor).setProvider(getProvider());
        }
        if (getSchemaLocation() != null)
        {
          ((ExtensionHook) elementVisitor).setSchemaLocation(getSchemaLocation());
        }
      }

      _attachedElementVisitor = elementVisitor;

      ExtensionRegistry extensionRegistry = ExtensionRegistry.getExtensionRegistry();
      DefaultElementContext context = (DefaultElementContext) extensionRegistry.createInitialContext();
      _helper.visitRecordedData(context, _attachedElementVisitor);
      _helper.clearRecordedData();
    }
  }

  /**
   * Returns true if an ElementVisitor is attached
   * @see #attachElementVisitor(ElementVisitor)
   * @return
   */
  public boolean isElementVisitorAttached()
  {
    synchronized(_attachmentLock)
    {
      return _attachedElementVisitor != null;
    }
  }
  
  /**
   * Returns the attached ElementVisitor.
   * 
   * @return null if visitor is not attached.
   */
  public ElementVisitor getAttachedElementVisitor() {
    synchronized(_attachmentLock) 
    {
      return _attachedElementVisitor;  
    }
  }
  
  
  private final DeferredElementVisitorHelper _helper = new DeferredElementVisitorHelper();
  private ElementVisitor _attachedElementVisitor = null;
  private Object _attachmentLock = new Object();
}
