/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.editor;

import java.util.prefs.Preferences;
import javax.ide.command.Context;
import javax.ide.model.Document;
import javax.ide.view.View;

/**
 * The <code>Editor</code> interface defines the methods to open and 
 * close editors, along with methods to save and restore editor specific 
 * settings.<p>
 *
 * Extensions that introduce new editor types need to implement this 
 * interface. A new editor is introduced by declaring it in the extension
 * deployment descriptor (EDD) file. There, the editor implementation class 
 * is associated with a unique string identifying the editor type.<p>
 */
public abstract class Editor extends View
{
  private Context _context;

  /**
   * This method is called to open the editor. This method is reponsible
   * for creating the editor, intializing it with the 
   * {@link javax.ide.model.Document} specified by the <code>context</code>.<p>
   *
   * This method should only be called by the {@link EditorManager}. The 
   * editor manager needs to keep track of all editors opened.<p>
   * 
   * This implementation stores the context for later retrieval via 
   * getContext().
   *
   * @param context The current context. The editor will be opened on 
   * the document specified in the context.
   */
  public void open( Context context )
  {
    _context = context;
  }

  /**
   * This method is called when the editor is closed by the user. This 
   * method is responsible for closing the editor. <p>
   *
   * This method should only be called by the {@link EditorManager}. The 
   * editor manager needs to keep track of all editors opened.
   */
  public abstract void close();

  /**
   * Restore the last known state of the editor. This method is called
   * by the {@link EditorManager} just before the editor is opened.<p>
   *
   * This method should only be called by the {@link EditorManager}. <p>
   *
   * @param preference Preferences saved when this editor was last closed.
   * These preferences are used for restoring the last know state of the
   * editor. Extensions are responsible for identifying the preference 
   * sub-node associated with this editor.
   */
  public abstract void restore( Preferences preference );

  /**
   * Save the current state of the editor so that it can be restored when the 
   * editor is re-opened on the same document. This method is called by the
   * {@link EditorManager} when an editor is about to be closed, or when
   * the <code>EditorManager</code> is shutting down.<p>
   *
   * This method should only be called by the {@link EditorManager}. <p>
   *
   * @param preference Preferences saved when this editor was last closed.
   * These preferences are used for restoring the last know state of the
   * editor. Extensions are responsible for identifying the preference 
   * sub-node associated with this editor.
   */
  public abstract void save( Preferences preference );
  
  public String getLabel()
  {
    Document doc = getContext().getDocument();
    if ( doc != null )
    {
      return doc.getLabel();
    }
    
    return super.getLabel();
  }
  
  /**
   * Get the context for this editor. This implementation returns the 
   * context passed to the open() method.
   * 
   * @return the current context.
   */
  public Context getContext()
  {
    return _context;
  }

}
