/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.command;

import javax.ide.Service;
import javax.ide.spi.ProviderNotFoundException;

/**
 * The <tt>CommandProcessor</tt> is responsible for managing the 
 * execution of the IDE commands and maintining the undo stack.<p>
 *
 * In general, all {@link Controller} implementations should use the 
 * IDE command processor to execute the commands that controller handles. <p>
 *
 * IDE service providers must extend this class and provide an 
 * implementation of the {@link #invoke(Command)} method. It is up to the 
 * IDE service provider implementation to decide whether it supports 
 * undo/redo on a per document, or global basis.
 * If the undo support provided is document based, implementors of this 
 * class must make sure that after a command is executed, the 
 * command processor checks the {@link Command#getAffectedDocuments} return 
 * value to detemine if other documents were affected by the current command 
 * execution.<p>
 *
 * If the {@link Command#getAffectedDocuments} returns one or more items, then
 * the undo stacks of the affected documents must be flushed in order to
 * maintain document consistency.
 * 
 * @see Command
 */
public abstract class CommandProcessor extends Service
{

  /**
   * Executes the actions associated with a specific command. A
   * <code>Controller</code> uses this method to tell the
   * <code>CommandProcessor</code> that a command should be executed.
   *
   * @param cmd   the command to execute.
   * @throws Exception if an error occurs processing the command.
   * @return Whether the specified executed {@link Command#OK} or 
   *    {@link Command#CANCEL}.
   */
  public abstract int invoke( Command cmd ) throws Exception;
  
  /**
   * Get the command processor implementation for this IDE.
   * 
   * @return the command processor implementation.
   */
  public static CommandProcessor getCommandProcessor()
  {
    try
    {
      return (CommandProcessor) getService( CommandProcessor.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No command processor." );
    }
  }
}
