/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.command;

import javax.ide.Identifiable;
import javax.ide.model.Document;

/**
 * The <code>Command</code> interface defines the interface of all command
 * objects. This interface consists of a procedure, <code>doit</code>, to
 * execute a command. Concrete Command subclasses implement the necessary
 * behavior to carry out a specific command.<p>
 *
 * This interface also provides additional methods necessary to implement
 * the undo service provided by the <code>CommandProcessor</code>. For the
 * undo mechanism it is necessary to distinguish three types of commands:<p>
 * 
 * <ul>
 *  <li><code>NO_CHANGE</code>: A command that require no undo.</li><p>
 *  <li><code>NORMAL</code>: A command that can be undone.</li><p>
 *  <li><code>NO_UNDO</code>: A command that cannot be undone, and which
 *                            prevents the undo of previously performed
 *                            normal commands.</li><p>
 * </ul>
 *
 * Commands are executed using the {@link javax.ide.command.CommandProcessor}.
 * As <code>NORMAL</code> commands are executed, they are recorded by the
 * command processor. This allows the command processor to perform undo and
 * redo operations.
 *
 * @see javax.ide.command.CommandProcessor
 */
public abstract class Command implements Identifiable
{
  private Context _context;
  private final int _type;

  /** 
   *  Indicates a command that can be undone. 
   */
  public static final int NORMAL = 0;

  /** 
   *  Indicates a command that requires no undo. 
   */
  public static final int NO_CHANGE = 1;

  /** 
   *  Indicates a command that cannot be undone, and which
   *  prevents the undo of previously performed normal commands.
   */
  public static final int NO_UNDO = 2;

  /**
   *  Command execution OK status
   */
  public static final int OK = 0;

  /**
   *  Command execution CANCEL status
   */
  public static final int CANCEL = 1;
  
  protected Command()
  {
    this( NORMAL );
  }
  
  protected Command( int type )
  {
    _type = type;
  }

  /** 
   *  Executes the actions associated with a specific command.
   *  When a command executes successfully, implementations should
   *  return OK, otherwise, return CANCEL or any other non-zero value.
   */
  public abstract int doit() throws Exception;

  /** 
   *  Called by the CommandProcessor to undo a command 
   *  When a command executes successfully, implementations should
   *  return OK, otherwise, return CANCEL or any other non-zero value.
   */
  public abstract int undo() throws Exception;

  /** 
   *  Gets the name of the command to display as the action to undo 
   */
  public abstract String getName();

  /** 
   *  Gets the command type 
   *  @return The command type. Can be one of the following values:
   *  {@link #NORMAL}, {@link #NO_UNDO}, or {@link #NO_CHANGE}.
   */
  public final int getType()
  {
    return _type;
  }

  /** 
   *  Gets the ide current {@link Context}. 
   */
  protected final Context getContext()
  {
    return _context;
  }

  /** 
   *  Sets {@link Context} associated with command 
   */
  public final void setContext(Context context)
  {
    _context = context;
  }

  /**
   *  In general, the command processor manages undo stacks on a per 
   *  {@link javax.ide.model.Document} basis.  When the execution of a 
   *  command affects more than the target document, the other affected 
   *  documents undo stacks must be be flushed in order to maintain document 
   *  consistency. This method should be called by the command processor
   *  implementations after a command is executed to determine if that 
   *  command affects other documents.<p>
   *
   *  The CommandProcessor uses the return value to clear the undo stack
   *  of the affected documents.<p>
   *
   *  For commands of the NORMAL and NO_UNDO type, this method should return an 
   *  array of documents affected by the execution of this command. 
   *  This array should not include the primary document on which this 
   *  command is operating, it should only include other documents affected 
   *  as a side effect of executing this command. The affected documents
   *  undo stack will be flushed.<p>
   *
   *  Commands of type NO_CHANGE should return an empty array. The default
   *  implementation returns an empty array.
   *
   *  @return the documents affected by this command, otherwise, an empty
   *  array. This method must not return <code>null</code>.
   */
  public Document[] getAffectedDocuments()
  {
    return new Document[0];
  }
}
