/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.build;

import java.util.EventObject;
import javax.ide.command.Context;

/**
 * Event object passed to the {@link BuildListener}s.
 */
public final class BuildEvent extends EventObject
{
  private final Context _context;
  private final boolean _success;

  /**
   * Constructor.
   *
   * @param buildSystem the build system where the event happened. Must not be
   *    null.
   * @param context the context to build. Must not be null.
   */
  public BuildEvent( BuildSystem buildSystem, Context context )
  {
    this( buildSystem, context, true );
  }

  /**
   * Constructor.
   *
   * @param buildSystem the build system where the event happened. Must not be
   *    null.
   * @param context the context to build. Must not be null.
   * @param success <code>true</code> if the build is succeding. <code>false</code>
   *    if it failed or was aborted.
   */
  public BuildEvent( BuildSystem buildSystem, Context context, boolean success )
  {
    super( buildSystem );
    
    _context = context;
    _success = success;
  }

  /**
   * Get the {@link BuildSystem} where the event happened.
   * This is functionally equivalent to casting the result of getSource() to 
   * a <code>BuildSystem</code> object.
   *
   * @return the {@link BuildSystem} that generated the event. Must not be null.
   */
  public BuildSystem getBuildSystem() 
  {
    return (BuildSystem)getSource();
  }

  /**
   * Get the context currently used to build. The context selection lists 
   * all documents being built. The context project provides the source 
   * and class path.
   *
   * @return the current context.
   */
  public Context getContext()
  {
    return _context;
  }

  /**
   * Flag indicating if the build was successful. Implementors must 
   * check this flag before proceeding with their pre or post build 
   * behavior. 
   *
   * @return <code>true</code> if build is successful. <code>false</code> if
   *    the build failed or was aborted.
   */
  public boolean isBuildSuccessful()
  {
    return _success;
  }
}

