/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 *
 * Test
 */
package javax.ide;

import java.net.URI;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.ide.build.BuildSystem;
import javax.ide.command.CommandProcessor;
import javax.ide.editor.EditorManager;
import javax.ide.extension.ExtensionRegistry;
import javax.ide.debug.Debugger;
import javax.ide.log.LogManager;
import javax.ide.model.DocumentFactory;
import javax.ide.menu.ActionRegistry;
import javax.ide.property.PropertyPageRegistry;
import javax.ide.spi.IDEListenerHook;
import javax.ide.util.MetaClass;
import javax.ide.util.Version;
import javax.ide.view.GUIUtilities;
import javax.ide.view.View;
import javax.ide.wizard.WizardManager;
import java.util.Iterator;
import javax.ide.spi.ProviderNotFoundException;

/**
 * This is the object that centralizes general IDE environment information and
 * services. Extensions use the IDE interface to access standard services.
 * <p>
 * Extensions can locate the IDE environment object using the 
 * {@link javax.ide.IDE#getIDE()} method.
 * <p>
 * IDE providers must extend this class and provide an implementation of 
 * {@link #getProductName()} and {@link #getUserHome()}.
 * 
 * @since 1.0
 */
public abstract class IDE extends Service
{
  private static final Version EDK_VERSION = new Version( "1.0" );
  
  private final Collection _listeners = new ArrayList();
  
  /**
   * Get the IDE product name.
   *
   * @return The product name of the IDE provider. Must not be null.
   */
  public abstract String getProductName();
  
  /**
   * Get the main window. 
   * 
   * @return the main window. Must not be null.
   */
  public abstract View getMainWindow();
  
  /**
   * Get the currently active view.
   * 
   * @return the currently active view. Must not be null.
   */
  public abstract View getActiveView();

  /**
   * Get Extension Development Kit version number.
   *
   * @return The current Extension Software Development Kit version
   * supported by the IDE provider. For this version of the ESDK, this must be
   * 1.0
   */
  public Version getEDKVersion()
  {
    return EDK_VERSION;
  }

  /**
   * Get the directory where the user work is saved. This is the default
   * directory under which end user documents are saved. Extension use the
   * returned value to suggest the filepath where a new document is going to
   * be created when the user invokes a document creation wizard.
   *
   * @return The directory where the user work is stored. Must not be null, 
   *  must be a file: scheme URI.
   */
  public abstract URI getUserHome();

  /**
   * Extension that need to lookup an action use {@link ActionRegistry} service.
   * 
   * @return The service used to find registered actions. Must not be null.
   */
  public final ActionRegistry getActionRegistry()
  {
    return ActionRegistry.getActionRegistry();
  }
  
  /**
   * Get the {@link ExtensionRegistry}.
   *
   * @return The extension registry service. Must not be null.
   */
  public final ExtensionRegistry getExtensionRegistry()
  {
    return ExtensionRegistry.getExtensionRegistry();
  }
  
  /**
   * Get the {@link DocumentFactory}.
   * 
   * @return The document factory. Must not be null.
   */
   public final DocumentFactory getDocumentFactory()
   {
     return DocumentFactory.getDocumentFactory();
   }

  /**
   * Get the {@link EditorManager}.
   *
   * @return The editor manager. Must not be null.
   */
  public final EditorManager getEditorManager()
  {
    return EditorManager.getEditorManager();
  }

  /**
   * Get the {@link CommandProcessor}.
   *
   * @return The command processor. Must not be null.
   */
  public final CommandProcessor getCommandProcessor()
  {
    return CommandProcessor.getCommandProcessor();
  }

  /**
   * Get the {@link LogManager}.
   *
   * @return The log manager. Must not be null.
   */
  public final LogManager getLogManager()
  {
    return LogManager.getLogManager();
  }

  /**
   * Get the {@link WizardManager}.
   *
   * @return The wizard manager. Must not be null.
   */
  public final WizardManager getWizardManager()
  {
    return WizardManager.getWizardManager();
  }
  
  /**
   * Get the {@link Debugger}.
   *
   * @return the debugger. Must not be null.
   */
  public final Debugger getDebugger()
  {
    return Debugger.getDebugger();
  }
  
  /**
   * Get the {@link BuildSystem}.
   *
   * @return the build system. Must not be null.
   */
  public final BuildSystem getBuildSystem()
  {
    return BuildSystem.getBuildSystem();
  }

  /**
   * Get the {@link PropertyPageRegistry}.
   *
   * @return the property page registry. Must not be null.
   */
  public final PropertyPageRegistry getPropertyPageRegistry()
  {
    return PropertyPageRegistry.getPropertyPageRegistry();
  }
  
  /**
   * Get the {@link GUIUtilities}.
   * 
   * @return the GUI utilities. These utilities allow extension writers to use
   * standard IDE dialogs in reporting extension specific errors, warnings, 
   * etc.. Must not be null.
   */
  public final GUIUtilities getGUIUtilities()
  {
    return GUIUtilities.getGUIUtilities();
  }
  
  protected final void addIDEListener( IDEListener listener )
  {
    _listeners.add( listener );
  }
  
  /**
   * Fire an IDEActivated event to all registered listeners.
   */
  protected final void fireIDEActivated()
  {
    final List listeners = new ArrayList( _listeners );
    
    IDEEvent event = new IDEEvent( this );
    for ( Iterator i = listeners.iterator(); i.hasNext(); )
    {
      final IDEListener listener = (IDEListener) i.next();
      listener.activated( event );
    }
  }

  /**
   * Fire an IDEDeactivated event to all registered listeners.
   */
  protected final void fireIDEDeactivated()
  {
    final List listeners = new ArrayList( _listeners );

    IDEEvent event = new IDEEvent( this );
    for ( Iterator i = listeners.iterator(); i.hasNext(); )
    {
      final IDEListener listener = (IDEListener) i.next();
      listener.deactivated( event );
    }
  }  
  
  protected void initialize()
  {
    IDEListenerHook hook = (IDEListenerHook)
      ExtensionRegistry.getExtensionRegistry().getHook( IDEListenerHook.ELEMENT );
    
    Collection listeners = hook.getListeners();
    for (Iterator i = listeners.iterator(); i.hasNext();)
    {
      MetaClass mc = (MetaClass) i.next();
      
      try
      {
        IDEListener ideListener = (IDEListener) mc.newInstance();
        addIDEListener( ideListener );
      }
      catch ( Exception e )
      {
        e.printStackTrace();
      }
    }
  }
  
  /**
   * Get the ide implementation.
   * 
   * @return the ide. Must not be null.
   */
  public static final IDE getIDE()
  {
    try
    {
      return (IDE) getService( IDE.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No IDE service" );
    }
  }
}
