/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#include <stdlib.h>

#include <debug/sahtrace.h>
#include <pcb/core.h>

#include "player_intf.h"

static pcb_timer_t *playTimer = NULL;

static void playlist_next_song(pcb_timer_t *timer, void *userdata) {
    SAH_TRACE_INFO("Song done, play next from playlist");
    object_t *current_song = (object_t *)userdata;
    pcb_timer_setUserData(timer,NULL);
    player_stop();

    // fetch next item from playlist and delete current item from playlist
    object_t *playlist = object_parent(current_song);
    object_t *next_song = object_nextInstance(current_song);
    object_delete(current_song);
    object_commit(playlist);

    if (!next_song) {
        // no next song available
        SAH_TRACE_NOTICE("No next song availabe, stop playing");
        return;
    }
    object_t *jukebox = object_parent(object_parent(playlist));
    object_t *song = object_getObjectByKey(jukebox, "Songs.Song.%d", object_parameterUInt32Value(next_song,"Index"));

    // restart the timer
    pcb_timer_setUserData(timer, next_song);
    pcb_timer_start(timer,object_parameterUInt32Value(song, "Duration") * 1000);
    char *uri = object_parameterCharValue(song, "Uri");
    player_set_uri(uri);
    free(uri);
    player_play();
    SAH_TRACE_INFO("Next song playing");
}

bool pcb_plugin_initialize(pcb_t *pcb, int  __attribute__((unused)) argc, char  __attribute__((unused)) *argv[]) {
    SAH_TRACE_INFO("Starting plugin");

    // create the play timer
    playTimer = pcb_timer_create();
    pcb_timer_setHandler(playTimer, playlist_next_song);

    plugin_config_t *cfg = pcb_pluginConfig(pcb);
    player_init(cfg->pcb, cfg->system_bus);

    return true;
}

void pcb_plugin_cleanup() {
    SAH_TRACE_INFO("Stopping plugin");
    // cleanup timer
    pcb_timer_destroy(playTimer);
}

function_exec_state_t __songs(function_call_t *fcall, argument_value_list_t *args, variant_t *retval) {
    // no arguments expected,
    // just clear the arghument list in case the caller has added arguments
    // if the argument list is not cleared, all arguments will be sent back as out arguments
    argument_valueClear(args);

    // get the available songs
    object_t *songs = object_getObjectByKey(fcall_object(fcall), "Songs.Song");

    // iterate over the list of all the songs (instances), and add them to the reply
    // first initialize the return value as an array
    variant_initialize(retval, variant_type_array);
    variant_list_t *song_list = variant_da_list(retval);

    object_t *song = NULL;
    object_for_each_instance(song, songs) {
        // create a variant list iterator containing a variant map
        variant_list_iterator_t *item = variant_list_iterator_createMap(NULL);
        // fetch the variant map point from the iterator
        variant_map_t *item_data = variant_list_iterator_da_map(item);
        // fill the variant map with all the parameters of the song
        object_getValues(song, item_data);
        // remove the uri
        variant_map_iterator_t *uri = variant_map_find(item_data,"Uri");
        variant_map_iterator_take(uri);
        // add to return value
        variant_list_append(song_list,item);
    }

    return function_exec_done;
}

function_exec_state_t __nowPlaying(function_call_t *fcall, argument_value_list_t *args, variant_t *retval) {
    // no arguments expected,
    // just clear the arghument list in case the caller has added arguments
    // if the argument list is not cleared, all arguments will be sent back as out arguments
    argument_valueClear(args);

    // this function is called on the object "Jukebox"
    object_t *jukebox = fcall_object(fcall);

    // The song being played is the first instance in the Playlist.Song object
    object_t *playlist = object_getObjectByKey(jukebox, "Playlist.Song");
    object_t *nowPlaying = object_firstInstance(playlist);
    // The "Index" parameter is pointing to the instance of the "Song.Song" object
    uint32_t index = object_parameterUInt32Value(nowPlaying, "Index");

    // Use the retrieved index to find the song object
    object_t *song = object_getObjectByKey(jukebox,"Songs.Song.%d", index);

    // convert the song object to a return value
    // initialize the retval to variant_map
    variant_initialize(retval, variant_type_map);
    // fetch the variant map pointer out of the variant
    variant_map_t *return_map = variant_da_map(retval);
    // fill the map with all parameters of the song object
    object_getValues(song, return_map);
    // remove the uri
    variant_map_iterator_t *uri = variant_map_find(return_map, "Uri");
    variant_map_iterator_take(uri);

    if (song) {
        // if a song is playing, populate the variant map
        uint32_t timeleft = pcb_timer_remainingTime(playTimer) / 1000;
        variant_map_addUInt32(return_map, "TimeRemaining", timeleft);
    }

    return function_exec_done;
}


function_exec_state_t __Songs_addSong(function_call_t *fcall, argument_value_list_t *args, variant_t *retval) {
    uint32_t attr = request_attributes(fcall_request(fcall));

    char *title = NULL;
    uint32_t duration = -1;
    char *artist = NULL;
    char *genre = NULL;

    // fetch all arguments
    argument_getChar(&title, args, attr, "title", NULL);
    argument_getUInt32(&duration, args, attr, "duration", -1);
    argument_getChar(&artist, args, attr, "artist", "");
    argument_getChar(&genre,args, attr, "genre", "");

    // clear the arghument list in case the caller has added extra arguments
    // if the argument list is not cleared, all arguments will be sent back as out arguments
    argument_valueClear(args);

    // create the new song, and set the parameters
    object_t *songs = object_getObjectByKey(fcall_object(fcall), "Song");
    object_t *newsong = object_createInstance(songs, 0, NULL);
    if (!newsong) {
        // failed to add song
        // return an error
        SAH_TRACE_ERROR("Failed to add song %s, %d:%s", title, pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    object_parameterSetCharValue(newsong, "Title", title);
    object_parameterSetUInt32Value(newsong, "Duration", duration);
    object_parameterSetCharValue(newsong, "Artist", artist);
    object_parameterSetCharValue(newsong, "Genre", genre);

    // commit the changes, if it fails do a rollback and return an error
    if (!object_commit(songs)) {
        SAH_TRACE_ERROR("Failed to commit %d:%s", pcb_error, error_string(pcb_error));
        object_rollback(songs);
        // return an error
        goto exit_error;
    }

    free(title);
    free(artist);
    free(genre);
    variant_initialize(retval, variant_type_unknown); // the unknown type is used for the void return value
    return function_exec_done;

exit_error:
    free(title);
    free(artist);
    free(genre);
    return function_exec_error;
}

function_exec_state_t __Songs_deleteSong(function_call_t *fcall, argument_value_list_t *args, variant_t *retval) {
    uint32_t attr = request_attributes(fcall_request(fcall));

    uint32_t index = 0;

    // fetch the index
    argument_getUInt32(&index, args, attr, "index", 0);

    // clear the argument list
    argument_valueClear(args);

    // remove the song
    object_t *songs = fcall_object(fcall);
    object_t *song = object_getObjectByKey(songs, "Song.%d", index);
    if (!song) {
        // song not found, just return
        goto exit_done;
    }

    if (!object_delete(song)) {
        // failed to delete the song,
        // return an error
        SAH_TRACE_ERROR("Failed to delete song %s %d:%s", object_name(song,0), pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    if (!object_commit(songs)) {
        SAH_TRACE_ERROR("Failed to commit %d:%s", pcb_error, error_string(pcb_error));
        object_rollback(songs);
        // return an error
        goto exit_error;
    }

exit_done:
    variant_initialize(retval, variant_type_unknown); // the unknown type is used for the void return value
    return function_exec_done;

exit_error:
    return function_exec_error;
}

function_exec_state_t __Playlist_addSong(function_call_t *fcall, argument_value_list_t *args, variant_t *retval) {
    uint32_t attr = request_attributes(fcall_request(fcall));

    uint32_t index = 0;

    // fetch the index
    argument_getUInt32(&index, args, attr, "index", 0);

    // clear the argument list
    argument_valueClear(args);

    // verify that the index is available
    object_t *jukebox = object_parent(fcall_object(fcall));
    object_t *song = object_getObjectByKey(jukebox, "Songs.Song.%d", index);
    if (!song) {
        // song with given index not found
        // return error
        goto exit_error;
    }

    object_t *playlist = object_getObjectByKey(fcall_object(fcall), "Song");
    // verify that the playlist already has songs
    bool startPlaying = !object_hasInstances(playlist);

    // add item to the playlist
    object_t *item = object_createInstance(playlist, 0, NULL);
    if (!item) {
        // failed to add song
        // return an error
        SAH_TRACE_ERROR("Failed to add song with index %d, %d:%s", index,pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    object_parameterSetUInt32Value(item, "Index", index);
    // commit the changes, if it fails do a rollback and return an error
    if (!object_commit(playlist)) {
        SAH_TRACE_ERROR("Failed to commit %d:%s", pcb_error, error_string(pcb_error));
        object_rollback(playlist);
        // return an error
        goto exit_error;
    }

    if (startPlaying) {
        SAH_TRACE_INFO("Start playing song");
        // set the instance object as user data of the playing timer
        pcb_timer_setUserData(playTimer, item);
        // start the timer using the duration of the song
        pcb_timer_start(playTimer, object_parameterUInt32Value(song, "Duration") * 1000); // set the time in ms
        // give the uri to the player
        const char *uri = object_da_parameterCharValue(song, "Uri");
        player_set_uri(uri);
        player_play();
    }

    variant_initialize(retval, variant_type_unknown); // the unknown type is used for the void return value
    return function_exec_done;

exit_error:
    return function_exec_error;
}

bool song_on_delete(object_t *object, object_t *instance) {
    // verify that the instance can be deleted
    // it can only be deleted when not in the playlist

    object_t *jukebox = object_parent(object_parent(object));
    object_t *playlist = object_getObjectByKey(jukebox, "Playlist.Song");
    string_t pattern;
    string_initialize(&pattern, 0);

    // search all instances in the playlist that has index of the song that is being deleted.
    string_appendFormat(&pattern, "Jukebox.Playlist.Song.*.Index=%s", object_name(instance,0));
    object_list_t *list = object_findObjects(playlist, string_buffer(&pattern), -1, search_attr_include_parameters|search_attr_include_values);
    string_cleanup(&pattern);
    if (list && !llist_isEmpty(list)) {
        SAH_TRACE_ERROR("Song can not be deleted in the playlist (count = %d)", llist_size(list));
        // it is in the playlist, do not remove it
        llist_cleanup(list);
        // by returning false, the object will not be deleted
        return false;
    }

    // ok to delete
    llist_cleanup(list);
    return true;
}
