/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#include <signal.h>

#include <debug/sahtrace.h>

#include <pcb/pcb_client.h>

#include "common.h"

static pcb_t *pcb_client = NULL;

static peer_info_t *client_initialize(const char *uri) {
    // 1. initialize connection library (eventsink)
    if (!connection_initLibrary()) {
        SAH_TRACE_ERROR("Failed to initialize event handling, exiting");
        goto exit_error;
    }

    // 2. Create a pcb context
    pcb_client = pcb_create("simple_client", 0, NULL);
    if (!pcb_client) {
        SAH_TRACE_ERROR("Failed to create pcb context (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    // 3. connect to argument specified in uri - should be a pcb uri
    peer_info_t *destination = pcb_client_connect(pcb_client, uri);
    if (!destination) {
        SAH_TRACE_ERROR("Failed to connect to %s (0x%8.8X - %s), exiting", uri, pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    return destination;

exit_error:
    // destroy pcb context
    pcb_destroy(pcb_client);
    pcb_client = NULL;
    // cleanup library
    connection_exitLibrary();

    return NULL;
}

int main(int argc, char *argv[]) {
    peer_info_t *destination = NULL;

    // initialize tracing: use app name, and trace to stderror
    sahTraceOpen(argv[0], TRACE_TYPE_STDERR);
    // set trace level to level warning (level 200)
    sahTraceSetLevel(TRACE_LEVEL_WARNING);

    // 1. verify arguments:
    // usage: pcb_simple_client URI OBJECT_PATH
    if (argc < 3) {
        SAH_TRACE_ERROR("Missing arguments, exiting");
        printf("Usage %s URI OBJECT_PATH\n",argv[0]);
        goto exit;
    }

    // 2. Initialize the client and pcb context
    destination = client_initialize(argv[1]);
    if (!destination) {
        goto exit;
    }

    // 3. get the object specified in argv[2]
    object_t *object = pcb_client_get_object(destination, argv[2]);
    if (!object) {
        SAH_TRACE_ERROR("Failed to get object to %s (0x%8.8X - %s), exiting",
                                                            argv[2],
                                                            pcb_client_error(pcb_client),
                                                            error_string(pcb_client_error(pcb_client)));
        goto exit_error;
    }

    // print the object
    print_object(object);

    // destroy pcb context
    pcb_destroy(pcb_client);
    connection_exitLibrary();
    return 0;

exit_error:
    // destroy pcb context
    pcb_destroy(pcb_client);
    // cleanup library
    connection_exitLibrary();
exit:
    // close tracing
    sahTraceClose();
    return -1;
}
