/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#include <signal.h>

#include <debug/sahtrace.h>

#include <pcb/core.h>
#include <pcb/utils.h>
#include <pcb/pcb_client.h>

#include "common.h"

static pcb_t *pcb_client = NULL;

static void client_signalHandler(int __attribute__((unused)) signal) {
    connection_setTerminated(true);
}

bool client_handle_reply_item(request_t __attribute__((unused)) *req,
                       reply_item_t *item,
                       pcb_t __attribute__((unused)) *pcb,
                       peer_info_t __attribute__((unused)) *from,
                       void __attribute__((unused)) *userdata) {
    SAH_TRACE_IN();

    switch(reply_item_type(item)) {
        case reply_type_object:
            print_object(reply_item_object(item));
        break;
        case reply_type_notification:
            print_notification(reply_item_notification(item));
        break;
        case reply_type_error:
            SAH_TRACE_ERROR("Request failed, exiting - error - %d - %s - %s",reply_item_error(item),
                                                                             reply_item_errorDescription(item),
                                                                             reply_item_errorInfo(item));
            connection_setTerminated(true);
            SAH_TRACE_OUT();
            return false;
        break;
        default:
        break;
    }

    SAH_TRACE_OUT();
    return true;
}

static peer_info_t *client_initialize(const char *uri) {
    // 1. initialize connection library (eventsink)
    if (!connection_initLibrary()) {
        SAH_TRACE_ERROR("Failed to initialize event handling, exiting");
        goto exit_error;
    }

    // 2. Create a pcb context
    pcb_client = pcb_create("simple_client", 0, NULL);
    if (!pcb_client) {
        SAH_TRACE_ERROR("Failed to create pcb context (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    // 3. set signal handlers
    connection_setSignalEventHandler(pcb_connection(pcb_client), SIGINT, client_signalHandler);
    connection_setSignalEventHandler(pcb_connection(pcb_client), SIGTERM, client_signalHandler);
    connection_setSignalEventHandler(pcb_connection(pcb_client), SIGPIPE, NULL); // ignore sigpipe

    // 4. connect to argument specified in uri - should be a pcb uri
    peer_info_t *destination = pcb_client_connect(pcb_client, uri);
    if (!destination) {
        SAH_TRACE_ERROR("Failed to connect to %s (0x%8.8X - %s), exiting", uri, pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    return destination;

exit_error:
    // destroy pcb context
    pcb_destroy(pcb_client);
    pcb_client = NULL;
    // cleanup library
    connection_exitLibrary();

    return NULL;
}

static request_t *client_create_send_request(peer_info_t *destination, const char *object_path) {
    // 1. create the request.
    request_t *request = request_create_getObject(object_path,                      // object path
                                                  -1,                               // request depth
                                                  request_getObject_all |           // request flags
                                                  request_notify_all |
                                                  request_no_object_caching);
    if (!request) {
        SAH_TRACE_ERROR("Failed to create request (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    // 2. Set the reply handler
    request_setReplyItemHandler(request, client_handle_reply_item);

    // 3. Send the request
    if (!pcb_sendRequest(pcb_client, destination, request)) {
        SAH_TRACE_ERROR("Failed to send request (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error_free_request;
    }

    return request;

exit_error_free_request:
    request_destroy(request);

exit_error:
    return NULL;
}

static void client_eventloop() {
    int retval = 0;

    SAH_TRACE_NOTICE("Entered event loop");

    // add all event sinks (connection_info_t) to a list
    llist_t cons;
    llist_initialize(&cons);
    llist_append(&cons, connection_getIterator(pcb_connection(pcb_client)));

    fd_set readset;
    fd_set writeset;

    while(!connection_isTerminated()) {
        // wait for an event on one of the event sinks
        retval = connection_waitForEvents_r(&cons, NULL, &readset, &writeset);
        if (retval < 0) {
            // an error has happened
            SAH_TRACE_ERROR("Error %d (%s)", pcb_error, error_string(pcb_error));
            // ignore the error and continue
            continue;
        }

        if (retval > 0) {
            // one or more events available in one of the event sinks
            // handle the events
            connection_handleEvents_r(&cons, &readset, &writeset);
        }

        if (retval == 0) {
            // Time out happened. This code should not be executed
            // no timeout is specfied in the connection_waitForEvents_r function
            SAH_TRACE_WARNING("Timeout !!");
        }
    }

    // cleanup the list of event sinks
    llist_cleanup(&cons);

    SAH_TRACE_NOTICE("Leaving event loop");

    return;
}

int main(int argc, char *argv[]) {
    peer_info_t *destination = NULL;
    request_t *request = NULL;
    int retval = 0;

    // initialize tracing: use app name, and trace to stderror
    sahTraceOpen(argv[0], TRACE_TYPE_STDERR);
    // set trace level to level warning (level 200)
    sahTraceSetLevel(TRACE_LEVEL_INFO);

    // 1. verify arguments:
    // usage: pcb_evented_client URI OBJECT_PATH
    if (argc < 3) {
        SAH_TRACE_ERROR("Missing arguments, exiting");
        printf("Usage %s URI OBJECT_PATH\n",argv[0]);
        retval = -1;
        goto exit;
    }

    // 2. Initialize the client and pcb context
    destination = client_initialize(argv[1]);
    if (!pcb_client) {
        retval = -1;
        goto exit;
    }

    // 3. create and send the request.
    request = client_create_send_request(destination, argv[2]);
    if (!request) {
        retval = -1;
        goto exit_cleanup;
    }

    // 4. start the event loop
    client_eventloop();

    // application stopped - start cleanup
    // destroy the request
    request_destroy(request);
exit_cleanup:
    // destroy pcb context
    pcb_destroy(pcb_client);
    // cleanup library
    connection_exitLibrary();
exit:
    // close tracing
    sahTraceClose();
    return retval;
}
