/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#if !defined(PCB_VALIDATOR_H)
#define PCB_VALIDATOR_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/core/types.h>

/**
@ingroup pcb_core
@file
@brief
Header file with validator type definitions and public validator functions
*/

/**
@ingroup pcb_core
@defgroup pcb_core_validator Validators
@{

@brief
Validator implementation

@details
Objects and parameters can contain a validator. The validator is used to
validate the value(s) and check constraints.

@section pcb_core_validator_standard Standard Validators
A small set of standard parameter validators is provided.\n
    - Enumeration validator: Checks that the value of a parameter is in a set of values
    - Range validator: Checks that the value of a parameter is within a certain range, for string parameters it checks that the length of the string is in this range.
    - Minimum validator: Checks that the value of a parameter is at least this value, for string parameters it checks that the length of the string is at least this value
    - Maximum validator: Checks that the value of a parameter is not higher than this value, for strings parameters it checks that the length of strings is not bigger then this value.
For each of these standard validators a param_validator_create function is available.\n

@section pcb_core_validator_custom Custom Validators
Custom validators can be created as well and not only for parameters but also for objects. To create custom parameter validator
use the function @ref param_validator_create_custom and to create a custom object validator use the @ref object_validator_create function.
You also need to write a callback function that is doing the real validation. Such a validation function must return true if validation was successful otherwise false.
Extra validation data can be provided when creating the validator and this data will be used when calling the validation function.

@section pcb_core_validator_add Adding Validators
Adding a validator to an object or parameter is done by using the @ref parameter_setValidator or @ref object_setValidator
*/

/**
@brief
Enumeration of the possible validator types.

@details

*/
typedef enum _parameter_validator_type {
    parameter_validator_custom = 0,  /**< A custom validator */
    parameter_validator_enum,        /**< The enum validator */
    parameter_validator_range,       /**< The range validator */
    parameter_validator_minimum,     /**< The minimum validator */
    parameter_validator_maximum,     /**< The maximum validator */
} parameter_validator_type_t;

/**
@brief
Parameter validation function definition

@details
Custom parameter validation functions must match this function signature.
Such a function must return false when the validation fails.
*/
typedef bool (* param_validation_handler_t) (parameter_t *parameter, void *validationData);

/**
@brief
Object validation function definition

@details
Custom object validation functions must match this function signature.
Such a function must return false when the validation fails.
*/
typedef bool (* object_validation_handler_t) (object_t *object, void *validationData);

/**
@}
*/

parameter_validator_t *param_validator_create_custom(param_validation_handler_t handler, void *validationData);
parameter_validator_t *param_validator_create_enum(variant_list_t *values);
parameter_validator_t *param_validator_create_range(variant_t *minimum, variant_t* maximum);
parameter_validator_t *param_validator_create_minimum(variant_t *minimum);
parameter_validator_t *param_validator_create_maximum(variant_t *maximum);

void param_validator_destroy(parameter_validator_t *validator);

object_validator_t *object_validator_create(object_validation_handler_t handler, void *validationData);
void object_validator_destroy(object_validator_t *validator);

parameter_validator_type_t param_validator_type(parameter_validator_t *validator);

const variant_t *param_validator_minimum(parameter_validator_t *validator);
const variant_t *param_validator_maximum(parameter_validator_t *validator);
variant_list_t *param_validator_values(parameter_validator_t *validator);
void *param_validator_data(parameter_validator_t *validator);

#ifdef __cplusplus
}
#endif

#endif
