/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#ifndef _PCB_CORE_REQUEST_PARAMETER_H
#define _PCB_CORE_REQUEST_PARAMETER_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/core/request.h>
#include <pcb/utils/variant.h>

/**
@brief
Set a bool request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a bool.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addBoolParameter(request_t *req, const char *name, bool value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setBool(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a int8_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a int8_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addInt8Parameter(request_t *req, const char *name, int8_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setInt8(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a int16_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a int16_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addInt16Parameter(request_t *req, const char *name, int16_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setInt16(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a int32_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a int32_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addInt32Parameter(request_t *req, const char *name, int32_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setInt32(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a int64_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a int64_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addInt64Parameter(request_t *req, const char *name, int64_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setInt64(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a uint8_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a uint8_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addUInt8Parameter(request_t *req, const char *name, uint8_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setUInt8(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a uint16_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a uint16_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addUInt16Parameter(request_t *req, const char *name, uint16_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setUInt16(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a uint32_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a uint32_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addUInt32Parameter(request_t *req, const char *name, uint32_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setUInt32(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a uint64_t request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a uint64_t.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addUInt64Parameter(request_t *req, const char *name, uint64_t value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setUInt64(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a double request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a double.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addDoubleParameter(request_t *req, const char *name, double value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setDouble(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a char * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a char *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addCharParameter(request_t *req, const char *name, const char * value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setChar(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a struct tm * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a struct tm *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addDateTimeParameter(request_t *req, const char *name, const struct tm * value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setDateTime(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a string_t * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a string_t *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addStringParameter(request_t *req, const char *name, const string_t * value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setString(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a variant_list_t * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a variant_list_t *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addListParameter(request_t *req, const char *name, variant_list_t * value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setListMove(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a variant_map_t * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a variant_map_t *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addMapParameter(request_t *req, const char *name, variant_map_t * value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setMapMove(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a int request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a int.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addFdParameter(request_t *req, const char *name, int value)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setFd(&v, value);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a void * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a void *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addByteArrayParameter(request_t *req, const char *name, const void * value, uint32_t size)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setByteArray(&v, value, size);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

/**
@brief
Set a string_list_t * request parameter

@details
This function is provided for convenience. It is setting a parameter in a request using a string_list_t *.

@param request the request of interest
@param name the name of the parameter to set
@param value the value of the parameter to set

@return
    - The converted value
*/
static inline bool request_addStringListParameter(request_t *req, const char *name, string_list_t * value, const char *separator)
{
    bool rc;
    variant_t v;
    variant_initialize(&v, variant_type_unknown);
    rc = variant_setStringList(&v, value, separator);
    if (rc)
        rc = request_addParameter(req, name, &v);
    variant_cleanup(&v);
    return rc;
}

#ifdef __cplusplus
}
#endif
#endif
