/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#if !defined(PCB_NOTIFICATION_H)
#define PCB_NOTIFICATION_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/utils/variant.h>
#include <pcb/core/types.h>

/**
@ingroup pcb_core
@file
@brief
Header file with notification type definitions and public notification functions
*/

/**
@ingroup pcb_core
@defgroup pcb_core_notification Notifications
@{

@brief
Notification implementation

@details
Notifications can be used as a signaling mechanism. Before a server can send a notification, at least one connection
to a bus or client must be established.\n

@section pcb_core_notification_standard Standard Notifications
Some standard notifications are defined, some of them are related to objects and data models, while others are mnore global.
    - Value changed: Send when a parameter has changed value
    - Object added: Send when an instance is added to a template object
    - Object deleted: Send when an instance is deleted.
    - Application started: Send by a server when it is ready to recieve requests.

@section pcb_core_notification_custom Custom Notifications
Custom notifications can be created and send to any client/server.\n

@section pcb_core_notification_parameter Notification Parameters
Each notification can contain zero, one or more parameters. These parameters exist out of two parts, the name and a value.
*/

/**
@brief
Enumeration of the notification types
*/
typedef enum _notification_types {
    // data model notifications
    notify_value_changed = 1,         /**< Value changed notification identifier */
    notify_object_added,              /**< Object added notification identifier */
    notify_object_deleted,            /**< Object deleted notification identifier */
    // application notifications
    notify_application_started = 10,  /**< Application started notification identifier */
    notify_application_stopping ,     /**< Application stopping notification identifier */
    notify_application_change_setting,/**< Application needs to change a setting identifier */
    notify_application_root_added,
    notify_application_root_deleted,
    // bus notifications
    notify_bus_interconnect = 20,     /**< Bus interconnect notification */
    // custom notifications
    notify_custom = 100,              /**< Custom notification base identifier */
} notification_type_t;

/**
@}
*/

notification_t *notification_create(uint32_t type);

notification_t *notification_createValueChanged(object_t *object, parameter_t *parameter);
notification_t *notification_createObjectAdded(object_t *object);
notification_t *notification_createObjectDeleted(object_t *object);

notification_t *notification_createApplicationStarted(const char *appName);
notification_t *notification_createApplicationStopping(const char *appName);
notification_t *notification_createApplicationChangeSetting(const char *settingName, const char *settingValue);
notification_t *notification_createApplicationRootAdded(const char *object_name);
notification_t *notification_createApplicationRootDeleted(const char *object_name);
notification_t *notification_createBusInterconnect(const char *appName, bool sync);

void notification_destroy(notification_t *notification);

uint32_t notification_type(const notification_t *notification);

bool notification_setName(notification_t *notification, const char *name);
const char *notification_name(notification_t *notification);

bool notification_setObjectPath(notification_t *notification, const char *objectpath);
const char *notification_objectPath(notification_t *notification);
const char *notification_objectName(notification_t *notification);

bool notification_addParameter(notification_t *notification, notification_parameter_t *param);

uint32_t notification_parameterCount(notification_t *notification);
notification_parameter_t *notification_firstParameter(const notification_t *notification);
notification_parameter_t *notification_nextParameter(notification_parameter_t *param);
notification_parameter_t *notification_prevParameter(notification_parameter_t *param);
notification_parameter_t *notification_lastParameter(notification_t *notification);

notification_parameter_t *notification_getParameter(const notification_t *notification, const char *name);

notification_parameter_t *notification_parameter_create(const char *name, const char *value);
notification_parameter_t *notification_parameter_createVariant(const char *name, const variant_t *value);
void notification_parameter_destroy(notification_parameter_t *param);

bool notification_parameter_setName(notification_parameter_t *param, const char *name);
bool notification_parameter_setValue(notification_parameter_t *param, const char *value);
bool notification_parameter_setVariant(notification_parameter_t *param, const variant_t *value);

const char *notification_parameter_name(notification_parameter_t *param);
char *notification_parameter_value(notification_parameter_t *param);
variant_t *notification_parameter_variant(notification_parameter_t *param);

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter value

@details
Get the notification parameter value.

@param n the notification of interest
@param name the name of the parameter to fetch

@return
    - The notification parameter value
*/
static inline variant_t *notification_da_parameterValue(const notification_t *n, const char *name)
{
    return notification_parameter_variant(notification_getParameter(n, name));
}

#include <pcb/core/notification_parameter.h>

#ifdef __cplusplus
}
#endif

#endif

