/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#if !defined(PCB_MAPPING_H)
#define PCB_MAPPING_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/utils/linked_list.h>
#include <pcb/utils/uri.h>
#include <pcb/utils/connection.h>

#include <pcb/core/parameter.h>

typedef enum _rule_type {
    rule_type_object = 1,
    rule_type_parameter,
    rule_type_parameter_request,
    rule_type_parameter_object,
    rule_type_function,
} rule_type_t;

typedef enum _rule_action {
    rule_action_accept = 1,
    rule_action_drop,
} rule_action_t;

/**
@brief
Parameter translate value handler

@details
A callback function can be added to the parameter which is called when the parameter value needs to be translated.
This kind of handler is mainly used by mapping plug-ins, so they can create a view on certain parameters.
*/
typedef bool (* parameter_translate_value_handler_t) (translate_direction_t direction, variant_t *client, variant_map_t *plugin);

/**
@brief
Parameter translate name handler

@details
A callback function can be added to mapping rules which is called when the parameter name needs to be translated.
This kind of handler is mainly used by mapping plug-ins, so they can create a view on certain parameters.
*/
typedef void (* parameter_translate_name_handler_t) (translate_direction_t direction, string_t *clientName, string_t *pluginName);

/**
@brief
Parameter translate type handler

@details
A callback function can be added to mapping rules which is called when the parameter type needs to be translated.
This kind of handler is mainly used by mapping plug-ins, so they can create a view on certain parameters.
*/
typedef void (* parameter_translate_type_handler_t) (translate_direction_t direction, parameter_t *param);

// object destinations (for mapping)
bool object_hasDestinations(object_t *object);
uint32_t object_destinationCount(object_t *object);
object_destination_t *object_map(object_t *object, const char *uri, bool recursive);
object_destination_t *object_addDestination(object_t *object, const char *uri);
object_destination_t *object_firstDestination(object_t *object);
object_destination_t *object_nextDestination(object_t *object, object_destination_t *reference);

bool request_translate(request_t *orig_req, request_t *trans_req, object_destination_t *dest);
bool object_translate(object_t *object, object_destination_t *dest);

object_destination_t *parameter_destination(parameter_t *parameter);
bool parameter_addMap(parameter_t *parameter, const char *destparam, variant_t *value);
bool parameter_setTranslator(parameter_t *parameter, parameter_translate_value_handler_t translatefn);

// destination management
void destination_delete(object_destination_t *dest);
bool destination_addParameter(object_destination_t *dest, const char *clientName, const char *destName);
bool destination_addFunction(object_destination_t *dest, const char *clientName, const char *destName);
peer_info_t *destination_peer(object_destination_t *dest);
const uri_t *destination_URI(object_destination_t *dest);
bool destination_setURI(object_destination_t *dest, const char *uri);

mapping_rule_t *destination_createRule(object_destination_t *dest, rule_type_t type, rule_action_t action);
mapping_rule_t *rule_createRule(mapping_rule_t *parent, rule_type_t type, rule_action_t action);
bool rule_setRegExp(mapping_rule_t *rule, const char *regexp);
bool rule_addModifierRename(mapping_rule_t *rule, parameter_translate_name_handler_t renameHandler);
bool rule_addModifierTranslate(mapping_rule_t *rule, parameter_translate_value_handler_t translateHandler);
bool rule_addModifierCast(mapping_rule_t *rule, parameter_translate_type_handler_t castHandler);

mapping_rule_t *rule_parent(mapping_rule_t *rule);

void rule_delete(mapping_rule_t *rule) ;

#ifdef __cplusplus
}
#endif

#endif
