/****************************************************************************
**
** Copyright (C) 2009 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** This file is part of the sah trace library.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#if !defined(__SAHTRACE_H__)
#define __SAHTRACE_H__

#ifdef __cplusplus
extern "C"
{
#endif

/**
@file

@brief
SAH Trace interface definition file
*/

/**
@defgroup sah_trace SAH Tracing Facility
@{

@brief
Soft At Home's Tracing Facility

@details
@section sah_trace_init Initializing the tracing facility
Typically in each of your applications that uses SAH libraries you will initialize the tracing functionality.
This is done by calling @ref sahTraceOpen, followed by @ref sahTraceSetLevel.\n
Also trace zones can be specified using @ref sahTraceAddZone\n
When @ref sahTraceClose is called, no more traces will be outputted. The global trace level can be changed
at any given time using @ref sahTraceSetLevel. \n

@section sah_trace_zones Trace Zones
Zones are used as a filter. To see trace messages from a certain zone, the zone must be activated.
This can be done by calling one or more times the @ref sahTraceAddZone function.\n

@section sah_trace_messages Providing Trace Messages
Use the SAH_TRACE_XXX macros to output a trace message to the output "device".\n
If you want to output a message for a specific zone use the SAH_TRACEZ_XXX macros.

@section sah_trace_threading SAH Trace and threads
Non of the provided functions are thread safe.

@section sah_trace_compiling SAH Trace at compile time
If you want to have sah traces enabled in your application or library, compile your application
or library with SAH_TRACES_ENABLED defined (-DSAHTRACES_ENABLED).\n
Optionally you can specify at compile time the maximum trace level by definig the SAHTRACES_LEVEL
(-DSAHTRACES_LEVEL=maxlevel).

*/

#include <stdarg.h>
#include <components.h>

/**
@brief
Device that has to be opened

@details
Provide one of these values to the @ref sahTraceOpen function
*/
typedef enum
{
  TRACE_TYPE_SYSLOG, /**< Output to the system log*/
  TRACE_TYPE_STDOUT, /**< Output to the stdout */
  TRACE_TYPE_STDERR, /**< Output to the stderr */
  TRACE_TYPE_FILE,   /**< Output to a file */
} sah_trace_type;

/**
@brief
Predefined trace levels

@details
These can be used in all trace functions and macros
*/
typedef enum
{
  TRACE_LEVEL_FATAL_ERROR = 0, /**< Fatal error, it is not possible to turn these messages off */
  TRACE_LEVEL_ERROR = 100,     /**< Error message */
  TRACE_LEVEL_WARNING = 200,   /**< Warning message */
  TRACE_LEVEL_NOTICE = 300,    /**< Notice message */
  TRACE_LEVEL_APP_INFO = 350,  /**< Application info message */
  TRACE_LEVEL_INFO = 400,      /**< Info message (normally used by libraries) */
  TRACE_LEVEL_CALLSTACK = 500  /**< Callstack messages */
} sah_trace_level;

/**
@brief
Timestamp format

@details
Predefined timestamp formats
*/
typedef enum
{
  TRACE_TIME_SYSTEM_SECONDS,   /**< System time in seconds.milliseconds */
  TRACE_TIME_APP_SECONDS,      /**< Application time in seconds.milliseconds. The aplication time is relative to the time the application has been started */
  TRACE_TIME_SYSTEM_TIME,      /**< System time in human readable form (HH:MM:SS) */
} sah_trace_timeformat;

typedef struct _sah_trace_zone sah_trace_zone_it;

/**
  @}
*/

// deprecated functions
void sahTraceInitialize(const char *ident);

// tracing function
void sahTraceOpen(const char *ident,sah_trace_type type);
void sahTraceOpenFile(const char *ident,const char *file);
void sahTraceClose(void);
int sahTraceIsOpen(void);

void sahTraceSetTimeFormat(sah_trace_timeformat format);

void sahTraceSetLevel(int tracelevel);
int sahTraceLevel(void);

const char *sahTraceIdentifier(void);

void sahTraceAddZone(int lvl,const char *name);
void sahTraceRemoveZone(const char *name);
void sahTraceRemoveAllZones(void);

sah_trace_zone_it *sahTraceFirstZone(void);
sah_trace_zone_it *sahTraceNextZone(sah_trace_zone_it *ref);
sah_trace_zone_it *sahTraceGetZone(const char *zone);
const char *sahTraceZoneName(sah_trace_zone_it *zone);
sah_trace_level sahTraceZoneLevel(sah_trace_zone_it *zone);
void sahTraceZoneSetLevel(sah_trace_zone_it *zone, int lvl);

sah_trace_type sahTraceType(void);

#if !defined(NO_DOXYGEN) // doxygen can not handle the attribute
void sahTrace(int tracelevel, const char *format, ...) __attribute__ ((format (printf,2,3)));
void sahTraceZ(int tracelevel, const char *zone, const char *format, ...) __attribute__ ((format (printf,3,4)));
#else
void sahTrace(int tracelevel, const char *format, ...);
void sahTraceZ(int tracelevel, const char *zone, const char *format, ...);
#endif
void sahTracev(int tracelevel, const char *format, va_list ap);
void sahTracevZ(int tracelevel, const char *zone, const char *format, va_list ap);

#define SAHTRACE_SOURCE_FORMAT       " - %s(%s@%s:%d)%s"
#define SAHTRACE_ZONE_FORMAT         "%s%-7.7s%s - "
#define SAHTRACE_MARK_FORMAT         "%s[%s]%s%s"

#define SAHTRACE_SOURCE              SAHTRACE_RESET, SAHTRACE_LBLUE, __FUNCTION__, __FILE__, __LINE__, SAHTRACE_RESET
#define SAHTRACE_ZONE(zone)          SAHTRACE_BLUE, zone, SAHTRACE_RESET
#define SAHTRACE_NO_ZONE             SAHTRACE_BLUE, "", SAHTRACE_RESET

// Color schemes are defined here
#if defined(CONFIG_SAH_TRACE_COLORS)|| defined(CONFIG_SAH_TRACE_COLORS_SIMPLE)
#define SAHTRACE_RESET   (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[0m":""
#define SAHTRACE_RED     (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[31m":""
#define SAHTRACE_LRED    (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[31;1m":""
#define SAHTRACE_YELLOW  (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[33;1m":""
#define SAHTRACE_GREEN   (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[32m":""
#define SAHTRACE_LGREEN  (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[32;1m":""
#define SAHTRACE_BLUE    (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[34m":""
#define SAHTRACE_LBLUE   (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[36m":""
#define SAHTRACE_BRIGHT  (sahTraceType() != TRACE_TYPE_SYSLOG && sahTraceType() != TRACE_TYPE_FILE)?"\033[1m":""
#else
#define SAHTRACE_RESET   ""
#define SAHTRACE_RED     ""
#define SAHTRACE_LRED    ""
#define SAHTRACE_YELLOW  ""
#define SAHTRACE_GREEN   ""
#define SAHTRACE_LGREEN  ""
#define SAHTRACE_BLUE    ""
#define SAHTRACE_LBLUE   ""
#define SAHTRACE_BRIGHT  ""
#endif

#if defined(CONFIG_SAH_TRACE_COLORS)
#define SAHTRACE_NEUTRAL             SAHTRACE_LBLUE, "-", SAHTRACE_RESET, SAHTRACE_RESET
#define SAHTRACE_FATAL_ERROR         SAHTRACE_RED, "X", SAHTRACE_RESET, SAHTRACE_RED
#define SAHTRACE_ERROR               SAHTRACE_LRED, "x", SAHTRACE_RESET, SAHTRACE_LRED
#define SAHTRACE_WARNING             SAHTRACE_YELLOW, "!", SAHTRACE_RESET, SAHTRACE_YELLOW
#define SAHTRACE_NOTICE              SAHTRACE_LGREEN, "n", SAHTRACE_RESET, SAHTRACE_LGREEN
#define SAHTRACE_APP_INFO            SAHTRACE_GREEN, "I", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_INFO                SAHTRACE_GREEN, "i", SAHTRACE_RESET, SAHTRACE_BRIGHT
#else
#if defined(CONFIG_SAH_TRACE_COLORS_SIMPLE)
#define SAHTRACE_NEUTRAL             SAHTRACE_LBLUE, "-", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_FATAL_ERROR         SAHTRACE_RED, "X", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_ERROR               SAHTRACE_LRED, "x", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_WARNING             SAHTRACE_YELLOW, "!", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_NOTICE              SAHTRACE_LGREEN, "n", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_APP_INFO            SAHTRACE_GREEN, "I", SAHTRACE_RESET, SAHTRACE_BRIGHT
#define SAHTRACE_INFO                SAHTRACE_GREEN, "i", SAHTRACE_RESET, SAHTRACE_BRIGHT
#else
#define SAHTRACE_NEUTRAL             "", "-", "", ""
#define SAHTRACE_FATAL_ERROR         "", "X", "", ""
#define SAHTRACE_ERROR               "", "x", "", ""
#define SAHTRACE_WARNING             "", "!", "", ""
#define SAHTRACE_NOTICE              "", "n", "", ""
#define SAHTRACE_APP_INFO            "", "I", "", ""
#define SAHTRACE_INFO                "", "i", "", ""
#endif
#endif

#define SAH_TRACE_DO_NOTHING

#define SAH_TRACE_FATAL(format, ...) sahTrace(TRACE_LEVEL_FATAL_ERROR, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s",SAHTRACE_NO_ZONE, SAHTRACE_FATAL_ERROR, ##__VA_ARGS__, SAHTRACE_SOURCE)

#if defined(SAHTRACES_ENABLED)

#if !defined(SAHTRACES_LEVEL)
#define SAHTRACES_LEVEL SAHTRACES_LEVEL_DEFAULT
#endif

#define SAH_TRACE(lvl,format,...) sahTrace(lvl, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_NEUTRAL, ##__VA_ARGS__, SAHTRACE_SOURCE)
#define SAH_TRACEZ(lvl,zone,format,...) sahTraceZ(lvl,zone, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_NEUTRAL, ##__VA_ARGS__, SAHTRACE_SOURCE)

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 100)
#define SAH_TRACE_ERROR(format, ...) sahTrace(TRACE_LEVEL_ERROR, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_ERROR, ##__VA_ARGS__, SAHTRACE_SOURCE)
#define SAH_TRACEZ_ERROR(zone,format, ...) sahTrace(TRACE_LEVEL_ERROR, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_ERROR, ##__VA_ARGS__, SAHTRACE_SOURCE)
#else
#define SAH_TRACE_ERROR(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_ERROR(zone, format, ...) SAH_TRACE_DO_NOTHING
#endif

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 200)
#define SAH_TRACE_WARNING(format, ...) sahTrace(TRACE_LEVEL_WARNING,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_WARNING, ##__VA_ARGS__, SAHTRACE_SOURCE)
#define SAH_TRACEZ_WARNING(zone,format, ...) sahTrace(TRACE_LEVEL_WARNING,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_WARNING, ##__VA_ARGS__, SAHTRACE_SOURCE)
#else
#define SAH_TRACE_WARNING(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_WARNING(zone,format, ...) SAH_TRACE_DO_NOTHING
#endif

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 300)
#define SAH_TRACE_NOTICE(format, ...) sahTrace(TRACE_LEVEL_NOTICE,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s",SAHTRACE_ZONE(""), SAHTRACE_NOTICE, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#define SAH_TRACEZ_NOTICE(zone,format, ...) sahTraceZ(TRACE_LEVEL_NOTICE,zone,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_NOTICE, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#else
#define SAH_TRACE_NOTICE(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_NOTICE(zone,format, ...) SAH_TRACE_DO_NOTHING
#endif

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 350)
#define SAH_TRACE_APP_INFO(format, ...) sahTrace(TRACE_LEVEL_APP_INFO, "%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_APP_INFO, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#define SAH_TRACEZ_APP_INFO(zone,format, ...) sahTraceZ(TRACE_LEVEL_APP_INFO,zone,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_APP_INFO, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#else
#define SAH_TRACE_APP_INFO(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_APP_INFO(zone,format, ...) SAH_TRACE_DO_NOTHING
#endif

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 400)
#define SAH_TRACE_INFO(format, ...) sahTrace(TRACE_LEVEL_INFO,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_INFO, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#define SAH_TRACEZ_INFO(zone,format, ...) sahTraceZ(TRACE_LEVEL_INFO,zone,"%s%-7.7s%s - %s[%s]%s%s" format "%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_INFO, ##__VA_ARGS__ , SAHTRACE_SOURCE)
#else
#define SAH_TRACE_INFO(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_INFO(zone,format, ...) SAH_TRACE_DO_NOTHING
#endif

#if !defined(SAHTRACES_LEVEL) || (SAHTRACES_LEVEL >= 500)
#define SAH_TRACE_IN() sahTrace(TRACE_LEVEL_CALLSTACK,"%s%-7.7s%s - >>>>>>>>>>%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_SOURCE)
#define SAH_TRACE_OUT() sahTrace(TRACE_LEVEL_CALLSTACK,"%s%-7.7s%s - <<<<<<<<<<%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(""), SAHTRACE_SOURCE)
#define SAH_TRACEZ_IN(zone) sahTraceZ(TRACE_LEVEL_CALLSTACK,zone,"%s%-7.7s%s - >>>>>>>>>>%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_SOURCE)
#define SAH_TRACEZ_OUT(zone) sahTraceZ(TRACE_LEVEL_CALLSTACK,zone,"%s%-7.7s%s - <<<<<<<<<<%s - %s(%s@%s:%d)%s", SAHTRACE_ZONE(zone), SAHTRACE_SOURCE)
#else
#define SAH_TRACE_IN() SAH_TRACE_DO_NOTHING
#define SAH_TRACE_OUT() SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_IN(zone) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_OUT(zone) SAH_TRACE_DO_NOTHING
#endif

#else

#if !defined(SAHTRACES_LEVEL)
#define SAHTRACES_LEVEL 0
#endif

#define SAH_TRACE(lvl,format,...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_ERROR(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_WARNING(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_NOTICE(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_APP_INFO(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_INFO(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACE_IN() SAH_TRACE_DO_NOTHING
#define SAH_TRACE_OUT() SAH_TRACE_DO_NOTHING

#define SAH_TRACEZ(lvl,zone,format,...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_ERROR(zone,format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_WARNING(zone,format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_NOTICE(format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_APP_INFO(zone,format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_INFO(zone,format, ...) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_IN(zone) SAH_TRACE_DO_NOTHING
#define SAH_TRACEZ_OUT(zone) SAH_TRACE_DO_NOTHING

#endif

#ifdef __cplusplus
}
#endif

#endif
