/****************************************************************************
* File:         $Workfile$
* Revision:     $Revision: 1.55 $
* Date:         $Date: 2012-10-03 12:27:08 $
*
* Overview:     
* Functions:
*
* Author:       Ian Tsao
* 
* -- Metanoia Copyright Notice --
*
* (C) Copyright 2005 Metanoia, Inc. All rights reserved.
* 
* Metanoia reserves the right to change specifications without notice. 
* Copyright 2005 Metanoia, Inc. All rights reserved. Preliminary Data-Sheet 
* indicates this product is in design and the specification may change. 
* Electrical parametrics have not been analyzed and are not specified. Do not 
* use this data sheet as a design reference document. Please contact Metanoia 
* for a current data sheet that may be used as a design reference. All 
* contents of this document are protected by copyright law and may not be 
* reproduced without the express written consent of Metanoia, Inc. Metanoia, 
* the Metanoia logo, and combinations thereof are trademarks of Metanoia, Inc. 
* Other product names used in this publication are for identification purposes 
* only and may be trademarks or registered trademarks of their respective 
* companies. The contents of this document are provided in connection with 
* Metanoia, Inc. products. Metanoia, Inc. has made best efforts to ensure that 
* the information contained herein is accurate and reliable. However, Metanoia, 
* Inc. makes no warranties, express or implied, as to the accuracy or 
* completeness of the contents of this publication and is providing this 
* publication 'as is'. Metanoia, Inc. reserves the right to make changes to 
* specifications and product descriptions at any time without notice, and to 
* discontinue or make changes to its products at any time without notice. 
* Metanoia, Inc. does not assume any liability arising out of the application 
* or use of any product or circuit, and specifically disclaims any and all 
* liability, including without limitation special, consequential, or 
* incidental damages.
* 
****************************************************************************/
#include "dsl.h"

#ifdef __KERNEL__
#include <linux/interrupt.h>
#include <linux/list.h>
#include <asm/bitops.h>

#ifdef KERNEL_2_6
#include <linux/timer.h>
#include <linux/sched.h>
#endif

#define SYNC_THRESHOLD 30
//#define CHECK_15M_BOUNDARY 1
#define CHECK_15M_BOUNDARY 15

struct timer_list timer1, timer2;
struct tm statistic_save_tm = {0, 0, 0, 0, 0, 0};
#endif


/**
 * \file 
 * \brief API set that metanoia provide
 */

/** This file is the main API that export to user.
 */

/**
 * Reset statistic buffer of each port
 * \param portid
 * 		port number
 */
void init_var_statistic(mt_uint32 portid)
{
	mt_uint8 i = portid;
	mt_uint8 j, k, l;

	// cur == -1 mean there is no data
	port_info[i].inv15_cur = -1;
	port_info[i].inv24_cur = -1;

	for(j=0;j<((MT_INV_MAX15-1)>>3)+1;j++) {
		port_info[i].inv15_exist[j]=0; 	
	}
	
	for(j=0;j<((MT_INV_MAX24-1)>>3)+1;j++) {
		port_info[i].inv24_exist[j]=0; 	
	}		
	
	// clear statistic data
	for(j=0;j<MT_IDX_LINE;j++) {
		// init 15 mins history buffer
		for(k=0;k<MT_INV_MAX15;k++) {
			for(l=0;l<MT_LINE_SIZE15;l++) {
				//line15[i][j][k][l]=0xffffffff;
				line15[i][j][k][l]=0;
			}
		}
		// init 15 mins init history buffer
		for(k=0;k<MT_INV_MAX15;k++) {
			for(l=0;l<MT_LINEINIT_SIZE15;l++) {
				//lineinit15[i][j][k][l]=0xffffffff;
				lineinit15[i][j][k][l]=0;
			}
		}
		// init 24 hour history buffer
		for(k=0;k<MT_INV_MAX24;k++) {
			for(l=0;l<MT_LINE_SIZE24;l++) {
				//line24[i][j][k][l]=0xffffffff;
				line24[i][j][k][l]=0;
			}
		}			
		// init 24 hour init history buffer
		for(k=0;k<MT_INV_MAX24;k++) {
			for(l=0;l<MT_LINEINIT_SIZE24;l++) {
				//line24[i][j][k][l]=0xffffffff;
				lineinit24[i][j][k][l]=0;
			}
		}
	}
	
	for(j=0;j<MT_IDX_CHAN;j++) {
		// init 15 mins history buffer
		for(k=0;k<MT_INV_MAX15;k++) {
			for(l=0;l<MT_CHAN_SIZE15;l++) {
				//chan15[i][j][k][l]=0xffffffff;
				chan15[i][j][k][l]=0;
			}
		}
		// init 24 hour history buffer
		for(k=0;k<MT_INV_MAX24;k++) {
			for(l=0;l<MT_CHAN_SIZE24;l++) {
				//chan24[i][j][k][l]=0xffffffff;
				chan24[i][j][k][l]=0;
			}
		}			
	}

//yalee980519 init global value 
//#ifdef TR069
	for (i=0;i<MAX_PORTS;i++) {
		for (j=0;j<TR069_SIZE_SHOWTIME ;j++) {
			reset_tr069_showtime_array(i, j);
		}
	}
//#endif //TR069
}


/**
 * Reset statistic data in DMT
 * \param portid
 * 		port number
 */
mt_ret clear_dmt_statistic(mt_uint32 portid)
{
	mt_ret ret;
	mt_uint8 tmp[10];
	mt_uint8 i, j;

	// clear DMT statistic data
	for(j=1;j<=MT_IDX_LINE;j++) {
		for(i=1;i<=MT_LINE_SIZE_CURRTABLE;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineCurrTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_LINEINIT_SIZE_CURRTABLE;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineInitCurrTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_LINE_SIZE15;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineHist15MinTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_LINE_SIZE24;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineHist1DayTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_LINEINIT_SIZE15;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineInitHist15MinTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_LINEINIT_SIZE24;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineInitHist1DayTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}
	}

	for(j=1;j<=MT_IDX_CHAN;j++) {
		for(i=1;i<=MT_CHAN_SIZE_CURRTABLE;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMChCurrTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_CHAN_SIZE15;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMChHist15MinTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}

		for(i=1;i<=MT_CHAN_SIZE24;i++) {
			sprintf(tmp, "%d.%d.%d.%d", MT_MIB_XdslGroup, MT_MIB_Xdsl2PMChHist1DTable, i, j);
			ret = set_port_mib(portid, tmp, "0");
			if(ret != MT_OK) goto end_clr_dmt_stat;
		}
	}

end_clr_dmt_stat:
	return ret;
}


/**
 * Initial procedure, must be called before any vdsllib API
 */
mt_ret init_port_info(void)
{
	mt_uint32 port;

	for(port=0;port<MAX_PORTS;port++) {
		// init port info structure data
		port_info[port].status = MT_OPSTATE_DISABLE;
		init_var_statistic(port);
	}	
	
	return MT_OK;
}


// -------------------------- MIB  -----------------------
/**
 * GetPortMIB get any MIB value on this port
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param pvalue
 * 		memory location to receive the value of the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- -DSL_ERR_NODATA : no this interval data
 *  - -DSL_ERR_PARAM : invalid parameter
 * \par Notice:
 * 
 */
mt_ret get_port_mib(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 i, ibyte, ibit;
	mt_uint8 start;
	mt_uint32 value;
	mt_uint32 groupid, tableid, paramid, idx, interval;

	if(portid == MT_PORT_BroadCast) {
		PRINTL(DBG_LVL_LIB, "Invalid portid=%i\n", portid);
		return -DSL_ERR_NOCHIP;
	}

	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	oid2gtip(loid, &groupid, &tableid, &paramid, &idx, &interval);
	
	/*dsl_er(check_param(MT_PAR_PORT, 0, portid));
	dsl_er(check_param(MT_PAR_GROUP, 0, groupid));
	dsl_er(check_param(MT_PAR_TABLE, groupid, tableid));
	dsl_er(check_param_idx(groupid, tableid, idx));*/
	
	if((interval >= 0) && (interval != 0xff)) {
		if(interval != 0) interval--;
		else interval = 0x7f;	//interval == 0
		PRINTL(DBG_LVL_LIB, "Interval=%i\n",interval);
		ibyte = interval / 8;
		ibit = interval - ibyte*8;
		//get history MIBs
		if(groupid == MT_MIB_XdslGroup) {
			//interval need to translate to interval_idx
			// interval_idx = inv15_cur-interval-1
			switch(tableid) {
				case MT_MIB_Xdsl2PMLineHist15MinTable:
					if((interval == 0x7f) && (paramid > 2) && (paramid <= 9)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMLineCurrTable;
						if(paramid == 9) paramid = 2;
						else paramid = paramid + 1;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX15) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv15_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv15_exist[((MT_INV_MAX15-1)/8+1)-1] & 1<<7) == 0) {	//check the indicate bit of the last buffer element to confirm the buffer is not scrolled
								start = 0;											//queue front
								interval = port_info[portid].inv15_cur - interval;	//reverse the interval to meet RFC requirement, port_info[portid].inv15_cur is the queue rear, to count interval bucket in the queue, we use "queue front" plus "reversed user inputted interval(port_info[portid].inv15_cur - interval)"
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv15_cur + 1) % MT_INV_MAX15;	//(port_info[portid].inv15_cur % MT_INV_MAX15) is the queue rear, i.e. ((port_info[portid].inv15_cur+1) % MT_INV_MAX15) is the queue front
								interval = MT_INV_MAX15 - (interval + 1);	//reverse the interval to meet RFC requirement, port_info[portid].inv15_cur is the queue rear, to count interval bucket in the queue, we use "queue front" plus "reversed user inputted interval(MT_INV_MAX15 - (interval + 1))"
							}
							value = line15[portid][idx-1][(interval+start)%MT_INV_MAX15][paramid-1];
						}
					}
					break;

				case MT_MIB_Xdsl2PMLineHist1DayTable:
					if((interval == 0x7f) && (paramid > 2) && (paramid <= 9)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMLineCurrTable;
						if(paramid == 9) paramid = 10;
						else paramid = paramid + 9;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX24) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv24_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv24_exist[((MT_INV_MAX24-1)/8+1)-1] & 1<<5) == 0) {	//check the indicate bit of the last buffer element to confirm the buffer is not scrolled
								start = 0;
								interval = port_info[portid].inv24_cur - interval;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv24_cur + 1) % MT_INV_MAX24;
								interval = MT_INV_MAX24 - (interval + 1);	//reverse the interval to meet RFC requirement
							}
							value = line24[portid][idx-1][(interval+start)%MT_INV_MAX24][paramid-1];
						}
					}
					break;

				case MT_MIB_Xdsl2PMLineInitHist15MinTable:
					if((interval == 0x7f) && (paramid > 2) && (paramid < 7)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMLineInitCurrTable;
						paramid = paramid + 1;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX15) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv15_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv15_exist[((MT_INV_MAX15-1)/8+1)-1] & 1<<7) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv15_cur - interval;	//reverse the interval to meet RFC requirement, port_info[portid].inv15_cur is the most recent interval data, we use backward counting to count the RFC expected interval from port_info[portid].inv15_cur
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv15_cur + 1) % MT_INV_MAX15;	//(port_info[portid].inv15_cur % MT_INV_MAX15) is the most recent interval data, i.e. ((port_info[portid].inv15_cur+1) % MT_INV_MAX15) is the last recent interval data and also is the starting of the interval queue
								interval = MT_INV_MAX15 - (interval + 1);	//reverse the interval to meet RFC requirement, because buffer scrolled, all interval buckets are filled with interval data, we use forward counting to count the most recent interval, that's why we use (MT_INV_MAX15 - (interval + 1)) instead of backward counting "interval"
							}
							value = lineinit15[portid][idx-1][(interval+start)%MT_INV_MAX15][paramid-1];
						}
					}
					break;

				case MT_MIB_Xdsl2PMLineInitHist1DayTable:				
					if((interval == 0x7f) && (paramid > 2) && (paramid < 7)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMLineInitCurrTable;
						paramid = paramid + 8;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX24) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv24_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv24_exist[((MT_INV_MAX24-1)/8+1)-1] & 1<<5) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv24_cur - interval;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv24_cur + 1) % MT_INV_MAX24;
								interval = MT_INV_MAX24 - (interval + 1);	//reverse the interval to meet RFC requirement
							}
							value = lineinit24[portid][idx-1][(interval+start)%MT_INV_MAX24][paramid-1];
						}
					}
					break;

				case MT_MIB_Xdsl2PMChHist15MinTable:
					if((interval == 0x7f) && (paramid > 3) && (paramid < 6)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMChCurrTable;
						paramid = paramid + 1;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX15) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_CHAN) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv15_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv15_exist[((MT_INV_MAX15-1)/8+1)-1] & 1<<7) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv15_cur - interval;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv15_cur + 1) % MT_INV_MAX15;
								interval = MT_INV_MAX15 - (interval + 1);	//reverse the interval to meet RFC requirement
							}
							value = chan15[portid][idx-1][(interval+start)%MT_INV_MAX15][paramid-1];
						}
					}
					break;

				case MT_MIB_Xdsl2PMChHist1DTable:
					if((interval == 0x7f) && (paramid > 3) && (paramid < 6)) {	//interval == 0, mapping to current table
						tableid = MT_MIB_Xdsl2PMChCurrTable;
						paramid = paramid + 6;
						sprintf((char *)loid, "%d.%d.%d.%d", groupid, tableid, paramid, idx);
						goto gpm_read_from_hpi;
					}
					else {
						if(interval > MT_INV_MAX24) return -DSL_ERR_NOINTERVAL;
						if(idx > MT_IDX_CHAN) return -DSL_ERR_NOPARAM;
						if((port_info[portid].inv24_exist[ibyte] & 1<<ibit) == 0) return -DSL_ERR_NOINTERVAL;
						else {
							if((port_info[portid].inv24_exist[((MT_INV_MAX24-1)/8+1)-1] & 1<<5) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv24_cur - interval;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv24_cur + 1) % MT_INV_MAX24;
								interval = MT_INV_MAX24 - (interval + 1);	//reverse the interval to meet RFC requirement
							}
							value = chan24[portid][idx-1][(interval+start)%MT_INV_MAX24][paramid-1];
						}
					}
				 	break;

				default:
					return -DSL_ERR_NOTABLE;
			}
		}
		else {
			PRINTL(DBG_LVL_LIB, "Only XdslGroup has interval data, interval of other Group must = 0");
			return -DSL_ERR_NOPARAM;
		}
	
		strvalue[0] = 0x00;	// insert mibsize high
		strvalue[1] = 0x01;	// insert mibsize low
		sprintf((char *)(strvalue+2), "%06x", value);
		PRINTL(DBG_LVL_LIB, "get_port_mib return %d\n", value);
		ret = MT_OK;
	}
	else {	//interval == 0xff or user inputted loid has no interval
		//yalee
		if(groupid == MT_MIB_XdslMetanoiaGroup && tableid == 2) {	//Layer 3 Private MIBs dedicate to TR069
			switch(paramid) {
//#ifdef TR069
				case 3:
					//xdsl2StatsShowtimeLinkRetrain xstr(2.2.3)
					value = gShowTimeLog[portid][TR069_LinkRetrain][TR069_SHOWTIME];
					break;

				case 4:
					//xdsl2StatsShowtimeInitErrors xstr(2.2.4)	
					value = gShowTimeLog[portid][TR069_InitErrors][TR069_SHOWTIME];
					break;

				case 5:
					//xdsl2StatsShowtimeInitTimeouts xstr(2.2.5)
					value = gShowTimeLog[portid][TR069_InitTimeouts][TR069_SHOWTIME];
					break;

				case 7:
					//xdsl2StatsShowtimeErroredSecs xstr(2.2.7)
					value = gShowTimeLog[portid][TR069_ErroredSecs][TR069_SHOWTIME];
					break;

				case 8:
					//xdsl2StatsShowtimeATUCErroredSecs xstr(2.2.8)
					value = gShowTimeLog[portid][TR069_ATUCErroredSecs][TR069_SHOWTIME];
					break;

				case 9:
					//xdsl2StatsShowtimeSeverelyErroredSecs xstr(2.2.9)
					value = gShowTimeLog[portid][TR069_SeverelyErroredSecs][TR069_SHOWTIME];
					break;

				case 10:
					//xdsl2StatsShowtimeATUCSeverelyErroredSecs xstr(2.2.10)
					value = gShowTimeLog[portid][TR069_ATUCSeverelyErroredSecs][TR069_SHOWTIME];
					break;

				case 11:
					//xdsl2StatsShowtimeFECErrors xstr(2.2.11)
					value = gShowTimeLog[portid][TR069_FECErrors][TR069_SHOWTIME];
					break;

				case 12:
					//xdsl2StatsShowtimeATUCFECErrors xstr(2.2.12)
					value = gShowTimeLog[portid][TR069_ATUCFECErrors][TR069_SHOWTIME];
					break;

				case 13:
					//xdsl2StatsShowtimeCRCErrors xstr(2.2.13)
					value = gShowTimeLog[portid][TR069_CRCErrors][TR069_SHOWTIME];
					break;

				case 14:
					//xdsl2StatsShowtimeATUCCRCErrors xstr(2.2.14)
					value = gShowTimeLog[portid][TR069_ATUCCRCErrors][TR069_SHOWTIME];
					break;

				case 17:
					//xdsl2StatsLastShowtimeLinkRetrain xstr(2.2.17)
					value = gShowTimeLog[portid][TR069_LinkRetrain][TR069_LAST_SHOWTIME];
					break;

				case 18:
					//xdsl2StatsLastShowtimeInitErrors xstr(2.2.18)
					value = gShowTimeLog[portid][TR069_InitErrors][TR069_LAST_SHOWTIME];
					break;

				case 19:
					//xdsl2StatsLastShowtimeInitTimeouts xstr(2.2.19)
					value = gShowTimeLog[portid][TR069_InitTimeouts][TR069_LAST_SHOWTIME];
					break;

				case 21:
					//xdsl2StatsLastShowtimeErroredSecs xstr(2.2.21)
					value = gShowTimeLog[portid][TR069_ErroredSecs][TR069_LAST_SHOWTIME];
					break;

				case 22:
					//xdsl2StatsLastShowtimeATUCErroredSecs xstr(2.2.22)
					value = gShowTimeLog[portid][TR069_ATUCErroredSecs][TR069_LAST_SHOWTIME];
					break;

				case 23:
					//xdsl2StatsLastShowtimeSeverelyErroredSecs xstr(2.2.23)
					value = gShowTimeLog[portid][TR069_SeverelyErroredSecs][TR069_LAST_SHOWTIME];
					break;

				case 24:
					//xdsl2StatsLastShowtimeATUCSeverelyErroredSecs xstr(2.2.24)
					value = gShowTimeLog[portid][TR069_ATUCSeverelyErroredSecs][TR069_LAST_SHOWTIME];
					break;

				case 25:
					//xdsl2StatsLastShowtimeFECErrors xstr(2.2.25)
					value = gShowTimeLog[portid][TR069_FECErrors][TR069_LAST_SHOWTIME];
					break;

				case 26:
					//xdsl2StatsLastShowtimeATUCFECErrors xstr(2.2.26)
					value = gShowTimeLog[portid][TR069_ATUCFECErrors][TR069_LAST_SHOWTIME];
					break;

				case 27:
					//xdsl2StatsLastShowtimeCRCErrors xstr(2.2.27)
					value = gShowTimeLog[portid][TR069_CRCErrors][TR069_LAST_SHOWTIME];
					break;

				case 28:
					//xdsl2StatsLastShowtimeATUCCRCErrors xstr(2.2.28)
					value = gShowTimeLog[portid][TR069_ATUCCRCErrors][TR069_LAST_SHOWTIME];
					break;

				case 29:
					//xdsl2StatsShowtimeOtTimeElapsed xstr(2.2.29)
					value = gShowTimeLog[portid][TR069_OtTimeElapsed][TR069_SHOWTIME];
					break;

				case 30:
					//xdsl2StatsShowtimeOtFecs xstr(2.2.30)
					value = gShowTimeLog[portid][TR069_OtFecs][TR069_SHOWTIME];
					break;

				case 31:
					//xdsl2StatsShowtimeOtLoss xstr(2.2.31)
					value = gShowTimeLog[portid][TR069_OtLoss][TR069_SHOWTIME];
					break;

				case 32:
					//xdsl2StatsShowtimeOtUas xstr(2.2.32)
					value = gShowTimeLog[portid][TR069_OtUas][TR069_SHOWTIME];
					break;

				case 33:
					//xdsl2StatsShowtimeRtTimeElapsed xstr(2.2.33)
					value = gShowTimeLog[portid][TR069_RtTimeElapsed][TR069_SHOWTIME];
					break;

				case 34:
					//xdsl2StatsShowtimeRtFecs xstr(2.2.34)
					value = gShowTimeLog[portid][TR069_RtFecs][TR069_SHOWTIME];
					break;

				case 35:
					//xdsl2StatsShowtimeRtLoss xstr(2.2.35)
					value = gShowTimeLog[portid][TR069_RtLoss][TR069_SHOWTIME];
					break;

				case 36:
					//xdsl2StatsShowtimeRtUas xstr(2.2.36)
					value = gShowTimeLog[portid][TR069_RtUas][TR069_SHOWTIME];
					break;

				case 37:
					//xdsl2StatsLastShowtimeRtLoss xstr(2.2.37)
					value = gShowTimeLog[portid][TR069_RtLoss][TR069_LAST_SHOWTIME];
					break;
//#endif	//TR069

				case 38:
					//xdsl2StatsTotalFullInits xstr(2.2.38)
					if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
					else {
						sprintf((char *)loid, "%s.%d", xdsl2PMLInitCurr1DayFullInits, idx);
						ret = get_port_data_logic(portid, loid, strvalue);
						value = simple_strtoul((char *)(strvalue+2), NULL, 16);
						for(i=0;i<MT_INV_MAX24;i++) {
							ibyte = i / 8;
							ibit = i - ibyte*8;
							if((port_info[portid].inv24_exist[ibyte] & 1<<ibit) == 0) break;	//interval not exist

							if((port_info[portid].inv24_exist[((MT_INV_MAX24-1)/8+1)-1] & 1<<5) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv24_cur - i;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv24_cur + 1) % MT_INV_MAX24;
								interval = MT_INV_MAX24 - (i + 1);	//reverse the interval to meet RFC requirement
							}

							paramid = MT_MIB_xdsl2PMLHistInit1DFullInits;
							value += lineinit24[portid][idx-1][(interval+start)%MT_INV_MAX24][paramid-1];
						}
					}
					break;

				case 39:
					//xdsl2StatsTotalFailedFullInits xstr(2.2.39)
					if(idx > MT_IDX_LINE) return -DSL_ERR_NOPARAM;
					else {
						sprintf((char *)loid, "%s.%d", xdsl2PMLInitCurr1DayFailedFullInits, idx);
						ret = get_port_data_logic(portid, loid, strvalue);
						value = simple_strtoul((char *)(strvalue+2), NULL, 16);
						for(i=0;i<MT_INV_MAX24;i++) {
							ibyte = i / 8;
							ibit = i - ibyte*8;
							if((port_info[portid].inv24_exist[ibyte] & 1<<ibit) == 0) break;	//interval not exist

							if((port_info[portid].inv24_exist[((MT_INV_MAX24-1)/8+1)-1] & 1<<5) == 0) {	//buffer not scrolled
								start = 0;
								interval = port_info[portid].inv24_cur - i;	//reverse the interval to meet RFC requirement
							}
							else {	//buffer scrolled
								start = (port_info[portid].inv24_cur + 1) % MT_INV_MAX24;
								interval = MT_INV_MAX24 - (i + 1);	//reverse the interval to meet RFC requirement
							}

							paramid = MT_MIB_xdsl2PMLHistInit1DFailedFullInits;
							value += lineinit24[portid][idx-1][(interval+start)%MT_INV_MAX24][paramid-1];
						}
					}
					break;

				default:
					return -DSL_ERR_NOPARAM;
			}

			strvalue[0] = 0x00;	// insert mibsize high byte
			strvalue[1] = 0x01;	// insert mibsize low byte
			sprintf((char *)(strvalue+2), "%06x", value);
			ret = MT_OK;
		}
		else {
gpm_read_from_hpi:
			while(1) {	//make sure we can gain the HPI access authority
				if(test_and_set_bit(0, &atomic_ops) == FALSE) {
					ret = get_port_data_logic(portid, loid, strvalue);
					clear_bit(0, &atomic_ops);
					break;
				}
				else {
					MYDELAY(50);
				}
			}
		}
	}

	return ret;
}


/**
 * SetPortMIB sets a parameter in the MIB for the selected port.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param pvalue
 * 		memory location with the value to be set into the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 * \par Notice:
 * 		some MIB can't be set randomly, in order to protect system, SetPortMIB will filter it. and response with error\n
 * 		SetPortMIB don't support interval paramter
 *  - filter table list: LineConfProfile table, AlarmConfProfile Table
 */
mt_ret set_port_mib(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;

	dsl_er(check_param(MT_PAR_PORT, 0, portid));
		 
	while(1) {	//make sure we can gain the HPI access authority
		if(test_and_set_bit(0, &atomic_ops) == FALSE) {
			ret = set_port_data_logic(portid, loid, strvalue);
			clear_bit(0, &atomic_ops);
			break;
		}
		else {
			MYDELAY(50);
		}
	}

	return ret;
}


/**
 * GetPhyStatus
 * \param phy_status
 * 	    defined in  MT_OPSTATE_*
 * \return 
 * 	- MT_OK: success
 * 	- < 0 : error code
 */
mt_ret get_phy_status(mt_uint32 portid, mt_uint8 *phy_status, mt_uint8 *conn_status)
{
	mt_ret ret = MT_OK;
	mt_uint8 pdata[10];
	mt_uint32 value;

	dsl_er(check_param(MT_PAR_PORT,0,portid));
	
	if(port_info[portid].status == MT_OPSTATE_DISABLE) {
		if(phy_status != NULL) *phy_status = MT_OPSTATE_DISABLE;
		if(conn_status != NULL) *conn_status = connect_status;
		return MT_OK;
	}

	if(phy_status != NULL) {
		if(in_interrupt()) {	//request comes from interrupt handler
			*phy_status = port_info[portid].status;
		}
		else {
			while(1) {	//make sure we can gain the HPI access authority
				if(test_and_set_bit(0, &atomic_ops) == FALSE) {
					ret = get_port_data_logic(portid, (mt_uint8 *)(xdslTerminal), pdata);
					if(ret == MT_OK) {
						value = simple_strtoul((char *)(pdata+2), NULL, 16);
						//Showtime: MT_MIB_noDefect=1, value= 1
						//Idle: MT_MIB_lossOfSignal=1, value= 110110
						//Training: MT_MIB_lossOfLink=1,MT_MIB_lossOfFraming=0,value= 110010
						PRINTL(DBG_LVL_LIB, "Phystatus = 0x%08x\n", value);
						value = (value >> MT_MIB_HandshakeIdle) & 0x00000003;
						switch(value) {
							case IDLE:
								*phy_status = MT_OPSTATE_IDLE;
								break;
							case HANDSHAKE:
								*phy_status = MT_OPSTATE_HANDSHAKE;
								break;
							case TRAINING:
								*phy_status = MT_OPSTATE_TRAINING;
								break;
							case SHOWTIME:
								*phy_status = MT_OPSTATE_SHOWTIME;
								break;
							default:
								PRINTL(DBG_LVL_LIB, "Impossible PhyStatus at Port%d!!\n", portid);
								break;
						}
					}
					else *phy_status = MT_OPSTATE_ERROR;

					clear_bit(0, &atomic_ops);
					break;
				}
				else {
					MYDELAY(50);
				}
			}
		}
	}

	if(conn_status != NULL) *conn_status = connect_status;

	return ret;
}


/**
 * get port's OpState
 * \param portid
 * 		port ID , start from 1
 * \param *state
 * 		return OpState defined in MT_OPSTATE_* to user
 * \return
 * 	- MT_OK: success
 * 	- < 0 : error code
 * 		
 */
mt_ret get_op_status(mt_uint32 portid, mt_uint8 *state)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	*state = port_info[portid].status;
	return MT_OK;
}


// -------------------------- statistic routine -----------------------
/**
 * This API must be called every 15 Min
 * \n get statistic data from firmware every 15 min
 * \return 
 * 	- MT_OK: success
 * 	- < 0 : error code
 */
mt_ret mt_perf_15min(void)
{
	mt_ret ret = MT_OK;
	mt_uint8 ibyte, ibit;
	mt_uint8 loid_tmp[SMALL_BUF_SIZE];
	mt_uint32 i, j;
	mt_uint32* pvalue;
	mt_uint32 inv15_tmp;

	// save 15mins statistic data 
	for(i=0;i<MAX_PORTS;i++) {
		port_info[i].inv15_cur++;
		// count bucket indicator position
		inv15_tmp = port_info[i].inv15_cur % MT_INV_MAX15;
		
		ibyte = inv15_tmp / 8;
		ibit = inv15_tmp - ibyte*8;
		
		if(port_info[i].status == MT_OPSTATE_DISABLE) {
			port_info[i].inv15_exist[ibyte] &= ~(1 << ibit);	// indicate that this interval bucket does not exist 
		}
		else {
			port_info[i].inv15_exist[ibyte] |= 1 << ibit;	// indicate that this interval bucket exist
			// Line
			for(j=1;j<=MT_IDX_LINE;j++) {
				pvalue = line15[i][j-1][inv15_tmp];	// point to current interval bucket data buffer
				*pvalue++ = j;
				*pvalue++ = port_info[i].inv15_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMLHist15MMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_LINE_SIZE15-2, pvalue);
				if( ret != MT_OK ) return ret;
			}
			// LineInit
			for(j=1;j<=MT_IDX_LINE;j++) {
				pvalue = lineinit15[i][j-1][inv15_tmp];	// point to current interval bucket data buffer
				*pvalue++ = port_info[i].inv15_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMLHistInit15MMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_LINEINIT_SIZE15-1, pvalue);
				if( ret != MT_OK ) return ret;
			}
			// Chan
			for(j=1;j<=MT_IDX_CHAN;j++) {
				if(j == 2 || j == 4) continue;	// skip logging bearer channels 2 and 4
				pvalue = chan15[i][j-1][inv15_tmp];
				*pvalue++ = j;
				*pvalue++ = port_info[i].inv15_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMChHist15MMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_CHAN_SIZE15-2, pvalue);
				if( ret != MT_OK ) return ret;
			}
		}
	}

	return MT_OK;
}


/**
 * This API must be called every 24 Hour
 * \n get statistic data from firmware every 24 hour
 * \param portid
 * 		port ID number
 * \return 
 * 	- MT_OK: success
 * 	- < 0 : error code
 */
 
mt_ret mt_perf_24hour(void)
{
	mt_ret ret;
	mt_uint8 ibyte, ibit;
	mt_uint8 loid_tmp[SMALL_BUF_SIZE];
	mt_uint32 i, j;
	mt_uint32* pvalue;
	mt_uint32 inv24_tmp;

	// save 24hours statistic data 
	for(i=0;i<MAX_PORTS;i++) {
		port_info[i].inv24_cur++;
		inv24_tmp = port_info[i].inv24_cur % MT_INV_MAX24;
		
		ibyte = inv24_tmp / 8;
		ibit = inv24_tmp - ibyte*8;
		
		if(port_info[i].status == MT_OPSTATE_DISABLE) {
			port_info[i].inv24_exist[ibyte] &= ~(1 << ibit);
		}
		else {
			port_info[i].inv24_exist[ibyte] |= 1 << ibit;
			// Line
			for(j=1;j<=MT_IDX_LINE;j++) {
				pvalue = line24[i][j-1][inv24_tmp];
				*pvalue++ = j;
				*pvalue++ = port_info[i].inv24_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMLHist1DMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_LINE_SIZE24-2, pvalue);
				if( ret != MT_OK ) return ret;
			}
			// LineInit
			for(j=1;j<=MT_IDX_LINE;j++) {
				pvalue = lineinit24[i][j-1][inv24_tmp];
				*pvalue++ = port_info[i].inv24_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMLHistInit1DMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_LINEINIT_SIZE24-1, pvalue);
				if( ret != MT_OK ) return ret;
			}
			// Chan
			for(j=1;j<=MT_IDX_CHAN;j++) {
				if(j == 2 || j == 4) continue;	// skip logging bearer channels 2 and 4
				pvalue = chan24[i][j-1][inv24_tmp];
				*pvalue++ = j;
				*pvalue++ = port_info[i].inv24_cur + 1;
				sprintf((char *)loid_tmp, "%s.%d", xdsl2PMChHist1DMonitoredTime, j);
				ret = get_port_data_logic_value_array(i, loid_tmp, MT_CHAN_SIZE24-2, pvalue);
				if( ret != MT_OK ) return ret;
			}			
		}
	}

	return MT_OK;
}


//yalee980519
//#ifdef TR069
mt_ret count_showtime_log_fun(mt_uint32 portid, mt_uint8 *loid , mt_uint32 *gArray)
{
	mt_ret ret;
	mt_uint8 strvalue[9];
	mt_uint32 currentValue;
	mt_uint32 currentShowtimeValue;

	ret = gpdl_body(portid, loid, strvalue);
	if (ret == MT_OK)
	{
		currentValue = simple_strtoul((char *)(strvalue+2), NULL , 16);
		currentShowtimeValue = currentValue - gArray[TR069_INIT_SHOWTIME] + gArray[TR069_HISTORY_SHOWTIME];
		//yalee:over run, firmware clear mib (initValue > currentValue) or 
		//currentShowtimeVlue < gArray[TR069_SHOWTIME] 
		//log showtime value to History and set initValue to zero
		if ((currentValue < gArray[TR069_INIT_SHOWTIME]) || (currentShowtimeValue < gArray[TR069_SHOWTIME]))
		{ 
			gArray[TR069_HISTORY_SHOWTIME] += (gArray[TR069_SHOWTIME] - gArray[TR069_HISTORY_SHOWTIME]);
			gArray[TR069_INIT_SHOWTIME] = 0;
		}
		//showtimeValue = currentValue - initValue + historyValue;
		gArray[TR069_SHOWTIME] = currentValue - gArray[TR069_INIT_SHOWTIME] + gArray[TR069_HISTORY_SHOWTIME];

		return MT_OK;
	}

	return ret;
}


mt_ret count_showtime_log(mt_uint32 portid)
{
	//oid array order must match TR069_ShowTime_Mib_Array in if.h
	char *loid[TR069_ARRAY_SIZE] = {
		xdsl2PMLInitCurr1DayFullInits,//LinkRetrain:xdsl2PMLInitCurr1DayFullInits0
		xdsl2PMLInitCurr1DayFailedFullInits,//InitErrors:xdsl2PMLInitCurr1DayFailedFullInits0
		xdsl2PMLInitCurr1DayFailedShortInits,//InitTimeouts:xdsl2PMLInitCurr1DayFailedShortInits0
		xdsl2PMLCurr1DayEs1,//ErroredSecs:xdsl2PMLCurr1DayEs1
		xdsl2PMLCurr1DayEs,//ATUCErroredSecs:xdsl2PMLCurr1DayEs
		xdsl2PMLCurr1DaySes1,//SeverelyErroredSecs:xdsl2PMLCurr1DaySes1
		xdsl2PMLCurr1DaySes,//ATUCSeverelyErroredSecs:xdsl2PMLCurr1DaySes
		xdsl2PMChCurr1DayCorrectedBlocks2,//FECErrors:xdsl2PMChCurr1DayCorrectedBlocks2
		xdsl2PMChCurr1DayCorrectedBlocks,//ATUCFECErrors:xdsl2PMChCurr1DayCorrectedBlocks
		xdsl2PMChCurr1DayCodingViolations2,//CRCrrors:xdsl2PMChCurr1DayCodingViolations2
		xdsl2PMChCurr1DayCodingViolations,//ATUCCRCErrors:xdsl2PMChCurr1DayCodingViolations
		xdsl2PMLCurr1DayTimeElapsed,
		xdsl2PMLCurr1DayFecs,
		xdsl2PMLCurr1DayLoss,
		xdsl2PMLCurr1DayUas,
		xdsl2PMLCurr1DayTimeElapsed1,
		xdsl2PMLCurr1DayFecs1,
		xdsl2PMLCurr1DayLoss1,
		xdsl2PMLCurr1DayUas1
	};						
	mt_uint32 i = 0;

 	/*tr069:LOID
	ReceiveBlocks:soon
	TransmitBlocks:soon*/
	for (i = 0 ; i < TR069_ARRAY_SIZE ; i++){
		count_showtime_log_fun(portid, (mt_uint8 *)(loid[i]), gShowTimeLog[portid][i]);
	}

	return MT_OK;
}


mt_ret reset_tr069_showtime_array(mt_uint32 portid ,int type)
{
//type define:
//        TR069_SHOWTIME,
//        TR069_LAST_SHOWTIME,
//        TR069_INIT_SHOWTIME,
//        TR069_HISTORY_SHOWTIME
	int i = 0;
	for (i = 0 ; i < TR069_ARRAY_SIZE ; i++){
		gShowTimeLog[portid][i][type] = 0;
	}

	return MT_OK;
}


mt_ret count_last_showtime_log(int portid)
{
	int i = 0;
	for (i = 0 ; i < TR069_ARRAY_SIZE ; i++){
		gShowTimeLog[portid][i][TR069_LAST_SHOWTIME] = gShowTimeLog[portid][i][TR069_SHOWTIME];
	}

	return MT_OK;
}


mt_ret count_init_showtime_log(int portid)
{
	mt_uint8 strvalue[9];
	mt_uint32 i = 0;
	//oid array order must match TR069_ShowTime_Mib_Array in if.h
	char *oid[TR069_ARRAY_SIZE] = {
		xdsl2PMLInitCurr1DayFullInits,//LinkRetrain:xdsl2PMLInitCurr1DayFullInits0
		xdsl2PMLInitCurr1DayFailedFullInits,//InitErrors:xdsl2PMLInitCurr1DayFailedFullInits0
		xdsl2PMLInitCurr1DayFailedShortInits,//InitTimeouts:xdsl2PMLInitCurr1DayFailedShortInits0
		xdsl2PMLCurr1DayEs1,//ErroredSecs:xdsl2PMLCurr1DayEs1
		xdsl2PMLCurr1DayEs,//ATUCErroredSecs:xdsl2PMLCurr1DayEs
		xdsl2PMLCurr1DaySes1,//SeverelyErroredSecs:xdsl2PMLCurr1DaySes1
		xdsl2PMLCurr1DaySes,//ATUCSeverelyErroredSecs:xdsl2PMLCurr1DaySes
		xdsl2PMChCurr1DayCorrectedBlocks2,//FECErrors:xdsl2PMChCurr1DayCorrectedBlocks2
		xdsl2PMChCurr1DayCorrectedBlocks,//ATUCFECErrors:xdsl2PMChCurr1DayCorrectedBlocks
		xdsl2PMChCurr1DayCodingViolations2,//CRCrrors:xdsl2PMChCurr1DayCodingViolations2
		xdsl2PMChCurr1DayCodingViolations,//ATUCCRCErrors:xdsl2PMChCurr1DayCodingViolations
		xdsl2PMLCurr1DayTimeElapsed,
		xdsl2PMLCurr1DayFecs,
		xdsl2PMLCurr1DayLoss,
		xdsl2PMLCurr1DayUas,
		xdsl2PMLCurr1DayTimeElapsed1,
		xdsl2PMLCurr1DayFecs1,
		xdsl2PMLCurr1DayLoss1,
		xdsl2PMLCurr1DayUas1
	};						

	//log InitShowtime array and reset HistoryShowtime array
	for (i = 0 ; i < TR069_ARRAY_SIZE ; i++){
		if (MT_OK == gpdl_body(portid, (mt_uint8 *)(oid[i]), strvalue))
		{
			//**Traning**//
			gShowTimeLog[portid][i][TR069_INIT_SHOWTIME] = simple_strtoul((char *)(strvalue+2), NULL , 16);
		}
	}

	return MT_OK;
}
//#endif	//TR069


mt_ret mt_perf_showtime(void)
{
	mt_ret ret = MT_OK;
	mt_uint8 state;
	mt_uint8 strvalue[9];
	mt_uint32 value = 0;
	mt_uint32 portid, tmp_value;
//#ifdef TR069
	static mt_uint8 vdsl_showtime_flag[MAX_PORTS];
//#endif	//TR069
	//static mt_uint8 count = 0;
	//static mt_uint32 showtimeBitFlag = 0;
	
	/**yalee:one time of 5 sec**/
	/*if (count != 5) {
		count++;
		return MT_OK;
	}*/
	/**reset count**/
	//count = 0;

	for (portid = 0 ; portid < MAX_PORTS ; portid++)
	{
		get_op_status(portid, &state);
		if(state == MT_OPSTATE_DISABLE) {
			PRINTL(DBG_LVL_LIB, "Port%i disabled\n", portid);
			continue;
		}

		// vdsl2loid.h: vdslTerminal:2.1.5
		ret = gpdl_body(portid, (mt_uint8 *)(xdslTerminal), strvalue);
		if(ret == MT_OK) {	//record port_status for interrupt handler
			value = simple_strtoul((char *)(strvalue+2), NULL, 16);
			//Showtime: MT_MIB_noDefect=1, value= 1
			//Idle: MT_MIB_lossOfSignal=1, value= 110110
			//Training: MT_MIB_lossOfLink=1,MT_MIB_lossOfFraming=0,value= 110010
			PRINTL(DBG_LVL_LIB, "Phystatus = 0x%08x\n", value);
			tmp_value = (value >> MT_MIB_HandshakeIdle) & 0x00000003;
			switch(tmp_value) {
				case IDLE:
					if(port_info[portid].status != MT_OPSTATE_IDLE) PRINTL(DBG_LVL_STATUS, "VDSL Port%d Status: IDLE\n", portid);
					port_info[portid].status = MT_OPSTATE_IDLE;
					break;
				case HANDSHAKE:
					if(port_info[portid].status != MT_OPSTATE_HANDSHAKE) PRINTL(DBG_LVL_STATUS, "VDSL Port%d Status: HANDSHAKE\n", portid);
					port_info[portid].status = MT_OPSTATE_HANDSHAKE;
					break;
				case TRAINING:
					if(port_info[portid].status != MT_OPSTATE_TRAINING) PRINTL(DBG_LVL_STATUS, "VDSL Port%d Status: TRAINING\n", portid);
					port_info[portid].status = MT_OPSTATE_TRAINING;
					break;
				case SHOWTIME:
					if(port_info[portid].status != MT_OPSTATE_SHOWTIME) PRINTL(DBG_LVL_STATUS, "VDSL Port%d Status: SHOWTIME\n", portid);
					port_info[portid].status = MT_OPSTATE_SHOWTIME;
					break;
				default:
					PRINTL(DBG_LVL_LIB, "Impossible PhyStatus at Port%d!!\n", portid);
					break;
			}
		}
		else {
			port_info[portid].status = MT_OPSTATE_ERROR;
			continue;
		}

//#ifdef TR069
		if( value & 0x00000200 ) {	
			//Showtime log
			count_showtime_log(portid);
			vdsl_showtime_flag[portid] = 1;
			//showtimeBitFlag |= (1 << portid );
		}else{
			//Training log
			//log InitShowtime array and clear HistoryShowtime array
			count_init_showtime_log(portid);
			reset_tr069_showtime_array(portid, TR069_HISTORY_SHOWTIME);

			//if(showtimeBitFlag &= (1 << portid))
			if(vdsl_showtime_flag[portid] == 1)
			{
				//ShowTime->Training
				//log last showtime value and clear showtime value
				count_last_showtime_log(portid);
				reset_tr069_showtime_array(portid, TR069_SHOWTIME);
			}
				vdsl_showtime_flag[portid] = 0;
			//	showtimeBitFlag &= (~(1 << portid ));
		}
//#endif	//TR069
	}

	return MT_OK;
}


/**
 * ClearPortStats clears port statistics
 * \param portid
 * 		port ID number 
 */
mt_ret clear_port_stats(mt_uint32 portid)
{
	mt_ret ret = MT_OK;

	// clear stats in history buffer
	init_var_statistic(portid);
	
	// clear stats in DMT
	ret = clear_dmt_statistic(portid);

	return ret;
} 


#ifdef __KERNEL__
int statistic_thread_handler(void *dummy)
{
	mt_ret ret;
	struct timeval tv;
	struct tm tm;
	unsigned long jiffies_tmp;

#ifdef KERNEL_2_6
	daemonize("vdsl_statistic");
	allow_signal(SIGKILL);
#else
	daemonize();
#endif

	PRINTL(DBG_LVL_LIB, "In statistic_thread_handler\n");

	while(!signal_pending(current)) {
		do_gettimeofday(&tv);
		ret = __offtime(&(tv.tv_sec), 0, &tm);
		if(ret == MT_OK) {
			if((tm.tm_min % CHECK_15M_BOUNDARY == 0) && (tm.tm_sec > SYNC_THRESHOLD)) {
				if(boottime.tm_min == tm.tm_min &&
					boottime.tm_hour == tm.tm_hour &&
					boottime.tm_mday == tm.tm_mday &&
					boottime.tm_mon == tm.tm_mon &&
					boottime.tm_year == tm.tm_year) {
					PRINTL(DBG_LVL_LIB, "statistic_timer_handler: boot skip\n");
					goto buttom_of_statistic;
				}
				else {
					if(statistic_save_tm.tm_min == tm.tm_min &&
						statistic_save_tm.tm_hour == tm.tm_hour &&
						statistic_save_tm.tm_mday == tm.tm_mday &&
						statistic_save_tm.tm_mon == tm.tm_mon &&
						statistic_save_tm.tm_year == tm.tm_year) {	//statistic has been saved 5 sec ago, so skip saving
						PRINTL(DBG_LVL_LIB, "statistic_timer_handler: not save min:%d sec:%d\n", tm.tm_min, tm.tm_sec);
					}
					else {
						if(test_and_set_bit(0, &atomic_ops) == FALSE) {
							PRINTL(DBG_LVL_LIB, "statistic_timer_handler: save min:%d sec:%d\n", tm.tm_min, tm.tm_sec);
							statistic_save_tm.tm_min = tm.tm_min;
							statistic_save_tm.tm_hour = tm.tm_hour;
							statistic_save_tm.tm_mday = tm.tm_mday;
							statistic_save_tm.tm_mon = tm.tm_mon;
							statistic_save_tm.tm_year = tm.tm_year;
		
							mt_perf_15min();	//15mins save
		
							if(tm.tm_hour == 0 && tm.tm_min == 0) {
								mt_perf_24hour();	//24hours save
								ret = time_calibrate();	//sync VDSL time with system time once per day
								if(ret != MT_OK) PRINTL(DBG_LVL_LIB, "Time calibration failed!!\n");
							}

							clear_bit(0, &atomic_ops);
						}
						else
							PRINTL(DBG_LVL_LIB, "statistic_timer_handler: access violation\n");
					}
				}
			}
		}
		else
			PRINTD("statistic_timer_handler: time transform failed!\n");
	
		//PRINTD("y:%d m:%d d:%d h:%d m:%d s:%d\n", boottime.tm_year, boottime.tm_mon, boottime.tm_mday, boottime.tm_hour, boottime.tm_min, boottime.tm_sec);
		//PRINTD("y:%d m:%d d:%d h:%d m:%d s:%d\n", tm.tm_year, tm.tm_mon, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

buttom_of_statistic:
//yalee980520
		if(test_and_set_bit(0, &atomic_ops) == FALSE) {
			mt_perf_showtime();
			//kernel_thread(mt_perf_showtime, NULL, NULL);
			clear_bit(0, &atomic_ops);
		}

		jiffies_tmp = jiffies;
		while(1) {
			MSLEEP(1000);
			if((jiffies - jiffies_tmp) >= (5 * HZ)) break;	//make sure the thread sleep enough for 5 seconds
		}
	}	
		
	complete_and_exit(&thread_exited, 1);
}

#ifndef KERNEL_2_6
/**
 *      msleep - sleep for a number of milliseconds
 *      @msecs: number of milliseconds to sleep
 *
 *      Issues schedule_timeout call for the specified number
 *      of milliseconds.
 *
 *      LOCKING:
 *      None.
 */
void mysleep(unsigned long msecs)
{
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_timeout(((HZ * msecs) / 1000) + 1);
}
#endif	//KERNEL_2_6
#endif	//__KERNEL__

EXPORT_SYMBOL(get_phy_status);
EXPORT_SYMBOL(get_port_mib);

// -------------------------- Debug -----------------------
// -------------------------- Misc -----------------------
