#include "dsll2.h"

#ifdef __KERNEL__
#include <linux/time.h>
#else
#include <stdio.h>
#endif

mt_uint8 pbuffer[BUF_SIZE];

/**
 * \file 
 * \brief communication module interface API2
 */

/**
 * Get mib2 value and convert to int, specify mib entry index
 */
mt_ret get_mib2_int_idx(mt_uint32 portid, mt_uint8 * loid, int * value, mt_uint32 idx){
	mt_uint8 loidf[OID_MAX_LEN];

	sprintf((char *)loidf, "%s.%d", loid, idx);

	return get_mib2_int(portid, loidf, value);
}

/**
 * Get mib2 value and convert to int
 */
mt_ret get_mib2_int(mt_uint32 portid, mt_uint8 * loid, int * value){
	mt_ret ret;

#ifdef __KERNEL__
	ret = gpdl_body(portid, loid, pbuffer);
#else
	ret = get_port_mib(portid, loid, pbuffer);
#endif

	if (ret == DSL_ERR_OK)	
#ifdef  __KERNEL__
		*value = simple_strtoul((char *)(pbuffer+2), NULL, 16);
#else
		*value = strtoul(pbuffer+2, NULL, 16);
#endif
	if (*value & 0x800000) *value |= 0xff000000;

	return ret;
}

/**
 * Get mib2 value and convert to uint32
 */
mt_ret get_mib2_uint32(mt_uint32 portid, mt_uint8 * loid, mt_uint32 * value){
	mt_ret ret;

#ifdef __KERNEL__
	ret = gpdl_body(portid, loid, pbuffer);
#else
	ret = get_port_mib(portid, loid, pbuffer);
#endif

	if (ret == DSL_ERR_OK)	
#ifdef  __KERNEL__
		*value = simple_strtoul((char *)(pbuffer+2), NULL, 16);
#else
		*value = strtoul((char *)(pbuffer+2), NULL, 16);
#endif

	return ret;
}

/**
 * Get mib2 value and convert to uint8, specify mib entry index
 */
mt_ret get_mib2_uint8_idx(mt_uint32 portid, mt_uint8 * loid, mt_uint8 * value, mt_uint32 toread, mt_uint32 * read, mt_uint32 idx){
	mt_uint8 loidf[OID_MAX_LEN];

	sprintf((char *)loidf, "%s.%d", loid, idx);

	return get_mib2_uint8(portid, loidf, value, toread, read);
}

/**
 * Get mib2 value and convert to uint8
 * \param portid: port Id, start from 0
 * \param loid: logic oid, defined in header file
 * \param value: pointer to the buffer to receives converted value
 * \param toread: number of bytes to be read from this mib entry
 * \param read: pointer to the number of bytes read
 * \return
 * 	- 0 : success
 * 	- other : error code 
 */
mt_ret get_mib2_uint8(mt_uint32 portid, mt_uint8 * loid, mt_uint8 * value, mt_uint32 toread, mt_uint32 * read){
	mt_ret ret;
	mt_uint32 i, size, remainder;
	mt_uint32 address, mibsize, mibtype;
	mt_uint8 remote = 0, loidtmp[OID_MAX_LEN], *dot;
	mt_uint32 idx,rmtread,rmttoread;

	dot = (mt_uint8 *)strchr((char *)loid, '.');
	if (dot == NULL) return DSL_ERR_NOTOK;
	dot = (mt_uint8 *)strchr((char *)(dot+1), '.');
	if (dot == NULL) return DSL_ERR_NOTOK;
	dot = (mt_uint8 *)strchr((char *)(dot+1), '.');
	if (dot != NULL) {
		strncpy((char *)loidtmp, (char *)loid, dot - loid);
#ifdef  __KERNEL__
		idx = simple_strtoul((char *)(dot+1), NULL, 10);
#else
		idx = strtoul((char *)(dot+1), NULL, 10);
#endif		
		if(idx & 0x80) {
			remote = 1;
			idx &= 0x7F;
		}
		//printf("loidtmp %s, remote %x\n", loidtmp, remote);
		sprintf((char *)loidtmp, "%s.%d", loidtmp, idx);
		//printf("loidtmp %s, idx %d\n", loidtmp, idx);
	}else
		strcpy((char *)loidtmp, (char *)loid);
	
	
	ret = get_mib2_info(loidtmp, &address, &mibsize, &mibtype);
	if (mibsize <= toread/3){
		size = mibsize;
		remainder = 0;
	}else{
		size = toread / 3;
		remainder = toread % 3;
	}
	//printf("size %d, remainder %d\n", size, remainder);
	if (remote){
		rmtread = 0;
		while ( rmtread < size){
			rmttoread = (size - rmtread < 128) ? (size - rmtread) : 128;
			ret = get_port_mem_remote(portid, address + rmtread, rmttoread, value + rmtread * 3);
			if (ret != DSL_ERR_OK) break;
			rmtread += rmttoread;
		}
		if (read) *read = rmtread * 3;
	}else{
		for (i = 0; i < size; i++){
			ret = get_port_mem(portid, address + i, mibtype, value + i * 3);
			if (ret != DSL_ERR_OK) break;
		}
		if (read) *read = i * 3;
	}
	
	if (DSL_ERR_OK == ret && remainder > 0){
		if (remote)
			ret = get_port_mem_remote(portid, address + size, 1, pbuffer);
		else
			ret = get_port_mem(portid, address + size, mibtype, pbuffer);
		value += size * 3;
		for(i = 0; i < remainder; i++){
			*value++ = pbuffer[i];
		}
		if (read) *read += remainder;
	}

	return ret;
}

/**
 * get_vdsl_rate gets various VDSL linerate and return to user
 * \param portid
 * 		port ID number
 * \param rate_type
 * 		Type of linerate to get  
 * 		- MT_UPSTREAM: target VDSL upstream linerate
 * 		- MT_DOWNSTREAM: target VDSL downstream linerate
 * 		- MT_CURRENT_UPSTREAM: current VDSL upstream linerate
 * 		- MT_CURRENT_DOWNSTREAM: current VDSL downstream linerate
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_rate(mt_uint32 portid,mt_uint8 rate_type,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	switch(rate_type)
		{
			case MT_UPSTREAM:
				 ret = get_port_mib(portid,"0.2.2.2.3",strvalue);
				 break;
			case MT_DOWNSTREAM:
				 ret = get_port_mib(portid,"0.2.2.2.1",strvalue);
				 break;
			case MT_CURRENT_UPSTREAM:
				 ret = get_port_mib(portid,"0.2.2.2.3",strvalue);
				 break;
			case MT_CURRENT_DOWNSTREAM:
				 ret = get_port_mib(portid,"0.2.2.2.1",strvalue);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}*/


/**
 * set_vdsl_rate sets various VDSL linerate from user
 * \param portid
 * 		port ID number
 * \param rate_type
 * 		Type of linerate to set  
 * 		- MT_UPSTREAM: target VDSL upstream linerate
 * 		- MT_DOWNSTREAM: target VDSL downstream linerate
 * \param pvalue
 * 		buffer containing the value to set, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret set_vdsl_rate(mt_uint32 portid,mt_uint8 rate_type,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	switch(rate_type)
		{
			case MT_UPSTREAM:
				 ret = set_port_mib(portid,"Set US",strvalue);
				 break;
			case MT_DOWNSTREAM:
				 ret = set_port_mib(portid,"Set DS",strvalue);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}*/


/**
 * get_vdsl_snrmargin gets various VDSL SNR and return to user
 * \param portid
 * 		port ID number
 * \param snr_type
 * 		Type of SNR to get  
 * 		- MT_UPSTREAM: target VDSL upstream SNR
 * 		- MT_DOWNSTREAM: target VDSL downstream SNR
 * 		- MT_CURRENT_UPSTREAM: current VDSL upstream SNR
 * 		- MT_CURRENT_DOWNSTREAM: current VDSL downstream SNR
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_snrmargin(mt_uint32 portid,mt_uint8 snr_type,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	switch(snr_type)
		{
			case MT_UPSTREAM:
				 ret = get_port_mib(portid,"Get SNR US",strvalue);
				 break;
			case MT_DOWNSTREAM:
				 ret = get_port_mib(portid,"Get SNR DS",strvalue);
				 break;
			case MT_CURRENT_UPSTREAM:
				 ret = get_port_mib(portid,"Get Current SNR US",strvalue);
				 break;
			case MT_CURRENT_DOWNSTREAM:
				 ret = get_port_mib(portid,"Get Current SNR DS",strvalue);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}*/


/**
 * set_vdsl_snrmargin sets various VDSL SNR from user
 * \param portid
 * 		port ID number
 * \param snr_type
 * 		Type of SNR to set  
 * 		- MT_UPSTREAM: target VDSL upstream SNR
 * 		- MT_DOWNSTREAM: target VDSL downstream SNR
 * \param pvalue
 * 		buffer containing the value to set, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret set_vdsl_snrmargin(mt_uint32 portid,mt_uint8 snr_type,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	switch(snr_type)
		{
			case MT_UPSTREAM:
				 ret = set_port_mib(portid,"Set SNR US",strvalue);
			    break;
			case MT_DOWNSTREAM:
				 ret = set_port_mib(portid,"Set SNR DS",strvalue);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}*/


/**
 * get_vdsl_interleave gets VDSL Interleave delay and return to user
 * \param portid
 * 		port ID number
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_interleave(mt_uint32 portid,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	ret = get_port_mib(portid,"Get interleave",strvalue);
	return ret;
}*/


/**
 * set_vdsl_interleave sets VDSL Interleave delay from user
 * \param portid
 * 		port ID number
 * \param pvalue
 * 		buffer containing the value to set, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret set_vdsl_interleave(mt_uint32 portid,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	ret = set_port_mib(portid,"Set interleave",strvalue);
	return ret;
}*/


/**
 * get_vdsl_current_crc gets various VDSL current CRC and return to user
 * \param portid
 * 		port ID number
 * \param crc_type
 * 		Type of CRC to get  
 * 		- MT_UPSTREAM: current VDSL upstream CRC
 * 		- MT_DOWNSTREAM: current VDSL downstream CRC
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_current_crc(mt_uint32 portid,mt_uint8 crc_type,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	switch(crc_type)
		{
			case MT_UPSTREAM:
				 ret = get_port_mib(portid,"Get CRC US",strvalue);
			    break;
			case MT_DOWNSTREAM:
				 ret = get_port_mib(portid,"Get CRC DS",strvalue);
			    break;
			default:
				ret = DSL_ERR_MISC;
				break;
		}
	return ret;
}*/


/**
 * get_vdsl_current_status gets VDSL current operating status and return to user
 * \param portid
 * 		port ID number
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_current_status(mt_uint32 portid,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	ret = get_port_mib(portid,"Get status",strvalue);
	return ret;
}*/


/**
 * get_vdsl_fwversion gets VDSL Firmware Version and return to user
 * \param portid
 * 		port ID number
 * \param pvalue
 * 		buffer to receive the returned value, value is in hex-string format, for example: "123def"
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
/*mt_ret get_vdsl_fwversion(mt_uint32 portid,mt_uint8 *strvalue)
{
	mt_uint32 ret = MT_OK;
	ret = get_port_mib(portid,"Get frimware version",strvalue);
	return ret;
}*/

//mt_ret get_vdsl_info
#if 0
/**
 * get_xdsl_pwrmngstate gets xdsl current power management state and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_pwrmngstate(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2LineStatusPwrMngState,value);
	return ret;
}

/**
 * get_xdsl_attainablerate gets xdsl attainable data rate(kbps) and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get  
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_attainablerate(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;

	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusAttainableRateUs,value);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusAttainableRateDs,value);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_actatp gets xdsl actual Aggregate Transmit Power (0.1 dBm) and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get  
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actatp(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusActAtpUs,value);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusActAtpDs,value);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_actprofile gets the xdsl profile in use and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 8a(0),8b(1),8c(2),8d(3),12a(4),12b(5),17a(6),30a(7)
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actprofile(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2LineStatusActProfile,value);
	return ret;
}

/**
 * get_xdsl_actsnrmode gets that xdsl virtual noise is enabled or disabled and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get  
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 1: virtual noise is disabled
 * 		- 2: virtual noise is enabled
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actsnrmode(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusActSnrModeUs,value);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusActSnrModeDs,value);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_eleclen gets xdsl electrical length (0.1 dB) and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_eleclen(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2LineStatusElectricalLength,value);
	return ret;
}

/**
 * get_xdsl_trellis gets whether trellis coding is in use and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get  
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 0: not used
 * 		- 1: used
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_trellis(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusTrellisUs,value);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int(portid,xdsl2LineStatusTrellisDs,value);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_actualce gets xdsl cyclic extension used (N/32 samples) and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actualce(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2LineStatusActualCe,value);
	return ret;
}

/**
 * get_xdsl_actdatarate gets xdsl actual net data rate(kbps) and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actdatarate(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDataRate,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDataRate,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDataRate,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDataRate,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_predatarate gets xdsl actual previous net data rate(kbps) and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_predatarate(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPrevDataRate,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPrevDataRate,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPrevDataRate,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPrevDataRate,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_actdelay gets xdsl actual one-way interleaving delay and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actdelay(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDelay,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDelay,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDelay,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActDelay,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_actinp gets xdsl actual INP and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_actinp(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActInp,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActInp,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActInp,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusActInp,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_inpreport gets xdsl INP reporting mode and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 1: computed using formula
 * 		- 2: estimated by Xtur
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_inpreport(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusInpReport,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusInpReport,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusInpReport,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusInpReport,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_nfec gets xdsl actual size of Reed-Solomon codeword (bytes) and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_nfec(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusNFec,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusNFec,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusNFec,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusNFec,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_rfec gets xdsl actual number of Reed-Solomon redundancy bytes (bytes) and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_rfec(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusRFec,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusRFec,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusRFec,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusRFec,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_lsymb gets xdsl actual number of bits per symbol (bits) and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_lsymb(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLSymb,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLSymb,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLSymb,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLSymb,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_intlvdepth gets xdsl actual interleaving Depth and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_intlvdepth(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvDepth,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvDepth,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvDepth,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvDepth,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_intlvblock gets xdsl actual interleaving block length and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_intlvblock(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvBlock,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvBlock,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvBlock,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusIntlvBlock,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_lpath gets xdsl actual latency path and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 0: FAST path
 * 		- 1: INTERLEAVED path
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_lpath(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLPath,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLPath,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLPath,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusLPath,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_atmstatus gets xdsl current status of ATM Data Path and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 0: noDefect
 * 		- 1: noCellDelineation
 * 		- 2: lossOfDelineation
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_atmstatus(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusAtmStatus,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusAtmStatus,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusAtmStatus,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusAtmStatus,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_ptmstatus gets xdsl current status of PTM Data Path and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 0: noDefect
 * 		- 1: outOfSync
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ptmstatus(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPtmStatus,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPtmStatus,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPtmStatus,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2ChStatusPtmStatus,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
		}
	return ret;
}

/**
 * get_xdsl_scdirection gets xdsl direction of subcarrier and return to user
 * \param portid
 * 		port ID number
 * \param index
 * 		Index of direction to get, range: 1 ~ 2 
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_scdirection(mt_uint32 portid, mt_uint8 idx, mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int_idx(portid, xdsl2SCStatusDirection, value, idx);
	return ret;
}

/**
 * get_xdsl_linscale gets xdsl scale factor to H(f) linear value and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linscale(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLinScale,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLinScale,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_linscgpsize gets xdsl number of SC per group used to report H(f) linear values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linscgpsize(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLinScGroupSize,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLinScGroupSize,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_logmt gets xdsl the number of symbols used to measure the Hlog(f) values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_logmt(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLogMt,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLogMt,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_logscgpsize gets xdsl number of SC per group used to report H(f) logarithmic values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_logscgpsize(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLogScGroupSize,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusLogScGroupSize,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_qlnmt gets xdsl the number of symbols used to measure the QLN(f) values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_qlnmt(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusQlnMt,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusQlnMt,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_qlnscgpsize gets xdsl number of SC per group used to report Quiet Line Noise values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_qlnscgpsize(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusQlnScGroupSize,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusQlnScGroupSize,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_snrmtime gets xdsl the number of symbols used to measure the SNR(f) values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_snrmtime(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusSnrMtime,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusSnrMtime,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_snrscgpsize gets xdsl number of SC per group used to report SNR values and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_snrscgpsize(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusSnrScGroupSize,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusSnrScGroupSize,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_rowstatus gets xdsl row status and return to user
 * \param portid
 * 		port ID number
 * \param direction
 * 		direction to get
 * 		- MT_DIR_UPSTREAM: upstream
 * 		- MT_DIR_DOWNSTREAM: downstream
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_rowstatus(mt_uint32 portid,mt_uint8 direction,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_DIR_UPSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusRowStatus,value, 1);
				 break;
			case MT_DIR_DOWNSTREAM:
				 ret = get_mib2_int_idx(portid,xdsl2SCStatusRowStatus,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_linebandid gets xdsl line band id and return to user
 * \param portid
 * 		port ID number
 * \param index
 * 		Index of band to get, range: 1 ~ 11  
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- MT_BAND_UPSTREAM: whole upstream
 * 		- MT_BAND_DOWNSTREAM: whole downstream
 * 		- MT_BAND_US0: upstream band number 0
 *  	- MT_BAND_DS1: downstream band number 1
 * 		- MT_BAND_US1: upstream band number 1
 * 		- MT_BAND_DS2: downstream band number 2
 * 		- MT_BAND_US2: upstream band number 2
 * 		- MT_BAND_DS3: downstream band number 3
 * 		- MT_BAND_US3: upstream band number 3
 * 		- MT_BAND_DS4: downstream band number 4
 * 		- MT_BAND_US4: upstream band number 4 
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linebandid(mt_uint32 portid,mt_uint8 index,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int_idx(portid,xdsl2LineBand,value, index);
	return ret;
}

/**
 * get_xdsl_linebandlnatten gets xdsl line band Line Attenuation (0.1 dB)and return to user
 * \param portid
 * 		port ID number
 * \param band
 * 		band to get
 * 		- MT_BAND_UPSTREAM: whole upstream
 * 		- MT_BAND_DOWNSTREAM: whole downstream
 * 		- MT_BAND_US0: upstream band number 0
 *  	- MT_BAND_DS1: downstream band number 1
 * 		- MT_BAND_US1: upstream band number 1
 * 		- MT_BAND_DS2: downstream band number 2
 * 		- MT_BAND_US2: upstream band number 2
 * 		- MT_BAND_DS3: downstream band number 3
 * 		- MT_BAND_US3: upstream band number 3
 * 		- MT_BAND_DS4: downstream band number 4
 * 		- MT_BAND_US4: upstream band number 4 
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linebandlnatten(mt_uint32 portid,mt_uint8 band,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch (band){
	case MT_BAND_UPSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 10);
		break;
	case MT_BAND_DOWNSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 11);
		break;
	case MT_BAND_US0:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 1);
		break;
	case MT_BAND_DS1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 6);
		break;
	case MT_BAND_US1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 2);
		break;
	case MT_BAND_DS2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 7);
		break;
	case MT_BAND_US2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 3);
		break;
	case MT_BAND_DS3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 8);
		break;
	case MT_BAND_US3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 4);
		break;
	case MT_BAND_DS4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 9);
		break;
	case MT_BAND_US4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusLnAtten,value, 5);
		break;
	default:
		ret = DSL_ERR_MISC;
		break;
	}
	return ret;
}

/**
 * get_xdsl_linebandsigatten gets xdsl line band Signal Attenuation (0.1 dB)and return to user
 * \param portid
 * 		port ID number
 * \param band
 * 		band to get
 * 		- MT_BAND_UPSTREAM: whole upstream
 * 		- MT_BAND_DOWNSTREAM: whole downstream
 * 		- MT_BAND_US0: upstream band number 0
 *  	- MT_BAND_DS1: downstream band number 1
 * 		- MT_BAND_US1: upstream band number 1
 * 		- MT_BAND_DS2: downstream band number 2
 * 		- MT_BAND_US2: upstream band number 2
 * 		- MT_BAND_DS3: downstream band number 3
 * 		- MT_BAND_US3: upstream band number 3
 * 		- MT_BAND_DS4: downstream band number 4
 * 		- MT_BAND_US4: upstream band number 4
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linebandsigatten(mt_uint32 portid,mt_uint8 band,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch (band){
	case MT_BAND_UPSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 10);
		break;
	case MT_BAND_DOWNSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 11);
		break;
	case MT_BAND_US0:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 1);
		break;
	case MT_BAND_DS1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 6);
		break;
	case MT_BAND_US1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 2);
		break;
	case MT_BAND_DS2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 7);
		break;
	case MT_BAND_US2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 3);
		break;
	case MT_BAND_DS3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 8);
		break;
	case MT_BAND_US3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 4);
		break;
	case MT_BAND_DS4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 9);
		break;
	case MT_BAND_US4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSigAtten,value, 5);
		break;
	default:
		ret = DSL_ERR_MISC;
		break;
	}
	return ret;
}

/**
 * get_xdsl_linebandsnr gets xdsl line band SNR margin (0.1 dB)and return to user
 * \param portid
 * 		port ID number
 * \param band
 * 		band to get
 * 		- MT_BAND_UPSTREAM: whole upstream
 * 		- MT_BAND_DOWNSTREAM: whole downstream
 * 		- MT_BAND_US0: upstream band number 0
 *  	- MT_BAND_DS1: downstream band number 1
 * 		- MT_BAND_US1: upstream band number 1
 * 		- MT_BAND_DS2: downstream band number 2
 * 		- MT_BAND_US2: upstream band number 2
 * 		- MT_BAND_DS3: downstream band number 3
 * 		- MT_BAND_US3: upstream band number 3
 * 		- MT_BAND_DS4: downstream band number 4
 * 		- MT_BAND_US4: upstream band number 4
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_linebandsnr(mt_uint32 portid,mt_uint8 band,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch (band){
	case MT_BAND_UPSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 10);
		break;
	case MT_BAND_DOWNSTREAM:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 11);
		break;
	case MT_BAND_US0:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 1);
		break;
	case MT_BAND_DS1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 6);
		break;
	case MT_BAND_US1:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 2);
		break;
	case MT_BAND_DS2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 7);
		break;
	case MT_BAND_US2:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 3);
		break;
	case MT_BAND_DS3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 8);
		break;
	case MT_BAND_US3:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 4);
		break;
	case MT_BAND_DS4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 9);
		break;
	case MT_BAND_US4:
		ret = get_mib2_int_idx(portid,xdsl2LineBandStatusSnrMargin,value, 5);
		break;
	default:
		ret = DSL_ERR_MISC;
		break;
	}
	return ret;
}

/**
 * get_xdsl_seglinreal gets xdsl real component of Hlin value and return to user
 * \param portid
 * 		port ID number
 * \param site
 * 		site to get
 * 		- MT_SITE_LOCAL: local site
 * 		- MT_SITE_REMOTE: remote site
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512*3 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_seglinreal(mt_uint32 portid,mt_uint8 site,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch(site)
		{
			case MT_SITE_LOCAL:
				 ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLinReal, value, 512*3, NULL);
				 break;
			case MT_SITE_REMOTE:
				 ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLinReal, value, 512*3, NULL, 1 + INDEX_REMOTE);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_seglinimg gets xdsl imaginary component of Hlin value and return to user
 * \param portid
 * 		port ID number
 * \param site
 * 		site to get
 * 		- MT_SITE_LOCAL: local site
 * 		- MT_SITE_REMOTE: remote site
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512*3 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_seglinimg(mt_uint32 portid,mt_uint8 site,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch(site)
		{
			case MT_SITE_LOCAL:
				 ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLinImg, value, 512*3, NULL);
				 break;
			case MT_SITE_REMOTE:
				 ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLinImg, value, 512*3, NULL, 1 + INDEX_REMOTE);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_seglog gets xdsl real H(f) logarithmic values in dB and return to user
 * \param portid
 * 		port ID number
 * \param site
 * 		site to get
 * 		- MT_SITE_LOCAL: local site
 * 		- MT_SITE_REMOTE: remote site
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512*3 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_seglog(mt_uint32 portid,mt_uint8 site,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch(site)
		{
			case MT_SITE_LOCAL:
				 ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLog, value, 512*3, NULL);
				 break;
			case MT_SITE_REMOTE:
				 ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLog, value, 512*3, NULL, 1 + INDEX_REMOTE);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_segqln gets xdsl real Quiet Line Noise values in dBm/Hz and return to user
 * \param portid
 * 		port ID number
 * \param site
 * 		site to get
 * 		- MT_SITE_LOCAL: local site
 * 		- MT_SITE_REMOTE: remote site
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_segqln(mt_uint32 portid,mt_uint8 site,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch(site)
		{
			case MT_SITE_LOCAL:
				 ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentQln, value, 512, NULL);
				 break;
			case MT_SITE_REMOTE:
				 ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentQln, value, 512, NULL, 1 + INDEX_REMOTE);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_segsnr gets xdsl the SNR Margin per SC group and return to user
 * \param portid
 * 		port ID number
 * \param site
 * 		site to get
 * 		- MT_SITE_LOCAL: local site
 * 		- MT_SITE_REMOTE: remote site
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_segsnr(mt_uint32 portid,mt_uint8 site,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch(site)
		{
			case MT_SITE_LOCAL:
				 ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentSnr, value, 512, NULL);
				 break;
			case MT_SITE_REMOTE:
				 ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentSnr, value, 512, NULL, 1 + INDEX_REMOTE);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_segbitsgainalloc gets the bits/gain allocation per SC and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an hex array
 * 		buffer size must be >= 512*3 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_segbitsgainalloc(mt_uint32 portid,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentBitsGainAlloc, value, 512*3, NULL);
	return ret;
}

/**
 * set_xdsl_segment sets the segment of the SC status information provided by this row
 * \param portid
 * 		port ID number
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret set_xdsl_segment(mt_uint32 portid,mt_uint32 seg)
{
	mt_uint32 ret = MT_OK;
	mt_uint8 row[6];
	sprintf(row, "%06x", seg);
#ifdef __KERNEL__
	ret = spdl_body(portid, xdsl2SCStatusSegment, row);
#else	
	ret = set_port_mib(portid, xdsl2SCStatusSegment, row);
#endif
	return ret;
}

/**
 * get_xdsl_g994vendorid gets xdsl G.994.1 vendor id and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an octec string
 * 		buffer size must be >= 9 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_g994vendorid(mt_uint32 portid,mt_uint8 unit,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch (unit){
	case MT_UNIT_XTUC:
		ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, value, 9, NULL, 1);
		break;
	case MT_UNIT_XTUR:
		ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, value, 9, NULL, 2);
		break;
	default:
		 ret = DSL_ERR_MISC;
		 break;
	}

	return ret;
}

/**
 * get_xdsl_sysvendorid gets xdsl system vendor id and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an octec string
 * 		buffer size must be >= 9 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_sysvendorid(mt_uint32 portid,mt_uint8 unit,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	switch (unit){
	case MT_UNIT_XTUC:
		ret = get_mib2_uint8_idx(portid, xdsl2LInvSystemVendorId, value, 9, NULL, 1);
		break;
	case MT_UNIT_XTUR:
		ret = get_mib2_uint8_idx(portid, xdsl2LInvSystemVendorId, value, 9, NULL, 2);
		break;
	default:
		ret = DSL_ERR_MISC;
		break;
	}

	return ret;
}

/**
 * get_xdsl_vernumber gets xTU version number and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an octec string
 * 		buffer size must be >= 36 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_vernumber(mt_uint32 portid,mt_uint8 unit,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	mt_uint8 buf[MIBSTRSIZE],loid[OID_MAX_LEN];
	
	switch (unit){
	case MT_UNIT_XTUC:
		sprintf(loid, "%s.1", xdsl2LInvVersionNumber);
		break;
	case MT_UNIT_XTUR:
		sprintf(loid, "%s.2", xdsl2LInvVersionNumber);
		break;
	default:
		return DSL_ERR_MISC;
	}
#ifdef __KERNEL__
	ret = gpdl_body(portid, loid, buf);
#else	
	ret = get_port_mib(portid, loid, buf);
#endif
	memcpy(value, buf + 2, 36);
	
	return ret;
}

/**
 * get_xdsl_serialnumber gets xTU serial number and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an octec string
 * 		buffer size must be >= 66 bytes
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_serialnumber(mt_uint32 portid,mt_uint8 unit,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	mt_uint8 buf[MIBSTRSIZE],loid[OID_MAX_LEN];
	
	switch (unit){
	case MT_UNIT_XTUC:
		sprintf(loid, "%s.1", xdsl2LInvSerialNumber);
		break;
	case MT_UNIT_XTUR:
		sprintf(loid, "%s.2", xdsl2LInvSerialNumber);
		break;
	default:
		return DSL_ERR_MISC;
	}
#ifdef __KERNEL__
	ret = gpdl_body(portid, loid, buf);
#else	
	ret = get_port_mib(portid, loid, buf);
#endif
	memcpy(value, buf + 2, 66);
	
	return ret;
}

/**
 * get_xdsl_selftest gets xTU self-test result and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 		- 0: self-test passed
 * 		- 1: self-test failed
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_selftest(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	
	switch (unit){
	case MT_UNIT_XTUC:
		ret = get_mib2_int_idx(portid, xdsl2LInvSelfTestResult, value, 1);
		break;
	case MT_UNIT_XTUR:
		ret = get_mib2_int_idx(portid, xdsl2LInvSelfTestResult, value, 2);
		break;
	default:
		ret = DSL_ERR_MISC;
		break;
	}
	
	return ret;
}

/**
 * get_xdsl_txcapability gets xTU transmission system capability list and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an octec string
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_txcapability(mt_uint32 portid,mt_uint8 unit,mt_uint8 *value)
{
	mt_uint32 ret = MT_OK;
	mt_uint8 buf[MIBSTRSIZE],loid[OID_MAX_LEN];
	
	switch (unit){
	case MT_UNIT_XTUC:
		sprintf(loid, "%s.1", xdsl2LInvTransmissionCapabilities);
		break;
	case MT_UNIT_XTUR:
		sprintf(loid, "%s.2", xdsl2LInvTransmissionCapabilities);
		break;
	default:
		return DSL_ERR_MISC;
	}
#ifdef __KERNEL__
	ret = gpdl_body(portid, loid, buf);
#else	
	ret = get_port_mib(portid, loid, buf);
#endif
	memcpy(value, buf + 2, ((buf[0] << 8) | buf[1])*6);
	
	return ret;
}

/**
 * get_xdsl_line_15m_timeelapsed gets xdsl 15 min time seconds and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_timeelapsed(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MTimeElapsed,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MTimeElapsed,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_15m_fecs gets xdsl current 15 min FECs and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_fecs(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MFecs,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MFecs,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_15m_es gets xdsl current 15 min error seconds and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_es(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MEs,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MEs,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_15m_ses gets xdsl current 15 min Ses and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_ses(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MSes,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MSes,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_15m_loss gets xdsl current 15 min LOSs and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_loss(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MLoss,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MLoss,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_15m_uas gets xdsl current 15 min Unavailability State and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_15m_uas(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MUas,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr15MUas,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_timeelapsed gets xdsl 1 day time seconds and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_timeelapsed(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayTimeElapsed,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayTimeElapsed,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_fecs gets xdsl current 1 day FECs and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_fecs(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayFecs,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayFecs,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_es gets xdsl current 1 day error seconds and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_es(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayEs,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayEs,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_ses gets xdsl current 1 day Ses and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_ses(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DaySes,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DaySes,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_loss gets xdsl current 1 day LOSs and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_loss(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayLoss,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayLoss,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_line_1day_uas gets xdsl current 1 day Unavailability State and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_line_1day_uas(mt_uint32 portid,mt_uint8 unit,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(unit)
		{
			case MT_UNIT_XTUC:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayUas,value, 1);
				 break;
			case MT_UNIT_XTUR:
				 ret = get_mib2_int_idx(portid,xdsl2PMLCurr1DayUas,value, 2);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_ch_15m_timeelapsed gets xdsl 15 min total elapsed seconds in this interval and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_15m_timeelapsed(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MTimeElapsed,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MTimeElapsed,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MTimeElapsed,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MTimeElapsed,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}


/**
 * get_xdsl_ch_15m_cv gets xdsl 15 min count of CRC and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_15m_cv(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCodingViolations,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCodingViolations,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCodingViolations,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCodingViolations,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_ch_15m_fec gets xdsl 15 min count of FEC and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_15m_fec(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCorrectedBlocks,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCorrectedBlocks,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCorrectedBlocks,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr15MCorrectedBlocks,value, 4);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_ch_1day_timeelapsed gets xdsl 1 day total elapsed seconds in this interval and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_1day_timeelapsed(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayTimeElapsed,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayTimeElapsed,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayTimeElapsed,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayTimeElapsed,value, 4);
				 break;				 
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_ch_1day_cv gets xdsl 1 day count of CRC and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_1day_cv(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCodingViolations,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCodingViolations,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCodingViolations,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCodingViolations,value, 4);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_ch_1day_fec gets xdsl 1 day count of FEC and return to user
 * \param portid
 * 		port ID number
 * \param channel
 * 		channel to get  
 * 		- MT_CH_US0~MT_CH_US1: upstream channel
 * 		- MT_CH_DS0~MT_CH_DS1: downstream channel
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_ch_1day_fec(mt_uint32 portid,mt_uint8 channel,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	switch(channel)
		{
			case MT_CH_US0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCorrectedBlocks,value, 1);
				 break;
			case MT_CH_US1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCorrectedBlocks,value, 2);
				 break;
			case MT_CH_DS0:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCorrectedBlocks,value, 3);
				 break;
			case MT_CH_DS1:
				 ret = get_mib2_int_idx(portid,xdsl2PMChCurr1DayCorrectedBlocks,value, 4);
				 break;
			default:
				 ret = DSL_ERR_MISC;
				 break;
		}
	return ret;
}

/**
 * get_xdsl_15m_init_validinterval gets xdsl 15 mins init valid intervals and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_validinterval(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MValidIntervals0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_invalidinterval gets xdsl 15 mins init invalid intervals and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_invalidinterval(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MInvalidIntervals0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_timeelapsed gets xdsl 15 mins init elapsed seconds in this interval and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_timeelapsed(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MTimeElapsed0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_fullinit gets xdsl 15 mins count of full initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_fullinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MFullInits0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_failfullinit gets xdsl 15 mins count of failed full initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_failfullinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MFailedFullInits0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_shortinit gets xdsl 15 mins count of short initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_shortinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MShortInits0,value);
	return ret;
}

/**
 * get_xdsl_15m_init_failshortinit gets xdsl 15 mins count of failed short initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_15m_init_failshortinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr15MFailedShortInits0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_validinterval gets xdsl 1 day init valid intervals and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_validinterval(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayValidIntervals0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_invalidinterval gets xdsl 1 day init invalid intervals and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_invalidinterval(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayInvalidIntervals0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_timeelapsed gets xdsl 1 day init elapsed seconds in this interval and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_timeelapsed(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayTimeElapsed0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_fullinit gets xdsl 1 day count of full initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_fullinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayFullInits0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_failfullinit gets xdsl 1 day count of failed full initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_failfullinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayFailedFullInits0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_shortinit gets xdsl 1 day count of short initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_shortinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayShortInits0,value);
	return ret;
}

/**
 * get_xdsl_1day_init_failshortinit gets xdsl 1 day count of failed short initializations and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_1day_init_failshortinit(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid,xdsl2PMLInitCurr1DayFailedShortInits0,value);
	return ret;
}

/**
 * get_xdsl_threshold gets xdsl threshold and return to user
 * \param portid
 * 		port ID number
 * \param unit
 * 		Termination unit to get  
 * 		- MT_UNIT_XTUC: Central Office
 * 		- MT_UNIT_XTUR: Remote site
 * \param value
 * 		buffer to receive the returned value
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_xdsl_threshold(mt_uint32 portid,mt_uint8 direction,PMCOUNTER *value)
{
	mt_uint32 ret = MT_OK;
	switch(direction)
		{
			case MT_UNIT_XTUC:
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinFecs, &value->fecs)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinEs, &value->es)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinSes, &value->ses)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinLoss, &value->loss)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinUas, &value->uas)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2ChAlarmConfProfileXtucThresh15MinCodingViolations, &value->cv)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2ChAlarmConfProfileXtucThresh15MinCorrected, &value->fec)) != MT_OK) break;
				break;
			case MT_UNIT_XTUR:
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinFecs, &value->fecs)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinEs, &value->es)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinSes, &value->ses)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinLoss, &value->loss)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinUas, &value->uas)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2ChAlarmConfProfileXturThresh15MinCodingViolations, &value->cv)) != MT_OK) break;
				if ((ret = get_mib2_int(portid, xdsl2ChAlarmConfProfileXturThresh15MinCorrected, &value->fec)) != MT_OK) break;
				break;
			default:
				ret = DSL_ERR_MISC;
				break;
		}
	return ret;
}


/**
 * get_firmwarever gets firmware version and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_firmwarever(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid, xdslFirmwareVer, value);
	return ret;
}

/**
 * get_terminal gets termial information and return to user
 * \param portid
 * 		port ID number
 * \param value
 * 		buffer to receive the returned value, value is an integer
 * 
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_terminal(mt_uint32 portid,mt_uint32 *value)
{
	mt_uint32 ret = MT_OK;
	ret = get_mib2_int(portid, xdslTerminal, value);
	return ret;
}
#endif
#ifdef __KERNEL__
/*EXPORT_SYMBOL(get_vdsl_rate);
EXPORT_SYMBOL(set_vdsl_rate);
EXPORT_SYMBOL(get_vdsl_snrmargin);
EXPORT_SYMBOL(set_vdsl_snrmargin);
EXPORT_SYMBOL(get_vdsl_interleave);
EXPORT_SYMBOL(set_vdsl_interleave);
EXPORT_SYMBOL(get_vdsl_current_crc);
EXPORT_SYMBOL(get_vdsl_current_status);
EXPORT_SYMBOL(get_vdsl_fwversion);*/

EXPORT_SYMBOL(get_mib2_int_idx);
EXPORT_SYMBOL(get_mib2_int);
EXPORT_SYMBOL(get_mib2_uint32);
EXPORT_SYMBOL(get_mib2_uint8_idx);
EXPORT_SYMBOL(get_mib2_uint8);
#if 0
EXPORT_SYMBOL(get_xdsl_pwrmngstate);
EXPORT_SYMBOL(get_xdsl_attainablerate);
EXPORT_SYMBOL(get_xdsl_actatp);
EXPORT_SYMBOL(get_xdsl_actprofile);
EXPORT_SYMBOL(get_xdsl_actsnrmode);
EXPORT_SYMBOL(get_xdsl_eleclen);
EXPORT_SYMBOL(get_xdsl_trellis);
EXPORT_SYMBOL(get_xdsl_actualce);

EXPORT_SYMBOL(get_xdsl_actdatarate);
EXPORT_SYMBOL(get_xdsl_predatarate);
EXPORT_SYMBOL(get_xdsl_actdelay);
EXPORT_SYMBOL(get_xdsl_actinp);
EXPORT_SYMBOL(get_xdsl_inpreport);
EXPORT_SYMBOL(get_xdsl_nfec);
EXPORT_SYMBOL(get_xdsl_rfec);
EXPORT_SYMBOL(get_xdsl_lsymb);
EXPORT_SYMBOL(get_xdsl_intlvdepth);
EXPORT_SYMBOL(get_xdsl_intlvblock);
EXPORT_SYMBOL(get_xdsl_lpath);
EXPORT_SYMBOL(get_xdsl_atmstatus);
EXPORT_SYMBOL(get_xdsl_ptmstatus);

EXPORT_SYMBOL(get_xdsl_scdirection);
EXPORT_SYMBOL(get_xdsl_linscale);
EXPORT_SYMBOL(get_xdsl_linscgpsize);
EXPORT_SYMBOL(get_xdsl_logmt);
EXPORT_SYMBOL(get_xdsl_logscgpsize);
EXPORT_SYMBOL(get_xdsl_qlnmt);
EXPORT_SYMBOL(get_xdsl_qlnscgpsize);
EXPORT_SYMBOL(get_xdsl_snrmtime);
EXPORT_SYMBOL(get_xdsl_snrscgpsize);
//EXPORT_SYMBOL(get_xdsl_scattainrate);
EXPORT_SYMBOL(get_xdsl_rowstatus);

EXPORT_SYMBOL(get_xdsl_linebandid);
EXPORT_SYMBOL(get_xdsl_linebandlnatten);
EXPORT_SYMBOL(get_xdsl_linebandsigatten);
EXPORT_SYMBOL(get_xdsl_linebandsnr);

EXPORT_SYMBOL(get_xdsl_seglinreal);
EXPORT_SYMBOL(get_xdsl_seglinimg);
EXPORT_SYMBOL(get_xdsl_seglog);
EXPORT_SYMBOL(get_xdsl_segqln);
EXPORT_SYMBOL(get_xdsl_segsnr);
EXPORT_SYMBOL(get_xdsl_segbitsgainalloc);

EXPORT_SYMBOL(set_xdsl_segment);

EXPORT_SYMBOL(get_xdsl_g994vendorid);
EXPORT_SYMBOL(get_xdsl_sysvendorid);
EXPORT_SYMBOL(get_xdsl_vernumber);
EXPORT_SYMBOL(get_xdsl_serialnumber);
EXPORT_SYMBOL(get_xdsl_selftest);
EXPORT_SYMBOL(get_xdsl_txcapability);

EXPORT_SYMBOL(get_xdsl_line_15m_timeelapsed);
EXPORT_SYMBOL(get_xdsl_line_15m_fecs);
EXPORT_SYMBOL(get_xdsl_line_15m_es);
EXPORT_SYMBOL(get_xdsl_line_15m_ses);
EXPORT_SYMBOL(get_xdsl_line_15m_loss);
EXPORT_SYMBOL(get_xdsl_line_15m_uas);
EXPORT_SYMBOL(get_xdsl_line_1day_timeelapsed);
EXPORT_SYMBOL(get_xdsl_line_1day_fecs);
EXPORT_SYMBOL(get_xdsl_line_1day_es);
EXPORT_SYMBOL(get_xdsl_line_1day_ses);
EXPORT_SYMBOL(get_xdsl_line_1day_loss);
EXPORT_SYMBOL(get_xdsl_line_1day_uas);

EXPORT_SYMBOL(get_xdsl_ch_15m_timeelapsed);
EXPORT_SYMBOL(get_xdsl_ch_15m_cv);
EXPORT_SYMBOL(get_xdsl_ch_15m_fec);
EXPORT_SYMBOL(get_xdsl_ch_1day_timeelapsed);
EXPORT_SYMBOL(get_xdsl_ch_1day_cv);
EXPORT_SYMBOL(get_xdsl_ch_1day_fec);

EXPORT_SYMBOL(get_xdsl_15m_init_validinterval);
EXPORT_SYMBOL(get_xdsl_15m_init_invalidinterval);
EXPORT_SYMBOL(get_xdsl_15m_init_timeelapsed);
EXPORT_SYMBOL(get_xdsl_15m_init_fullinit);
EXPORT_SYMBOL(get_xdsl_15m_init_failfullinit);
EXPORT_SYMBOL(get_xdsl_15m_init_shortinit);
EXPORT_SYMBOL(get_xdsl_15m_init_failshortinit);
EXPORT_SYMBOL(get_xdsl_1day_init_validinterval);
EXPORT_SYMBOL(get_xdsl_1day_init_invalidinterval);
EXPORT_SYMBOL(get_xdsl_1day_init_timeelapsed);
EXPORT_SYMBOL(get_xdsl_1day_init_fullinit);
EXPORT_SYMBOL(get_xdsl_1day_init_failfullinit);
EXPORT_SYMBOL(get_xdsl_1day_init_shortinit);
EXPORT_SYMBOL(get_xdsl_1day_init_failshortinit);

EXPORT_SYMBOL(get_xdsl_threshold);

EXPORT_SYMBOL(get_firmwarever);
EXPORT_SYMBOL(get_terminal);
#endif
#endif

