/****************************************************************************
* File:         $Workfile$
* Revision:     $Revision: 1.49 $
* Date:         $Date: 2012-05-23 12:44:15 $
*
* Overview:
* Functions:
*
* Author:       Ian Tsao
*
* -- Metanoia Copyright Notice --
*
* (C) Copyright 2005 Metanoia, Inc. All rights reserved.
*
* Metanoia reserves the right to change specifications without notice.
* Copyright 2005 Metanoia, Inc. All rights reserved. Preliminary Data-Sheet
* indicates this product is in design and the specification may change.
* Electrical parametrics have not been analyzed and are not specified. Do not
* use this data sheet as a design reference document. Please contact Metanoia
* for a current data sheet that may be used as a design reference. All
* contents of this document are protected by copyright law and may not be
* reproduced without the express written consent of Metanoia, Inc. Metanoia,
* the Metanoia logo, and combinations thereof are trademarks of Metanoia, Inc.
* Other product names used in this publication are for identification purposes
* only and may be trademarks or registered trademarks of their respective
* companies. The contents of this document are provided in connection with
* Metanoia, Inc. products. Metanoia, Inc. has made best efforts to ensure that
* the information contained herein is accurate and reliable. However, Metanoia,
* Inc. makes no warranties, express or implied, as to the accuracy or
* completeness of the contents of this publication and is providing this
* publication 'as is'. Metanoia, Inc. reserves the right to make changes to
* specifications and product descriptions at any time without notice, and to
* discontinue or make changes to its products at any time without notice.
* Metanoia, Inc. does not assume any liability arising out of the application
* or use of any product or circuit, and specifically disclaims any and all
* liability, including without limitation special, consequential, or
* incidental damages.
*
****************************************************************************/
#include "dsll2.h"
#include "mib2.h"

#ifdef __KERNEL__
#include <linux/time.h>
#include <asm/io.h>
#include <asm/bitops.h>

#if KERNEL_2_6 //yalee980114 for ralink
#include <linux/jiffies.h>
#if defined (RALINK_RT3052) || defined (RALINK_RT3352) || defined (VITESS_VSC7501)
#include <asm/semaphore.h>
#elif defined (ICPLUS_IP3210) || defined (ATHEROS_AR93XX) || defined (REALTEK_RTL8672)
#include <linux/semaphore.h>
#endif
#endif	//KERNEL_2_6

#else	//__KERNEL__
#include <stdio.h>
#endif	//__KERNEL__

//#include "fake_mem.h"
//mt_uint8 fake_mib[10] = {0x00, 0x01, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x00, 0x00};
#define HOST_MSG_ADDR 0x12
#define HOST_MSG_CTRL 0x12
#define ISSUE_CMD "1"
#define NUM_SEC_PER_15MIN 900
#define NUM_SEC_PER_1DAY 86400
#if defined (REALTEK_RTL8672)
#define ERASE_DELAY 7	//3.5 seconds, we use 0.5 second as delay unit
#else
#define ERASE_DELAY 1	//0.5 seconds, we use 0.5 second as delay unit
#endif
#define RETRY_COUNT 2
#ifdef OPENWRT
static DEFINE_SEMAPHORE(mr_sem_api);
#else
static DECLARE_MUTEX(mr_sem_api);
#endif	//OPENWRT
static int sem_ret;
static mt_uint8 mib2_first_access = 1;
static mt_uint8 bindrv_load = 0;


/**
 * \file
 * \brief communication module interface API
 */

// -------------------------- VDSL L2 API -----------------------
/**
 * hpi_cmd_model issue the given command id to corresponding DMT
 * \param portid
 * 		port ID number
 * \param cmdid
 * 		command ID in string format
 * \return
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret hpi_cmd_model(mt_uint32 portid, mt_uint8 *cmdid)
{
	mt_uint8 i = 0;
	mt_uint8 pdata[10];
	mt_uint32 value;
	mt_ret ret;

	ret = gpdl_body(portid, (mt_uint8 *)(xdslHostCommandStatus), pdata);
	if(ret != MT_OK) return ret;
	value = simple_strtoul((char *)(pdata+2), NULL, 16);
	while(value) {	//check if current command status is "1"(last command not complied)
		if(i++ == 100) {	//check for 100 times and give up
			PRINTD("Issue host command failed!!\n");
			return DSL_ERR_NOTOK;
		}

		//PRINTD("Waiting to issue connect command...\n");
		MYDELAY(20);

		ret = gpdl_body(portid, (mt_uint8 *)(xdslHostCommandStatus), pdata);
		if(ret != MT_OK) return ret;
		value = simple_strtoul((char *)(pdata+2), NULL, 16);
	}

	ret = spdl_body(portid, (mt_uint8 *)(xdslHostCommand), cmdid);
	if(ret != MT_OK) return ret;
	return spdl_body(portid, (mt_uint8 *)(xdslHostCommandStatus), (mt_uint8 *)(ISSUE_CMD));
}


/**
 * connect_port activates the selected port so that it would seek a connection
 * \param portid
 * 		port ID number
 * \return
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret connect_port(mt_uint32 portid)
{
	mt_ret ret;

	ret = hpi_cmd_model(portid, (mt_uint8 *)(CONNECT_PORT));
	PRINTL(DBG_LVL_IF, "connect_port return %i\n", ret);
	if(ret == MT_OK)
		port_info[portid].status = MT_OPSTATE_HANDSHAKE;

	connect_status = TRUE;

	return  ret;
}


/**
 * DisconnectPort inactivates the selected port so that it stops responding to far end connection attempts
 * \param portid
 * 		port ID number
 * \return
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret disconnect_port(mt_uint32 portid)
{
	mt_ret ret;

	ret = hpi_cmd_model(portid, (mt_uint8 *)(DISCONNECT_PORT));
	PRINTL(DBG_LVL_IF, "disconnect_port return %i\n", ret);
	if(ret == MT_OK)
		port_info[portid].status = MT_OPSTATE_IDLE;

	connect_status = FALSE;

	//MYDELAY(500);

	return ret;
}


/**
 * ShutdownSystem drops all the connections and brings system to low power mode. Only way to bring it back to normal state is to execute InitSystem
 * \return
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret shutdown_system(void)
{
	mt_uint8 chipid;

	for(chipid=0;chipid<MAX_CHIPS;chipid++)
		shutdown_chip(chipid);

	return MT_OK;
}


/**
 * ShutdownChip drops connections of all chips and brings them to low power mode. Only way to bring it back to normal state is to execute InitChip or InitSystem
 * \param chipid
 * 		chip ID number
 * 		- 0 : success
 * 		- < 0 : error code
 */
mt_ret shutdown_chip(mt_uint32 chipid)
{
	mt_ret ret;

	ret = hpi_cmd_model(chipid, (mt_uint8 *)(SHUTDOWN_CHIP));
	PRINTL(DBG_LVL_IF, "disconnect_port return %i\n",ret);
	if(ret == MT_OK)
		port_info[chipid].status = MT_OPSTATE_IDLE;

	MYDELAY(500);

	return ret;
}


/**
 * Init Modem by chipid
 */
mt_ret init_modem(mt_uint32 chipid)
{
	mt_ret ret;

	ret = hpi_cmd_model(chipid, (mt_uint8 *)(INIT_MODEM));
	PRINTL(DBG_LVL_IF, "disconnect_port return %i\n",ret);
	if(ret == MT_OK)
		port_info[chipid].status = MT_OPSTATE_IDLE;

	MYDELAY(500);

	return ret;
}

/**
 * Reset Modem by chipid
 */
mt_ret reset_modem(mt_uint32 chipid)
{
	mt_ret ret;

	ret = hpi_cmd_model(chipid, (mt_uint8 *)(RESET_MODEM));
	PRINTL(DBG_LVL_IF, "disconnect_port return %i\n",ret);
	if(ret == MT_OK)
		port_info[chipid].status = MT_OPSTATE_IDLE;

	MYDELAY(500);

	return ret;
}


int check_sum(void)
{
	int chksum, toggle;
	int i;
	int bd_len;
	int i1;
	chksum = 0;
	toggle = 0;

	i = 0;
	bd_len = BD_HDR_SIZE + GET_PKG_INT(bd[BD_tbl_max]) + GET_PKG_INT(bd[BD_par_max]) * GET_PKG_INT(bd[BD_tbl_max]);
	for(i=2;i<bd_len;i++) {
		i1 = GET_PKG_INT(bd[i]);
		if(i1 > 0) {
			if(toggle) chksum = chksum << 1;
			chksum = chksum + i1;
			chksum = chksum & 0X7fffffff;
			toggle = ! toggle;
		}
		else {
			chksum = chksum + 1;
		}
	}

	return chksum;
}


mt_ret get_mib2_header(mt_uint32 portid)
{
	mt_ret ret = MT_OK;
	mt_uint8 pdata[DMT_WORD];
	mt_uint8 strbuf1[7], strbuf2[7];
	mt_uint32 address;
	mt_uint32 i;
	//int chk_value;

	if(mib2_first_access) {
		if(bindrv_load == 0) {
			PRINTL(DBG_LVL_IF, "Binary Driver doesn't load!!\n");
			return DSL_ERR_BINDRVNOTLOAD; //no driver
		}

		//check identification
		if(GET_PKG_INT(bd[BD_identification]) != BD_signature) {
			PRINTL(DBG_LVL_IF, "Binary Driver Signature mismatch!!\n");
			return DSL_ERR_BINDRVSIGNATURE; //identification fail
		}

		//check checksum
		/*chk_value = check_sum();
		if(chk_value != GET_PKG_INT(bd[BD_chksum])) {
			PRINTL(DBG_LVL_IF, "Checksum Error!!\n");
			return DSL_ERR_BINDRVCHKSUM;
		}*/

		// check mib revision
		ret = new_dsp_get(portid, GET_PKG_INT(bd[BD_mib_start]), MT_MEMTYPE_X, pdata);	//memtype always be 0 means memtype always be X memory
		sprintf((char *)strbuf1, "%02X%02X%02X", pdata[0], pdata[1], pdata[2]);
		sprintf((char *)strbuf2, "%06X", GET_PKG_INT(bd[BD_mib_version]));
		if(strncmp((char *)strbuf1, (char *)strbuf1, 6) != 0) {
			PRINTL(DBG_LVL_IF, "mib version not compatiable!!\n");
			ret = DSL_ERR_REVISION; // version not compatiable
		}
		
		// get table start address
		for(i=0;i<GET_PKG_INT(bd[BD_tbl_fw]);i++){
			address = GET_PKG_INT(bd[BD_tbl_start])+i;
			ret = new_dsp_get(portid, address, MT_MEMTYPE_X, pdata);	//memtype always be 0 means memtype always be X memory
			add_start[i] = (pdata[1] << 8) + pdata[2];
		}

		mib2_first_access = 0;

		// temporarily disable end mark checking
		/*if(ret == MT_OK){
			// check end mark
			address = GET_PKG_INT(bd[BD_mib_start])+GET_PKG_INT(bd[BD_mib_size])+1+BD_GRP_MAX+GET_PKG_INT(bd[BD_tbl_max]);
			ret = new_dsp_get(portid, address, MT_MEMTYPE_X, pdata);	//memtype always be 0 means memtype always be X memory
			sprintf(strbuf1, "%02X%02X%02X", pdata[0], pdata[1], pdata[2]);
			if(strncmp(strbuf1, end_mark, 6) != 0) {
				PRINTL(DBG_LVL_IF, "end marker not correct\n");
				ret = DSL_ERR_ENDMARK; // end marker not correct
			}
		}*/
	}

	return ret;
}


/**
 * trans_mib_addr translate VDSL2 MIB into the corresponding memory address.
 * \param groupid
 * 		identifies MIB group
 * 		- 0: vdsl group
 * 		- 1: vdslNotifications group
 * 		- 2: vdslMCM group
 * \param tableid
 * 		MIB table ID from vdslmib.h
 * \param index
 * 		index type (vdsl(97), fast(125), interleaved(124))
 * \param paramid
 * 		MIB parameter ID from vdslmib.h
 * \param address
 * 		translated address
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 */
int trans_mib_addr(mt_uint32 groupid, mt_uint32 tableid, mt_uint32 idx,
					  mt_uint32 paramid, mt_uint32 *address)
{
	int tbl_real, par_real, idx_real, tbl_hdr;//, idx_idx;
	int tbl_real_last;
	int tbl_max, par_max;
	int idx_rev;
	int par_addr1, par_addr0, par_value1, par_value0;
	int mibsize;
	int first = 1;
	int i, idx_grp, idx_tbl, idx_size;

	//printk("groupid: %d, tableid: %d, idx: %d, paramid: %d\n", groupid, tableid, idx, paramid);
	
	tableid--;
	paramid--;

	tbl_max = GET_PKG_INT(bd[BD_tbl_max]);
	par_max = GET_PKG_INT(bd[BD_par_max]);
	tbl_hdr = -1;
	for(i=0;i<tbl_max;i++) {
		idx_real = GET_PKG_INT(bd[BD_HDR_SIZE+i]);
		idx_grp = (idx_real & 0xe0000000) >> 29;
		idx_tbl = (idx_real & 0x1f000000) >> 24;

		if(idx_grp == groupid && idx_tbl == tableid) {
			tbl_hdr = i;
			break;
		}

	}
	if(tbl_hdr == -1) return -1;
	

	tbl_real = tbl_hdr;
	// grp+table+reverse+idx

	idx_real = GET_PKG_INT(bd[BD_HDR_SIZE + tbl_hdr]);
	// grp[31:29],tbl[28:24],idx_size[23:8],rev_idx[7:4],idx[3:0]
	idx_size = (idx_real & 0x00ffff00 ) >> 8;
	idx_rev = (idx_real & 0xf0) >> 4;
	//idx_idx = idx_real & 0xf ;

	tbl_real_last = tbl_real;
	par_real = paramid;
	while(par_real >= par_max) {
		if(first) {
			if(idx_rev > 0) { // this table parmeter count>64
				tbl_real = tbl_max-idx_rev;
			}
			first=0;
		}
		else {
			tbl_real_last = tbl_real;
			tbl_real--;
		}
		par_real -= par_max;
	}

	par_addr1 = BD_HDR_SIZE + tbl_max + par_max * tbl_real + par_real;
	par_value1 = GET_PKG_INT(bd[par_addr1]);
	if(par_real != 0) {
		par_addr0 = par_addr1 - 1;
		par_value0 = GET_PKG_INT(bd[par_addr0]);
	}
	else {
		if(first) {
			par_value0 = 0;
		}
		else{ // rev row with col=0
			par_addr0 = BD_HDR_SIZE + tbl_max + par_max * (tbl_real_last) + par_max - 1;
			par_value0 = GET_PKG_INT(bd[par_addr0]);
		}
	}

	mibsize = par_value1 - par_value0;
	
	*address = add_start[tbl_hdr] + par_value0 + (idx-1) * idx_size;

	//printk("mibsize: %x, adddress: %x\n", mibsize, *address);

	return mibsize;
}


/**
 * oid2logic convert standard VDSL2 MIB OID to DMT logic OID.
 * \param oid
 * 		MIB Object ID string
 * 
 * \return
 * 	- 0 : convertion failed
 * 	- 1 : convertion success
 */
int oid2logic(char *oid, char *loid){
	mt_uint8 i;
	mt_uint8 str_find = 1;
	//mt_uint8 version = 1;
	char* oid_list[MAXOID]={
		"1.1.1.1", "1.1.2.1", "1.2.2.1", "1.2.3.1", "1.2.4.1",
		"1.2.5.1", "1.3.1.1", "1.5.1.1.1", "1.5.1.2.1", "1.5.1.3.1",
		"1.5.1.4.1", "1.5.2.1.1", "1.5.3.1.1", "1.5.3.2.1", "1.5.3.3.1",
		"1.4.1.1.1", "1.4.1.2.1", "1.4.1.3.1", "1.4.1.4.1", "1.4.1.5.1",
		"1.4.1.6.1", "1.4.2.1.1", "1.4.2.2.1", "1.4.2.3.1", "0.0.0",
		"1.1", "1.2",	//ghs
		"2.1",			//metanoia
		"3.1", "3.2"	//metanoia2
	};
	char* oid_logic[MAXOID]={
		"0.1", "0.2", "0.3", "0.4", "0.5",
		"0.6", "0.7", "0.8", "0.9", "0.10",
		"0.11", "0.12", "0.13", "0.14", "0.15",
		"0.16", "0.17", "0.18", "0.19", "0.20",
		"0.21", "0.22", "0.23", "0.24", "0.0",
		"1.1", "1.2",
		"2.1",
		"3.1", "3.2"
	};

	for(i=0;i<MAXOID;i++) {
		/*str_find = strstr(oid, oid_list[i]);
		if(str_find == NULL) continue;
		else break;*/
		str_find = strncmp(oid_list[i], oid, strlen(oid_list[i]));
		if(str_find == 0) break;	// target oid substring found
	}
	if(str_find == 0) {
		if(strlen(oid) >= OID_MAX_LEN) return DSL_ERR_NOTOK;
		strcpy(loid, oid_logic[i]);
		strcat(loid, oid+strlen(oid_list[i]));
		//printk("OID: %s LOID: %s\n", oid, loid);
		return MT_OK;
	}

	return DSL_ERR_NODATA;
}


/**
 * oid2gtip convert DMT logic OID string to integer group, table, parameter, index format
 * \param oid
 * 		MIB Object ID string
 * \param groupid
 * 		identifies MIB group
  * \param tableid
 * 		MIB table ID from vdslmib.h
 * \param index
 * 		index type (vdsl(97), fast(125), interleaved(124))
 * \param paramid
 * 		MIB parameter ID from vdslmib.h 
 * 
 * \return
 * 	- FALSE : convertion failed
 * 	- TRUE : convertion success
 */
mt_ret oid2gtip(mt_uint8 *oid, mt_uint32 *groupid, mt_uint32 *tableid, mt_uint32 *paramid, mt_uint32 *idx, mt_uint32 *interval)
{
	mt_uint8 oidtmp[OID_MAX_LEN];
	char *token;
#if KERNEL_2_6 //yalee980119 for ralink
	char *bp = (char *)oidtmp;
#endif	

	strcpy((char *)oidtmp, (char *)oid);
	//printk("oid: %s\n", oidtmp);

#if KERNEL_2_6//strsep relpace strtok in kernel 2.6
	token = strsep(&bp, ".");
	if(token == NULL) return FALSE;
	*groupid = simple_strtoul(token, NULL, 10);

	token = strsep(&bp, ".");
	if(token == NULL) return FALSE;
	*tableid = simple_strtoul(token, NULL, 10);

	token = strsep(&bp, ".");
	if(token == NULL) return FALSE;
	*paramid = simple_strtoul(token, NULL, 10);

	token = strsep(&bp, ".");
	if(token == NULL) *idx = 1;
	else *idx = simple_strtoul(token, NULL, 10);

	if(interval != NULL) {
		token = strsep(&bp, ".");
		if(token == NULL) *interval = 0xff;
		else *interval = simple_strtoul(token, NULL, 10);
	}
#else //KERNEL_2_4
	token = strtok((char *)oidtmp, ".");
	if(token == NULL) return FALSE;
	*groupid = simple_strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) return FALSE;
	*tableid = simple_strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) return FALSE;
	*paramid = simple_strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) *idx = 1;
	else *idx = simple_strtoul(token, NULL, 10);

	if(interval != NULL) {
		token = strtok(NULL, ".");
		if(token == NULL) *interval = 0xff;
		else *interval = simple_strtoul(token, NULL, 10);
	}
#endif

	//if(interval != NULL) printk("g:%d, t:%d, p:%d, i:%d, i:%d\n", *groupid, *tableid, *paramid, *idx, *interval);
	//else printk("g:%d, t:%d, p:%d, i:%d\n", *groupid, *tableid, *paramid, *idx);
	
	return TRUE;
}


/**
 * get_mib2_info returns the information for user queried MIB OID.
 * \param loid
 * 		MIB Object ID string
 * \param address
 * 		translated address
 * \param mibsize
 * 		translated mibsize
 * \param mibtype
 * 		translated mibtype
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 */
mt_ret get_mib2_info(mt_uint8 *loid, mt_uint32 *address, mt_uint32 *mibsize, mt_uint32 *mibtype)
{
	mt_ret ret = MT_OK;
	mt_uint32 groupid, tableid, idx, paramid;
	
	if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
		ret = get_mib2_header(0);	//always use port 1 to check mib2 header status
		if(ret != MT_OK) {
			PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
			goto end_get_mib2_info;
		}

		*mibsize = trans_mib_addr(groupid, tableid, idx, paramid, address);
		if(*mibsize == 0xffffffff) { 
			ret = DSL_ERR_NOSUCHMIB;
			goto end_get_mib2_info;
		}

		*mibtype = 0;	//we can't translate mibtype for now, so we make the default value to "0"
	}
	else {
		PRINTL(DBG_LVL_IF, "Incomplete OID\n");
		ret =  DSL_ERR_NOTOK;
	}

end_get_mib2_info:
	return ret;
}


/**
 * set_port_data sets a parameter in the MIB set for the selected port.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param value
 * 		new value for the parameter
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 */
mt_ret set_port_data(mt_uint32 portid, mt_uint8 *oid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 state;
	mt_uint8 pdata[DMT_WORD], tmp[7];
	mt_uint8 loid[OID_MAX_LEN];
	mt_uint32 i, value;
	mt_uint32 port;
	mt_uint32 address;
	mt_uint32 groupid, tableid, idx, paramid;
	mt_uint32 mibsize, varsize;
	
	sem_ret = down_interruptible(&mr_sem_api);

	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	dsl_status(portid);

	memset(tmp, 0, sizeof(tmp));

	/*fake_mib[2] = strvalue[0];
	fake_mib[3] = strvalue[1];
	fake_mib[4] = strvalue[2];
	fake_mib[5] = strvalue[3];
	fake_mib[6] = strvalue[4];
	fake_mib[7] = strvalue[5];
	fake_mib[8] = strvalue[6];
	return MT_OK;*/

	if(portid == MT_PORT_BroadCast) {
		for(port = 0; port < MAX_PORTS; port++) {
			get_op_status(port, &state);
			if(state == MT_OPSTATE_DISABLE) {
				PRINTL(DBG_LVL_IF, "Port%i disabled\n", port);
				continue;
			}

			if(oid2logic((char *)oid, (char *)loid) == MT_OK) {
				if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
					ret = get_mib2_header(port);
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
						continue;
					}
					
					mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
					if(mibsize == 0xffffffff) {
						ret = DSL_ERR_NOSUCHMIB;
						goto end_set_port_data;
					}

					varsize = strlen((char *)strvalue);
					if(varsize > (mibsize * 6)) {
						ret = DSL_ERR_DATASIZE;
						goto end_set_port_data;
					}
					if(varsize > 6 && (varsize % 6) > 0) {
						ret = DSL_ERR_ALIGNMENT;
						goto end_set_port_data;
					}

					for(i=0;i<varsize;i+=6) {
						strncpy((char *)tmp, (char *)(strvalue+i), 6);
						value = simple_strtoul((char *)tmp, NULL, 16);
						pdata[0] = (value & 0xff0000)>>16;
						pdata[1] = (value & 0x00ff00)>>8;
						pdata[2] = value & 0x0000ff;
						
						ret = new_dsp_set(port, address+i/6, MT_MEMTYPE_X, pdata);	// memtype always be X memory
						if(ret != MT_OK) {
							PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", port);
							break;
						}
						//fake_mem[address+i/6] = value;	// remove this line and open above section after FW ready
					}
				}
				else {
					PRINTL(DBG_LVL_IF, "Incomplete OID\n");
					ret =  DSL_ERR_NOTOK;
				}
			}
			else {
				PRINTL(DBG_LVL_IF, "Incompatible OID\n");
				ret =  DSL_ERR_NOTOK;
			}
		}
	}
	else {
		get_op_status(portid, &state);
		if(state == MT_OPSTATE_DISABLE) {
			PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
			ret = DSL_ERR_NOTOK;
		}
		else {
			if(oid2logic((char *)oid, (char *)loid) == MT_OK) {
				if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
					ret = get_mib2_header(portid);
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
						goto end_set_port_data;
					}
		
					mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
					if(mibsize == 0xffffffff) { 
						ret = DSL_ERR_NOSUCHMIB;
						goto end_set_port_data;
					}

					varsize = strlen((char *)strvalue);
					if(varsize > (mibsize * 6)) {
						ret = DSL_ERR_DATASIZE;
						goto end_set_port_data;
					}
					if(varsize > 6 && (varsize % 6) > 0) {
						ret = DSL_ERR_ALIGNMENT;
						goto end_set_port_data;
					}

					for(i=0;i<varsize;i+=6) {
						strncpy((char *)tmp, (char *)(strvalue+i), 6);
						value = simple_strtoul((char *)tmp, NULL, 16);
						pdata[0] = (value & 0xff0000)>>16;
						pdata[1] = (value & 0x00ff00)>>8;
						pdata[2] = value & 0x0000ff;
						
						ret = new_dsp_set(portid, address+i/6, MT_MEMTYPE_X, pdata);	// memtype always be X memory
						if(ret != MT_OK) {
							PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", portid);
							break;
						}
						//fake_mem[address+i/6] = value;	// remove this line and open above section after FW ready
					}
				}
				else {
					PRINTL(DBG_LVL_IF, "Incomplete OID\n");
					ret =  DSL_ERR_NOTOK;
				}
			}
			else {
				PRINTL(DBG_LVL_IF, "Incompatible OID\n");
				ret =  DSL_ERR_NOTOK;
			}
		}
	}

end_set_port_data:
	up(&mr_sem_api);

	return ret;
}


mt_ret spdl_body(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 state;
	mt_uint8 pdata[DMT_WORD], tmp[7];
	mt_uint32 i, value;
	mt_uint32 port;
	mt_uint32 address;
	mt_uint32 groupid, tableid, idx, paramid;
	mt_uint32 mibsize, varsize;
	
	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	dsl_status(portid);
	
	memset(tmp, 0, sizeof(tmp));

	if(portid == MT_PORT_BroadCast) {
		for(port = 0; port < MAX_PORTS; port++) {
			get_op_status(port, &state);
			if(state == MT_OPSTATE_DISABLE) {
				PRINTL(DBG_LVL_IF, "Port%i disabled\n", port);
				continue;
			}

			if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
				ret = get_mib2_header(port);
				if(ret != MT_OK) {
					PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
					continue;
				}
				
				mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
				if(mibsize == 0xffffffff) {
					ret = DSL_ERR_NOSUCHMIB;
					goto end_set_port_data_logic;
				}

				varsize = strlen((char *)strvalue);
				if(varsize > (mibsize * 6)) {
					ret = DSL_ERR_DATASIZE;
					goto end_set_port_data_logic;
				}
				if(varsize > 6 && (varsize % 6) > 0) {
					ret = DSL_ERR_ALIGNMENT;
					goto end_set_port_data_logic;
				}

				if(varsize <= 6) {
					strncpy((char *)tmp, (char *)strvalue, varsize);
					value = simple_strtoul((char *)tmp, NULL, 16);
					pdata[0] = (value & 0xff0000)>>16;
					pdata[1] = (value & 0x00ff00)>>8;
					pdata[2] = value & 0x0000ff;
					
					ret = new_dsp_set(port, address, MT_MEMTYPE_X, pdata);	// memtype always be X memory
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", port);
					}
				}
				else {	//varsize > 6
					for(i=0;i<varsize;i+=6) {
						strncpy((char *)tmp, (char *)(strvalue+i), 6);
						value = simple_strtoul((char *)tmp, NULL, 16);
						pdata[0] = (value & 0xff0000)>>16;
						pdata[1] = (value & 0x00ff00)>>8;
						pdata[2] = value & 0x0000ff;
						
						ret = new_dsp_set(port, address+i/6, MT_MEMTYPE_X, pdata);	// memtype always be X memory
						if(ret != MT_OK) {
							PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", port);
							break;
						}
						//fake_mem[address+i/6] = value;	// remove this line and open above section after FW ready
					}
				}
			}
			else {
				PRINTL(DBG_LVL_IF, "Incomplete OID\n");
				ret =  DSL_ERR_NOTOK;
			}
		}
	}
	else {
		get_op_status(portid, &state);
		if(state == MT_OPSTATE_DISABLE) {
			PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
			ret = DSL_ERR_NOTOK;
		}
		else {
			if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
				ret = get_mib2_header(portid);
				if(ret != MT_OK) {
					PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
					goto end_set_port_data_logic;
				}
	
				mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
				if(mibsize == 0xffffffff) {
					ret = DSL_ERR_NOSUCHMIB;
					goto end_set_port_data_logic;
				}

				varsize = strlen((char *)strvalue);
				if(varsize > (mibsize * 6)) {
					ret = DSL_ERR_DATASIZE;
					goto end_set_port_data_logic;
				}
				if(varsize > 6 && (varsize % 6) > 0) {
					ret = DSL_ERR_ALIGNMENT;
					goto end_set_port_data_logic;
				}

				if(varsize <= 6) {
					strncpy((char *)tmp, (char *)strvalue, varsize);
					value = simple_strtoul((char *)tmp, NULL, 16);
					pdata[0] = (value & 0xff0000)>>16;
					pdata[1] = (value & 0x00ff00)>>8;
					pdata[2] = value & 0x0000ff;
					
					ret = new_dsp_set(portid, address, MT_MEMTYPE_X, pdata);	// memtype always be X memory
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", portid);
					}
				}
				else {	//varsize > 6
					for(i=0;i<varsize;i+=6) {
						strncpy((char *)tmp, (char *)(strvalue+i), 6);
						value = simple_strtoul((char *)tmp, NULL, 16);
						pdata[0] = (value & 0xff0000)>>16;
						pdata[1] = (value & 0x00ff00)>>8;
						pdata[2] = value & 0x0000ff;
						
						ret = new_dsp_set(portid, address+i/6, MT_MEMTYPE_X, pdata);	// memtype always be X memory
						if(ret != MT_OK) {
							PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", portid);
							break;
						}
						//fake_mem[address+i/6] = value;	// remove this line and open above section after FW ready
					}
				}
			}
			else {
				PRINTL(DBG_LVL_IF, "Incomplete OID\n");
				ret =  DSL_ERR_NOTOK;
			}
		}
	}

end_set_port_data_logic:
	return ret;
}


/**
 * set_port_data_logic sets a logic parameter in the MIB set for the selected port.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param value
 * 		new value for the parameter
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 */
mt_ret set_port_data_logic(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
#if KERNEL_2_6
	char *bp = (char *)loid;
#endif
	mt_ret ret = MT_OK;
	mt_uint8 memtype;
	mt_uint8 pdata[DMT_WORD];
	mt_uint8 *token;
	mt_uint32 address, value;

	sem_ret = down_interruptible(&mr_sem_api);

	if(loid[0] == 'm' && loid[1] == '.') {	//special leading oid group number denote to memory write
#if KERNEL_2_6
		token = (mt_uint8 *)strsep(&bp, ".");	//skip leading "m."

		token = (mt_uint8 *)strsep(&bp, ".");
		if(token == NULL) return 0;
		memtype = (mt_uint8)simple_strtoul((char *)token, NULL, 16);

		token = (mt_uint8 *)strsep(&bp, ".");
		if(token == NULL) return 0;
		address = simple_strtoul((char *)token, NULL, 16);

		value = simple_strtoul((char *)strvalue, NULL, 16);
		pdata[0] = (value & 0xff0000)>>16;
		pdata[1] = (value & 0x00ff00)>>8;
		pdata[2] = value & 0x0000ff;
#else
		token = (mt_uint8 *)strtok(loid, ".");	//skip leading "m."

		token = (mt_uint8 *)strtok(NULL, ".");
		if(token == NULL) return 0;
		memtype = (mt_uint8)simple_strtoul((char *)token, NULL, 16);

		token = (mt_uint8 *)strtok(NULL, ".");
		if(token == NULL) return 0;
		address = simple_strtoul((char *)token, NULL, 16);

		value = simple_strtoul((char *)strvalue, NULL, 16);
		pdata[0] = (value & 0xff0000)>>16;
		pdata[1] = (value & 0x00ff00)>>8;
		pdata[2] = value & 0x0000ff;
#endif

		ret = new_dsp_set(portid, address, memtype, pdata);
	}
	else
		ret = spdl_body(portid, loid, strvalue);

	up(&mr_sem_api);

	return ret;
}


/**
 * get_port_data copies one parameter value in the MIB set to user space.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param pvalue
 * 		memory location to receive the value of the parameter in the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 	  
 */
mt_ret get_port_data(mt_uint32 portid, mt_uint8 *oid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 state;
	mt_uint8 pdata[DMT_WORD], tmp[7];
	mt_uint8 loid[OID_MAX_LEN];
	mt_uint32 i, value;
	mt_uint32 address;
	mt_uint32 groupid, tableid, idx, paramid;
	mt_uint32 mibsize;

	if(portid == MT_PORT_BroadCast) {
		PRINTL(DBG_LVL_IF, "Invalid portid=%i\n", portid);
		return -DSL_ERR_NOCHIP;
	}

	sem_ret = down_interruptible(&mr_sem_api);

	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	dsl_status(portid);

	memset(tmp, 0, sizeof(tmp));

	/*strvalue[0] = fake_mib[0];
	strvalue[1] = fake_mib[1];
	strvalue[2] = fake_mib[2];
	strvalue[3] = fake_mib[3];
	strvalue[4] = fake_mib[4];
	strvalue[5] = fake_mib[5];
	strvalue[6] = fake_mib[6];
	strvalue[7] = fake_mib[7];*/
	/*strvalue[8] = 0x36;
	strvalue[9] = 0x35;
	strvalue[10] = 0x34;
	strvalue[11] = 0x33;
	strvalue[12] = 0x32;
	strvalue[13] = 0x31;*/
	/*strvalue[8] = fake_mib[8];
	return MT_OK;*/

	get_op_status(portid, &state);
	if(state == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
		ret =  DSL_ERR_NOTOK;
	}
	else {
		strvalue[0] = 0x00;
		strvalue[1] = 0x00;	// let's set MIB size as "0" for default
		strvalue[2] = 0x30;
		strvalue[3] = 0x30;
		strvalue[4] = 0x30;
		strvalue[5] = 0x30;
		strvalue[6] = 0x30;
		strvalue[7] = 0x30;
		strvalue[8] = 0x00;

		//printk("ORG_OID: %s\n", oid);

		if(oid2logic((char *)oid, (char *)loid) == MT_OK) {
			if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
				ret = get_mib2_header(portid);
				if(ret != MT_OK) {
					PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
					goto end_get_port_data;
				}
				//printk("loid: %s\n", loid);
				strvalue[0] = 0xFF;	// tag "0xFF" for later strcat()
				strvalue[1] = 0xFF;	// tag "0xFF" for later strcat()
				strvalue[2] = 0x00;	// tag "0x00" for later strcat()
				mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
				if(mibsize == 0xffffffff) {
					ret = DSL_ERR_NOSUCHMIB;
					goto end_get_port_data;
				}
				//printk("address: %d\n", address);
				//printk("mibsize: %d\n", mibsize);
				if(mibsize > BUF_SIZE_DMT_WORD) {	// this "if statement" can be removed once MIB2 OID access all fixed 
					ret = DSL_ERR_DATASIZE;
					goto end_get_port_data;
				}
				//printk("before getting MIB\n");
				for(i=0;i<mibsize;i++) {
					ret = new_dsp_get(portid, address+i, MT_MEMTYPE_X, pdata);	// memtype always be X memory
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", portid);
						break;
					}

					value = (pdata[0] << 16) + (pdata[1] << 8) + pdata[2];
					//value = fake_mem[address+i];	// remove this line and open above section after FW ready
					sprintf((char *)tmp, "%06x", value);
					strcat((char *)strvalue, (char *)tmp);
				}
				//printk("after getting MIB\n");
				strvalue[0] = (i >> 8) & 0xff;
				strvalue[1] = i & 0xff;

				/*strvalue[0] = 0x0;
				strvalue[1] = 0x1;
				strvalue[2] = 0x31;
				strvalue[3] = 0x32;
				strvalue[4] = 0x33;
				strvalue[5] = 0x34;
				strvalue[6] = 0x35;
				strvalue[7] = 0x36;
				strvalue[8] = 0x00;*/
			}
			else {
				PRINTL(DBG_LVL_IF, "Incomplete OID\n");
				ret =  DSL_ERR_NOTOK;
			}
		}
		else {
			PRINTL(DBG_LVL_IF, "Incompatible OID\n");
			ret =  DSL_ERR_NOTOK;
		}
	}

end_get_port_data:
	up(&mr_sem_api);

	return ret;
}


mt_ret gpdl_body(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 state, remote = 0;
	mt_uint8 tmp[7];
	mt_uint8 *pdata;
	mt_uint32 i, value;
	mt_uint32 address;
	mt_uint32 groupid, tableid, idx, paramid;
	mt_uint32 mibsize;

	if(portid == MT_PORT_BroadCast) {
		PRINTL(DBG_LVL_IF, "Invalid portid=%i\n", portid);
		return -DSL_ERR_NOCHIP;
	}

	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	dsl_status(portid);

	if((pdata = (mt_uint8 *)kmalloc(BUF_SIZE, GFP_KERNEL)) == NULL) return -ENOMEM;

	memset(tmp, 0, sizeof(tmp));

	get_op_status(portid, &state);
	if(state == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
		ret =  DSL_ERR_NOTOK;
	}
	else {
		strvalue[0] = 0x00;
		strvalue[1] = 0x00;	// let's set MIB size as "0" for default
		strvalue[2] = 0x30;
		strvalue[3] = 0x30;
		strvalue[4] = 0x30;
		strvalue[5] = 0x30;
		strvalue[6] = 0x30;
		strvalue[7] = 0x30;
		strvalue[8] = 0x00;

		//printk("LOGIC_OID: %s\n", loid);

		if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
			if(idx & 0x80) {
				remote = 1;
				idx &= 0x7F;
			}
			ret = get_mib2_header(portid);
			if(ret != MT_OK) {
				PRINTL(DBG_LVL_IF, "Error getting MIB2 header\n");
				goto end_get_port_data_logic;
			}
			strvalue[0] = 0xFF;	// tag "0xFF" for later strcat()
			strvalue[1] = 0xFF;	// tag "0xFF" for later strcat()
			strvalue[2] = 0x00;	// tag "0x00" for later strcat()
			mibsize = trans_mib_addr(groupid, tableid, idx, paramid, &address);
			if(mibsize == 0xffffffff) {
				ret = DSL_ERR_NOSUCHMIB;
				goto end_get_port_data_logic;
			}
			//printk("g:%d, t:%d, p:%d, i:%d\n", groupid, tableid, paramid, idx);
			//printk("address: 0x%x\n", address);
			//printk("mibsize: %d\n", mibsize);
			if(mibsize > BUF_SIZE_DMT_WORD) {	// this "if statement" can be removed once MIB2 OID access all fixed 
				ret = DSL_ERR_DATASIZE;
				goto end_get_port_data_logic;
			}
			//printk("before getting MIB\n");
			if(remote) {	// get remote logic MIB
				printk("remote!!\n");
				ret = gpmr_body(portid, address, mibsize, pdata);
				for(i=0;i<mibsize;i++) {
					value = (pdata[0+i*3] << 16) + (pdata[1+i*3] << 8) + pdata[2+i*3];
					sprintf((char *)tmp, "%06x", value);
					memcpy(strvalue+2+i*6, tmp, 6);
				}
			}
			else {	// get local logic MIB
				for(i=0;i<mibsize;i++) {
					ret = new_dsp_get(portid, address+i, MT_MEMTYPE_X, pdata);	// memtype always be X memory
					if(ret != MT_OK) {
						PRINTL(DBG_LVL_IF, "Port%i MIB2 access error\n", portid);
						break;
					}
	
					value = (pdata[0] << 16) + (pdata[1] << 8) + pdata[2];
					//value = fake_mem[address+i];	// remove this line and open above section after FW ready
					sprintf((char *)tmp, "%06x", value);
					strcat((char *)strvalue, (char *)tmp);
				}
			}
			//printk("after getting MIB\n");
			strvalue[0] = (i >> 8) & 0xff;	// insert mibsize high
			strvalue[1] = i & 0xff;	// insert mibsize low

			/*strvalue[0] = 0x0;
			strvalue[1] = 0x1;
			strvalue[2] = 0x31;
			strvalue[3] = 0x32;
			strvalue[4] = 0x33;
			strvalue[5] = 0x34;
			strvalue[6] = 0x35;
			strvalue[7] = 0x36;
			strvalue[8] = 0x00;*/
		}
		else {
			PRINTL(DBG_LVL_IF, "Incomplete OID\n");
			ret =  DSL_ERR_NOTOK;
		}
	}

end_get_port_data_logic:
	kfree(pdata);
	return ret;
}


/**
 * get_port_data_logic copies one logic parameter value in the MIB set to user space.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string, use group.table.(param+128).[index] to get remote logic parameter
 * \param pvalue
 * 		memory location to receive the value of the parameter in the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 	  
 */
mt_ret get_port_data_logic(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
#if KERNEL_2_6
	char *bp = (char *)loid;
#endif
	mt_ret ret = MT_OK;
	mt_uint8 memtype;
	mt_uint8 pdata[DMT_WORD];
	mt_uint8 *token;
	mt_uint32 address, value;

	/*if(in_interrupt()) {
		if(down_trylock(&mr_sem_api)) return DSL_ERR_NOTOK;
	}
	else {
		sem_ret = down_interruptible(&mr_sem_api);
	}*/
	sem_ret = down_interruptible(&mr_sem_api);

	if(loid[0] == 'm' && loid[1] == '.') {	//special leading oid group number denote to memory read
		strvalue[0] = 0x00;
		strvalue[1] = 0x00;	// let's set MIB size as "0" for default
		strvalue[2] = 0x30;
		strvalue[3] = 0x30;
		strvalue[4] = 0x30;
		strvalue[5] = 0x30;
		strvalue[6] = 0x30;
		strvalue[7] = 0x30;
		strvalue[8] = 0x00;
		
#if KERNEL_2_6
		token = (mt_uint8 *)strsep(&bp, ".");	//skip leading "m."

		token = (mt_uint8 *)strsep(&bp, ".");
		if(token == NULL) return 0;
		memtype = (mt_uint8)simple_strtoul((char *)token, NULL, 16);

		token = (mt_uint8 *)strsep(&bp, ".");
		if(token == NULL) return 0;
		address = simple_strtoul((char *)token, NULL, 16);
#else
		token = (mt_uint8 *)strtok(loid, ".");	//skip leading "m."

		token = (mt_uint8 *)strtok(NULL, ".");
		if(token == NULL) return 0;
		memtype = (mt_uint8)simple_strtoul((char *)token, NULL, 16);

		token = (mt_uint8 *)strtok(NULL, ".");
		if(token == NULL) return 0;
		address = simple_strtoul((char *)token, NULL, 16);
#endif

		ret = new_dsp_get(portid, address, memtype, pdata);
		if(ret == MT_OK) {
			value = (pdata[0] << 16) + (pdata[1] << 8) + pdata[2];
			sprintf((char *)(strvalue+2), "%06x", value);
			strvalue[0] = 0;	// insert mibsize high
			strvalue[1] = 1;	// insert mibsize low
		}
	}
	else
		ret = gpdl_body(portid, loid, strvalue);

	up(&mr_sem_api);

	return ret;
}


/**
 * get_port_data_logic_array copies current parameter value to user space.
 * \param portid
 * 		port ID number
 * \param loid
 * 		identifies MIB Logic OID from vdsl2mib.h
 * \param count
 * 		array count 
 * \param strvalue
 * 		memory location to receive the value of the parameter in the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 	  
 */
mt_ret get_port_data_logic_value_array(mt_uint32 portid, mt_uint8 *loid, mt_uint32 count, mt_uint32 *pvalue)
{
	mt_ret ret = MT_OK;
	mt_uint8 pdata[10];
	mt_uint8 loidtmp[OID_MAX_LEN];
	mt_uint32 i = 0;
	mt_uint32 groupid = 0, tableid = 0, paramid = 0, idx = 0;
	
	if(oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL)) {
		for(i=0;i<count;i++){
			sprintf((char *)loidtmp, "%d.%d.%d.%d", groupid, tableid, paramid+i, idx);
			ret = gpdl_body(portid, loidtmp, pdata);
			if(ret != MT_OK) break;
			*pvalue++ = simple_strtoul((char *)(pdata+2), NULL, 16);
		}
	}
	else
		ret = DSL_ERR_NOTOK;

	return ret;
}


/**
 * SetMemData sets the data to the specified address for the selected port.
 * \param portid
 * 		port ID number
 * \param iAddress
 * 		memory address to write
 * \param iMemType
 * 		memory type (X-memory(0x00), Y-memory(0x01), P-memory(0x02))
 * \param pData
 * 		new Data for the memory address
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 	  
 * \notice
 * 		not supported	  
 */
mt_ret set_port_mem(mt_uint32 portid, mt_uint32 address, mt_uint8 memtype, mt_uint8 *pdata)
{
	mt_uint8 state;
	mt_uint32 ret = MT_OK;
	
	dsl_er(check_param(MT_PAR_PORT, 0, portid));	
	dsl_status(portid);

	get_op_status(portid, &state);
	if(state == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
		ret =  DSL_ERR_NOTOK;
	}
	else {
		while(1) {	//make sure we can gain the HPI access authority
			if(test_and_set_bit(0, &atomic_ops) == FALSE) {
				sem_ret = down_interruptible(&mr_sem_api);
				ret = new_dsp_set(portid, address, memtype, pdata);
				up(&mr_sem_api);
				clear_bit(0, &atomic_ops);
				break;
			}
			else {
				MYDELAY(50);
			}		
		}
	}

	return ret;
}


/**
 * GetMemData copies current data in the specified address to user space.
 * \param portid
 * 		port ID number
 * \param address
 * 		memory address to read
 * \param memtype
 * 		memory type (X-memory(0x00), Y-memory(0x01), P-memory(0x02))
 * \param pdata
 * 		memory location to receive the data in the specified address
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 * \notice
 * 		not supported	  
 */
mt_ret get_port_mem(mt_uint32 portid, mt_uint32 address, mt_uint8 memtype, mt_uint8 *pdata)
{
	mt_uint8 state;
	mt_uint32 ret = MT_OK;
	
	dsl_er(check_param(MT_PAR_PORT, 0, portid));	
	dsl_status(portid);

	get_op_status(portid, &state);
	if(state == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_IF, "Port%i disabled\n", portid);
		ret =  DSL_ERR_NOTOK;
	}
	else {
		while(1) {	//make sure we can gain the HPI access authority
			if(test_and_set_bit(0, &atomic_ops) == FALSE) {
				sem_ret = down_interruptible(&mr_sem_api);
				ret = new_dsp_get(portid, address, memtype, pdata);
				up(&mr_sem_api);
				clear_bit(0, &atomic_ops);
				break;
			}
			else {
				MYDELAY(50);
			}		
		}
	}

	return ret;
}


mt_ret gpmr_body(mt_uint32 portid, mt_uint32 address, mt_uint32 length, mt_uint8 *pdata)
{
	mt_ret ret = MT_OK;
	mt_uint8 tmp[7];
	mt_uint8 *buf;
	mt_uint8 res = 0x02; // 0->success, 1->timeout, other->fail
	static mt_uint32 contentid = 128;
	mt_uint32 type, rsptype;
	mt_uint32 wait, i, toread, value, read, retry;

	dsl_er(check_param(MT_PAR_PORT, 0, portid));	
	dsl_status(portid);

	if ((buf = (mt_uint8 *)kmalloc(BUF_SIZE, GFP_KERNEL)) == NULL) return -ENOMEM;

	memset(tmp, 0, sizeof(tmp));
	read = 0;
	while (read < length) {
		toread = (length - read < 100) ? (length - read) : 100; // CfgParamsTempBuf size is 128
		retry = 0;
		while (retry++ < 3){
			if (contentid <= 127) contentid = 128;
			if (contentid >= 1000000) contentid = 128; else contentid++;
			
			type = 0x200000 + contentid;
		
			sprintf((char *)buf, "%06x%06x%06x", type, address+read, toread);
			MYDELAY(10);
			if ((ret = spdl_body(portid, (mt_uint8 *)(VmeDslCmdReg), buf)) != DSL_ERR_OK)
				goto end_get_port_mem_remote;
	
			// request cmd
			MYDELAY(10);
			if ((ret = spdl_body(portid, (mt_uint8 *)(VmeDslCmd), (mt_uint8 *)("81407d"))) != DSL_ERR_OK)
				goto end_get_port_mem_remote;
			
			MYDELAY(500);
			res = 0x01;
			for (wait = 0; wait < 100; wait++) {
				ret = gpdl_body(portid, (mt_uint8 *)(VmeDslCmd), buf);
				if (ret != DSL_ERR_OK) {
					res = 0x02;
					break;
				}

				if (memcmp(buf + 2, "51", 2) == 0) {
					MYDELAY(10);
	
					ret = gpdl_body(portid, (mt_uint8 *)(VmeRxNsfRdId), buf);
					if (DSL_ERR_OK == ret){
						memcpy(tmp, buf+2, 6);
						rsptype = simple_strtoul((char *)tmp, NULL, 16);

						if (type == rsptype)
							res = 0x00;
						else{
							res = 0x03;
							continue;
						}
					}
					else
						res = 0x04;
	
					break;
				} else if (memcmp(buf + 2, "49", 2) == 0) {
					res = 0x05;
					break;
				} 
				else
					MYDELAY(50);
			}
		
			if (res == 0x00)
				break;

			printk("gpmr_bdy: retry %d, res %x\n", retry, res);
		} // while (retry)
	
		if (res != 0) {
			ret = DSL_ERR_NOTOK;
			goto end_get_port_mem_remote;
		}
		
		MYDELAY(10);
		if ((ret = gpdl_body(portid, (mt_uint8 *)(CfgParamsTempBuf), buf)) != DSL_ERR_OK)
			goto end_get_port_mem_remote;
		
		for (i = 0; i < toread; i++){
			memcpy(tmp, buf+2+i*6, 6);
			value = simple_strtoul((char *)tmp, NULL, 16);
			//printk("value %x\n", value);
			*pdata++ = (value >> 16) & 0xff;
			*pdata++ = (value >> 8) & 0xff;
			*pdata++ = value & 0xff;
		}

		read += toread;
	}
	
end_get_port_mem_remote:
	kfree(buf);
	return ret;
}


mt_ret get_port_mem_remote(mt_uint32 portid, mt_uint32 address, mt_uint32 length, mt_uint8 *pdata)
{
	mt_uint32 ret = MT_OK;

	dsl_er(check_param(MT_PAR_PORT, 0, portid));	
	dsl_status(portid);

	sem_ret = down_interruptible(&mr_sem_api);
	ret = gpmr_body(portid, address, length, pdata);
	up(&mr_sem_api);
	
	return ret;
}


void set_mapper_ports(mt_uint32 chipid, mt_uint32 ports)
{
	mt_uint32 c_map = 0;
	
	if(ports == 0) {
		map[chipid] = 0;
	}
	else {
		if(ports > 32) {
			PRINTL(DBG_LVL_MAPPER,"set_mapper_ports: Too many ports per chip %i.\n", ports);
			return;
 		}	

		while(ports) {
			c_map |= 1 << (--ports);
		}

 		map[chipid] = c_map;
 		PRINTL(DBG_LVL_MAPPER,"set_mapper_ports: map[%i] set to %x.\n", chipid, c_map);
	}
}


/**
 * Download Firmware
 * \param chipid
 * 		chipid that firmware will download
 * \param cmdID
 * 		Download Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size
 * \param buffer
 * 		The buffer need to be downloaded
 * \notice
 * 		Firmware download sequence is MT_FIRMWARE_START,MT_FIRMWARE_CONT,MT_FIRMWARE_END
 *
 */
mt_ret download_firmware(mt_uint8 chipid, mt_uint8 cmdID, mt_uint16 bufsize, mt_uint8* buffer)
{
	mt_ret ret;
	mt_uint32 chip;

	dsl_er(check_param(MT_PAR_CHIP, 0, chipid));

	while(1) {	//make sure we can gain the HPI access authority
		if(test_and_set_bit(0, &atomic_ops) == FALSE) {
			sem_ret = down_interruptible(&mr_sem_api);

			if( cmdID == MT_FIRMWARE_START ) {
#ifdef SPI_ACCESS
#if defined (ICPLUS_IP3210)
				//bufsize is used as firmwareid here
				if(bufsize == 1) *databit_clr = COCPE;
				else *databit_set = COCPE;
#endif
#endif	//SPI_ACCESS

				//some initial job
				if(auto_reset) {
					PRINTL(DBG_LVL_IF, "Resetting Chip...\n");
					ret = reset_chip(chipid);
					if(ret != MT_OK) return ret;
				}

				mib2_first_access = 1;
				bindrv_load = 0;
			}

			if( cmdID == MT_FIRMWARE_CONT && (bufsize > 0) ) {
#ifndef SPI_ACCESS
				ret = download_chip(chipid, bufsize, buffer );
				if(ret != MT_OK) return ret;
#endif	//SPI_ACCESS
			}

			if( cmdID == MT_FIRMWARE_END ) {
				PRINTL(DBG_LVL_CHIP, "\n");	//this "\n" corresponds to end of the "." in download_chip() in chip.c

#ifdef XENOS
				//delay 100ms for waiting firmware download complete, this delay corresponds to end of firmware download in download_chip() in chip.c
				MYDELAY(100);
#endif
#ifdef HOPE
				//delay 100ms for waiting firmware download complete, this delay corresponds to end of firmware download in download_chip() in chip.c
				MYDELAY(100);
#endif
#ifdef BOARD_A5
				//delay 100ms for waiting firmware download complete, this delay corresponds to end of firmware download in download_chip() in chip.c
				MYDELAY(100);
#endif

				//this port status change corresponds to end of firmware download in download_chip() in chip.c,
				//by setting the chip status to idle, all the L3 and part of L2 APIs will be working
				set_status_idle_chip(chipid);

				if(chipid == MT_PORT_BroadCast) {
					for(chip = 0; chip < MAX_PORTS; chip++) {
						set_mapper_ports(chip, MAX_PORTS);
						err_recovery[chip] = 0;
					}
				}
				else {
					set_mapper_ports(chipid, MAX_PORTS);
					err_recovery[chipid] = 0;
				}
			}

			up(&mr_sem_api);
			clear_bit(0, &atomic_ops);
			break;
		}
		else {
			MYDELAY(50);
		}
	}

	return MT_OK;
}


mt_ret time_calibrate(void)
{
	mt_ret ret;
	mt_uint8 tmp[7];
	mt_uint8 state;
	mt_uint32 port;
	struct timeval tv;
	struct tm tm;

	sem_ret = down_interruptible(&mr_sem_api);

	//Synchronize Firmware 15 Min and One Day counter with wall clock
	for(port=0;port<MAX_PORTS;port++) {
		get_op_status(port, &state);
		if(state == MT_OPSTATE_DISABLE) {
			PRINTL(DBG_LVL_IF, "Port%i disabled\n", port);
			continue;
		}

		do_gettimeofday(&tv);
		ret = __offtime(&(tv.tv_sec), 0, &tm);
		if(ret != MT_OK) {
			PRINTD("Time transform failed!\n");
			return ret;
		}

		sprintf((char *)tmp, "%x", NUM_SEC_PER_15MIN - (((tm.tm_min % 15) * 60) + tm.tm_sec));
		ret = spdl_body(port, (mt_uint8 *)(xdslFifteenMinCnt), tmp);
		if(ret != MT_OK) PRINTD("Firmware 15 Min synchronization failed!!\n");

		sprintf((char *)tmp, "%x", NUM_SEC_PER_1DAY - ((tm.tm_hour*3600) + (tm.tm_min*60) + tm.tm_sec));
		ret = spdl_body(port, (mt_uint8 *)(xdslOneDayCnt), tmp);
		if(ret != MT_OK) PRINTD("Firmware One Day synchronization failed!!\n");
	}

	up(&mr_sem_api);

	return MT_OK;
}


/**
 * Load Firmware binary driver
 * \param cmdID
 * 		Load Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size
 * \param buffer
 * 		The buffer need to be downloaded
 * \notice
 * 		Firmware binary driver load sequence is MT_FIRMWARE_START,MT_FIRMWARE_CONT,MT_FIRMWARE_END
 *
 */
mt_ret load_firmware_bindrv(mt_uint8 cmdID, mt_uint16 bufsize, mt_uint8* buffer)
{
	mt_ret ret = MT_OK;
	mt_uint8 *ptr = (mt_uint8 *)bd;
	static mt_uint32 bytesread;

	if( cmdID == MT_FIRMWARE_START ) {
		bytesread = 0;
		memset(ptr, 0, sizeof(bd));
	}

	if( cmdID == MT_FIRMWARE_CONT && (bufsize > 0) ) {
		memcpy(ptr+bytesread, buffer, bufsize);
		bytesread += bufsize;
	}

	if( cmdID == MT_FIRMWARE_END ) {
		/*int i;

		for(i=0;i<0xffffff;i++);
		for(i=0;i<256;i++) {
			if(i % 16 == 0) printk("\n");
			printk("%02x ", ptr[i]);
		}
		for(i=BUF_SIZE;i<BUF_SIZE+256;i++) {
			if((i-8) % 16 == 0) printk("\n");
			printk("%02x ", ptr[i]);
		}
		printk("\n");*/

		bindrv_load = 1;

		while(1) {	//make sure we can gain the HPI access authority
			if(test_and_set_bit(0, &atomic_ops) == FALSE) {
				ret = time_calibrate();	//Synchronize Firmware 15 Min and One Day counter with wall clock
				if(ret != MT_OK) PRINTL(DBG_LVL_IF, "Time calibration failed!!\n");

				clear_bit(0, &atomic_ops);
				break;
			}
			else {
				MYDELAY(50);
			}
		}
	}

	return ret;
}


#if defined (SPI_ACCESS)
/**
 * Flash SPI flash memory
 * \param cmdID
 * 		Flash Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size
 * \param buffer
 * 		The buffer need to be flashed
 * \notice
 * 		Firmware flash sequence is MT_FIRMWARE_START,MT_FIRMWARE_CONT,MT_FIRMWARE_END
 *
 */
mt_ret flash_firmware(mt_uint8 cmdID, mt_uint16 bufsize, mt_uint8* buffer)
{
	mt_ret ret = MT_OK;
	mt_uint8 loop, retry;
	mt_uint32 gpio_value;
	static mt_uint32 address;
	volatile mt_uint32 *gpio_addr;

	sem_ret = down_interruptible(&mr_sem_api);

	if( cmdID == MT_FIRMWARE_START ) {
		//some initial job
		if(auto_reset) {
			PRINTL(DBG_LVL_IF, "Resetting Chip...\n");
			ret = reset_chip_low(0);	//reset the only chip and pull DMT RESET to low for flash accessing
			if(ret != MT_OK) return ret;
		}

		//config selected pin as SPI output/input pin
#if defined (ICPLUS_IP3210)
		gpio_addr = (volatile mt_uint32 *)ioremap(0xA300500C, 0x2);
		gpio_value = *gpio_addr;
		gpio_value |= COCPE | RESET | SCKB | SC2B | SPI_SRDB;	//output
		gpio_value &= ~(SPI_STDB);	//input
#elif defined (RALINK_RT3352)
		gpio_addr = (volatile mt_uint32 *)ioremap(0x10000624, 0x2);
		gpio_value = *gpio_addr;
		gpio_value |= RESET | SCKB | SC2B | SPI_SRDB;	//output
		gpio_value &= ~(SPI_STDB);	//input
#elif defined (ATHEROS_AR93XX)
		gpio_addr = (volatile mt_uint32 *)ioremap(0x18040000, 0x2);
		gpio_value = *gpio_addr;
		gpio_value &= ~(RESET | SCKB | SC2B | SPI_SRDB);	//output
		gpio_value |= SPI_STDB;	//input
#elif defined (REALTEK_RTL8672)
		gpio_addr = (volatile mt_uint32 *)(0xB8003508);
		gpio_value = *gpio_addr;
		gpio_value |= RESET_PIN | SCKB_PIN | SC2B_PIN | SPI_SRDB_PIN;	//output
		gpio_value &= ~(SPI_STDB_PIN);	//input
#endif
		*gpio_addr = gpio_value;

		//erase entire flash
		spi_erase_all();

		//delay for waiting flash erase complete
		for(loop=0;loop<ERASE_DELAY;loop++) {	//we can delay 1 seconds for 4Mb flash or 2.5 seconds for 8Mb flash
			printk(".");

			MYDELAY(500);
		}

		address = 0x00000000;
	}

	if( cmdID == MT_FIRMWARE_CONT && (bufsize > 0) ) {
		for(retry=0;retry<RETRY_COUNT;retry++) {
			spi_write_data(address, bufsize, buffer);
			ret = spi_check_data(address, bufsize, buffer);
			if(ret == MT_OK) break;
		}

#if defined (REALTEK_RTL8672)
		//flash update hack for realtek RTL8672
		ret = MT_OK;
#endif

		address += bufsize;

		printk(".");
	}

	if( cmdID == MT_FIRMWARE_END ) {
		//config selected pin as DMT output/input pin
#if defined (ICPLUS_IP3210)
		gpio_addr = (volatile mt_uint32 *)ioremap(0xA300500C, 0x2);
		gpio_value = *gpio_addr;
		gpio_value |= COCPE | RESET | SCKB | SC2B | DMT_SRDB;	//output
		gpio_value &= ~(DMT_STDB);	//input
#elif defined (RALINK_RT3352)
		gpio_addr = (volatile mt_uint32 *)ioremap(0x10000624, 0x2);
		gpio_value = *gpio_addr;
		gpio_value |= RESET | SCKB | SC2B | DMT_SRDB;	//output
		gpio_value &= ~(DMT_STDB);	//input
#elif defined (ATHEROS_AR93XX)
		gpio_addr = (volatile mt_uint32 *)ioremap(0x18040000, 0x2);
		gpio_value = *gpio_addr;
		gpio_value &= ~(RESET | SCKB | SC2B | DMT_SRDB);	//output
		gpio_value |= DMT_STDB;	//input
#elif defined (REALTEK_RTL8672)
		gpio_addr = (volatile mt_uint32 *)(0xB8003508);
		gpio_value = *gpio_addr;
		gpio_value |= RESET_PIN | SCKB_PIN | SC2B_PIN | DMT_SRDB_PIN;	//output
		gpio_value &= ~(DMT_STDB_PIN);	//input
#endif
		*gpio_addr = gpio_value;

		//Release DMT RESET
		ret = reset_chip_high(0);
	}

	up(&mr_sem_api);

	return ret;
}
#endif //defined (SPI_ACCESS)


EXPORT_SYMBOL(connect_port);
EXPORT_SYMBOL(disconnect_port);
