/****************************************************************************
* File:         $Workfile$
* Revision:     $Revision: 1.17 $
* Date:         $Date: 2012-05-23 12:44:14 $
*
* Overview:     
* Functions:
*
* Author:       Ian Tsao
* 
* -- Metanoia Copyright Notice --				
*
* (C) Copyright 2005 Metanoia, Inc. All rights reserved.
* 
* Metanoia reserves the right to change specifications without notice. 
* Copyright 2005 Metanoia, Inc. All rights reserved. Preliminary Data-Sheet 
* indicates this product is in design and the specification may change. 
* Electrical parametrics have not been analyzed and are not specified. Do not 
* use this data sheet as a design reference document. Please contact Metanoia 
* for a current data sheet that may be used as a design reference. All 
* contents of this document are protected by copyright law and may not be 
* reproduced without the express written consent of Metanoia, Inc. Metanoia, 
* the Metanoia logo, and combinations thereof are trademarks of Metanoia, Inc. 
* Other product names used in this publication are for identification purposes 
* only and may be trademarks or registered trademarks of their respective 
* companies. The contents of this document are provided in connection with 
* Metanoia, Inc. products. Metanoia, Inc. has made best efforts to ensure that 
* the information contained herein is accurate and reliable. However, Metanoia, 
* Inc. makes no warranties, express or implied, as to the accuracy or 
* completeness of the contents of this publication and is providing this 
* publication 'as is'. Metanoia, Inc. reserves the right to make changes to 
* specifications and product descriptions at any time without notice, and to 
* discontinue or make changes to its products at any time without notice. 
* Metanoia, Inc. does not assume any liability arising out of the application 
* or use of any product or circuit, and specifically disclaims any and all 
* liability, including without limitation special, consequential, or 
* incidental damages.
* 
****************************************************************************/
#include "dsll3.h"
#include "boardioctl.h"
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
#define GPIO_DEV "/dev/gpio"
#endif //defined (RALINK_RT3052) || defined (RALINK_RT3352)
#define OPEN_VDSL(vdslfd) ({vdslfd=open("/dev/vdsl", O_RDWR);if(vdslfd == -1){printf("vdsl device open failed.\n");return -1;}})
#define OPEN_BOARD(boardfd) ({boardfd=open("/dev/board", O_RDWR);if(boardfd == -1){printf("board device open failed.\n");return -1;}})
#define EOC_CMD_HEADER_SIZE 3
#define FW_HEADER_SIZE 4
#define FW_BINDRV_HEADER_SIZE 3
#define FW_FLASH_HEADER_SIZE 3
#define SPMA_HEADER_SIZE 4

int vdslfd;
int boardfd;
char ioctl_buf[SMALL_BUF_SIZE];
char ioctl_big_buf[BUF_SIZE];
mt_ret ret;

/**
 * \file 
 * \brief communication module interface API
 */
#ifndef __KERNEL__
/**
 * Get libdsl.so version
 */
mt_ret get_lib_version(mt_uint8 *version){
	strcpy(version, LIBVERSION);

	return MT_OK;
}
#endif	//__KERNEL__


mt_uint32 reset_chip(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));
	
#ifdef VITESS_VSC7501
	system("echo out > /proc/gpio/GPIOA25");
	system("echo clear > /proc/gpio/GPIOA25");
	system("echo set > /proc/gpio/GPIOA25");
#endif

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_RESET, ioctl_buf);
	close(vdslfd);


	return ret;
}


mt_uint32 reset_chip_high(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));
	
#ifdef VITESS_VSC7501
	system("echo out > /proc/gpio/GPIOA25");
	system("echo clear > /proc/gpio/GPIOA25");
	system("echo set > /proc/gpio/GPIOA25");
#endif

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_RESET_HIGH, ioctl_buf);
	close(vdslfd);

	return ret;
}


mt_uint32 reset_chip_low(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));

#ifdef VITESS_VSC7501
	system("echo out > /proc/gpio/GPIOA25");
	system("echo clear > /proc/gpio/GPIOA25");
	system("echo set > /proc/gpio/GPIOA25");
#endif

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_RESET_LOW, ioctl_buf);
	close(vdslfd);

	return ret;
}


// -------------------------- Ian L2 API -----------------------
/**
 * ConnectPort activates the selected port so that it would seek a connection
 * \param portid
 * 		port ID number 
 * \return 
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret connect_port(mt_uint32 portid)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=portid;
	ret = ioctl(vdslfd, VDSL_CONNECT_PORT, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * DisconnectPort inactivates the selected port so that it stops responding to far end connection attempts
 * \param portid
 * 		port ID number
 * \return 
 * 	- 0 : success
 * 	- < 0 : error code
 */
mt_ret disconnect_port(mt_uint32 portid)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=portid;
	ret = ioctl(vdslfd, VDSL_DISCONNECT_PORT, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * ShutdownSystem drops all the connections and brings system to low power mode. Only way to bring it back to normal state is to execute InitSystem
 * \return 
 * 	- 0 : success
 * 	- < 0 : error code 	
 */
mt_ret shutdown_system(void)
{
	OPEN_VDSL(vdslfd);
	ret = ioctl(vdslfd, VDSL_SHUTDOWN_SYSTEM, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * ShutdownChip drops all of the chip¡?�s connections and brings it to low power mode. Only way to bring it back to normal state is to execute InitChip or InitSystem
 * \param chipid
 * 		chip ID number 
 * 		- 0 : success
 * 		- < 0 : error code 	
 */		
mt_ret shutdown_chip(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_SHUTDOWN_CHIP, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * Init Modem by chipid
 */
mt_ret init_modem(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_INIT_MODEM, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * Reset Modem by chipid
 */
mt_ret reset_modem(mt_uint32 chipid)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=chipid;
	ret = ioctl(vdslfd, VDSL_RESET_MODEM, ioctl_buf);
	close(vdslfd);

	return ret;
}


mt_ret get_mib2_header(mt_uint32 portid)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=portid;
	ret = ioctl(vdslfd, VDSL_GET_MIB2_HEADER, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * get_mib2_info returns the information for user queried MIB OID.
 * \param loid
 * 		MIB Object ID string
 * \param address
 * 		translated address
 * \param mibsize
 * 		translated mibsize
 * \param mibtype
 * 		translated mibtype
 * 
 * \return
 * 	- 0 : success
 * 	- < 0 : error code 
 */
mt_ret get_mib2_info(mt_uint8 *loid, mt_uint32 *address, mt_uint32 *mibsize, mt_uint32 *mibtype)
{
	OPEN_VDSL(vdslfd);
	strcpy(ioctl_big_buf, loid);
	ret = ioctl(vdslfd, VDSL_GET_MIB2_INFO, ioctl_big_buf);
	memcpy(address, ioctl_big_buf, sizeof(mt_uint32));
	memcpy(mibsize, ioctl_big_buf+4, sizeof(mt_uint32));
	memcpy(mibtype, ioctl_big_buf+8, sizeof(mt_uint32));
	close(vdslfd);

	return ret;
}


/**
 * set_port_data sets a parameter in the MIB for the selected port.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param value
 * 		new value for the parameter
 * 
 * \return
 * 	- 0 : success
 * 	- <0 : error code 
 */
mt_ret set_port_data(mt_uint32 portid, mt_uint8 *oid, mt_uint8 *strvalue)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));	

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=portid;
	strcpy(ioctl_big_buf+1, oid);
	strcpy(ioctl_big_buf+1+strlen(oid)+1, strvalue);
	ret = ioctl(vdslfd, VDSL_SET_PORT_DATA, ioctl_big_buf);
	close(vdslfd);

	return ret;
}


/**
 * set_port_data_logic sets a logic parameter in the MIB set for the selected port.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param value
 * 		new value for the parameter
 * 
 * \return
 * 	- 0 : success
 * 	- <0 : error code 
 */
mt_ret set_port_data_logic(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));	

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=portid;
	strcpy(ioctl_big_buf+1, loid);
	strcpy(ioctl_big_buf+1+strlen(loid)+1, strvalue);
	ret = ioctl(vdslfd, VDSL_SET_PORT_DATA_LOGIC, ioctl_big_buf);
	close(vdslfd);

	return ret;
}


/**
 * get_port_data copies current parameter value to user space.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param pvalue
 * 		memory location to receive the value of the parameter in the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- <0 : error code 	  
 */
mt_ret get_port_data(mt_uint32 portid, mt_uint8 *oid, mt_uint8 *strvalue)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));	

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=portid;
	strcpy(ioctl_big_buf+1, oid);
	ret = ioctl(vdslfd, VDSL_GET_PORT_DATA, ioctl_big_buf);
	memcpy(strvalue, ioctl_big_buf, BUF_SIZE);
	close(vdslfd);

	return ret;
}


/**
 * get_port_data_logic copies one logic parameter value in the MIB set to user space.
 * \param portid
 * 		port ID number
 * \param oid
 * 		MIB Object ID string
 * \param pvalue
 * 		memory location to receive the value of the parameter in the MIB
 * 
 * \return
 * 	- 0 : success
 * 	- <0 : error code 	  
 */
mt_ret get_port_data_logic(mt_uint32 portid, mt_uint8 *loid, mt_uint8 *strvalue)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));	

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=portid;
	strcpy(ioctl_big_buf+1, loid);
	ret = ioctl(vdslfd, VDSL_GET_PORT_DATA_LOGIC, ioctl_big_buf);
	memcpy(strvalue, ioctl_big_buf, BUF_SIZE);
	close(vdslfd);

	return ret;
}


/**
 * set_port_mem sets the data to the specified address for the selected port.
 * \param portid
 * 		port ID number
 * \param iAddress
 * 		memory address to write
 * \param iMemType
 * 		memory type (X-memory(0x00), Y-memory(0x01), P-memory(0x02))
 * \param pData
 * 		new Data for the memory address
 * \return
 * 	- 0 : success
 * 	- <0 : error code 	  
 * \notice
 * 		not supported	  
 */
mt_ret set_port_mem(mt_uint32 portid, mt_uint32 address, mt_uint8 memtype, mt_uint8 *pdata)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=portid;
	ioctl_buf[1]=(address & 0xff0000)>>16;
	ioctl_buf[2]=(address & 0x00ff00)>>8;
	ioctl_buf[3]=address & 0x0000ff;
	ioctl_buf[4]=memtype;
	ioctl_buf[5]=pdata[0];
	ioctl_buf[6]=pdata[1];
	ioctl_buf[7]=pdata[2];
	ret = ioctl(vdslfd, VDSL_SET_PORT_MEM, ioctl_buf);
	close(vdslfd);

	return ret;
}


/**
 * get_port_mem copies current data in the specified address to user space.
 * \param portid
 * 		port ID number
 * \param address
 * 		memory address to read
 * \param memtype
 * 		memory type (X-memory(0x00), Y-memory(0x01), P-memory(0x02))
 * \param pdata
 * 		memory location to receive the data in the specified address
 * 
 * \return
 * 	- 0 : success
 * 	- <0 : error code 
 * \notice
 * 		not supported	  
 */
mt_ret get_port_mem(mt_uint32 portid, mt_uint32 address, mt_uint8 memtype, mt_uint8 *pdata)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_buf[0]=portid;
	ioctl_buf[1]=(address & 0xff0000)>>16;
	ioctl_buf[2]=(address & 0x00ff00)>>8;
	ioctl_buf[3]=address & 0x0000ff;
	ioctl_buf[4]=memtype;
	ret = ioctl(vdslfd, VDSL_GET_PORT_MEM, ioctl_buf);
	pdata[0]=ioctl_buf[0];
	pdata[1]=ioctl_buf[1];
	pdata[2]=ioctl_buf[2];
	close(vdslfd);

	return ret;
}


mt_ret get_port_mem_remote(mt_uint32 portid, mt_uint32 address, mt_uint32 length, mt_uint8 *pdata)
{
	dsl_er(check_param(MT_PAR_PORT,0,portid));

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=portid;
	ioctl_big_buf[1]=(address & 0xff0000)>>16;
	ioctl_big_buf[2]=(address & 0x00ff00)>>8;
	ioctl_big_buf[3]=address & 0x0000ff;
	ioctl_big_buf[4]=(length & 0x00ff00)>>8;
	ioctl_big_buf[5]=length & 0x0000ff;
	ret = ioctl(vdslfd, VDSL_GET_PORT_MEM_REMOTE, ioctl_big_buf);
	memcpy(pdata, ioctl_big_buf, length*3);
	close(vdslfd);

	return ret;
}


/**
 * Download Firmware
 * \param chipid
 * 		chipid that firmware will download
 * \param cmdID 
 * 		Download Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size 
 * \param buffer
 * 		The buffer need to be downloaded
 * \notice 
 * 		Firmware download sequence is MT_FIRMWARE_START, MT_FIRMWARE_CONT, MT_FIRMWARE_END
 * 		
 */
mt_ret download_firmware(mt_uint8 chipid, mt_uint8 cmdID, mt_uint16 bufsize,mt_uint8* buffer)
{
	dsl_er(check_param(MT_PAR_PORT,0,chipid));

	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=chipid;
	ioctl_big_buf[1]=cmdID;
	ioctl_big_buf[2]=(bufsize & 0xff00)>>8;
	ioctl_big_buf[3]=bufsize & 0x00ff;
	if(cmdID == MT_FIRMWARE_CONT) memcpy(ioctl_big_buf+FW_HEADER_SIZE, buffer, bufsize);
	ret = ioctl(vdslfd, VDSL_DOWNLOAD_FIRMWARE, ioctl_big_buf);
	close(vdslfd);

#ifdef VITESS_VSC7501
	if(cmdID == MT_FIRMWARE_START) {
		system("echo out > /proc/gpio/GPIOA25");
		system("echo clear > /proc/gpio/GPIOA25");
		system("echo set > /proc/gpio/GPIOA25");
		sleep(1);
	}
#endif

	return ret;
}


/**
 * Load Firmware binary driver
 * \param cmdID 
 * 		Load Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size 
 * \param buffer
 * 		The buffer need to be downloaded
 * \notice 
 * 		Firmware binary driver load sequence is MT_FIRMWARE_START,MT_FIRMWARE_CONT,MT_FIRMWARE_END
 *
 */
mt_ret load_firmware_bindrv(mt_uint8 cmdID, mt_uint16 bufsize, mt_uint8* buffer)
{
	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=cmdID;
	ioctl_big_buf[1]=(bufsize & 0xff00)>>8;
	ioctl_big_buf[2]=bufsize & 0x00ff;
	memcpy(ioctl_big_buf+FW_BINDRV_HEADER_SIZE, buffer, bufsize);
	ret = ioctl(vdslfd, VDSL_LOAD_FIRMWARE_BINDRV, ioctl_big_buf);
	close(vdslfd);

	return ret;
}


/**
 * Flash SPI flash memory
 * \param cmdID
 * 		Flash Cmd ID MT_FIRMWARE_*
 * \param bufsize
 * 		Buffer size
 * \param buffer
 * 		The buffer need to be flashed
 * \notice
 * 		Firmware flash sequence is MT_FIRMWARE_START,MT_FIRMWARE_CONT,MT_FIRMWARE_END
 *
 */
mt_ret flash_firmware(mt_uint8 cmdID, mt_uint16 bufsize, mt_uint8* buffer)
{
	OPEN_VDSL(vdslfd);
	ioctl_big_buf[0]=cmdID;
	ioctl_big_buf[1]=(bufsize & 0xff00)>>8;
	ioctl_big_buf[2]=bufsize & 0x00ff;
	memcpy(ioctl_big_buf+FW_FLASH_HEADER_SIZE, buffer, bufsize);
	ret = ioctl(vdslfd, VDSL_FLASH_FIRMWARE, ioctl_big_buf);
	close(vdslfd);

	return ret;
}
