/****************************************************************************
* File:         $Workfile$
* Revision:     $Revision: 1.238 $
* Date:         $Date: 2012-05-23 12:44:14 $
*
* Overview:     
* Functions:
*
* Author:       Ian Tsao
* 
* -- Metanoia Copyright Notice --
*
* (C) Copyright 2005 Metanoia, Inc. All rights reserved.
* 
* Metanoia reserves the right to change specifications without notice. 
* Copyright 2005 Metanoia, Inc. All rights reserved. Preliminary Data-Sheet 
* indicates this product is in design and the specification may change. 
* Electrical parametrics have not been analyzed and are not specified. Do not 
* use this data sheet as a design reference document. Please contact Metanoia 
* for a current data sheet that may be used as a design reference. All 
* contents of this document are protected by copyright law and may not be 
* reproduced without the express written consent of Metanoia, Inc. Metanoia, 
* the Metanoia logo, and combinations thereof are trademarks of Metanoia, Inc. 
* Other product names used in this publication are for identification purposes 
* only and may be trademarks or registered trademarks of their respective 
* companies. The contents of this document are provided in connection with 
* Metanoia, Inc. products. Metanoia, Inc. has made best efforts to ensure that 
* the information contained herein is accurate and reliable. However, Metanoia, 
* Inc. makes no warranties, express or implied, as to the accuracy or 
* completeness of the contents of this publication and is providing this 
* publication 'as is'. Metanoia, Inc. reserves the right to make changes to 
* specifications and product descriptions at any time without notice, and to 
* discontinue or make changes to its products at any time without notice. 
* Metanoia, Inc. does not assume any liability arising out of the application 
* or use of any product or circuit, and specifically disclaims any and all 
* liability, including without limitation special, consequential, or 
* incidental damages.
* 
****************************************************************************/
/**
 * \file
 * \brief DSL CLI main program
 * 
 * \n The functionality of this program is
 * - <1> unit test of each functionality of xDSLAPI
 * - <2> debug mode server
 * - <3> sample program for user to integrate with their AP
 * - <4> sample lib routing that user may need it.
 * 
 */
//cfgsh include
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <unistd.h>
//dslman include
#include <time.h>
#ifdef IOCTL_INTRFCE
#include "ioctlmeta.h"
#include "dsll3.h"
#include "dsltype.h"
#include "dslerr.h"
#include "vdsllib.h"
#include "vdsl2loid.h"
#include "awardm.h"
#else
#include "dsl.h"
#endif
#include "xmlparse.h"
#include "bsp.h"
#include "clitr069.h"
#include "if2.h"
#include "maketime.h"
#include "cmdprocess.h"
#include "dslcli.h"

#include "offtimeuser.h"

#ifdef CO_API
#if defined (TEMPLATE_PROFILE)
#include "profilemgr.h"
#endif //defined (TEMPLATE_PROFILE) 
#endif //CO_API

//cfgsh define
#define xmalloc malloc

//dslman define
#define MT_CHECK_ID 1
#define MT_CHECK_IDVALUE 2
#define LIST_QUESTION_MARK_CMD
#define BILLION 1000000000L
#define PNAMETMP 128
#define PNAMESIZE 65536
#define BLKSIZE 512
#define EOC_BUF_SIZE 128

//**yalee961224**//
#define FATAL do { fprintf(stderr, "Error at line %d, file %s (%d) [%s]\n", \
__LINE__, __FILE__, errno, strerror(errno)); exit(1); } while(0)
#define MAP_SIZE 1024
#define MAP_MASK (MAP_SIZE-1)

#define INDEX_REMOTE 128
#define	LINKTIME_VER 130000

// -------------------------- Global Declaration -----------------------
//cfgsh global
char const rl_basic_word_break_characters[] = " \t\n\"\\'`@$><=;|&{(";

/* The name of this program, as taken from argv[0]. */
char * progname;

/* The command valid in the current context */
cmdt * commands = root_cmd;

/* Current command level function pointer */
int (* cmdf_dir)(unsigned char * buffer,	struct cmdt * pcmd) = com_root;

/* The prompt line */
char prompt[PROMPT_SIZE];

/* The CLI hostname presented in prompt line */
char host_name[HNAME_SIZE];

/* show prompt and online help? */
int show_prompt = 1;

/* silent mode identifier */
int silent = 0;

/* static mode identifier */
int statical = 0;

//dslman global
#ifdef BOARD_A3
#define SMI_DUMP_IGMP() smi_dump_igmp_a3()
#else
#define SMI_DUMP_IGMP()
#endif

mt_uint8 portcnt;
mt_uint8 portbuf[OID_MAX_LEN], portnum[SMALL_BUF_SIZE];
mt_uint8 pbuffer[BUF_SIZE];

//Statistic define
char *statistic_table[44] = {
	//Xdsl2PMLineHist1DayTable
	"PMLHist1DUnit",
	"PMLHist1DInterval",
	"PMLHist1DMonitoredTime",
	"PMLHist1DFecs",
	"PMLHist1DEs",
	"PMLHist1DSes",
	"PMLHist1DLoss",
	"PMLHist1DUas",
	"PMLHist1DValidInterval",
	//Xdsl2PMLineInitHist1DayTable
	"PMLHistInit1DInterval",
	"PMLHistInit1DMonitoredTime",
	"PMLHistInit1DFullInits",
	"PMLHistInit1DFailedFullInits",
	"PMLHistInit1DShortInits",
	"PMLHistInit1DFailedShortInits",
	"PMLHistInit1DValidInterval",
	//Xdsl2PMChHist1DTable
	"PMChHist1DUnit",
	"PMChHist1DInterval",
	"PMChHist1DMonitoredTime",
	"PMChHist1DCodingViolations",
	"PMChHist1DCorrectedBlocks",
	"PMChHist1DValidInterval",
	//Xdsl2PMLineHist15MinTable
	"PMLHist15MUnit",
	"PMLHist15MInterval",
	"PMLHist15MMonitoredTime",
	"PMLHist15MFecs",
	"PMLHist15MEs",
	"PMLHist15MSes",
	"PMLHist15MLoss",
	"PMLHist15MUas",
	"PMLHist15MValidInterval",
	//Xdsl2PMLineInitHist15MinTable
	"PMLHistInit15MInterval",
	"PMLHistInit15MMonitoredTime",
	"PMLHistInit15MFullInits",
	"PMLHistInit15MFailedFullInits",
	"PMLHistInit15MShortInits",
	"PMLHistInit15MFailedShortInits",
	"PMLHistInit15MValidInterval",
	//Xdsl2PMChHist15MinTable
	"PMChHist15MUnit",
	"PMChHist15MInterval",
	"PMChHist15MMonitoredTime",
	"PMChHist15MCodingViolations",
	"PMChHist15MCorrectedBlocks",
	"PMChHist15MValidInterval"
};

static int print_mib_uint32(mt_uint32 portid, const char * fmt, mt_uint8 * loid, char * name, char * desc, int idx);
static int IsRT(mt_uint32 portid);
static int IsValidMibValue(int value);

// jerry: used by silent mode
#ifdef __USE_SOCKET__
char xmlparserbuf[MSG_BUFFER_SIZE];
#endif


//=======================dslman subroutines=========================

#ifdef XML_MIN_SIZE
// -------------------------- XML Parse Functions -----------------------
/**
 * XML tag extracter\n
 * >= 0 : success, value is atts position\n
 * < 0 : fail
 */
int posIdValue(const char **atts,const char *id,const char *value,int checkType)
{
	int i;
	for (i = 0; atts[i]; i += 2) {
		switch( checkType )
		{
			case MT_CHECK_ID:
				if( strcmp(atts[i],id)==0 )
				{
					return i;
				}
				break;
			case MT_CHECK_IDVALUE:
				if( strcmp(atts[i],id)==0 && strcmp(atts[i+1],value)==0 )
				{
					return i;
				}
				break;
			default:
				break;
		}
		
	}
	return -1;
  
} 

#ifdef __USE_SOCKET__
/**
 * XML parser calling and Web commands execution function
 */
void startElement(void *userData, const char *name, const char **atts)
{
	int cliexit;
	int *depthPtr = userData;
	int pos;
	int id_tmp;
	unsigned char buffer[CMD_BUFFER_SIZE];

	if(strcmp(name,"ACTION") == 0) {
		pos = posIdValue(atts,"id","",MT_CHECK_ID);
		if( pos >=0 ) {
			id_tmp = pos+1;
			pos = posIdValue(atts,"cmd","",MT_CHECK_ID);
			if(pos >= 0) {
				commands = root_cmd;
				bzero(buffer, sizeof(char) * CMD_BUFFER_SIZE);
				bzero(msg_buf, sizeof(char) * MSG_BUFFER_SIZE);

				strcpy(buffer, atts[pos+1]);
#ifdef OPEN_LOG
				syslog(1, buffer);
#endif
				cliexit = execute_line(stripwhite(buffer), commands);
#ifdef OPEN_LOG
				syslog(1, msg_buf);
#endif
				if ( strstr(msg_buf, "</number>") != NULL) {
					msg_buf[strlen(msg_buf)-9] = msg_buf[strlen(msg_buf)];
					sprintf(xmlparserbuf, "%s", msg_buf);
					syslog(1, "exit code = %d", cliexit);
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				else if ( strstr(msg_buf, "</MIB>") != NULL) {
					sprintf(xmlparserbuf, "<REPLY id=\"%s\" status=\"%d\">%s%s" , 
								atts[id_tmp], cliexit == 0 ? 1 : 0, msg_buf, "</REPLY>");
					syslog(1, "exit code = %d", cliexit);
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				else if(msg_buf[strlen(msg_buf)-1] == '|') {
					msg_buf[strlen(msg_buf)-1] = msg_buf[strlen(msg_buf)];
					sprintf(xmlparserbuf, "<REPLY id=\"%s\" status=\"%d\" msg=\"%s%s", 
								atts[id_tmp], cliexit == 0 ? 1 : 0, msg_buf, "\"></REPLY>");
					syslog(1, "exit code = %d", cliexit);
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				else {
					sprintf(xmlparserbuf, "<REPLY id=\"%s\" status=\"%d\" msg=\"%s%s", 
								atts[id_tmp], cliexit == 0 ? 1 : 0, msg_buf, "\"></REPLY>");
					syslog(1, "exit code = %d", cliexit);
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
			}
			else {
				sprintf(xmlparserbuf, "<REPLY id=\"%s\" status=\"0\" msg=\"cmd not found!!\"></REPLY>", atts[id_tmp]);
			}
		}
		else {
			sprintf(xmlparserbuf, "<REPLY id=\"\" status=\"0\" msg=\"id not found!!\"></REPLY>");
		}
	}
	else {
		sprintf(xmlparserbuf, "<REPLY id=\"\" status=\"0\" msg=\"ACTION not found!!\"></REPLY>");
	}

	*depthPtr += 1;
}
#else
/**
 * XML parser calling and Web commands execution function
 */
void startElement(void *userData, const char *name, const char **atts)
{
	int cliexit;
	int *depthPtr = userData;
	int pos;
	int id_tmp;
	unsigned char buffer[CMD_BUFFER_SIZE];

	if(strcmp(name,"ACTION") == 0) {
		pos = posIdValue(atts,"id","",MT_CHECK_ID);
		if( pos >=0 ) {
			id_tmp = pos+1;
			pos = posIdValue(atts,"cmd","",MT_CHECK_ID);
			if(pos >= 0) {
				commands = root_cmd;
				bzero(buffer, sizeof(char) * CMD_BUFFER_SIZE);
				bzero(msg_buf, sizeof(char) * MSG_BUFFER_SIZE);

				//syslog(1, atts[id_tmp]);
				strcpy(buffer, atts[pos+1]);
#ifdef OPEN_LOG
				syslog(1, buffer);
#endif
				cliexit = execute_line(stripwhite(buffer), commands);
				//syslog(1, "In startElement");
				//syslog(1, "msg_buf");
#ifdef OPEN_LOG
				syslog(1, msg_buf);
#endif
				//syslog(1, "End of msg_buf");
				//if(msg_buf[strlen(msg_buf)-1] == '#') {
				if ( strstr(msg_buf, "</number>") != NULL) {
					//msg_buf[strlen(msg_buf)-1] = msg_buf[strlen(msg_buf)];
					msg_buf[strlen(msg_buf)-9] = msg_buf[strlen(msg_buf)];
					PRINTD("%s" , msg_buf);
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				//else if(msg_buf[strlen(msg_buf)-1] == '>') {
				else if ( strstr(msg_buf, "</MIB>") != NULL) {
					PRINTD("<REPLY id=\"%s\" status=\"%d\">" , atts[id_tmp], cliexit == 0 ? 1 : 0 );
					PRINTD("%s" , msg_buf);
					PRINTD("</REPLY>");
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				else if(msg_buf[strlen(msg_buf)-1] == '|') {
					PRINTD("<REPLY id=\"%s\" status=\"%d\" msg=\"", atts[id_tmp], cliexit == 0 ? 1 : 0 );
					msg_buf[strlen(msg_buf)-1] = msg_buf[strlen(msg_buf)];
					PRINTD("%s" , msg_buf);
					PRINTD("\"></REPLY>");
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
				else {
					PRINTD("<REPLY id=\"%s\" status=\"%d\" msg=\"", atts[id_tmp], cliexit == 0 ? 1 : 0 );
					PRINTD("%s" , msg_buf);
					PRINTD("\"></REPLY>");
#ifdef OPEN_LOG
					syslog(1, "End of PRINTD");
#endif
				}
			}
			else {
				PRINTD("<REPLY id=\"%s\" status=\"0\" msg=\"cmd not found!!\"></REPLY>", atts[id_tmp]);
			}
		}
		else {
			PRINTD("<REPLY id=\"\" status=\"0\" msg=\"id not found!!\"></REPLY>");
		}
	}
	else {
		PRINTD("<REPLY id=\"\" status=\"0\" msg=\"ACTION not found!!\"></REPLY>");
	}

	*depthPtr += 1;
}
#endif


/**
 * XML parser ending calling function
 */
void endElement(void *userData, const char *name)
{
	int *depthPtr = userData;
	*depthPtr -= 1;
}
#endif	//XML_MIN_SIZE


// -------------------------- User Library -----------------------

/**
 * Print uint32 mib entry
 * \param portid: port Id, start from 0
 * \param fmt: print format, ex: "%-24s|%-24s %s"
 * \param loid: logic oid, defined in header file
 * \param name: mib entry name
 * \param desc: mib entry description
 * \param idx: mib entry index
 * \return
 * 	- 0 : success
 * 	- other : error code 
 */
static int print_mib_uint32(mt_uint32 portid, const char * fmt, mt_uint8 * loid, char * name, char * desc, int idx)
{
	int ret , value;
	mt_uint8 loidf[OID_MAX_LEN], valuestr[64];

	sprintf(loidf, "%s.%d", loid, idx);
	if ((ret = get_mib2_int(portid, loidf, &value)) == DSL_ERR_OK){
		if (IsValidMibValue(value))
			sprintf(valuestr, "%d", value);
		else
			strcpy(valuestr, "NA");
			
		PRINTL(DBG_LVL_ALWAYS, fmt, name, valuestr, desc);
		PRINTL(DBG_LVL_XML, "%s=%s|", name, valuestr);
	}

	return ret;
}


static int IsRT(mt_uint32 portid){
	mt_uint32 site;

	get_mib2_int(portid, xdslTerminal, &site);

	return (site & 0x00000001) ? 1 : 0;
}


static int IsValidMibValue(int value){
	/* 0x7FFFFE:8388606, 0x7FFFFF:8388607, 0xFFFFCC:16777164 */
	return (value == 0x7FFFFE || value == 0x7FFFFF || value == 0xFFFFCC) ? 0 : 1;
}


/**
 * Display a block of 32-bit memory
 */
void dispmem(mt_uint32* pbuf,mt_uint32 count)
{
	int i;
	
	for(i=0;i<count;i++)
	{
		PRINTD("%x ",*pbuf++);
	}
}


/**
 * Display a block of 8-bit memory
 */
void dispmem8(mt_uint8* pbuf,mt_uint32 count)
{
	int i;
	
	for(i=0;i<count;i++)
	{
		PRINTD("%x ",*pbuf++);
	}
}


mt_ret process_port(void)
{
	mt_uint8 multi, error, len;
	mt_uint32 value;
	mt_uint32 i, j, k;
	
	portcnt = 0;
	multi = 0;
	error = 0;
	len = strlen(portbuf);
	for(i=0,j=0;i<len;i++) {
		if(portbuf[i] == ',' || i+1 == len) {
			value = strtod(portbuf+j, NULL);
			if(value == 0) {
				PRINTL(DBG_LVL_CLI, "Port number process error!!");
				error = 1;
				break;
			}

			if(multi) {
				for(;k<=value;k++) {
					portnum[portcnt++] = k;
				}

				multi = 0;
			}
			else {
				portnum[portcnt++] = value;
			}

			j = i+1;
		}
		else if(portbuf[i] == '-') {
			value = strtod(portbuf+j, NULL);
			if(value == 0) {
				PRINTL(DBG_LVL_CLI, "Port number process error!!");
				error = 1;
				break;
			}
			
			k = value;
			j = i+1;
			multi = 1;
		}
	}
	
	//for(i=0;i<portcnt;i++) printf("%d ", portnum[i]);
	//printf("\n");

	if(error) return DSL_ERR_NOTOK;
	else return DSL_ERR_OK;
}


#ifdef DEBUG_API
/**
 * We need delay in many case, the delay unit maybe clock, micro-second, second. And different system have different mechanism to delay.
 * \param type
 * 	 	1: Second\n
 * 	 	2: CPU clock
 */
void dsldelay(mt_uint8 type, mt_uint32 count)
{
	mt_uint32 i;

	switch( type ){
		case 1:
			i=25000000*count;	//25000000 loops about 1 seconds, the timing is measured on Award2, not on other boards
			while(i--);
			break;
		case 2:
			i=25000*count;	//25000 loops about 1 miliseconds, the timing is measured on Award2, not on other boards
			while(i--);
			break;
		default:
			break;
	}
}
#endif // DEBUG_API


// -------------------------- general cmd function -----------------------

/**
 * dslcli command execution entry point
 */
int execute(unsigned char * pbuf, cmdt * pcmd)
{
	int i = 0;
	unsigned char cmd[MT_CMD_BufferSize];

	//buffer overflow problem <== Ian solved this problem
	sscanf(pbuf, "%s", cmd);
	while(pcmd[i].cmd != NULL){
		//PRINTD("cmd:%s\n",pcmd[i]);
		if(strcmp(pcmd[i].cmd, cmd) == 0){
			if(strcmp("?", cmd) == 0)
				return (*pcmd[i].cmdf)(pbuf + strlen(pcmd[i].cmd) + 1, pcmd);
			else
				return (*pcmd[i].cmdf)(pbuf + strlen(pcmd[i].cmd) + 1, pcmd[i].sub);
		}
		i++;
	}
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "No such command.");
	return DSL_ERR_NOCMD;
}


/**
 * dslcli multi-commands parser
 */
int execute_line(unsigned char * pbuf, cmdt * pcmd)
{
	int i, len;
	int is_dir_cmd;
	int ifexit;
	int (*tmpf_dir)(unsigned char * buffer,	struct cmdt * pcmd);
	cmdt *tmp_cmd_tbl = NULL;
	unsigned char cmd[MT_CMD_BufferSize];
	unsigned char opt[MT_OPT_BufferSize];
	char command[MT_CMD_BufferSize];
	char line[MT_CMD_BufferSize];
	unsigned char *buffer = cmd;
	unsigned char *token, *ptr;
	FILE *streamout;

	if(pbuf[0] == '#' || pbuf[0] == '\n' || pbuf[0] == '\0')	//comment line or blank line, skip it!!
		return DSL_ERR_OK;

	/*token = strstr(pbuf, "#");
	if(token != NULL) *token = '\0';	//strip comment from command*/

	//check if inputted command is "all-spaces" command
	for(i = 0, len = strlen(pbuf); i < len; i++) {
		if(pbuf[i] != ' ')
			break;
	}
	if((i == len || i == len-1) && (pbuf[len-1] == ' ' || pbuf[len-1] == '\n')) {	//if yes, skip "all-spaces" command
		return DSL_ERR_OK;
	}

	//add variable parser to support read variable through cfgmgr
	bzero(buffer, sizeof(char) * MT_CMD_BufferSize);
	len = strlen(pbuf);
	ptr = pbuf;
	token = strstr(pbuf, "$");
	while(token != NULL) {
		*token++ = '\0';
		strcat(buffer, ptr);
		ptr = token;
		//PRINTD("buffer: %s\n", buffer);

		token = strstr(ptr, "$");
		if(token != NULL) {
			*token = '\0';
			//PRINTD("ptr: %s\n", ptr);
		}
		else {	//token == NULL, ending character "$" of variable no found, this is illegal
			buffer[0] = '\0';	//this make strlen(buffer) equals 0
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Variable format error!!");
			break;
		}

		sprintf(command, ""CFGMGR_PATH" -r %s", ptr);
		if((streamout = popen(command, "r")) != NULL) {
			if(fgets(line, 256, streamout) != NULL) {
				stripwhite(line);
				strcat(buffer, line);
			}
			pclose(streamout);
		}

		ptr = ++token;
		token = strstr(ptr, "$");
	}

	if(strlen(buffer)) {//strlen(buffer) != 0 means we catched variables
		strcat(buffer, ptr);
		strcpy(pbuf, buffer);
		PRINTL(DBG_LVL_ALWAYS, "%s", pbuf);
	}

	if(pbuf[0] == '#')	//after variable replacing, the line became comment line, skip it!!
		return DSL_ERR_OK;

	if(strlen(pbuf) > MT_CMD_BufferSize) {	//input command too long, truncate length to MT_CMD_BufferSize
		pbuf[MT_CMD_BufferSize-1] = '\0';
	}

	while(strlen(pbuf) > 0) {
	//for(j = 0; j < 10 && (strlen(pbuf) > 0); j++){
		//PRINTD("\"%s\"\n", pbuf);
		sscanf(pbuf, "%s", cmd);
		i = 0;
		ifexit = DSL_ERR_NOCMD;
		is_dir_cmd = 0;
		while(commands[i].cmd != NULL){
			//PRINTD("command=%s\n", commands[i].cmd);
			if(strcmp(commands[i].cmd, cmd) == 0){
				//after running execute(), commands pointer
				//will be changed, so keep the old command
				//table pointer here
				tmp_cmd_tbl = commands;
				ifexit = execute(pbuf, commands);

				//tmp_cmd_tbl[i].sub == NULL means the current
				//executed is the last command, so discard
				//the rest tokens
				if(ifexit || tmp_cmd_tbl[i].sub == NULL)
					goto end_execute_line;

				//check if the executed command is a directory command,
				//if yes, mark it and save the directory level function
				if(tmp_cmd_tbl[i].sub != NULL) {
					is_dir_cmd = 1;
					tmpf_dir = tmp_cmd_tbl[i].cmdf;
				}

				pbuf = pbuf + strlen(tmp_cmd_tbl[i].cmd) + 1;
				pbuf = stripwhite(pbuf);

				//if current executed command have an option
				//and inputted command has the option token,
				//then skip the option token to keep executing
				//the next command
				if((strlen(pbuf) > 0) && tmp_cmd_tbl[i].complete_func != NULL){
					sscanf(pbuf, "%s", opt);
					pbuf = pbuf + strlen(opt) + 1;
				}

				break;
			}
			i++;
		}
		if(ifexit == DSL_ERR_NOCMD && commands[i].cmd == NULL){
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "No such command.");
			break;
		}
	}

end_execute_line:
	//if the last command is not a directory command then
	//change "commands" back to the original command directory
	//else sync current directory level
	if(!is_dir_cmd) cmdf_dir(NULL, NULL);
	else cmdf_dir = tmpf_dir;

	return ifexit;
}


//=======================cfgsh subroutines=========================

/* **************************************************************** */
/*                                                                  */
/*                       Utility Functions                          */
/*                                                                  */
/* **************************************************************** */


/** Set the prompt */
void set_prompt(char * prompt_add) {

  if(prompt_add)
    snprintf(prompt, PROMPT_SIZE, "%s %s%s", host_name, prompt_add, PROMPT);
  else
    snprintf(prompt, PROMPT_SIZE, "%s%s", host_name, PROMPT);

  return;
}


/** dslcli config initializer */
int init_config (int setup) {
	char command[64];
	char line[256];
	FILE *streamout;

	sprintf(command, ""CFGMGR_PATH" -r CLI_HOSTNAME");
	if((streamout = popen(command, "r")) != NULL) {
		if(fgets(line, 256, streamout) != NULL) {
			stripwhite(line);
			strcpy(host_name, line);
		}
		pclose(streamout);
	}

#if defined (CO_API) && defined (TEMPLATE_PROFILE)
	INIT_LIST_HEAD(&vdsl_all_profile_head);
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)

	// load ATM PVC settings from board.conf
	parse_pvc_rule();

	return DSL_ERR_OK;
}


/** A safe(r) replacement for the system() call */
int safesystem (char * filename, char * cmdline[])
{
  int pid;
  int status;

#if !defined (VIA_SAMSUNG) && !defined (REALTEK_RTL8672)
  pid = fork();
#else
  pid = vfork();
#endif
  if (pid < 0) {
    perror(cmdline[0]);
  } else if(pid == 0) {
    execve(filename, cmdline, NULL);
    /* wtf? execve just failed on us */
    exit(1);
  }

  /* FIXME: use waitpid instead so that if some other child retunrs
     we'll not get confused. Doesn't really matter for our use here. */

  wait(&status);

  if(WIFEXITED(status))
    return WEXITSTATUS(status);

  return DSL_ERR_NOTOK;
}


/* **************************************************************** */
/*                                                                  */
/*                  Interface to Readline Completion                */
/*                                                                  */
/* **************************************************************** */


#ifdef DEBUG_API
/** Interface tab completion matches function */
char ** interface_completion_matches(const char * text, char * dummy, int start)
{
  //if(word_num(start, rl_line_buffer) == 1)
  //  return rl_completion_matches(text, interface_generator);

  return NULL;
}


/** Route tab completion matches  function */
char ** route_completion_matches(const char * text, char * dummy, int start)
{
  return NULL;

}
#endif //DEBUG_API


//=======================dslcli cmd tables=========================
/** Return to root mode */
int com_root (unsigned char * pbuf, cmdt * pcmd)
{
	commands = root_cmd;
	set_prompt(NULL);
	return DSL_ERR_OK;
}


int com_help (unsigned char * pbuf, cmdt * pcmd)
{
	int ll_help;
	ll_help = help(pbuf,pcmd);
	return DSL_ERR_OK;
}


int com_exit (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_EXITCLI;
}


int com_home (unsigned char * pbuf, cmdt * pcmd)
{
	commands = root_cmd;
	set_prompt(NULL);
	return DSL_ERR_OK;
}


int com_exitf (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_EXITCLI;
}


mt_ret oid2gtip(mt_uint8 *oid, mt_uint32 *groupid, mt_uint32 *tableid, mt_uint32 *paramid, mt_uint32 *idx, mt_uint32 *interval)
{
	mt_uint8 *token;
	mt_uint8 oidtmp[OID_MAX_LEN];

	strcpy(oidtmp, oid);
	//printf("oid: %s\n", oidtmp);

	token = strtok(oidtmp, ".");
	if(token == NULL) return FALSE;
	*groupid = strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) return FALSE;
	*tableid = strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) return FALSE;
	*paramid = strtoul(token, NULL, 10);

	token = strtok(NULL, ".");
	if(token == NULL) *idx = 1;
	else *idx = strtoul(token, NULL, 10);

	if(interval != NULL) {
		token = strtok(NULL, ".");
		if(token == NULL) *interval = 0;
		else *interval = strtoul(token, NULL, 10);
	}

	//if(interval != NULL) printf("g:%d, t:%d, p:%d, i:%d, i:%d\n", *groupid, *tableid, *paramid, *idx, *interval);
	//else printf("g:%d, t:%d, p:%d, i:%d\n", *groupid, *tableid, *paramid, *idx);

	return TRUE;
}


/**
 * Show VPI/VCI VLAN mapping
 * \par Usage:
 * 		cmd <pvc>
 * \param pvc: Permanent Virtual Circuit to show
 * \return
 *
 */
int show_pvc_rule(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 i, tmp[32];
	mt_uint8 pvc, vpi, prio, encap, atmqos, activation;
	mt_uint16 vci, vlanid;
	mt_uint32 pcr, scr, mbs;

	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "PVC\tVPI\tVCI\tVLAN_ID\tQueue_Priority\tATM_Encapsulation\tATM_QoS\tPCR\tSCR\tMBS\tActivation");
	for(i=0;i<NUM_OF_PVC;i++) {
		sprintf(pbuffer, "PVC%d\t%d\t%d\t%d\t%d\t\t", i, atm_pvc[i].vpi, atm_pvc[i].vci, atm_pvc[i].vlanid, atm_pvc[i].prio);

		if(atm_pvc[i].encap == VC_MUX)
			strcat(pbuffer, "VC-MUX");
		else if(atm_pvc[i].encap == LLC)
			strcat(pbuffer, "LLC");

		strcat(pbuffer, "\t\t\t");

		/*if(atm_pvc[i].atmqos == CBR)
			strcat(pbuffer, "CBR");
		else if(atm_pvc[i].atmqos == VBR_RT)
			strcat(pbuffer, "VBR-RT");
		else if(atm_pvc[i].atmqos == VBR_NRT)
			strcat(pbuffer, "VBR-NRT");
		else if(atm_pvc[i].atmqos == UBR)
			strcat(pbuffer, "UBR");*/
		if(atm_pvc[i].atmqos == UBR)
			strcat(pbuffer, "UBR");

		sprintf(tmp, "\t%d\t%d\t%d\t", atm_pvc[i].pcr, atm_pvc[i].scr, atm_pvc[i].mbs);
		strcat(pbuffer, tmp);

		if(atm_pvc[i].active == TRUE)
        	strcat(pbuffer, "Yes");
       	else
       		strcat(pbuffer, "No");

        PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "%s", pbuffer);
	}

	return ret;
}


#ifdef WITH_STATISTIC
/**
 * Show port statistic
 * \par Usage:
 * 		cmd <portid> <loid>
 * \param portid: port Id, start from 1
 * \param loid: Logic MIB Object ID represented in x.y.z format
 * \return
 *
 */
int show_portstat_by_oid(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 phy_status;
	mt_uint8 not_empty;
	mt_uint8 tmp[7];
	mt_uint8 buf[BUF_SIZE];
	mt_uint8 loid[OID_MAX_LEN];
	mt_uint8 loidtmp[BUF_SIZE];
	mt_uint32 portid;
	mt_uint32 j, size, value;
	mt_uint32 groupid, tableid, idx, paramid, interval;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "show_portstat: start");
#endif

	pbuffer[0] = 0x0;
	pbuffer[1] = 0x0;

	ret = sscanf(pbuf, "%i %s", &portid, loid);
	if( ret != 2 ) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <portid> <loid>");
		return -DSL_ERR_NOPASS;
	}
	//PRINTL(DBG_LVL_ALWAYS, "portid=%i,loid=%s", portid, loid);

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_ALWAYS, "Port%d disabled", portid+1);
		ret = DSL_ERR_NOTOK;
	}
	else {
		ret = oid2gtip(loid, &groupid, &tableid, &paramid, &idx, NULL);
		if(ret == FALSE) {
			PRINTD("Insufficient OID field!!\n");
			return DSL_ERR_NOTOK;
		}

		if(groupid != MT_MIB_XdslGroup) {	//group has no interval
			return DSL_ERR_NOTOK;
		}

		not_empty = 0;
		//PRINTL(DBG_LVL_XML, "<row>");
		if(tableid == MT_MIB_Xdsl2PMLineHist1DayTable || tableid == MT_MIB_Xdsl2PMLineInitHist1DayTable || tableid == MT_MIB_Xdsl2PMChHist1DTable) {
			size = MT_INV_MAX24;

			PRINTL(DBG_LVL_ALWAYS, "=====1Day Statistic=====", value);
			for(interval=1;interval<=size;interval++) {
				sprintf(loidtmp, "%d.%d.%d.%d.%d", groupid, tableid, paramid, idx, interval);
				//printf("loid: %s\n", loidtmp);
				ret = get_port_mib(portid, loidtmp, pbuffer);
				if(ret == DSL_ERR_OK) {
					not_empty = 1;
					value = strtoul(pbuffer+2, NULL, 16);
					//PRINTL(DBG_LVL_ALWAYS, "LOID:\t%s\t%d", loidtmp, value);
					if(interval == 1) {
						PRINTL(DBG_LVL_ALWAYS, "\n%d.%d.%d.%d", groupid, tableid, paramid, idx);
						PRINTL(DBG_LVL_ALWAYS, "Past");
					}

					PRINTL(DBG_LVL_ALWAYS, "%02dD\t%d", interval, value);
				}
				else {
					//PRINTL(DBG_LVL_ALWAYS, "Get statistic failed!");
				}
			}
			if(not_empty) {
				PRINTL(DBG_LVL_ALWAYS, "");
			}
		}

		else if(tableid == MT_MIB_Xdsl2PMLineHist15MinTable || tableid == MT_MIB_Xdsl2PMLineInitHist15MinTable || tableid == MT_MIB_Xdsl2PMChHist15MinTable) {
			size = MT_INV_MAX15;

			PRINTL(DBG_LVL_ALWAYS, "=====15Min Statistic=====", value);
			for(interval=1, j=0;interval<=size;interval++) {
				sprintf(loidtmp, "%d.%d.%d.%d.%d", groupid, tableid, paramid, idx, interval);
				//printf("loid: %s\n", loidtmp);
				ret = get_port_mib(portid, loidtmp, pbuffer);
				if(ret == DSL_ERR_OK) {
					not_empty = 1;
					value = strtoul(pbuffer+2, NULL, 16);
					//PRINTL(DBG_LVL_ALWAYS, "LOID:\t%s\t%d", loidtmp, value);
					if(interval == 1) {
						PRINTL(DBG_LVL_ALWAYS, "\n%d.%d.%d.%d", groupid, tableid, paramid, idx);
						PRINTL(DBG_LVL_ALWAYS, "Past\t15M\t30M\t45M\t60M");
					}

					if(j == 0) {
						bzero(buf, BUF_SIZE);
						sprintf(loidtmp, "%02dH", interval/4);
						strcat(buf, loidtmp);
					}

					sprintf(loidtmp, "\t%d", value);
					strcat(buf, loidtmp);

					if(++j == 4) {
						j = 0;
						PRINTL(DBG_LVL_ALWAYS, "%s", buf);
					}
				}
				else {
					PRINTL(DBG_LVL_ALWAYS, "Get statistic failed!");
				}
			}
			if(not_empty && j != 0) {
				PRINTL(DBG_LVL_ALWAYS, "%s", buf);
			}
			if(not_empty) {
				PRINTL(DBG_LVL_ALWAYS, "");
			}
		}
		else	//table has no interval
			return DSL_ERR_NOTOK;
	}

	return ret;
}

/**
 * Show port statistic
 * \par Usage:
 * 		cmd <testid> <portid>
 * \param testid: test item Id, Id 1 is 15Min, Id 2 is 1Day.
 * \param portid: port Id, start from 1
 * \return
 *
 */
int show_portstat(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 phy_status;
	mt_uint8 not_empty;
	mt_uint8 tmp[7];
	mt_uint8 buf[BUF_SIZE], xml_buf[BUF_SIZE];
	mt_uint8 loidtmp[BUF_SIZE];
	mt_uint8 paramsize, idxsize, intrvlsize;
	mt_uint32 id, portid;
	mt_uint32 i, size, value;
	mt_uint32 groupid, tableid, idx, paramid, interval;
	mib_table_set statistic[6] = {
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineHist1DayTable, 9, MT_IDX_LINE, MT_INV_MAX24},
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineInitHist1DayTable, 7, MT_IDX_LINE, MT_INV_MAX24},
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMChHist1DTable, 6, MT_IDX_CHAN, MT_INV_MAX24},
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineHist15MinTable, 9, MT_IDX_LINE, MT_INV_MAX15},
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMLineInitHist15MinTable, 7, MT_IDX_LINE, MT_INV_MAX15},
			{MT_MIB_XdslGroup, MT_MIB_Xdsl2PMChHist15MinTable, 6, MT_IDX_CHAN, MT_INV_MAX15}
	};

	// jerry
	struct tm *tp;
	struct timeval tv;
	time_t buf_time;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "show_portstat: start");
#endif

	pbuffer[0] = 0x0;
	pbuffer[1] = 0x0;

	ret = sscanf(pbuf, "%i %i", &id, &portid);
	if( ret != 2 ) {
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> <portid>, port id start from 1");
		PRINTL(DBG_LVL_ALWAYS, "  1: Get 15Min Statistic");
		PRINTL(DBG_LVL_ALWAYS, "  2: Get 1Day Statistic");
		return DSL_ERR_OK;
	}
	//PRINTL(DBG_LVL_ALWAYS, "test id=%i,portid=%i", id, portid);

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE) {
		PRINTL(DBG_LVL_ALWAYS, "Port%d disabled", portid+1);
		ret = DSL_ERR_NOTOK;
	}
	else {
		// jerry
		gettimeofday(&tv, NULL);
		buf_time = tv.tv_sec;
		tp = (struct tm*)malloc(sizeof(struct tm));
		memset(tp, 0, sizeof(struct tm));
		offtimeuser(&buf_time, 0, tp);

		if(id == 1) {
			//15Min History
			ret = get_port_mib(portid, xdsl2PMLCurr15MTimeElapsed, pbuffer);
			if(ret == DSL_ERR_OK) value = strtoul(pbuffer+2, NULL, 16);
			else PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Get 15Min Time Elapsed failed!");

			PRINTL(DBG_LVL_ALWAYS, "Current 15Min Time Elapsed: %02d:%02d", value/60, value%60);
			PRINTL(DBG_LVL_XML, "port=%d|15MinElapsed=%d", portid+1, value);

			PRINTL(DBG_LVL_ALWAYS, "15Min History\t\tmtime\tfecs\tes\tses\tloss\tuas\tvalid\tinit\tfailed\tcv\tfec");
			PRINTL(DBG_LVL_XML, "|15MinHistory=mtime,fecs,es,ses,loss,uas,valid,init,failed,cv,fec");

			// jerry: boundary issue
			if ((buf_time % SECS_15_MINS) < SECS_LOG_DELAY)
				buf_time -= (buf_time % SECS_15_MINS + SECS_15_MINS);
			else 
				buf_time -= buf_time % SECS_15_MINS;
			
			for(interval=0;interval<=MT_INV_MAX15;interval++) {
				bzero(buf, BUF_SIZE);
				bzero(xml_buf, BUF_SIZE);

				// jerry
				sprintf(buf, "%02d:%02d\t\t", tp->tm_hour, tp->tm_min);
				offtimeuser(&buf_time, 0, tp);
				buf_time -= SECS_15_MINS;
#ifdef __jerry__
				if(interval == 0) sprintf(buf, "Current      ");
				else sprintf(buf, "Previous %2d  ", interval);
#endif
				for(idx=1;idx<=MT_IDX_LINE;idx++) {
					if(idx == 1) {
						strcat(buf, "XTUC");
						sprintf(xml_buf, "|%d.XTUC=", interval);
					}
					else {
						bzero(buf, BUF_SIZE);
						bzero(xml_buf, BUF_SIZE);
						strcat(buf, "\t        XTUR");
						sprintf(xml_buf, "|%d.XTUR=", interval);
					}
					for(i=3;i<6;i++) {
						groupid = statistic[i].group;
						tableid = statistic[i].table;
						paramsize = statistic[i].param;
						idxsize = statistic[i].index;

						if((idxsize == MT_IDX_CHAN) && (idx == 2)) idx++;	// we process bearer channel 1 and 3 only

						for(paramid=1;paramid<=paramsize;paramid++) {
							sprintf(loidtmp, "%d.%d.%d.%d.%d", groupid, tableid, paramid, idx, interval);
							if(strncmp(loidtmp, xdsl2PMLHist15MUnit, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHist15MInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit15MInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit15MMonitoredTime, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit15MShortInits, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit15MFailedShortInits, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit15MValidInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist15MUnit, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist15MInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist15MMonitoredTime, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist15MValidInterval, 6) == 0) continue;

							ret = get_port_mib(portid, loidtmp, pbuffer);
							if(ret == DSL_ERR_OK) {
								value = strtoul(pbuffer+2, NULL, 16);
								//PRINTL(DBG_LVL_ALWAYS, "LOID:\t%s\t%d", loidtmp, value);

								sprintf(loidtmp, "\t%d", value);
								strcat(buf, loidtmp);
								sprintf(loidtmp, "%d,", value);
								strcat(xml_buf, loidtmp);
							}
							else {
								//PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Get statistic failed!");
								ret = DSL_ERR_OK;
								goto end_portstat;
							}
						}

					}

					PRINTL(DBG_LVL_ALWAYS, "%s", buf);
					xml_buf[strlen(xml_buf)-1] = xml_buf[strlen(xml_buf)];
					PRINTL(DBG_LVL_XML, "%s", xml_buf);
				}
			}
		}
		else if(id == 2) {
			//1Day History
			ret = get_port_mib(portid, xdsl2PMLCurr1DayTimeElapsed, pbuffer);
			if(ret == DSL_ERR_OK) value = strtoul(pbuffer+2, NULL, 16);
			else PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Get 1Day Time Elapsed failed!");
			PRINTL(DBG_LVL_ALWAYS, "Current 1Day Time Elapsed: %02d:%02d:%02d", value/3600, (value%3600)/60, value%60);
			PRINTL(DBG_LVL_XML, "port=%d|1DayElapsed=%d", portid+1, value);

			PRINTL(DBG_LVL_ALWAYS, "1Day History\t\tmtime\tfecs\tes\tses\tloss\tuas\tvalid\tinit\tfailed\tcv\tfec");
			PRINTL(DBG_LVL_XML, "|1DayHistory=mtime,fecs,es,ses,loss,uas,valid,init,failed,cv,fec");

			// jerry: boundary issue
			if ((buf_time % SECS_PER_DAY) < SECS_LOG_DELAY)
				buf_time -= (buf_time % SECS_PER_DAY + SECS_PER_DAY + SECS_PER_DAY);
			else 
				buf_time -= (buf_time % SECS_PER_DAY + SECS_PER_DAY);

			for(interval=0;interval<=MT_INV_MAX24;interval++) {
				bzero(buf, BUF_SIZE);
				bzero(xml_buf, BUF_SIZE);
				
				// jerry
				sprintf(buf, "%04d/%02d/%02d\t", (tp->tm_year + 1900), tp->tm_mon, tp->tm_mday);
				offtimeuser(&buf_time, 0, tp);
				buf_time -= SECS_PER_DAY;
				
#ifdef __jerry__
				if(interval == 0) sprintf(buf, "Current      ");
				else sprintf(buf, "Previous %2d  ", interval);
#endif
				for(idx=1;idx<=MT_IDX_LINE;idx++) {
					if(idx == 1) {
						strcat(buf, "XTUC");
						sprintf(xml_buf, "|%d.XTUC=", interval);
					}
					else {
						bzero(buf, BUF_SIZE);
						bzero(xml_buf, BUF_SIZE);
						strcat(buf, "\t        XTUR");
						sprintf(xml_buf, "|%d.XTUR=", interval);
					}
					for(i=0;i<3;i++) {
						groupid = statistic[i].group;
						tableid = statistic[i].table;
						paramsize = statistic[i].param;
						idxsize = statistic[i].index;

						if((idxsize == MT_IDX_CHAN) && (idx == 2)) idx++;	// we process bearer channel 1 and 3 only

						for(paramid=1;paramid<=paramsize;paramid++) {
							sprintf(loidtmp, "%d.%d.%d.%d.%d", groupid, tableid, paramid, idx, interval);
							if(strncmp(loidtmp, xdsl2PMLHist1DUnit, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHist1DInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit1DInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit1DMonitoredTime, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit1DShortInits, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit1DFailedShortInits, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMLHistInit1DValidInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist1DUnit, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist1DInterval, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist1DMonitoredTime, 6) == 0) continue;
							if(strncmp(loidtmp, xdsl2PMChHist1DValidInterval, 6) == 0) continue;

							ret = get_port_mib(portid, loidtmp, pbuffer);
							if(ret == DSL_ERR_OK) {
								value = strtoul(pbuffer+2, NULL, 16);
								//PRINTL(DBG_LVL_ALWAYS, "LOID:\t%s\t%d", loidtmp, value);

								sprintf(loidtmp, "\t%d", value);
								strcat(buf, loidtmp);
								sprintf(loidtmp, "%d,", value);
								strcat(xml_buf, loidtmp);
							}
							else {
								//PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Get statistic failed!");
								ret = DSL_ERR_OK;
								goto end_portstat;
							}
						}

					}

					PRINTL(DBG_LVL_ALWAYS, "%s", buf);
					xml_buf[strlen(xml_buf)-1] = xml_buf[strlen(xml_buf)];
					PRINTL(DBG_LVL_XML, "%s", xml_buf);
				}
			}
		}
		free(tp);
	}

end_portstat:
	return ret;
}
#endif	//WITH_STATISTIC


get_roid_port(mt_uint8 *rawoid, mt_uint32 *portid, mt_uint8 *stripoid)
{
	mt_uint8 *token, i, str_find = 1;
	mt_uint8 oidtmp[OID_MAX_LEN], oidtarget[OID_MAX_LEN];
	mt_uint8* oid_list[MAXOID]={
		"1.1.1.1", "1.1.2.1", "1.2.2.1", "1.2.3.1", "1.2.4.1",
		"1.2.5.1", "1.3.1.1", "1.5.1.1.1", "1.5.1.2.1", "1.5.1.3.1",
		"1.5.1.4.1", "1.5.2.1.1", "1.5.3.1.1", "1.5.3.2.1", "1.5.3.3.1",
		"1.4.1.1.1", "1.4.1.2.1", "1.4.1.3.1", "1.4.1.4.1", "1.4.1.5.1",
		"1.4.1.6.1", "1.4.2.1.1", "1.4.2.2.1", "1.4.2.3.1", "0.0.0",
		"1.1", "1.2",	//ghs
		"2.1",			//metanoia
		"3.1", "3.2"	//metanoia2
	};
	index_structure idx_layout[MAXOID]={
		{1,1}, {1,2}, {1,2}, {1,2}, {1,2},
		{1,3}, {1,2}, {0,1}, {0,1}, {0,2},
		{0,3}, {0,1}, {0,1}, {0,1}, {0,1},
		{1,2}, {1,1}, {1,3}, {1,3}, {1,2},
		{1,2}, {1,2}, {1,3}, {1,3}, {0,0},
		{1,1}, {1,1},
		{1,1},
		{1,1}, {1,1}
	};

	for(i=0;i<MAXOID;i++) {
		str_find = strncmp(oid_list[i], rawoid, strlen(oid_list[i]));
		if(str_find == 0) break;	// target oid substring found
	}
	if(str_find == 0) {
		if(strlen(rawoid) >= OID_MAX_LEN) return DSL_ERR_NOTOK;
		if(idx_layout[i].ifIndex == 0) return DSL_ERR_NODATA;

		strcpy(oidtarget, oid_list[i]);	//attach ....group.table

		strcpy(oidtmp, rawoid+strlen(oid_list[i])+1);	//param.index1.index2....
		token = strtok(oidtmp, ".");	//param
		if(token != NULL) {
			strcat(oidtarget, ".");	//attach "."
			strcat(oidtarget, token);	//attach param
		}
		else return DSL_ERR_NOTOK;

		token = strtok(NULL, ".");	//index1
		if(token != NULL) *portid = strtoul(token, NULL, 10);	//convert index1 to portid
		else return DSL_ERR_NOTOK;

		if(idx_layout[i].totalIndex > 1) {
			while(1) {
				token = strtok(NULL, ".");	//indexN
				if(token != NULL) {
					strcat(oidtarget, ".");	//attach "."
					strcat(oidtarget, token);	//attach indexN
				}
				else break;
			}
		}

		strcpy(stripoid, oidtarget);
		//printf("RAW OID: %s, STRIP OID: %s, PORTID: %d\n", rawoid, stripoid, *portid);

		return DSL_ERR_OK;
	}

	return DSL_ERR_NODATA;
}


/**
 * Show MIB entry
 * \par Usage:
 * 		cmd <oid>
 * \param oid: Real MIB Object ID represented in x.y.z format
 * \return
 *
 */
int show_mibentry(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 phy_status;
	mt_uint8 tmp[7];
	mt_uint8 oid[OID_MAX_LEN];
	mt_uint8 strvalue[BUF_SIZE];
	mt_uint32 portid;
	mt_uint32 i, size, value;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "show_mib2entry: start");
#endif

	pbuffer[0] = 0x0;
	pbuffer[1] = 0x0;

	ret = sscanf(pbuf, "%s", oid);
	if( ret != 1 ) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <oid>");
		return -DSL_ERR_NOPASS;
	}
	//PRINTL(DBG_LVL_ALWAYS, "oid=%s", oid);

	if(get_roid_port(oid, &portid, oid) != DSL_ERR_OK) return DSL_ERR_NOTOK;

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		ret = get_port_data(portid, oid, pbuffer);
	}

	if(ret == DSL_ERR_OK) {
		sprintf(strvalue, "MIB2 value: ");
		strcat(strvalue, pbuffer+2);
		PRINTL(DBG_LVL_ALWAYS, "%s", strvalue);
		value = strtoul(pbuffer+2, NULL, 16);
		PRINTL(DBG_LVL_XML, "<MIB idx=\"0\" oid=\"%s\" value=\"%d\"></MIB>", oid, value);
		/*tmp[6] = 0;
		size = (pbuffer[0] << 8) | pbuffer[1];
		for(i=0;i<size;i++) {
			strncpy(tmp, pbuffer+2+i*6, 6);
			value = strtoul(tmp, NULL, 16);
			PRINTL(DBG_LVL_ALWAYS, "%06x", value);
		}*/
	}
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting MIB value.");

	return ret;
}


/**
 * Show Logic MIB entry
 * \par Usage:
 * 		cmd <portid> <oid>
 * \param portid: port Id, start from 1
 * \param loid: Logic MIB Object ID represented in x.y.z format
 * \return
 *
 */
int show_logic_mibentry(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 phy_status;
	mt_uint8 tmp[7];
	mt_uint8 loid[OID_MAX_LEN];
	mt_uint8 strvalue[BUF_SIZE];
	mt_uint32 portid;
	mt_uint32 i, size, value;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "show_mib2entry: start");
#endif

	pbuffer[0] = 0x0;
	pbuffer[1] = 0x0;

	ret = sscanf(pbuf, "%i %s", &portid, loid);
	if( ret != 2 ) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <portid> <loid>");
		return -DSL_ERR_NOPASS;
	}
	//PRINTL(DBG_LVL_ALWAYS, "portid=%i,loid=%s", portid, loid);

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		ret = get_port_mib(portid, loid, pbuffer);
	}

	if(ret == DSL_ERR_OK) {
		sprintf(strvalue, "MIB2 value: ");
		strcat(strvalue, pbuffer+2);
		PRINTL(DBG_LVL_ALWAYS, "%s", strvalue);
		value = strtoul(pbuffer+2, NULL, 16);
		PRINTL(DBG_LVL_XML, "<MIB idx=\"0\" loid=\"%s\" value=\"%d\"></MIB>", loid, value);
		/*tmp[6] = 0;
		size = (pbuffer[0] << 8) | pbuffer[1];
		for(i=0;i<size;i++) {
			strncpy(tmp, pbuffer+2+i*6, 6);
			value = strtoul(tmp, NULL, 16);
			PRINTL(DBG_LVL_ALWAYS, "%06x", value);
		}*/
	}
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting MIB value.");

	return ret;
}


/** show pm counter help function */
static mt_ret show_counter(mt_uint32 portid, const char * fmt, char * entry[][2])
{
	mt_ret ret = DSL_ERR_OK;
	mt_uint32 i, co, rs;

	i = 0;
	while (entry[i][0] != NULL){
		ret = get_mib2_int_idx(portid, entry[i][1], &co, 1);
		ret = get_mib2_int_idx(portid, entry[i][1], &rs, 2);

		if (ret != DSL_ERR_OK)break;

		PRINTL(DBG_LVL_ALWAYS, fmt, entry[i][0], co, rs);
		PRINTL(DBG_LVL_XML, "%s=%d %d|", entry[i][0], co, rs);

		i++;
	}

	return ret;
}


#ifdef WITH_ADV
static mt_ret show_hist_ctr(mt_uint32 portid, mt_uint32 interval, char * oid[])
{
	mt_ret ret = DSL_ERR_OK;
	mt_uint32 near, far;
	char faroid[32];
	int i = 0;
	const char * counters[] = {
			"FECS", "ES", "SES", "LOSS", "UAS"
	};

	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "      %-14s%-14s", "Near End", "Far End");

	while (oid[i] != NULL){
		ret = get_mib2_int(portid, oid[i], &near);
		if (ret != DSL_ERR_OK) break;
		sprintf(faroid, "%s.%d", oid[i], INDEX_REMOTE);
		ret = get_mib2_int(portid, faroid , &far);
		if (ret != DSL_ERR_OK) break;

		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "%-5s|%-14d%-14d", counters[i], near, far);

		i++;
	}

	return ret;
}
#endif //WITH_ADV


#ifdef VIA_SAMSUNG
/**
 * Show xml line status
 * \par Usage:
 * 		cmd <portid>
 * \param portid: port Id, start from 1
 * \return
 * 	- 0 : success
 * 	- other : error code
 */
int com_interface_xdsl_show_xmlstatus(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt_b = "%-24s|%-24s %s";
	const char * outfmt_a = "%-24s|%-14s%-14s %s";
	mt_uint8 phy_status, advusstr[32],advdsstr[32];
	mt_uint32 portid, advus, advds, firmver;
	int ret, advidx;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		ret = get_phy_status(portid, &phy_status, NULL);
		if (ret == DSL_ERR_OK){
			switch (phy_status){
			case MT_OPSTATE_IDLE:
				PRINTL(DBG_LVL_XML, "Status=Idle|");
				break;
			case MT_OPSTATE_HANDSHAKE:
				PRINTL(DBG_LVL_XML, "Status=Handshake|");
				break;
			case MT_OPSTATE_TRAINING:
				PRINTL(DBG_LVL_XML, "Status=Traning|");
				break;
			case MT_OPSTATE_SHOWTIME:
				PRINTL(DBG_LVL_XML, "Status=Showtime|");
				break;
			}

			print_mib_uint32(portid, outfmt_b, xdsl2ChStatusActDataRate, "Actual Data Rate", "(US) Kb/s", 1);
			print_mib_uint32(portid, outfmt_b, xdsl2ChStatusActDataRate, "Actual Data Rate", "(DS) Kb/s", 3);
			print_mib_uint32(portid, outfmt_b, xdsl2LineBandStatusSnrMargin, "SNR Margin", "(US) 0.1dB", 10);
			print_mib_uint32(portid, outfmt_b, xdsl2LineBandStatusSnrMargin, "SNR Margin", "(DS) 0.1dB", 11);
			get_mib2_int(portid, xdslFirmwareVer, &firmver);
			PRINTL(DBG_LVL_XML, "Firmware Version=%-x|", firmver);
		}
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting counter value.");

	return ret;
}
#endif //VIA_SAMSUNG


/**
 * Show line status summary
 * \par Usage:
 * 		cmd
 * \return
 * 	- 0 : success
 * 	- other : error code
 */
int com_interface_xdsl_show_summary(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-4d|%9d %7d/%-7d%-12s%6d/%-6d%7d/%-7d%6d/%-6d%d/%-d";
	mt_uint8 phy_status, status[16];
	mt_uint32 portid, firmver, linktime, rateus, rateds, inpus, inpds, delayus, delayds, crcus, crcds, txpkt, rxpkt;
	int ret;
	FILE *streamout;
	char line[8], command[256];

	PRINTL(DBG_LVL_ALWAYS, "%-4s|%-10s%-15s%-12s%-13s%-15s%-13s%s"
	, "Port", "Link time", "Rate US/DS", "Status", "INP US/DS", "Delay US/DS", "CRC US/DS", "TxPkts/RxPkts");

	for ( portid = 0; portid < MAX_PORTS; portid++) {
		dsl_er(check_param(MT_PAR_PORT, 0, portid));

		ret = get_phy_status(portid, &phy_status, NULL);

		if (ret == DSL_ERR_OK && phy_status != MT_OPSTATE_DISABLE && phy_status != MT_OPSTATE_ERROR){
			get_mib2_int(portid, xdsl2ChStatusActDataRate, &rateus);
			get_mib2_int(portid, xdsl2ChStatusActDataRate2, &rateds);


			sprintf(command, ""CFGMGR_PATH" -r V_DISABLE_P%d", portid+1);
			if((streamout = popen(command, "r")) != NULL) {
				if(fgets(line, 8, streamout) != NULL) {
					stripwhite(line);
				}
				pclose(streamout);
			}

			if(*line != '#'){
				strcpy(status, "Disable");
			}else{
				switch (phy_status){
				case MT_OPSTATE_IDLE:
					strcpy(status, "Idle");
					break;
				case MT_OPSTATE_HANDSHAKE:
					strcpy(status, "Handshake");
					break;
				case MT_OPSTATE_TRAINING:
					strcpy(status, "Training");
					break;
				case MT_OPSTATE_SHOWTIME:
					strcpy(status, "Showtime");
					break;
				default:
					strcpy(status, "Unkown");
					break;
				}
			}
			get_mib2_int(portid, xdslFirmwareVer, &firmver);
			if ( MT_OPSTATE_SHOWTIME == phy_status && firmver >= LINKTIME_VER){
				if (IsRT(portid))
					get_mib2_int(portid, xdsl2StatsShowtimeRtTimeElapsed, &linktime);
				else
					get_mib2_int(portid, xdsl2StatsShowtimeOtTimeElapsed, &linktime);
			}else
				linktime = 0;


			get_mib2_int(portid, xdsl2ChStatusActInp, &inpus);
			get_mib2_int(portid, xdsl2ChStatusActInp2, &inpds);

			get_mib2_int(portid, xdsl2ChStatusActDelay, &delayus);
			get_mib2_int(portid, xdsl2ChStatusActDelay2, &delayds);

			get_mib2_int(portid, xdsl2MetanoiaOtCurrCodingViolation0, &crcus);
			get_mib2_int(portid, xdsl2MetanoiaOtCurrCodingViolation01, &crcds);

			get_mib2_int(portid, xdslMetanoiaOtTxPktCntr, &txpkt);
			get_mib2_int(portid, xdslMetanoiaOtRxPktCntr, &rxpkt);


		}else{
			if (ret != DSL_ERR_OK || phy_status == MT_OPSTATE_ERROR)
				strcpy(status, "Error");
			else
				strcpy(status, "Power Off");
			linktime = rateus = rateds = inpus = inpds = delayus = delayds = crcus = crcds = txpkt = rxpkt = 0;
		}

		PRINTL(DBG_LVL_ALWAYS, outfmt, portid+1, linktime, rateus, rateds, status, inpus, inpds, delayus, delayds, crcus, crcds, txpkt, rxpkt);
		PRINTL(DBG_LVL_XML, "Port=%d|Link time=%d|Rate US/DS=%d/%d|Status=%s|INP US/DS=%d/%d|Delay US/DS=%d/%d|CRC US/DS=%d/%d|TxPkts=%d|RxPkts=%d|"
				, portid+1, linktime, rateus, rateds, status, inpus, inpds, delayus, delayds, crcus, crcds, txpkt, rxpkt);
	}

	return DSL_ERR_OK;
}


int com_interface_xdsl_show_status(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt_b = "%-24s|%-24s %s";
	const char * outfmt_a = "%-24s|%-14s%-14s %s";
	const char * profile_str[] = {"8a", "8b", "8c", "8d", "12a", "12b", "17a", "30a"};
	char * bsitem[][3] = {
		{xdsl2ChStatusActDataRate, "Actual Data Rate", "(US) Kb/s"},
		{xdsl2ChStatusActDataRate2, "Actual Data Rate", "(DS) Kb/s"},
		{xdsl2LineBandStatusSnrMargin9, "SNR Margin", "(US) 0.1dB"},
		{xdsl2LineBandStatusSnrMargin10, "SNR Margin", "(DS) 0.1dB"},
		{xdsl2LineBandStatusSigAtten9, "Signal Attenuation", "(US) 0.1dB"},
		{xdsl2LineBandStatusSigAtten10, "Signal Attenuation", "(DS) 0.1dB"},
		{xdsl2LineBandStatusLnAtten9, "Line Attenuation", "(US) 0.1dB"},
		{xdsl2LineBandStatusLnAtten10, "Line Attenuation", "(DS) 0.1dB"},
		{NULL,NULL,NULL}
	};
	char * advitem[][5] = {
			/* desc, us, ds */
			{"Actual delay", xdsl2ChStatusActDelay, xdsl2ChStatusActDelay2, "ms", NULL},
			{"Actual INP", xdsl2ChStatusActInp, xdsl2ChStatusActInp2, "0.1 symbols", NULL},
			{"Previous Data Rate", xdsl2ChStatusPrevDataRate, xdsl2ChStatusPrevDataRate2, "Kbps", NULL},
			{"Attainable Rate", xdsl2LineStatusAttainableRateUs, xdsl2LineStatusAttainableRateDs, "Kbps", NULL},
			{"Transmit Power", xdsl2LineStatusActAtpUs, xdsl2LineStatusActAtpDs, "0.1 dBm", NULL},
			{"Trellis", xdsl2LineStatusTrellisUs, xdsl2LineStatusTrellisDs, "", NULL},
			{"Interleaving Depth", xdsl2ChStatusIntlvDepth, xdsl2ChStatusIntlvDepth2, "", NULL},
			{"Interleaving Block", xdsl2ChStatusIntlvBlock, xdsl2ChStatusIntlvBlock2, "", NULL},
			{"SNR Margin", xdsl2LineBandStatusSnrMargin, "", "(US0,--) 0.1 dB", "SNR Margin 0"},
			{"SNR Margin", xdsl2LineBandStatusSnrMargin1, xdsl2LineBandStatusSnrMargin5, "(US1,DS1) 0.1 dB", "SNR Margin 1"},
			{"SNR Margin", xdsl2LineBandStatusSnrMargin2, xdsl2LineBandStatusSnrMargin6, "(US2,DS2) 0.1 dB", "SNR Margin 2"},
			{"SNR Margin", xdsl2LineBandStatusSnrMargin3, xdsl2LineBandStatusSnrMargin7, "(US3,DS3) 0.1 dB", "SNR Margin 3"},
			{"SNR Margin", xdsl2LineBandStatusSnrMargin4, xdsl2LineBandStatusSnrMargin8, "(US4,DS4) 0.1 dB", "SNR Margin 4"},
			{"Signal Attenuation", xdsl2LineBandStatusSigAtten, "", "(US0,--) 0.1 dB", "Signal Attenuation 0"},
			{"Signal Attenuation", xdsl2LineBandStatusSigAtten1, xdsl2LineBandStatusSigAtten5, "(US1,DS1) 0.1 dB", "Signal Attenuation 1"},
			{"Signal Attenuation", xdsl2LineBandStatusSigAtten2, xdsl2LineBandStatusSigAtten6, "(US2,DS2) 0.1 dB", "Signal Attenuation 2"},
			{"Signal Attenuation", xdsl2LineBandStatusSigAtten3, xdsl2LineBandStatusSigAtten7, "(US3,DS3) 0.1 dB", "Signal Attenuation 3"},
			{"Signal Attenuation", xdsl2LineBandStatusSigAtten4, xdsl2LineBandStatusSigAtten8, "(US4,DS4) 0.1 dB", "Signal Attenuation 4"},
			{"Line Attenuation", xdsl2LineBandStatusLnAtten, "", "(US0,--) 0.1 dB", "Line Attenuation 0"},
			{"Line Attenuation", xdsl2LineBandStatusLnAtten1, xdsl2LineBandStatusLnAtten5, "(US1,DS1) 0.1 dB", "Line Attenuation 1"},
			{"Line Attenuation", xdsl2LineBandStatusLnAtten2, xdsl2LineBandStatusLnAtten6, "(US2,DS2) 0.1 dB", "Line Attenuation 2"},
			{"Line Attenuation", xdsl2LineBandStatusLnAtten3, xdsl2LineBandStatusLnAtten7, "(US3,DS3) 0.1 dB", "Line Attenuation 3"},
			{"Line Attenuation", xdsl2LineBandStatusLnAtten4, xdsl2LineBandStatusLnAtten8, "(US4,DS4) 0.1 dB", "Line Attenuation 4"},
			{"Showtime Full Inits", xdsl2StatsShowtimeLinkRetrain, xdsl2StatsShowtimeLinkRetrain, "", NULL},
			{"Showtime CV", xdsl2MetanoiaOtCurrCodingViolation0, xdsl2MetanoiaOtCurrCodingViolation01, "", NULL},
			{"Showtime FEC", xdsl2MetanoiaOtCurrCorrectedBlocks0, xdsl2MetanoiaOtCurrCorrectedBlocks01, "", NULL},
			{"Showtime Fecs", xdsl2MetanoiaOtCurrFecs0, xdsl2MetanoiaOtCurrFecs01, "", NULL},
			{"Showtime Es", xdsl2MetanoiaOtCurrEs0, xdsl2MetanoiaOtCurrEs01, "", NULL},
			{"Showtime Ses", xdsl2MetanoiaOtCurrSes0, xdsl2MetanoiaOtCurrSes01, "", NULL},
			{"Showtime Loss", xdsl2MetanoiaOtCurrLoss0, xdsl2MetanoiaOtCurrLoss01, "", NULL},
			{"Showtime Uas", xdsl2MetanoiaOtCurrUas0, xdsl2MetanoiaOtCurrUas01, "", NULL},
			{"15M Elapsed time", xdsl2PMLCurr15MTimeElapsed, xdsl2PMLCurr15MTimeElapsed1, "secs", NULL},
			{"15M FECS", xdsl2PMLCurr15MFecs, xdsl2PMLCurr15MFecs1, "", NULL},
			{"15M ES", xdsl2PMLCurr15MEs, xdsl2PMLCurr15MEs1, "", NULL},
			{"15M SES", xdsl2PMLCurr15MSes, xdsl2PMLCurr15MSes1, "", NULL},
			{"15M LOSS", xdsl2PMLCurr15MLoss, xdsl2PMLCurr15MLoss1, "", NULL},
			{"15M UAS", xdsl2PMLCurr15MUas, xdsl2PMLCurr15MUas1, "", NULL},
			{"15M Full Inits", xdsl2PMLInitCurr15MFullInits, xdsl2PMLInitCurr15MFullInits, "", NULL},
			{"15M CV", xdsl2PMChCurr15MCodingViolations, xdsl2PMChCurr15MCodingViolations2, "", NULL},
			{"15M FEC", xdsl2PMChCurr15MCorrectedBlocks, xdsl2PMChCurr15MCorrectedBlocks2, "", NULL},
			{"1Day Elapsed time", xdsl2PMLCurr1DayTimeElapsed, xdsl2PMLCurr1DayTimeElapsed1, "secs", NULL},
			{"1Day FECS", xdsl2PMLCurr1DayFecs, xdsl2PMLCurr1DayFecs1, "", NULL},
			{"1Day ES", xdsl2PMLCurr1DayEs, xdsl2PMLCurr1DayEs1, "", NULL},
			{"1Day SES", xdsl2PMLCurr1DaySes, xdsl2PMLCurr1DaySes1, "", NULL},
			{"1Day LOSS", xdsl2PMLCurr1DayLoss, xdsl2PMLCurr1DayLoss1, "", NULL},
			{"1Day UAS", xdsl2PMLCurr1DayUas, xdsl2PMLCurr1DayUas1, "", NULL},
			{"1Day Full Inits", xdsl2PMLInitCurr1DayFullInits, xdsl2PMLInitCurr1DayFullInits, "", NULL},
			{"1Day CV", xdsl2PMChCurr1DayCodingViolations, xdsl2PMChCurr1DayCodingViolations2, "", NULL},
			{"1Day FEC", xdsl2PMChCurr1DayCorrectedBlocks, xdsl2PMChCurr1DayCorrectedBlocks2, "", NULL},
			{"Total FECS", xdsl2MetanoiaOtFecs0, xdsl2MetanoiaOtFecs01, "", NULL},
			{"Total ES", xdsl2MetanoiaOtEs0, xdsl2MetanoiaOtEs01, "", NULL},
			{"Total SES", xdsl2MetanoiaOtSes0, xdsl2MetanoiaOtSes01, "", NULL},
			{"Total LOSS", xdsl2MetanoiaOtLoss0, xdsl2MetanoiaOtLoss01, "", NULL},
			{"Total UAS", xdsl2MetanoiaOtUas0, xdsl2MetanoiaOtUas01, "", NULL},
			{"Total CV", xdsl2MetanoiaOtCodingViolations0, xdsl2MetanoiaOtCodingViolations01, "", NULL},
			{"Total FEC", xdsl2MetanoiaOtCorrectedBlocks0, xdsl2MetanoiaOtCorrectedBlocks01, "", NULL},
			{NULL, NULL, NULL, NULL, NULL}};
	mt_uint8 phy_status, usstr[32],dsstr[32];
	int ret, advidx, bsidx, portid, firmver, profile, usvalue, dsvalue;
	FILE *streamout;
	char line[8], command[256];

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	ret = get_phy_status(portid, &phy_status, NULL);
	if(ret != DSL_ERR_OK || phy_status == MT_OPSTATE_DISABLE || phy_status == MT_OPSTATE_ERROR){
		if (ret != DSL_ERR_OK || phy_status == MT_OPSTATE_ERROR){
			PRINTL(DBG_LVL_ALWAYS, "Status of Line %u: Error", portid+1);
			PRINTL(DBG_LVL_XML, "Status=Error|");
		}else{
			PRINTL(DBG_LVL_ALWAYS, "Status of Line %u: Power Off", portid+1);
			PRINTL(DBG_LVL_XML, "Status=Power Off|");
		}
		ret = DSL_ERR_NOTOK;
	}
	else{
			sprintf(command, ""CFGMGR_PATH" -r V_DISABLE_P%d", portid+1);
			if((streamout = popen(command, "r")) != NULL) {
				if(fgets(line, 8, streamout) != NULL) {
					stripwhite(line);
				}
				pclose(streamout);
			}

			if(*line != '#'){
				PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Disable", portid+1);
				PRINTL(DBG_LVL_XML, "Status=Disable|");

			}else{
				switch (phy_status){
				case MT_OPSTATE_IDLE:
					PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Idle", portid+1);
					PRINTL(DBG_LVL_XML, "Status=Idle|");
					break;
				case MT_OPSTATE_HANDSHAKE:
					PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Handshake", portid+1);
					PRINTL(DBG_LVL_XML, "Status=Handshake|");
					break;
				case MT_OPSTATE_TRAINING:
					PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Training", portid+1);
					PRINTL(DBG_LVL_XML, "Status=Traning|");
					break;
				case MT_OPSTATE_SHOWTIME:
					PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Showtime", portid+1);
					PRINTL(DBG_LVL_XML, "Status=Showtime|");
					break;
				default:
					PRINTL(DBG_LVL_ALWAYS, "Status (Basic) of Line %u: Unkown", portid+1);
					PRINTL(DBG_LVL_XML, "Status=Unkown|");
					break;
				}
			}

			if (MT_OPSTATE_SHOWTIME == phy_status){
				get_mib2_int(portid, xdsl2LineStatusActProfile, &profile);
				PRINTL(DBG_LVL_ALWAYS, "Profile: %s", profile_str[profile]);
				PRINTL(DBG_LVL_XML,"Profile=%s|", profile_str[profile]);
			}else{
				PRINTL(DBG_LVL_ALWAYS, "Profile: NA");
				PRINTL(DBG_LVL_XML,"Profile=NA|");
			}


			PRINTL(DBG_LVL_ALWAYS, "%-24s|%-24s %s", "", "Value", "Description");
			for (bsidx =0 ; bsitem[bsidx][0] != NULL; bsidx++){
				ret = get_mib2_int(portid, bsitem[bsidx][0], &usvalue);
				if (MT_OK != ret)
						strcpy(usstr, "FAILED");
				else {
					if (IsValidMibValue(usvalue))
						sprintf(usstr, "%d", usvalue);
					else
						strcpy(usstr, "NA");
				}
				PRINTL(DBG_LVL_ALWAYS, outfmt_b, bsitem[bsidx][1], usstr, bsitem[bsidx][2]);

				ret = get_mib2_int(portid, bsitem[++bsidx][0], &dsvalue);
				if (MT_OK != ret)
						strcpy(dsstr, "FAILED");
				else {
					if (IsValidMibValue(dsvalue))
						sprintf(dsstr, "%d", dsvalue);
					else
						strcpy(dsstr, "NA");
				}
				PRINTL(DBG_LVL_ALWAYS, outfmt_b, bsitem[bsidx][1], dsstr, bsitem[bsidx][2]);

				PRINTL(DBG_LVL_XML, "%s=%s %s|", bsitem[bsidx][1], usstr, dsstr);
			}
			print_mib_uint32(portid, outfmt_b, xdsl2LineStatusElectricalLength, "Electrical Length", "0.1 dB", 1);
			get_mib2_int(portid, xdslFirmwareVer, &firmver);
			if ( MT_OPSTATE_SHOWTIME == phy_status && firmver >= LINKTIME_VER){
				if (IsRT(portid))
					print_mib_uint32(portid, outfmt_b, xdsl2StatsShowtimeRtTimeElapsed, "Link Time", "second(s)", 1);
				else
					print_mib_uint32(portid, outfmt_b, xdsl2StatsShowtimeOtTimeElapsed, "Link Time", "second(s)", 1);
			}else{
				PRINTL(DBG_LVL_ALWAYS, outfmt_b, "Link Time", "0", "second(s)");
				PRINTL(DBG_LVL_XML, "Link Time=0|");
			}

			PRINTL(DBG_LVL_ALWAYS, "%-24s|%06x", "Firmware Version", firmver);
			PRINTL(DBG_LVL_XML, "Firmware Version=%-x|", firmver);
			PRINTL(DBG_LVL_ALWAYS, "\nStatus (Advance) of Line %u:", portid+1);
			PRINTL(DBG_LVL_ALWAYS, "%24s %-14s%-14s", "", "Upstream", "Downstream");

			for (advidx = 0; advitem[advidx][0] != NULL; advidx++){
				if (strlen(advitem[advidx][1]) == 0)
					strcpy(usstr, "--");
				else{
					ret = get_mib2_int(portid, advitem[advidx][1], &usvalue);

					if (MT_OK != ret)
						strcpy(usstr, "FAILED");
					else {
						if (IsValidMibValue(usvalue))
							sprintf(usstr, "%d", usvalue);
						else
							strcpy(usstr, "NA");
					}
				}

				if (strlen(advitem[advidx][2]) == 0)
					strcpy(dsstr, "--");
				else{
					ret = get_mib2_int(portid, advitem[advidx][2], &dsvalue);

					if (MT_OK != ret)
						strcpy(dsstr, "FAILED");
					else {
						if (IsValidMibValue(dsvalue))
							sprintf(dsstr, "%d", dsvalue);
						else
							strcpy(dsstr, "NA");
					}
				}

				PRINTL(DBG_LVL_ALWAYS, outfmt_a, advitem[advidx][0], usstr, dsstr, advitem[advidx][3]);
				if (advitem[advidx][4] == NULL)
					PRINTL(DBG_LVL_XML, "%s=%s %s|", advitem[advidx][0], usstr, dsstr);
				else
					PRINTL(DBG_LVL_XML, "%s=%s %s|", advitem[advidx][4], usstr, dsstr);
			}

			ret = DSL_ERR_OK;

#if 0//yalee980617:for gohead web reach roid
			printf("Line Rate, (US) Kb/s %s.1\n",xdslMetanoiaOtCurrLineRate);
			printf("Line Rate, (DS) Kb/s %s.2\n",xdslMetanoiaOtCurrLineRate);
			printf("Actual Data Rate, (US) Kb/s %s.1\n",xdsl2ChStatusActDataRate);
			printf("Actual Data Rate, (DS) Kb/s %s.3\n",xdsl2ChStatusActDataRate);
			printf("SNR Margin (US) 0.1dB %s.10\n",xdsl2LineBandStatusSnrMargin);
			printf("SNR Margin (US) 0.1dB %s.11\n",xdsl2LineBandStatusSnrMargin);
			PRINTL(DBG_LVL_ALWAYS, "%24s %-14s%-14s", "", "Upstream", "Downstream");
			for (advidx = 0; advitem[advidx][0] != NULL; advidx++){
				PRINTL(DBG_LVL_ALWAYS, outfmt_a, advitem[advidx][0], advitem[advidx][1],
									advitem[advidx][2], advitem[advidx][3]);
			}
#endif

			/*get_port_mib(portid, xdsl2ChStatusActDataRate, pbuffer);
			get_port_mib(portid, xdsl2ChStatusActDataRate2, pbuffer);
			get_port_mib(portid, xdsl2LineBandStatusSnrMargin9, pbuffer);
			get_port_mib(portid, xdsl2ChStatusActDataRate, pbuffer);
			get_port_mib(portid, xdslFirmwareVer, pbuffer);*/
			/*get_port_mib(portid, "2.1.1", pbuffer);
			get_port_mib(portid, "2.1.2", pbuffer);
			get_port_mib(portid, "2.1.3", pbuffer);
			get_port_mib(portid, "2.1.4", pbuffer);
			get_port_mib(portid, "2.1.5", pbuffer);*/

			/*for (advidx = 0; advitem[advidx][0] != NULL; advidx++){
				get_port_mib(portid, advitem[advidx][1], pbuffer);
				get_port_mib(portid, advitem[advidx][2], pbuffer);
			}*/

			/*get_port_mib(portid, "1.1.129", pbuffer);
			printf("%02x %02x %02x %02x\n", pbuffer[0], pbuffer[1], pbuffer[6], pbuffer[7]);
			dispmem8(pbuffer, 8);
			printf("Done!!\n");*/

			/*get_port_mem_remote2(portid, 0x6c4e, 1, pbuffer);
			dispmem8(pbuffer, 8);
			printf("\n");*/
	}

	return ret;
}


int com_interface_xdsl_show_pm_line_curr (unsigned char * pbuf, cmdt * pcmd)
{
	const char * titlefmt = "%-20s %-24s%-24s";
	const char * etimefmt = "%-20s|%-24s%-24s";
	const char * countfmt = "%-20s|%-24d%-24d";

    char * entry15m[][2] = {
    		{"15M FECS", xdsl2PMLCurr15MFecs}, // FECS
    		{"15M ES", xdsl2PMLCurr15MEs}, // ES
    		{"15M SES", xdsl2PMLCurr15MSes}, // SES
    		{"15M LOSS", xdsl2PMLCurr15MLoss}, // LOSS
    		{"15M UAS", xdsl2PMLCurr15MUas}, // UAS
			{NULL, NULL}
	};

    char * entry1day[][2] = {
    		{"1Day FECS", xdsl2PMLCurr1DayFecs}, // FECS
    		{"1Day ES", xdsl2PMLCurr1DayEs}, // ES
    		{"1Day SES", xdsl2PMLCurr1DaySes}, // SES
    		{"1Day LOSS", xdsl2PMLCurr1DayLoss}, // LOSS
    		{"1Day UAS", xdsl2PMLCurr1DayUas}, // UAS
			{NULL, NULL}
	};

    mt_uint8 phy_status, etimestr_co[32], etimestr_rs[32];
	mt_uint32 portid, etime_co,etime_rs;
	int ret;
	PRINTL(DBG_LVL_IF, "Portdisabled\n");
	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "Current line counters of Line %u:", portid+1);
		PRINTL(DBG_LVL_ALWAYS, titlefmt, "", "Central Office", "Remote site");
		ret = get_mib2_int_idx(portid, xdsl2PMLCurr15MTimeElapsed, &etime_co, 1);
		ret = get_mib2_int_idx(portid, xdsl2PMLCurr15MTimeElapsed, &etime_rs, 2);
		sprintf(etimestr_co, "%d mins %d secs", etime_co / 60, etime_co % 60);
		sprintf(etimestr_rs, "%d mins %d secs", etime_rs / 60, etime_rs % 60);
		PRINTL(DBG_LVL_ALWAYS, etimefmt, "15M Elapsed time", etimestr_co, etimestr_rs);
		PRINTL(DBG_LVL_XML, "15M Elapsed time=%d %d|", etime_co, etime_rs);
		ret = show_counter(portid, countfmt, entry15m);
		PRINTL(DBG_LVL_XML, "|");
		ret = get_mib2_int_idx(portid, xdsl2PMLCurr1DayTimeElapsed, &etime_co, 1);
		ret = get_mib2_int_idx(portid, xdsl2PMLCurr1DayTimeElapsed, &etime_rs, 2);
		sprintf(etimestr_co, "%d hours %d mins %d secs", etime_co / 3600, (etime_co % 3600) / 60, etime_co % 60);
		sprintf(etimestr_rs, "%d hours %d mins %d secs", etime_rs / 3600, (etime_rs % 3600) / 60, etime_rs % 60);
		PRINTL(DBG_LVL_ALWAYS, etimefmt, "1Day Elapsed time", etimestr_co, etimestr_rs );
		PRINTL(DBG_LVL_XML, "1Day Elapsed time=%d %d|", etime_co, etime_rs);
		ret = show_counter(portid, countfmt, entry1day);
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting curent line counter.");

	return ret;
}


#ifdef WITH_ADV
int com_interface_xdsl_show_pm_15m_hist (unsigned char * pbuf, cmdt * pcmd)
{
    char * loid[] = {
			"0.4.1.3.4", // FECS
			"0.4.1.3.5", // ES
			"0.4.1.3.6", // SES
			"0.4.1.3.7", // LOSS
			"0.4.1.3.8", // UAS
			"0.4.2.2.4", // CV
			"0.4.2.2.5", // FEC
			NULL,
	};

	mt_uint8 phy_status;
	mt_uint32 portid, interval, elapsedtime;
	int ret;
#if 0
	ret = sscanf(pbuf, "%u %u", &portid, &interval);
	if (ret != 2){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> <interval>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		ret = get_mib2_int(portid, "0.4.1.3.3", &elapsedtime);
		if (ret == DSL_ERR_OK){
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "History 15-min counters of Line %u:", portid+1);
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Elapsed time : %d mins %d secs", elapsedtime / 60, elapsedtime % 60);

			ret = show_hist_ctr(portid, interval, loid);
		}
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting counter value.");
#endif
	return ret;
}


int com_interface_xdsl_show_pm_1d_hist (unsigned char * pbuf, cmdt * pcmd)
{
	// 4.1.4.x
    char * oid[] = {
			"0.4.1.4.4", // FECS
			"0.4.1.4.5", // ES
			"0.4.1.4.6", // SES
			"0.4.1.4.7", // LOSS
			"0.4.1.4.8", // UAS
			"0.4.2.3.4", // CV
			"0.4.2.3.5", // FEC
			NULL,
	};

	mt_uint8 phy_status;
	mt_uint32 portid, interval, elapsedtime;
	int ret;
#if 0
	ret = sscanf(pbuf, "%u %u", &portid, &interval);
	if (ret != 2){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> <interval>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		ret = get_mib2_int(portid, "0.4.1.4.3", &elapsedtime);
		if (ret == DSL_ERR_OK){
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "History 1-Day counters of Line %u:", portid+1);
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Elapsed time : %d hours %d mins %d secs", elapsedtime / 3600, (elapsedtime % 3600) / 60, elapsedtime % 60);

			ret = show_hist_ctr(portid, interval, oid);
		}
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting counter value.");
#endif
	return ret;
}


int com_interface_xdsl_show_delt (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_interface_xdsl_show_linetest (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}
#endif //WITH_ADV


int com_interface_xdsl_show_pm_ch_curr (unsigned char * pbuf, cmdt * pcmd)
{
	const char * etimefmt = "%-20s|%-24s";
	const char * countfmt = "%-20s|%-24d";

	mt_uint8 phy_status, elapsedtimestr[32];
	mt_uint32 portid, elapsedtime, idx, unit, usc, dsc, counter;
	int ret;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		usc = dsc = 0;
		PRINTL(DBG_LVL_ALWAYS, "Current channel counters of Line %u:", portid+1);
		for (idx = 1; idx <= 4; ){
			ret = get_mib2_int_idx(portid, xdsl2PMChCurrUnit, &unit, idx);
			if (ret == DSL_ERR_OK){
				printf("\n");

				if (unit == 1)
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Upstream:");
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Downstream:");

				ret = get_mib2_int_idx(portid, xdsl2PMChCurr15MTimeElapsed, &elapsedtime, idx);
				sprintf(elapsedtimestr, "%d mins %d secs", elapsedtime / 60, elapsedtime % 60);
				PRINTL(DBG_LVL_ALWAYS, etimefmt, "15M Elapsed time", elapsedtimestr);
				PRINTL(DBG_LVL_XML, "15M Elapsed time=%d|", elapsedtime);
				ret = get_mib2_int_idx(portid, xdsl2PMChCurr15MCodingViolations, &counter, idx);
				PRINTL(DBG_LVL_ALWAYS, countfmt, "15M CV", counter);
				PRINTL(DBG_LVL_XML, "15M CV=%d|", counter);
				ret = get_mib2_int_idx(portid, xdsl2PMChCurr15MCorrectedBlocks, &counter, idx);
				PRINTL(DBG_LVL_ALWAYS, countfmt, "15M FEC", counter);
				PRINTL(DBG_LVL_XML, "15M FEC=%d|", counter);
				ret = get_mib2_int_idx(portid, xdsl2PMChCurr1DayTimeElapsed, &elapsedtime, idx);
				sprintf(elapsedtimestr, "%d hours %d mins %d secs", elapsedtime / 3600, (elapsedtime % 3600) / 60, elapsedtime % 60);
				PRINTL(DBG_LVL_ALWAYS, etimefmt, "1Day Elapsed time", elapsedtimestr);
				PRINTL(DBG_LVL_XML, "1Day Elapsed time=%d|", elapsedtime);
				ret = get_mib2_int_idx(portid, xdsl2PMChCurr1DayCodingViolations, &counter, idx);
				PRINTL(DBG_LVL_ALWAYS, countfmt, "1Day CV", counter);
				PRINTL(DBG_LVL_XML, "1Day CV=%d|", counter);
				ret = get_mib2_int_idx(portid, xdsl2PMChCurr1DayCorrectedBlocks, &counter, idx);
				PRINTL(DBG_LVL_ALWAYS, countfmt, "1Day FEC", counter);
				PRINTL(DBG_LVL_XML, "1Day FEC=%d", counter);
				if (idx != 3)
					PRINTL(DBG_LVL_XML, ";");
				idx += 2;
			}
		}
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting current channel counter.");

	return ret;
}


int com_interface_xdsl_show_invent (unsigned char * pbuf, cmdt * pcmd)
{
	mt_uint32 portid;
	int ret, idx;
	mt_uint8 phy_status, loid[OID_MAX_LEN], g994_vid[16], sys_vid[9], vn[36], sn[66];
	mt_uint32 selftest, unit;
	mt_uint32 address, mibsize, mibtype;
	const char *selftestres[] = {"passed", "failed"};
	const char *xtu[] = {"Central Office", "Remote Site"};


	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "Line Inventory of Line %u:", portid+1);

		for (idx = 1; idx <= 2; idx++ ){
			sprintf(loid, "%s.%d", xdsl2LInvUnit, idx);
			ret = get_mib2_int(portid, loid, &unit);

			sprintf(loid, "%s.%d", xdsl2LInvG994VendorId, idx);
			ret = get_mib2_uint8(portid, loid, g994_vid, 9, NULL);

			sprintf(loid, "%s.%d", xdsl2LInvSystemVendorId, idx);
			ret = get_mib2_uint8(portid, loid, sys_vid, 9, NULL);

			memset(vn, 0x00, 36);
			sprintf(loid, "%s.%d", xdsl2LInvVersionNumber, idx);
			if ((ret = get_port_mib(portid, loid, pbuffer)) == DSL_ERR_OK)
				memcpy(vn, pbuffer + 2, 36);

			memset(sn, 0x00, 66);
			sprintf(loid, "%s.%d", xdsl2LInvSerialNumber, idx);
			if ((ret = get_port_mib(portid, loid, pbuffer)) == DSL_ERR_OK)
				memcpy(sn, pbuffer + 2, 66);

			sprintf(loid, "%s.%d", xdsl2LInvSelfTestResult, idx);
			ret = get_mib2_int(portid, loid, &selftest);

			sprintf(loid, "%s.%d", xdsl2LInvTransmissionCapabilities, idx);
			ret = get_port_mib(portid, loid, pbuffer);

			if (ret == DSL_ERR_OK){
				PRINTL(DBG_LVL_ALWAYS, "Unit	         |%-40s", xtu[unit-1]);
				PRINTL(DBG_LVL_ALWAYS, "G994vendorid     |Country code:%02x%02x,Vendor=%-40s", g994_vid[0], g994_vid[1], g994_vid + 2);
				PRINTL(DBG_LVL_ALWAYS, "Systemvendorid   |%-40s", sys_vid);
				PRINTL(DBG_LVL_ALWAYS, "version no.      |%-40s", vn);
				PRINTL(DBG_LVL_ALWAYS, "serial no.       |%-40s", sn);
				PRINTL(DBG_LVL_ALWAYS, "self-test result |%-40s", selftestres[selftest]);
				PRINTL(DBG_LVL_ALWAYS, "tx capabilities  |%-40s\n", pbuffer + 2);

				PRINTL(DBG_LVL_XML, "Unit=%s|", xtu[unit-1]);
				PRINTL(DBG_LVL_XML, "G994vendorid=Country code:%02x%02x,Vendor=%s|", g994_vid[0], g994_vid[1], g994_vid + 2);
				PRINTL(DBG_LVL_XML, "Systemvendorid=%s|", sys_vid);
				PRINTL(DBG_LVL_XML, "version no.=%s|", vn);
				PRINTL(DBG_LVL_XML, "serial no.=%s|", sn);
				PRINTL(DBG_LVL_XML, "self-test result=%s|", selftestres[selftest]);
				PRINTL(DBG_LVL_XML, "tx capabilities=%s|", pbuffer + 2);
			}
		}
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting inventory value.");

	return DSL_ERR_OK;
}


int com_interface_xdsl_show_threshold (unsigned char * pbuf, cmdt * pcmd)
{
	mt_uint32 portid;
	int ret;
	mt_uint8 phy_status;
	mt_uint32 fecs_o, es_o, ses_o, loss_o, uas_o, cv_o, fec_o;
	mt_uint32 fecs_r, es_r, ses_r, loss_r, uas_r, cv_r, fec_r;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinFecs, &fecs_o);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinEs, &es_o);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinSes, &ses_o);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinLoss, &loss_o);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXtucThresh15MinUas, &uas_o);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinFecs, &fecs_r);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinEs, &es_r);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinSes, &ses_r);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinLoss, &loss_r);
		get_mib2_int(portid, xdsl2LineAlarmConfProfileXturThresh15MinUas, &uas_r);
		get_mib2_int(portid, xdsl2ChAlarmConfProfileXtucThresh15MinCodingViolations, &cv_o);
		get_mib2_int(portid, xdsl2ChAlarmConfProfileXtucThresh15MinCorrected, &fec_o);
		get_mib2_int(portid, xdsl2ChAlarmConfProfileXturThresh15MinCodingViolations, &cv_r);
		get_mib2_int(portid, xdsl2ChAlarmConfProfileXturThresh15MinCorrected, &fec_r);

		PRINTL(DBG_LVL_ALWAYS, "Line %u Threshold (15min error seconds):", portid+1);
		PRINTL(DBG_LVL_ALWAYS, "     %-14s %-14s", "Central Office", "Remote site");
		PRINTL(DBG_LVL_ALWAYS, "FECS|%-14d %-14d", fecs_o, fecs_r);
		PRINTL(DBG_LVL_ALWAYS, "ES  |%-14d %-14d", es_o, es_r);
		PRINTL(DBG_LVL_ALWAYS, "SES |%-14d %-14d", ses_o, ses_r);
		PRINTL(DBG_LVL_ALWAYS, "LOSS|%-14d %-14d", loss_o, loss_r);
		PRINTL(DBG_LVL_ALWAYS, "UAS |%-14d %-14d", uas_o, uas_r);
		PRINTL(DBG_LVL_ALWAYS, "CV  |%-14d %-14d", cv_o, cv_r);
		PRINTL(DBG_LVL_ALWAYS, "FEC |%-14d %-14d", fec_o, fec_r);

		PRINTL(DBG_LVL_XML, "15MinFECS=%d %d|", fecs_o, fecs_r);
		PRINTL(DBG_LVL_XML, "15MinES=%d %d|", es_o, es_r);
		PRINTL(DBG_LVL_XML, "15MinSES=%d %d|", ses_o, ses_r);
		PRINTL(DBG_LVL_XML, "15MinLOSS=%d %d|", loss_o, loss_r);
		PRINTL(DBG_LVL_XML, "15MinUAS=%d %d|", uas_o, uas_r);
		PRINTL(DBG_LVL_XML, "15MinCV=%d %d|", cv_o, cv_r);
		PRINTL(DBG_LVL_XML, "15MinFEC=%d %d", fec_o, fec_r);
		// Can't find 1-day mib, but askey have

		ret = DSL_ERR_OK;
	}
	return ret;
}

int com_interface_xdsl_show_table(unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface_xdsl_show_table_cmd;
	set_prompt("(interface xdsl show table)");
	return DSL_ERR_OK;
}


//=======================interface_xdsl_show cmd tables=============
cmdt interface_xdsl_show_cmd[] = {
	{"summary", "Show line status summary", NULL, NULL, com_interface_xdsl_show_summary},
	{"status", "Show line status", NULL, NULL, com_interface_xdsl_show_status},
#ifdef VIA_SAMSUNG
	{"xmlstatus", "Show line status", NULL, NULL, com_interface_xdsl_show_xmlstatus},
#endif
	{"pm-line-curr","Show current counters of xdsl lines", NULL, NULL, com_interface_xdsl_show_pm_line_curr},
	{"pm-ch-curr", "Show current counters of xdsl channels", NULL, NULL, com_interface_xdsl_show_pm_ch_curr},
#ifdef WITH_ADV
	{"pm-15m-hist","Show history 15-min counters of xdsl lines", NULL, NULL,com_interface_xdsl_show_pm_15m_hist},
	{"pm-1d-hist", "Show history 1day counters of xdsl lines", NULL, NULL, com_interface_xdsl_show_pm_1d_hist},
	{"delt", "DELT loop diagnostic function", NULL, NULL, com_interface_xdsl_show_delt},
	{"linetest", "Line diagnostic function", NULL, NULL, com_interface_xdsl_show_linetest},
#endif
	{"invent", "Show inventory of xdsl lines", NULL, NULL, com_interface_xdsl_show_invent},
	{"threshold", "Show threshold of xdsl lines", NULL, NULL, com_interface_xdsl_show_threshold},
	{"table", "Show mib tables of xdsl line", interface_xdsl_show_table_cmd, NULL, com_interface_xdsl_show_table},
	{"oid","Show xDSL Logic MIB entry, <portid> <loid>", NULL, NULL, show_logic_mibentry},
	{"roid","Show xDSL Real MIB entry, <oid>", NULL, NULL, show_mibentry},
#ifdef WITH_STATISTIC
	{"statistic", "Show complete Port statistic information, <testid> <portid>", NULL, NULL, show_portstat},
	{"oidstatistic", "Show Port statistic information by LOID, <portid> <loid>", NULL, NULL, show_portstat_by_oid},
#endif	//WITH_STATISTIC
	{"pvc","Show PVC rules", NULL, NULL, show_pvc_rule},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", interface_xdsl_cmd, NULL, com_interface_xdsl},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


static void show_psdlevel(mt_uint8 * data, mt_uint8 pairs, mt_uint16 * range){
	float slop, psdlvl1, psdlvl2, psdlvl;
	int pair, scidx1, scidx2, tone, tone_stop, count, ridx;
	mt_uint8 end_hit, skip_band, tmp[7];

	end_hit = skip_band = FALSE;
	scidx1 =  scidx2 = count = ridx = 0;
	psdlvl1 = psdlvl2 = -140.0;
	for (pair = 0; pair < pairs; pair++){
		strncpy(tmp, data + pair*2*6, 6);
		scidx2 = strtoul(tmp, NULL, 16);
		strncpy(tmp, data + (pair*2+1)*6, 6);
		psdlvl2 = strtoul(tmp, NULL, 16) * 0.1 - 140;

		if (scidx2 != 0 || psdlvl2 != -140.0){
			PRINTL(DBG_LVL_XML, "%d=%.1f|", scidx2, psdlvl2);
		}

		if (psdlvl1 == -140.0 || psdlvl2 == -140.0) {
			scidx1 = scidx2;
			psdlvl1 = psdlvl2;
			continue;
		}

		if (end_hit == TRUE) {
			skip_band = TRUE;
			end_hit = FALSE;
		}
		else {	//end_hit == FALSE
			if (scidx2 == range[ridx]) {
				end_hit = TRUE;
				ridx++;
			}
		}

		if (silent != 1){
			if (skip_band == TRUE) {
				for (tone = scidx1+1; tone < scidx2; tone++){
					printf("%4d|-100.0 ", tone);

					if (count++ % 8 == 7)
						PRINTL(DBG_LVL_ALWAYS, "");
				}

				skip_band = FALSE;
			}
			else {
				slop = (psdlvl2 - psdlvl1)/(scidx2 - scidx1);

				//PRINTL(DBG_LVL_ALWAYS, "scidx1 %d, psdlvl1 %f, scidx2 %d, psdlvl2 %f, slop %f", scidx1, psdlvl1, scidx2, psdlvl2, slop);

				if (end_hit == TRUE)
					tone_stop = scidx2;
				else
					tone_stop = scidx2 - 1;

				for (tone = scidx1; tone <= tone_stop; tone++){
					psdlvl = psdlvl1 + slop * (tone - scidx1);
					printf("%4d|%-6.1f ", tone, psdlvl);
					//PRINTL(DBG_LVL_XML, "%d=%.1f|", tone, psdlvl);

					if (count++ % 8 == 7)
						PRINTL(DBG_LVL_ALWAYS, "");
				}
			}
		}

		scidx1 = scidx2;
		psdlvl1 = psdlvl2;
	}

	PRINTL(DBG_LVL_ALWAYS, "");
}


int com_interface_xdsl_show_table_line(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-24s|%-24s %s";
	int ret, numfields;
	mt_uint8 phy_status, tmp[7], cmdpsd[16];
	mt_uint16 range[5];
	mt_uint32 portid;

	numfields = sscanf(pbuf, "%u %s", &portid, cmdpsd);
	if (numfields < 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> [psd]");
		return -DSL_ERR_NOPASS;
	}
	if (numfields > 1){
		if (strcmp(cmdpsd, "psd") != 0){
			PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> [psd]");
			return -DSL_ERR_NOPASS;
		}
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		if (numfields == 1){
			PRINTL(DBG_LVL_ALWAYS, "xdsl2LineTable of Line %u:", portid+1);
			PRINTL(DBG_LVL_ALWAYS, "%-24s %-24s Description", "", "Value");
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusPwrMngState, "4-No power management", "", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusAttainableRateDs, "Attainable Rate Ds", "(kbps)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusAttainableRateUs, "Attainable Rate Us", "(kbps)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActAtpDs, "ATP Ds", "(0.1dB)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActAtpUs, "ATP Us", "(0.1dB)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActProfile, "Profile", "8a(0),8b(1),8c(2),8d(3),12a(4),12b(5),17a(6),30a(7)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActSnrModeDs, "Virtual noise Ds", "Disabled(1),Enabled(2)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActSnrModeUs, "Virtual noise Us", "Disabled(1),Enabled(2)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusElectricalLength, "Electrical Length", "(0.1dB)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusTrellisDs, "Trellis Ds", "Not used(0),Used(1)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusTrellisUs, "Trellis Us", "Not used(0),Used(1)", 1);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineStatusActualCe, "Cyclic extension", "N/32 samples", 1);
		}else{
			memset(pbuffer, '\0', sizeof(pbuffer));

			memset(range, 0, sizeof(range));
			ret = get_port_mib(portid, xdslMetanoiaOtMedley0Us, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[0] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaOtMedley1Us, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[1] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaOtMedley2Us, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[2] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaOtMedley3Us, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[3] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaOtMedley4Us, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[4] = strtoul(tmp, NULL, 16);
			//PRINTL(DBG_LVL_ALWAYS, "range: %x %x %x %x %x %x %x %x %x %x\n", range[0], range[1], range[2], range[3], range[4], range[5], range[6], range[7], range[8], range[9]);
			if ((ret = get_port_mib(portid, xdsl2LineStatusMrefPsdUs, pbuffer)) == DSL_ERR_OK){
				PRINTL(DBG_LVL_ALWAYS, "Psd level (dBm/Hz) Us : Tone No.|PSD level");
				PRINTL(DBG_LVL_XML, "Psd Level Us:");
				if (strncmp(pbuffer + 2, "c69a53", 6) != 0)
					show_psdlevel(pbuffer + 2, 32, range);
				else{
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "NA");
					PRINTL(DBG_LVL_XML, "|");
				}
			}

			memset(range, 0, sizeof(range));
			ret = get_port_mib(portid, xdslMetanoiaRtMedley0Ds, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[0] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaRtMedley1Ds, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[1] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaRtMedley2Ds, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[2] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaRtMedley3Ds, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[3] = strtoul(tmp, NULL, 16);
			ret = get_port_mib(portid, xdslMetanoiaRtMedley4Ds, pbuffer);
			strncpy(tmp, pbuffer + 2, 3);
			range[4] = strtoul(tmp, NULL, 16);
			//PRINTL(DBG_LVL_ALWAYS, "range: %x %x %x %x %x %x %x %x %x %x\n", range[0], range[1], range[2], range[3], range[4], range[5], range[6], range[7], range[8], range[9]);
			if ((ret = get_port_mib(portid, xdsl2LineStatusMrefPsdDs, pbuffer)) == DSL_ERR_OK){
				PRINTL(DBG_LVL_ALWAYS, "Psd level (dBm/Hz) Ds : Tone No.|PSD level");
				PRINTL(DBG_LVL_XML, "Psd Level Ds:");
				if (strncmp(pbuffer + 2, "c69a53", 6) != 0)
					show_psdlevel(pbuffer + 2, 48, range);
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "NA");
			}
		}
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting line table.");

	return ret;
}


int com_interface_xdsl_show_table_lineband(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-24s|%-24s %s";
	int ret, idx;
	mt_uint8 phy_status;
	mt_uint32 portid;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "xdsl2LineBandTable of Line %u:", portid+1);
		PRINTL(DBG_LVL_ALWAYS, "%-24s %-24s Description", "", "Value");
		for (idx = 1; idx < 12; idx++){
			ret = print_mib_uint32(portid, outfmt, xdsl2LineBand, "Band ID", "us(1),ds(2),us0(3),ds1(4),us1(5),ds2(6),us2(7),ds3(8),us3(9),ds4(10),us4(11)", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineBandStatusLnAtten, "Line Attenuation", "(0.1 dB)", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineBandStatusSigAtten, "Signal Attenuation", "(0.1 dB)", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2LineBandStatusSnrMargin, "SNR Margin", "(0.1 dB)", idx);
			PRINTL(DBG_LVL_ALWAYS, "");
		}
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting line band table.");

	return ret;
}


int com_interface_xdsl_show_table_channelstatus(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-40s|%-10s %s";
	int ret, idx, unit;
	mt_uint8 phy_status;
	mt_uint32 portid;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "xdsl2ChannelStatusTable of Line %u:", portid+1);

		for (idx = 1; idx < 4;){

			ret = get_mib2_int_idx(portid, xdsl2PMChCurrUnit, &unit, idx);
			if (ret == DSL_ERR_OK){
				PRINTL(DBG_LVL_ALWAYS,"");
				if (unit == 1)
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Upstream:");
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Downstream:");

				PRINTL(DBG_LVL_ALWAYS, "%-40s %-10s Description", "", "Value");
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusActDataRate, "Actual Data Rate", "(kbps)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusPrevDataRate, "Previous Data Rate", "(kbps)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusActDelay, "Actual delay", "(ms)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusActInp, "Actual INP", "(0.1 DMT symbol)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusInpReport, "INP reporting mode", "UsingFormula(1),ByXtur(2)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusNFec, "Size of Reed-Solomon codeword", "(bytes)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusRFec, "Number of Reed-Solomon redundancy bytes", "(bytes)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusLSymb, "Number of bits per symbol", "(bits)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusIntlvDepth, "Interleaving Depth", "", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusIntlvBlock, "Interleaving Block Length", "", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusLPath, "Latency Path", "FAST(0),INTERLEAVED(1)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusAtmStatus, "Status of ATM Data Path", "BITS:noDefect(0),noCellDelineation(1),lossOfDelineation(2)", idx);
				ret = print_mib_uint32(portid, outfmt, xdsl2ChStatusPtmStatus, "Status of PTM Data Path", "BITS:noDefect(0),outOfSync(1)", idx);
				if (idx == 1)
					msg_buf[strlen(msg_buf)-1] = ';';

				idx += 2;
			}
		} // for ()
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting channel status table.");

	return ret;
}


int com_interface_xdsl_show_table_scstatus(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-40s|%-24s %s";
	int ret, idx;
	mt_uint8 phy_status;
	mt_uint32 portid;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "xdsl2SCStatusTable of Line %u:", portid+1);
		PRINTL(DBG_LVL_ALWAYS, "%-40s %-24s Description", "", "Value");
		for (idx = 1; idx < 3; idx++){
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusDirection, "Direction", "upstream(1),downstream(2)", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusLinScale, "Scale factor to H(f) linear value", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusLinScGroupSize, "Number of SC per group (H(f) linear)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusLogMt, "Symbols to measure Hlog(f)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusLogScGroupSize, "Number of SC per group (H(f) log)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusQlnMt, "Symbols to measure QLN(f)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusQlnScGroupSize, "Number of SC per group (QLN)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusSnrMtime, "Symbols to measure SNR(f)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusSnrScGroupSize, "Number of SC per group (SNR)", "", idx);
			ret = print_mib_uint32(portid, outfmt, xdsl2SCStatusRowStatus, "Row Status", "", idx);
			PRINTL(DBG_LVL_ALWAYS, "");
		}
		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting scstatus table.");

	return ret;
}


static int show_scseglin(mt_uint32 portid, FILE * outfile)
{
	mt_ret ret;
	mt_uint8 real[512*3], img[512*3], output[64];
	mt_uint32 i, scale, outoffset, direction;
	float realV,imgV;

	ret = get_mib2_int(portid, xdsl2SCStatusLinScale, &scale);

	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLinReal, real, 512*3, NULL);
	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLinImg, img, 512*3, NULL);

	if (ret == DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS, "H(f) linear representation of Line %d :", portid+1);
		PRINTL(DBG_LVL_ALWAYS, "%9s %20s   %20s", "Tone No.", "Real", "Imaginary");
		if (outfile != NULL)
			fprintf(outfile, "Local:");
		for (i = 0; i < 512; i++){
			realV = (short)(real[i*3+1] << 8 | real[i*3+2]);
			imgV = (short)(img[i*3+1] << 8 | img[i*3+2]);
			if (realV != -0x8000 || imgV != -0x8000){
				outoffset = sprintf(output, "%4d-%4d|", i*8 + 1, (i+1)*8);
				if (outfile != NULL)
					fprintf(outfile, "%d-%d=", i*8 + 1, (i+1)*8);
				if (realV != -0x8000){
					outoffset += sprintf(output + outoffset, "%20.10f", realV*scale/32768/1024);
					if (outfile != NULL)
						fprintf(outfile, "%.10f", realV*scale/32768/1024);
				}else
					outoffset += sprintf(output + outoffset, "%20s", "");

				outoffset += sprintf(output + outoffset, "   ");
				if (outfile != NULL)
					fprintf(outfile, ",");

				if (imgV != -0x8000){
					outoffset += sprintf(output + outoffset, "%20.10f", imgV*scale/32768/1024);
					if (outfile != NULL)
						fprintf(outfile, "%.10f", imgV*scale/32768/1024);
				}

				PRINTL(DBG_LVL_ALWAYS, output);
				if (outfile != NULL)
					fprintf(outfile, "|");
			}
		} // for ()
	}

	ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, pbuffer, 9, NULL, IsRT(portid) ? 1 : 2); // get OT

	if (ret == DSL_ERR_OK){
		if ( strcmp(pbuffer + 2, "META") == 0 ){

			ret = get_mib2_int_idx(portid, xdsl2SCStatusLinScale, &scale, 1 + INDEX_REMOTE);

			ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLinReal, real, 512*3, NULL, 1 + INDEX_REMOTE);
			ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLinImg, img, 512*3, NULL, 1 + INDEX_REMOTE);

			if (ret == DSL_ERR_OK){
				PRINTL(DBG_LVL_ALWAYS, "H(f) linear representation of Line %d Remote:", portid+1);
				PRINTL(DBG_LVL_ALWAYS, "%9s %20s   %20s", "Tone No.", "Real", "Imaginary");
				if (outfile != NULL)
					fprintf(outfile, "Remote:");
				for (i = 0; i < 512; i++){
					realV = (short)(real[i*3+1] << 8 | real[i*3+2]);
					imgV = (short)(img[i*3+1] << 8 | img[i*3+2]);
					if (realV != -0x8000 || imgV != -0x8000){
						outoffset = sprintf(output, "%4d-%4d|", i*8 + 1, (i+1)*8);
						if (outfile != NULL)
							fprintf(outfile, "%d-%d=", i*8 + 1, (i+1)*8);
						if (realV != -0x8000){
							outoffset += sprintf(output + outoffset, "%20.10f", realV*scale/32768/1024);
							if (outfile != NULL)
								fprintf(outfile, "%.10f", realV*scale/32768/1024);
						}else
							outoffset += sprintf(output + outoffset, "%20s", "");

						outoffset += sprintf(output + outoffset, "   ");
						if (outfile != NULL)
							fprintf(outfile, ",");

						if (imgV != -0x8000){
							outoffset += sprintf(output + outoffset, "%20.10f", imgV*scale/32768/1024);
							if (outfile != NULL)
								fprintf(outfile, "%.10f", imgV*scale/32768/1024);
						}

						PRINTL(DBG_LVL_ALWAYS, output);
						if (outfile != NULL)
							fprintf(outfile, "|");
					}
				} // for ()
			}
		}else
			PRINTL(DBG_LVL_ALWAYS, "H(f) linear representation of Line %d Remote: Not Metanoia Device", portid+1);
	}

	return ret;
}


static int show_scseglog(mt_uint32 portid, FILE * outfile)
{
	mt_ret ret;
	mt_uint8 log_l[512*3], log_r[512*3], tmp[7], localfmt[24], remotefmt[24],localfmtxml[24], remotefmtxml[24];
	mt_uint32 direction, i, data, isrt;
	mt_uint8 * value, showremote;
	mt_uint16 log;

	isrt = IsRT(portid);

	if (isrt){
		strcpy(localfmt, "DS:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "US:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "DS:%d-%d=%.1f|");
		strcpy(remotefmtxml, "US:%d-%d=%.1f|");
	}else{
		strcpy(localfmt, "US:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "DS:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "US:%d-%d=%.1f|");
		strcpy(remotefmtxml, "DS:%d-%d=%.1f|");
	}

	ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, pbuffer, 9, NULL, isrt ? 1 : 2); // get OT
	showremote = (strcmp(pbuffer + 2, "META") == 0);

	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentLog, log_l, 512*3, NULL);

	if (showremote)
		ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentLog, log_r, 512*3, NULL, 1 + INDEX_REMOTE);

	PRINTL(DBG_LVL_ALWAYS, "SCStatusSegmentLog of Line %d:", portid+1);
	if (ret == DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS, "%14s %10s", "Tone No.", "dB");
		for (i = 0; i < 512; i++){
			log = log_l[i*3+1] << 8 | log_l[i*3+2];
			if (log != 1023){
				PRINTL(DBG_LVL_ALWAYS, localfmt, i*8 + 1, (i+1)*8, 6 - (float)log/10);
				if (outfile != NULL)
					fprintf(outfile, localfmtxml, i*8 + 1, (i+1)*8, 6 - (float)log/10);
			}
			else if (showremote){
				log = log_r[i*3+1] << 8 | log_r[i*3+2];
				if (log != 1023){
					PRINTL(DBG_LVL_ALWAYS, remotefmt, i*8 + 1, (i+1)*8, 6 - (float)log/10);
					if (outfile != NULL)
						fprintf(outfile, remotefmtxml, i*8 + 1, (i+1)*8, 6 - (float)log/10);
				}
			}
		} // for ()

		if (!showremote)
			PRINTL(DBG_LVL_ALWAYS, "%s: Not Metanoia Device", isrt?"US":"DS");
	}

	return ret;
}


static int show_scsegqln(mt_uint32 portid, FILE * outfile)
{
	mt_ret ret;
	mt_uint8 qln_l[512], qln_r[512], localfmt[24], remotefmt[24], localfmtxml[24], remotefmtxml[24];
	mt_uint32 direction, i, isrt;
	mt_uint8 showremote;

	isrt = IsRT(portid);

	if (isrt){
		strcpy(localfmt, "DS:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "US:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "DS:%d-%d=%.1f|");
		strcpy(remotefmtxml, "US:%d-%d=%.1f|");
	}else{
		strcpy(localfmt, "US:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "DS:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "US:%d-%d=%.1f|");
		strcpy(remotefmtxml, "DS:%d-%d=%.1f|");
	}

	ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, pbuffer, 9, NULL, isrt ? 1 : 2); // get OT
	showremote = (strcmp(pbuffer + 2, "META") == 0);

	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentQln, qln_l, 512, NULL);

	if (showremote)
		ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentQln, qln_r, 512, NULL, 1 + INDEX_REMOTE);

	PRINTL(DBG_LVL_ALWAYS, "SCStatusSegmentQln of Line %d:", portid+1);
	if (ret == DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS, "%14s %10s", "Tone No.", "dBm/Hz");
		for (i = 0; i < 512; i++){
			if (qln_l[i] != 0xff){
				PRINTL(DBG_LVL_ALWAYS, localfmt, i*8 + 1, (i+1)*8, -23 - (float)qln_l[i]/2);
				if (outfile != NULL)
					fprintf(outfile, localfmtxml, i*8 + 1, (i+1)*8, -23 - (float)qln_l[i]/2);
			}
			else if (showremote && qln_r[i] != 0xff){
				PRINTL(DBG_LVL_ALWAYS, remotefmt, i*8 + 1, (i+1)*8, -23 - (float)qln_r[i]/2);
				if (outfile != NULL)
					fprintf(outfile, remotefmtxml, i*8 + 1, (i+1)*8, -23 - (float)qln_r[i]/2);
			}
		} // for ()

		 if (!showremote)
			 PRINTL(DBG_LVL_ALWAYS, "%s: Not Metanoia Device", isrt?"US":"DS");
	}

	return ret;
}


static int show_scsegsnr(mt_uint32 portid, FILE * outfile)
{
	mt_ret ret;
	mt_uint8 snr_l[512], snr_r[512], localfmt[24], remotefmt[24], localfmtxml[24], remotefmtxml[24];
	mt_uint32 direction, i, isrt;
	mt_uint8 showremote;

	isrt = IsRT(portid);

	if (isrt){
		strcpy(localfmt, "DS:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "US:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "DS:%d-%d=%.1f|");
		strcpy(remotefmtxml, "US:%d-%d=%.1f|");
	}else{
		strcpy(localfmt, "US:  %4d-%4d|%10.1f");
		strcpy(remotefmt, "DS:  %4d-%4d|%10.1f");
		strcpy(localfmtxml, "US:%d-%d=%.1f|");
		strcpy(remotefmtxml, "DS:%d-%d=%.1f|");
	}

	ret = get_mib2_uint8_idx(portid, xdsl2LInvG994VendorId, pbuffer, 9, NULL, isrt ? 1 : 2); // get OT
	showremote = (strcmp(pbuffer + 2, "META") == 0);

	ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentSnr, snr_l, 512, NULL);

	if (showremote)
		ret = get_mib2_uint8_idx(portid, xdsl2SCStatusSegmentSnr, snr_r, 512, NULL, 1 + INDEX_REMOTE);

	PRINTL(DBG_LVL_ALWAYS, "SCStatusSegmentSnr of Line %d:", portid+1);
	if (ret == DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS, "%14s %10s", "Tone No.", "dB");
		for (i = 0; i < 512; i++){
			if (snr_l[i] != 0xff){
				PRINTL(DBG_LVL_ALWAYS, localfmt, i*8 + 1, (i+1)*8, -32 + (float)snr_l[i]/2);
				if (outfile != NULL)
					fprintf(outfile, localfmtxml, i*8 + 1, (i+1)*8, -32 + (float)snr_l[i]/2);
			}
			else if (showremote && snr_r[i] != 0xff){
				PRINTL(DBG_LVL_ALWAYS, remotefmt, i*8 + 1, (i+1)*8, -32 + (float)snr_r[i]/2);
				if (outfile != NULL)
					fprintf(outfile, remotefmtxml, i*8 + 1, (i+1)*8, -32 + (float)snr_r[i]/2);
			}
		} // for ()

		if (!showremote)
			PRINTL(DBG_LVL_ALWAYS, "%s: Not Metanoia Device", isrt?"US":"DS");
	}

	return ret;
}


static int show_scsegbitsalloc(mt_uint32 portid, FILE * outfile)
{
	mt_uint8 seg[6], bits[512*3], output[80];
	int ret, row, subcarrier, outoffset;

	PRINTL(DBG_LVL_ALWAYS, "SCStatusSegmentBitsAlloc of Line %d : Tone No.|Bits", portid+1);
	for (row = 1; row <= 8 ; row++){
		sprintf(seg, "%06x", row);
		if (ret = set_port_mib(portid, xdsl2SCStatusSegment, seg) != DSL_ERR_OK)
			break;
		sleep(1);
		if (ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentBitsGainAlloc, bits, 512*3, NULL) != DSL_ERR_OK)
			break;

		outoffset = 0;
		for (subcarrier = 0; subcarrier < 512; subcarrier++){
			if (bits[subcarrier*3] == 0xff){ // first byte per word
				outoffset += sprintf(output + outoffset, "%4d|%3d ", (row - 1) * 512 + subcarrier + 1, 0);
				if (outfile != NULL)
					fprintf(outfile, "%d=%d|", (row - 1) * 512 + subcarrier + 1, 0);
			}else{
				outoffset += sprintf(output + outoffset, "%4d|%3d ", (row - 1) * 512 + subcarrier + 1, bits[subcarrier*3]);
				if (outfile != NULL)
					fprintf(outfile, "%d=%d|", (row - 1) * 512 + subcarrier + 1, bits[subcarrier*3]);
			}
			if (subcarrier % 8 == 7) {
				PRINTL(DBG_LVL_ALWAYS, output);
				outoffset = 0;
			}
		}
	}

	return ret;
}


static int show_scseggainalloc(mt_uint32 portid, FILE * outfile)
{
	mt_uint8 seg[6], bits[512*3], output[128], xml[128];
	int ret, row, subcarrier, outoffset, xmloffset;
	mt_uint16 gain;
	float gainshow;
	int xmllen = 0;

	PRINTL(DBG_LVL_ALWAYS, "SCStatusSegmentGainAlloc of Line %d : Tone No.|Gain", portid+1);
	for (row = 1; row <= 8 ; row++){
		sprintf(seg, "%06x", row);
		if (ret = set_port_mib(portid, xdsl2SCStatusSegment, seg) != DSL_ERR_OK)
			break;
		sleep(1);
		if (ret = get_mib2_uint8(portid, xdsl2SCStatusSegmentBitsGainAlloc, bits, 512*3, NULL) != DSL_ERR_OK)
			break;

		outoffset = 0;
		xmloffset = 0;
		for (subcarrier = 0; subcarrier < 512; subcarrier++){
			gain = bits[subcarrier*3+1] << 8 | bits[subcarrier*3+2];
			if (gain == 0xffff){ // last 2bytes per word
				outoffset += sprintf(output + outoffset, "%4d|%6d ", (row - 1) * 512 + subcarrier + 1, 0);
				xmloffset += sprintf(xml + xmloffset, "%d ", 0);
			}else{
				gainshow = (float)gain/512;
				outoffset += sprintf(output + outoffset, "%4d|%6.5f ", (row - 1) * 512 + subcarrier + 1, gainshow);
				xmloffset += sprintf(xml + xmloffset, "%.5f ", gainshow);
			}
			if (subcarrier % 8 == 7) {
				PRINTL(DBG_LVL_ALWAYS, output);
				outoffset = 0;

				xml[xmloffset - 1] = '|';
				if (NULL != outfile)
					fprintf(outfile, "%d-%d=%s", (row - 1)*512 + subcarrier - 7 + 1, (row - 1)*512 + subcarrier + 1, xml);
				xmloffset = 0;
			}
		}
	}

	return ret;
}


static void show_xmlscsegment(const char * file, mt_uint32 idx)
{
	char buf[1024];
	FILE * fscdata;
	mt_uint32 toread;

	if (access(file, F_OK) == 0){
		fscdata = fopen(file, "r");

		if (NULL != fscdata){
			fseek(fscdata, (idx - 1) * 5000, SEEK_SET);
			toread = 5000;
			memset(buf, '\0', 1024);
			while (!feof(fscdata)){
				toread -= fread(buf, 1, (toread > 1023) ? 1023 : toread, fscdata);
				if (ferror(fscdata)){
					msg_buf[0] = '\0';
					PRINTL(DBG_LVL_XML, "File I/O Error");
					break;
				}
				PRINTL(DBG_LVL_XML, buf);
				if (0 == toread) break;
				memset(buf, '\0', 1023);
			}
			fclose(fscdata);
		}else
			PRINTL(DBG_LVL_XML, "Can not open data file");
	}else
		PRINTL(DBG_LVL_XML, "Please update data first");
}


static int checkentry(mt_uint8 * entry)
{
	return ( strcasecmp(entry, "lin") != 0 && strcasecmp(entry, "log") != 0
			&& strcasecmp(entry, "qln") != 0 && strcasecmp(entry, "snr") != 0
			&& strcasecmp(entry, "bitsalloc") != 0 && strcasecmp(entry, "gainalloc") != 0);
}


mt_ret generate_scstatus_output(mt_uint32 portid, mt_uint32 idx, mt_uint8 *entry, mt_uint8 *action, mt_uint8 *prefix)
{
	mt_uint8 mutex[32], outfile[32];
	int ret, mutfile;
	FILE * fscdata;
	struct stat st;

	sprintf(mutex, "/tmp/%s.mut", entry);
	if (access(mutex, F_OK) == -1){

		mutfile = creat(mutex, O_CREAT);
		if (mutfile != -1){
			sprintf(outfile, "/tmp/%s%s", prefix, entry);
			if (strcasecmp(action, "update") == 0){
				fscdata = fopen(outfile, "w");
				if (NULL != fscdata){
					if (strcasecmp(entry, "lin") == 0)
						ret = show_scseglin(portid, fscdata);
					else if (strcasecmp(entry, "log") == 0)
						ret = show_scseglog(portid, fscdata);
					else if (strcasecmp(entry, "qln") == 0)
						ret = show_scsegqln(portid, fscdata);
					else if (strcasecmp(entry, "snr") == 0)
						ret = show_scsegsnr(portid, fscdata);
					else if (strcasecmp(entry, "bitsalloc") == 0)
						ret = show_scsegbitsalloc(portid, fscdata);
					else if (strcasecmp(entry, "gainalloc") == 0)
						ret = show_scseggainalloc(portid, fscdata);

					fclose(fscdata);

					if (DSL_ERR_OK != ret){
						PRINTL(DBG_LVL_XML, "Update failed");
						remove(outfile);
					}
				}else
					PRINTL(DBG_LVL_XML, "File I/O Error");
			}else if (strcasecmp(action, "get") == 0){
				st.st_size = 0;
				stat(outfile, &st);
				if (idx > (st.st_size / 5000 + ((st.st_size % 5000) ? 1 : 0)) || idx == 0)
					PRINTL(DBG_LVL_XML, "Index is out of range");
				else
					show_xmlscsegment(outfile, idx);
			}else{ // size
				st.st_size = 0;
				stat(outfile, &st);
				PRINTL(DBG_LVL_XML, "%u", st.st_size / 5000 + ((st.st_size % 5000) ? 1 : 0));
			}


			close(mutfile);
			remove(mutex);
		}else
			PRINTL(DBG_LVL_XML, "File I/O Error");
	}else
		PRINTL(DBG_LVL_XML, "Busy");

	return ret;
}


/**
 * Show xdsl2SCStatusSegmentTable
 * \par Usage:
 * 		cmd <portid> <entry> <action> [idx]
 * \param portid: port Id, start from 1
 * \param entry: which entry, lin/log/qln/snr/bitsalloc/gainalloc
 * \param action: which action (only for xml), update/get/size
 * \return
 * 	- 0 : success
 * 	- other : error code
 */
int com_interface_xdsl_show_table_scstatussegment(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%40s|%-10d %s";
	int ret;
	mt_uint8 phy_status, entry[32], action[16];
	mt_uint32 portid, idx;

	idx = 0;
	ret = sscanf(pbuf, "%u %s %s %u", &portid, entry, action, &idx);
	if (silent && (ret < 3 || checkentry(entry)
		|| ( strcasecmp(action , "update") != 0 && strcasecmp(action, "get") != 0
		&& strcasecmp(action, "size") != 0))){
		PRINTL(DBG_LVL_XML, "Usage: cmd <portid> <lin/log/qln/snr/bitsalloc/gainalloc> <update/get <idx>/size>");
		return -DSL_ERR_NOPASS;
	}else if (!silent && (ret != 2 || checkentry(entry))){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> <lin/log/qln/snr/bitsalloc/gainalloc>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status != MT_OPSTATE_SHOWTIME){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %d not ShowTime", portid + 1);
		ret = DSL_ERR_NOTOK;
	}
	else{
		if (silent){
			ret = generate_scstatus_output(portid, idx, entry, action, NULL);
		}else{
			if (strcasecmp(entry, "lin") == 0)
				ret = show_scseglin(portid, NULL);
			else if (strcasecmp(entry, "log") == 0)
				ret = show_scseglog(portid, NULL);
			else if (strcasecmp(entry, "qln") == 0)
				ret = show_scsegqln(portid, NULL);
			else if (strcasecmp(entry, "snr") == 0)
				ret = show_scsegsnr(portid, NULL);
			else if (strcasecmp(entry, "bitsalloc") == 0)
				ret = show_scsegbitsalloc(portid, NULL);
			else if (strcasecmp(entry, "gainalloc") == 0)
				ret = show_scseggainalloc(portid, NULL);
		}

		ret = DSL_ERR_OK;
	}

	return ret;
}


int com_interface_xdsl_show_table_pmlineinitcurr(unsigned char * pbuf, cmdt * pcmd)
{
	const char * outfmt = "%-42s|%-24s %s";
	int ret;
	mt_uint8 phy_status;
	mt_uint32 portid;

	ret = sscanf(pbuf, "%u", &portid);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	}
	else{
		PRINTL(DBG_LVL_ALWAYS, "xdsl2PMLineInitCurrTable of Line %u:", portid+1);
		PRINTL(DBG_LVL_ALWAYS, "%-42s %-24s Description", "", "Value");
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr15MValidIntervals, "15M valid intervals", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr15MInvalidIntervals, "15M invalid intervals", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr15MTimeElapsed, "15M elapsed seconds", "(seconds)", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr15MFullInits, "15M count of full initializations", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr15MFailedFullInits, "15M count of failed full initializations", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr1DayValidIntervals, "1Day valid intervals", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr1DayInvalidIntervals, "1Day invalid intervals", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr1DayTimeElapsed, "1Day elapsed seconds", "(seconds)", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr1DayFullInits, "1Day count of full initializations", "", 1);
		ret = print_mib_uint32(portid, outfmt, xdsl2PMLInitCurr1DayFailedFullInits, "1Day count of failed full initializations", "", 1);

		ret = DSL_ERR_OK;
	}

	if(ret != DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "Error getting pmlineinitcurr table.");

	return ret;
}


//=======================interface_xdsl_show_table cmd tables=============
cmdt interface_xdsl_show_table_cmd[] = {
	{"line", "Show xdsl2LineTable", NULL, NULL, com_interface_xdsl_show_table_line},
	{"lineband", "Show xdsl2LineBandTable", NULL, NULL, com_interface_xdsl_show_table_lineband},
	{"channelstatus", "Show xdsl2ChannelStatusTable", NULL, NULL, com_interface_xdsl_show_table_channelstatus},
	{"scstatus", "Show xdsl2SCStatusTable", NULL, NULL, com_interface_xdsl_show_table_scstatus},
	{"scstatussegment", "Show xdsl2SCStatusSegmentTable", NULL, NULL, com_interface_xdsl_show_table_scstatussegment},
	{"pmlineinitcurr", "Show PMLineInitCurrTable", NULL, NULL, com_interface_xdsl_show_table_pmlineinitcurr},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL, com_help},
	{"exit", "Change to upper directory", interface_xdsl_show_cmd, NULL, com_interface_xdsl_show},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL}
};


#ifdef WITH_ADV
int com_diag_xdsl_lb (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_diag_xdsl_meitest (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


//=======================diag cmd tables=============
cmdt diag_xdsl_cmd[] = {
	{"lb","Change to ge directory", NULL, NULL, com_diag_xdsl_lb},
	{"meitest","Change to xdsl directory",NULL, NULL, com_diag_xdsl_meitest},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL, com_help},
	{"exit", "Change to upper directory", diag_cmd, NULL, com_diag},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


int com_diag_ge (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_diag_xdsl (unsigned char * pbuf, cmdt * pcmd)
{
	commands = diag_xdsl_cmd;
	set_prompt("(diag xdsl)");
	return DSL_ERR_OK;
}


//=======================diag cmd tables=============
cmdt diag_cmd[] = {
	{"ge","Change to ge directory", NULL/*diag_ge_cmd*/, NULL, com_diag_ge},
	{"xdsl","Change to xdsl directory",diag_xdsl_cmd, NULL,com_diag_xdsl},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", root_cmd, NULL, com_home},
	{"home", "Return to the root directory", root_cmd, NULL,com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};
#endif //WITH_ADV


mt_ret set_port_atm_pvc(mt_uint8 portid)
{
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 i, pvc_count = 0;
	mt_uint8 loid[OID_MAX_LEN];
	mt_uint32 value1, value2, value3;

	//setup pvc MIBs
	for(i=0;i<NUM_OF_PVC;i++) {
		if(atm_pvc[i].active == TRUE) {
			//setup pvc related MIBs
			//VPI VCI
			//Tx
			value1 = (atm_pvc[i].vpi << 16) | atm_pvc[i].vci;
			sprintf(pbuffer, "%06x", value1);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_CfgSarSegTransP0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - CfgSarSegTransP%d: 0x%s\n", loid, i, pbuffer);
			//Rx
			value1 = (atm_pvc[i].vpi << 16) | atm_pvc[i].vci;
			sprintf(pbuffer, "%06x", value1);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_CfgSarReamMapperP0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - CfgSarReamMapperP%d: 0x%s\n", loid, i, pbuffer);

			//VLAN
			//Tx
			value1 = 0x00FFF000;
			value1 |= atm_pvc[i].vlanid & 0x00000FFF;
			sprintf(pbuffer, "%06x", value1);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_CfgSarSegMapperP0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");

			//force to remove VLAN ID
			value2 = (CfgSarSegProcPx_Default & 0xFFFFC000) | (VLAN_OP_REMOVE << 12) | (atm_pvc[i].vlanid & 0x00000FFF);
			printf("OID: %s - CfgSarSegMapperP%d: 0x%s\n", loid, i, pbuffer);
			//Rx
			//force to tag(replace) VLAN ID
			value3 = (CfgSarReamProcPx_Default & 0xFFFFC000) | (VLAN_OP_REPLACE << 12) | (atm_pvc[i].vlanid & 0x00000FFF);

			//Packet Queue Priority
			//Tx

			//Rx


			//Encapsulation
			//Tx
			value2 &= ~(0x0081C000);
			if(atm_pvc[i].encap == VC_MUX) {
				value2 |= 4 << 14;
				value2 |= 0 << 23;
			}
			else if(atm_pvc[i].encap == LLC) {
				value2 |= 1 << 14;
				value2 |= 1 << 23;
			}
			sprintf(pbuffer, "%06x", value2);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_CfgSarSegProcP0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - CfgSarSegProcP%d: 0x%s\n", loid, i, pbuffer);
			//Rx
			value3 &= ~(0x00C1C000);
			if(atm_pvc[i].encap == VC_MUX) {
				value3 |= 4 << 14;
				value3 |= 0 << 22;
				value3 |= 1 << 23;
			}
			else if(atm_pvc[i].encap == LLC) {
				value3 |= 1 << 14;
				value3 |= 1 << 22;
				value3 |= 1 << 23;
			}
			sprintf(pbuffer, "%06x", value3);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_CfgSarReamProcP0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - CfgSarReamProcP%d: 0x%s\n", loid, i, pbuffer);

			//ATM QoS
			sprintf(pbuffer, "3");	//always use UBR for now
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_xdslMetanoiaAtmService0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - xdslMetanoiaAtmService%d: 0x%s\n", loid, i, pbuffer);

			//PCR CDVT SCR MBS
			sprintf(pbuffer, "%06x%06x%06x%06x", atm_pvc[i].pcr, 0x000000, atm_pvc[i].scr, atm_pvc[i].mbs);
			sprintf(loid, "%d.%d.%d", MT_MIB_XdslMetanoiaGroup, MT_MIB_XdslMetanoiaTable, MT_MIB_xdslMetanoiaAtmTfcParam0+i);
			ret = set_port_mib(portid, loid, pbuffer);
			if(ret == DSL_ERR_OK)
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
			printf("OID: %s - xdslMetanoiaAtmTfcParam%d: 0x%s\n", loid, i, pbuffer);

			pvc_count++;
		}
	}

	//We force to not use pause frame and drop packet when queue is full for now
	get_port_mib(portid, CfgM2Qosfr, pbuffer);
	value1 = strtoul(pbuffer+2, NULL, 16);
	value1 |= 0x000001E0;	//force Tx(switch to DMT) to drop packet
	sprintf(pbuffer, "%06x", value1);
	ret = set_port_mib(portid, CfgM2Qosfr, pbuffer);
	if(ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
	printf("OID: %s - CfgM2Qosfr: 0x%s\n", CfgM2Qosfr, pbuffer);

	//setup total pvc number
	if(pvc_count != 0) {
		pvc_count--;	//pvc_count set as 0 means using 1 PVC

		get_port_mib(portid, CfgSarfr, pbuffer);
		value1 = strtoul(pbuffer+2, NULL, 16);
		value1 &= ~(0x0000003F);
		value1 = ((pvc_count << 3) & 0x00000038) | (pvc_count & 0x00000007);	//Rx|Tx
		sprintf(pbuffer, "%06x", value1);
		ret = set_port_mib(portid, CfgSarfr, pbuffer);
		if(ret == DSL_ERR_OK)
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
		else
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
		printf("OID: %s - CfgSarfr: 0x%s\n", CfgSarfr, pbuffer);

		PRINTL(DBG_LVL_CLI, "Setting ADSL PVC...");
		ret = disconnect_port(portid);
		ret = connect_port(portid);
	}

	return ret;
}


/**
 * Set ADSL VPI/VCI VLAN mapping
 * \par Usage:
 * 		cmd <pvc> <vpi> <vci> <vlanid> <priority> <encapsulation> <atmqos> <pcr> <scr> <mbs> <activation>
 * \param pvc: Permanent Virtual Circuit to set
 * \param vpi: Virtual Path Identifier to map
 * \param vci: Virtual Circuit Identifier to map
 * \param vlanid: Virtual LAN Identifier to map
 * \param priority: set VLAN priority to specified packet priority queue, Queue0 - Queue4
 * \param encapsulation: set encapsulation for VC-MUX or LLC, 0: VC-MUX, 1: LLC
 * \param atmqos: set VPI/VCI to use specified ATM QoS, 0: CBR, 1: VBR-RT, 2: VBR-NRT, 3: UBR
 * \param pcr: set Peak Cell Rate for specified PVC
 * \param scr: set Sustained Cell Rate for specified PVC
 * \param mbs: set Maximum Burst Size for specified PVC
 * \param activation: set Activation for specified PVC
 * \return
 *
 */
int set_pvc_rule(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 command[64];
	mt_uint32 portid;
	mt_uint32 pvc, vpi, vci, vlanid, prio, encap, atmqos, pcr, scr, mbs, activation;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "set_vpi_vci_vlan_encap_prio: start");
#endif

	ret = sscanf(pbuf, "%i %i %i %i %i %i %i %i %i %i %i", &pvc, &vpi, &vci, &vlanid, &prio, &encap, &atmqos, &pcr, &scr, &mbs, &activation);
	if( ret != 11 )
	{
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <pvc> <vpi> <vci> <vlanid> <priority> <encapsulation> <atmqos> <pcr> <scr> <mbs> <activation>");
		PRINTL(DBG_LVL_ALWAYS, "  pvc: 0 - %d", NUM_OF_PVC-1);
		PRINTL(DBG_LVL_ALWAYS, "  vpi: 0 - 255");
		PRINTL(DBG_LVL_ALWAYS, "  vci: 32 - 65535");
		PRINTL(DBG_LVL_ALWAYS, "  vlanid: 1 - 4094");
		//PRINTL(DBG_LVL_ALWAYS, "  priority: 0 - 3");
		PRINTL(DBG_LVL_ALWAYS, "  priority: 0 - 0");
		PRINTL(DBG_LVL_ALWAYS, "  encapsulation: 0: VC-MUX, 1: LLC");
		//PRINTL(DBG_LVL_ALWAYS, "  atmqos: 0: CBR, 1: VBR-RT, 2: VBR-NRT, 3: UBR");
		PRINTL(DBG_LVL_ALWAYS, "  atmqos: 0: UBR");
		PRINTL(DBG_LVL_ALWAYS, "  pcr: 0 - 5500");
		PRINTL(DBG_LVL_ALWAYS, "  scr: 0 - 5500");
		PRINTL(DBG_LVL_ALWAYS, "  mbs: 0 - 65535");
		PRINTL(DBG_LVL_ALWAYS, "  activation: 0: Deactivate, 1: Activate");
		return -DSL_ERR_NOPASS;
	}
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "pvc=%i,vpi=%i,vci=%i,vlanid=%i,priority=%i,encapsulation=%i,atmqos=%i,pcr=%i,scr=%i,mbs=%i,activation=%i", pvc, vpi, vci, vlanid, prio, encap, atmqos, pcr, scr, mbs, activation);

	if(pvc >= NUM_OF_PVC) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed PVC number is 0 - %d", NUM_OF_PVC-1);
		return DSL_ERR_NOTOK;
	}
	if(vpi > VPI_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed VPI value is 0 - %d", VPI_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(vci < VCI_LIMIT1 || vci > VCI_LIMIT2) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed VCI value is %d - %d", VCI_LIMIT1, VCI_LIMIT2);
		return DSL_ERR_NOTOK;
	}
	if(vlanid < VLAN_LIMIT1 || vci > VLAN_LIMIT2) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed VLAN ID is %d - %d", VLAN_LIMIT1, VLAN_LIMIT2);
		return DSL_ERR_NOTOK;
	}
	if(prio > PRIO_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed Queue Priority value is 0 - %d", PRIO_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(encap > ENCAP_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed ATM Encapsulation value is 0 - %d", ENCAP_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(atmqos > ATMQOS_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed ATM QoS value is 0 - %d", ATMQOS_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(pcr > PCR_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed PCR value is 0 - %d", PCR_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(scr > SCR_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed SCR value is 0 - %d", SCR_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(mbs > MBS_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed MBS QoS value is 0 - %d", MBS_LIMIT);
		return DSL_ERR_NOTOK;
	}
	if(activation > ACTIVE_LIMIT) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Allowed Activation value is 0 - %d", ACTIVE_LIMIT);
		return DSL_ERR_NOTOK;
	}

	//we mask off SCR and MBS for CBR and UBR
	if(atmqos == CBR || atmqos == UBR) scr = mbs = 0;

	//check if PVC rules need to be applied
	if(atm_pvc[pvc].vpi != vpi ||
	   atm_pvc[pvc].vci != vci ||
	   atm_pvc[pvc].vlanid != vlanid ||
	   atm_pvc[pvc].prio != prio ||
	   atm_pvc[pvc].encap != encap ||
	   atm_pvc[pvc].atmqos != atmqos ||
	   atm_pvc[pvc].pcr != pcr ||
	   atm_pvc[pvc].scr != scr ||
	   atm_pvc[pvc].mbs != mbs ||
	   atm_pvc[pvc].active != activation) {
		//keep PVC rules
		atm_pvc[pvc].vpi = vpi;
		atm_pvc[pvc].vci = vci;
		atm_pvc[pvc].vlanid = vlanid;
		atm_pvc[pvc].prio = prio;
		atm_pvc[pvc].encap = encap;
		atm_pvc[pvc].atmqos = atmqos;	//we always use UBR for now
		atm_pvc[pvc].pcr = pcr;
		atm_pvc[pvc].scr = scr;
		atm_pvc[pvc].mbs = mbs;
		atm_pvc[pvc].active = activation;

		//apply PVC rules
		set_port_atm_pvc(0);

		//save PVC rules to board.conf
		sprintf(command, ""CFGMGR_PATH" -w ADSL_PVC%d=%x@%x@%x@%x@%x@%x@%x@%x@%x@%x", pvc, vpi, vci, vlanid, prio, encap, atmqos, pcr, scr, mbs, activation);
		ret = system(command);
	}

	PRINTL(DBG_LVL_CLI, "ADSL PVC%d Applied!!", pvc);

	return ret;
}


/**
 * Set Logic MIB entry
 * \par Usage:
 * 		cmd <portid> <groupid> <tableid> <index> <param> <value>
 * \param portid: port Id, start from 1
 * \param oid: MIB object Id present in x.y.z format
 * \param value: new MIB entry
 * \return
 *
 */
int set_logic_mibentry(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 loid[128];
	mt_uint32 portid;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "set_mibentry: start");
#endif

	portid = 1;

	ret = sscanf(pbuf, "%i %s %s", &portid, loid, pbuffer);
	if( ret != 3 ) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <portid> <loid> <value>");
		return -DSL_ERR_NOPASS;
	}
	PRINTL(DBG_LVL_ALWAYS, "portid=%i,loid=%s,value=%s", portid, loid, pbuffer);

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	ret = set_port_mib(portid, loid, pbuffer);
	if(ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");

	return ret;
}


#ifdef CO_API
/**
 * Set MIB entry
 * \par Usage:
 * 		cmd <portid> <groupid> <tableid> <index> <param> <value>
 * \param portid: port Id, start from 1
 * \param oid: MIB object Id present in x.y.z format
 * \param value: new MIB entry
 * \return
 *
 */
int set_mibentry(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 oid[OID_MAX_LEN];
	mt_uint32 portid;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "set_mibentry: start");
#endif

	portid = 1;

	ret = sscanf(pbuf, "%s %s", oid, pbuffer);
	if( ret != 2 ) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <oid> <value>");
		return -DSL_ERR_NOPASS;
	}
	//PRINTL(DBG_LVL_ALWAYS, "oid=%s,value=%s", oid, pbuffer);

	if(get_roid_port(oid, &portid, oid) != DSL_ERR_OK) return DSL_ERR_NOTOK;

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	ret = set_port_data(portid, oid, pbuffer);
	if(ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB value has set.");
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");

	return ret;
}


/**
 * Set Logic MIB bit
 * \par Usage:
 * 		cmd <portid> <groupid> <tableid> <index> <param> <bit> <value>
 * \param portid: port Id, start from 1
 * \param oid: MIB object Id present in x.y.z format
 * \param bit: specified bit number, start from 0
 * \param value: new bit value for MIB entry
 * \return
 *
 */
int set_logic_mibbit(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 loid[128];
	mt_uint32 portid;
	mt_uint32 bitmask, value, value1, value2;

#ifdef MT_TRACE
	PRINTL(DBG_LVL_ALWAYS, "set_mibbit: start");
#endif

	portid = 1;	bitmask = 0; value = 0;

	ret = sscanf(pbuf, "%i %s %x %x", &portid, loid, &bitmask, &value);
	if( ret != 4 )
	{
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Usage:cmd <portid> <loid> <bitmask> <value>");
		return -DSL_ERR_NOPASS;
	}
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "portid=%i,loid=%s,bitmask=%i,value=%i", portid, loid, bitmask, value);

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_port_mib(portid, loid, pbuffer);
	value2 = strtoul(pbuffer+2, NULL, 16);

	value1 = bitmask & value;	//make sure the bitmask and user inputted value are consistant
	value2 &= ~bitmask;	//mask off those bits we wana change form the original MIB value
	value2 |= value1;	//apply those bits we wana change into the original MIB value

	sprintf(pbuffer, "%06x", value2);
	ret = set_port_mib(portid, loid, pbuffer);
	if(ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "MIB bit value has set.");
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB bit value.");

	return ret;
}


int com_interface_xdsl_set_threshold (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}
#endif //CO_API


//=======================interface_xdsl_set cmd tables=============
cmdt interface_xdsl_set_cmd[] = {
#ifdef CO_API
//	{"threshold", "Set the threshold for 15-min or 1day counters of xdsl lines",NULL, NULL, com_interface_xdsl_set_threshold},
	{"oidbit","Set bit(s) in one xDSL Logic MIB entry, <portid> <loid> <bitmask> <value>", NULL, NULL, set_logic_mibbit},
	{"roid","Set xDSL Real MIB entry, <oid> <value>", NULL, NULL, set_mibentry},
#endif //CO_API
	{"oid","Set xDSL Logic MIB entry, <portid> <loid> <value>", NULL, NULL, set_logic_mibentry},
	{"pvc","Set PVC rules, <pvc> <vpi> <vci> <vlanid> <priority> <encapsulation> <atmqos> <pcr> <scr> <mbs> <activation>", NULL, NULL, set_pvc_rule},
//	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", interface_xdsl_cmd, NULL, com_interface_xdsl},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


#ifdef WITH_ADV
int com_interface_xdsl_diag_delt (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_interface_xdsl_diag_lbt (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_interface_xdsl_diag_loopactivate (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


//=======================interface_xdsl_diag cmd tables=============
cmdt interface_xdsl_diag_cmd[] = {
	{"delt","Specifies whether to force Loop diagnostic mode or not", NULL, NULL, com_interface_xdsl_diag_delt},
	{"lbt","This function loops a byte buffer over the EOC back again and compares the result",NULL, NULL,com_interface_xdsl_diag_lbt},
	{"loopactivate", "Activate the specified loop on the desired line",NULL, NULL,com_interface_xdsl_diag_loopactivate},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", interface_xdsl_cmd, NULL, com_interface_xdsl},
	{"home", "Return to the root directory", root_cmd, NULL,com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};
#endif //WITH_ADV

//#ifdef CO_API
int com_interface_xdsl_set (unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface_xdsl_set_cmd;
	set_prompt("(interface xdsl set)");
	return DSL_ERR_OK;
}
//#endif	//CO_API


int com_interface_xdsl_show (unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface_xdsl_show_cmd;
	set_prompt("(interface xdsl show)");
	return DSL_ERR_OK;
}


#ifdef WITH_ADV
int com_interface_xdsl_diag (unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface_xdsl_diag_cmd;
	set_prompt("(interface xdsl diag)");
	return DSL_ERR_OK;
}
#endif


//=======================interface_xdsl cmd tables=============
cmdt interface_xdsl_cmd[] = {
//#ifdef CO_API
	{"set","Set command for xdsl interfaces", interface_xdsl_set_cmd, NULL, com_interface_xdsl_set},
//#endif
	{"show","Show commands for xdsl lines",interface_xdsl_show_cmd, NULL,com_interface_xdsl_show},
#ifdef WITH_ADV
	{"diag", "Diag commands for xdsl line", interface_xdsl_diag_cmd, NULL, com_interface_xdsl_diag},
#endif
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", interface1_cmd, NULL, com_interface},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


#ifdef WITH_ADV
int com_interface_l2 (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}


int com_interface_port (unsigned char * pbuf, cmdt * pcmd)
{
	return DSL_ERR_OK;
}
#endif //WITH_ADV


int com_interface_xdsl (unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface_xdsl_cmd;
	set_prompt("(interface xdsl)");
	return DSL_ERR_OK;
}


//=======================interface cmd tables=============
cmdt interface1_cmd[] = {
#ifdef WITH_ADV
	{"l2","Arl related commands", NULL, NULL, com_interface_l2},
	{"port","Commands for ports",NULL, NULL, com_interface_port},
#endif //WITH_ADV
	{"xdsl", "Commands for xdsl interfaces?",interface_xdsl_cmd, NULL, com_interface_xdsl},
	{"help", "Help display an overview of the CLI syntax", NULL, NULL, com_help},
	{"exit", "Change to upper directory", root_cmd, NULL, com_home},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


#ifdef TEMPLATE_PROFILE
int com_show_xdsl_sprofile (unsigned char * pbuf, cmdt * pcmd)
{

	mt_ret ret;
	char options[256];
	char profile_name[256];
	char buffer[256];

	ret = sscanf(pbuf,"%s %s", options, profile_name);

	if (ret != 2 && ret != 1) {
		PRINTL(DBG_LVL_ALWAYS, "Usage:	cmd <options> <profile_name>");
		PRINTL(DBG_LVL_ALWAYS, "options:SYS,BND,VNS,PSD,RFI,DPB,ALL");
		return -DSL_ERR_NOPASS;
	}

	if(strcmp(options,"SYS")==0) {
		get_package_profile_name(1,0, SYS_PREFIX,NULL);
	}
	else if(strcmp(options,"BND")==0) {
		get_package_profile_name(1,0, BND_PREFIX,NULL);
	}
	else if(strcmp(options,"VNS")==0) {
		get_package_profile_name(1,0, VNS_PREFIX,NULL);
	}
	else if(strcmp(options,"PSD")==0) {
		get_package_profile_name(1,0, PSD_PREFIX,NULL);
	}
	else if(strcmp(options,"RFI")==0) {
		get_package_profile_name(1,0, RFI_PREFIX,NULL);
	}
	else if(strcmp(options,"DPB")==0) {
		get_package_profile_name(1,0, DPB_PREFIX,NULL);
	}
	else if(strcmp(options,"ALL")==0) {
		/*show package support bandplan*/
		get_package_profile_name(1,0, SYS_PREFIX,NULL);
		get_package_profile_name(1,0, BND_PREFIX,NULL);
		get_package_profile_name(1,0, VNS_PREFIX,NULL);
		get_package_profile_name(1,0, PSD_PREFIX,NULL);
		get_package_profile_name(1,0, RFI_PREFIX,NULL);
		get_package_profile_name(1,0, DPB_PREFIX,NULL);
	}
	else {
		PRINTL(DBG_LVL_ALWAYS, "Usage:	cmd <options> <profile_name>");
		PRINTL(DBG_LVL_ALWAYS, "options:SYS,BND,VNS,PSD,RFI,DPB,ALL");
		return -DSL_ERR_NOPASS;
	}

	return DSL_ERR_OK;
}
#endif //TEMPLATE_PROFILE


//=======================show_xdsl cmd tables=============
cmdt show_xdsl_cmd[] = {
	//interface xdsl show tables
	{"summary", "Show line status summary", NULL, NULL, com_interface_xdsl_show_summary},
	{"status", "Show line status", NULL, NULL, com_interface_xdsl_show_status},
	{"pvc","Show PVC rules", NULL, NULL, show_pvc_rule},
#ifdef VIA_SAMSUNG
	{"xmlstatus", "Show line status", NULL, NULL, com_interface_xdsl_show_xmlstatus},
#endif
	{"pm-line-curr","Show current counters of xdsl lines", NULL, NULL, com_interface_xdsl_show_pm_line_curr},
	{"pm-ch-curr", "Show current counters of xdsl channels", NULL, NULL, com_interface_xdsl_show_pm_ch_curr},
#ifdef WITH_ADV
	{"pm-15m-hist","Show history 15-min counters of xdsl lines", NULL, NULL,com_interface_xdsl_show_pm_15m_hist},
	{"pm-1d-hist", "Show history 1day counters of xdsl lines", NULL, NULL, com_interface_xdsl_show_pm_1d_hist},
	{"delt", "DELT loop diagnostic function", NULL, NULL, com_interface_xdsl_show_delt},
	{"linetest", "Line diagnostic function", NULL, NULL, com_interface_xdsl_show_linetest},
#endif
	{"invent", "Show inventory of xdsl lines", NULL, NULL, com_interface_xdsl_show_invent},
	{"threshold", "Show threshold of xdsl lines", NULL, NULL, com_interface_xdsl_show_threshold},
//	{"table", "Show mib tables of xdsl line", interface_xdsl_show_table_cmd, NULL, com_interface_xdsl_show_table},
	{"line", "Show xdsl2LineTable", NULL, NULL, com_interface_xdsl_show_table_line},
	{"lineband", "Show xdsl2LineBandTable", NULL, NULL, com_interface_xdsl_show_table_lineband},
	{"channelstatus", "Show xdsl2ChannelStatusTable", NULL, NULL, com_interface_xdsl_show_table_channelstatus},
	{"scstatus", "Show xdsl2SCStatusTable", NULL, NULL, com_interface_xdsl_show_table_scstatus},
	{"scstatussegment", "Show xdsl2SCStatusSegmentTable", NULL, NULL, com_interface_xdsl_show_table_scstatussegment},
	{"pmlineinitcurr", "Show PMLineInitCurrTable", NULL, NULL, com_interface_xdsl_show_table_pmlineinitcurr},
#ifdef TEMPLATE_PROFILE
//	{"profile", "Show Profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslprofile_show},
	{"line-template", "Show Line Template profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdsllinetemplate_show},
	{"line-profile", "Show Line Profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdsllineprofile_show},
	{"chan-profile", "Show Channel Profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslchanprofile_show},
	{"alarm-template", "Show Alarm Template names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslalarmtemplate_show},
	{"line-alarm-profile", "Show Line Alarm Profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdsllinealarmprofile_show},
	{"chan-alarm-profile", "Show Channel Alarm Profile names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslchanalarmprofile_show},
	{"port", "Show port of a specified profile",NULL, NULL,com_profiles_vdslport_show},
	{"virtual-noise", "Show Virtaul Noise names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslexternprofile_vnshow},
	{"dpbopsd", "Show DpboPsd names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslexternprofile_dpbshow},
	{"rfi-bands", "Show RfiBands names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslexternprofile_rfishow},
	{"psd", "Show Psd names or show detail information of a specified profile",NULL, NULL,com_profiles_vdslexternprofile_psdshow},
	{"sprofile", "Show system profile names ",NULL, NULL,com_show_xdsl_sprofile},
#endif //TEMPLATE_PROFILE
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", root_cmd, NULL, com_home},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


//****************Profile Sub Function**************//
/**
 * get_package_profile_name gets Firmware package profile name by profile id and return to user
 * \param option
 * 		option for distinguish calling from CLI or API
 * \param profileid
 * 		profile id, id start from 1
 * \param prefix
 * 		prefix for identifying profile type
 * \param buffer
 * 		buffer to receive the returned profile name, if id is out of boundary, length of buffer will be "0"
 *
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_package_profile_name(mt_uint8 option, mt_uint16 profileid, mt_uint8 *prefix, mt_uint8 *buffer)
{
#define PROFILE_NAME_LEN_SHIFT DMT_WORD
	//read/write
	int fh;
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 pname[PNAMESIZE];
	mt_uint8 buf[PNAMETMP];
	mt_uint8 tmp[DMT_WORD];
	mt_uint32 i, j;
	mt_uint32 length;
	mt_uint32 start, count, bytesread;

	fh = open(PACKAGE_PATH, O_RDONLY);

	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "firmware package file \"%s\" was not found.", PACKAGE_PATH);
		return DSL_ERR_NOTOK;
	}

	// obtain profile count
	lseek(fh, PROFILE_COUNT, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get profile count
	//printf("profile count: %d\n", count);

	// obtain profile start
	lseek(fh, PROFILE_START, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	start = GET_DMT_WORD(tmp);	// get start

	if(option) {	// for CLI
		if(!strcmp(SYS_PREFIX, prefix)) strcpy(buf, "System");
		else if(!strcmp(BND_PREFIX, prefix)) strcpy(buf, "Band");
		else if(!strcmp(VNS_PREFIX, prefix)) strcpy(buf, "Virtual Noise");
		else if(!strcmp(PSD_PREFIX, prefix)) strcpy(buf, "PSD");
		else if(!strcmp(RFI_PREFIX, prefix)) strcpy(buf, "RFI Band");
		else if(!strcmp(DPB_PREFIX, prefix)) strcpy(buf, "DPBO PSD");
		sprintf(pname, "Supported %s profile:", buf);
		for(i=0;i<count;i++) {
			// obtain profile starting offset
			lseek(fh, start+(i*PROFILE_HDR_LEN)+PROFILE_NAME_LEN_SHIFT, SEEK_SET);
			while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			length = GET_DMT_WORD(tmp);	// get profile name string length
			//printf("profile name string length: 0x%08x\n", length);
			while (((bytesread = read(fh, buf, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			buf[length] = '\0';	// add NULL terminator in the string end
			if(strncmp(prefix, buf, PREFIX_LEN) != 0) continue;	// skip the prefix mismatched profile name
			strcat(pname, "\n");
			strcat(pname, buf+PREFIX_LEN);
			PRINTL(DBG_LVL_XML, "%s|", buf+PREFIX_LEN);
		}
		PRINTL(DBG_LVL_CLI, "%s", pname);
	}
	else {	// for API
		if(profileid == 0 || profileid > count) {
			buffer[0] = '\0';
		}
		else {
			for(i=0,j=0;i<count;i++) {
				// obtain profile starting offset
				lseek(fh, start+(i*PROFILE_HDR_LEN)+PROFILE_NAME_LEN_SHIFT, SEEK_SET);
				while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
				length = GET_DMT_WORD(tmp);	// get profile name string length
				//printf("profile name string length: 0x%08x\n", length);
				while (((bytesread = read(fh, buf, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
				buf[length] = '\0';	// add NULL terminator in the string end
				if(strncmp(prefix, buf, PREFIX_LEN) != 0) {	//prefix not matched
					if(j == 0) continue;	// j == 0 means prefix match never happened, skip the prefix mismatched profile name
					else {	// j != 0 means matched prefix is over but user given profileid is not reached, return with not OK
						buffer[0] = '\0';
						ret = DSL_ERR_NOTOK;
						break;
					}
				}

				if(++j == profileid) {
					strcpy(buffer, buf+PREFIX_LEN);
					break;
				}
			}
		}
	}

	close(fh);

	return ret;
}


/**
 * set_package_profile_by_id applys profile into DMT from Firmware package by profile id
 * \param portid
 * 		port ID number
 * \param profileid
 * 		id of profile to be applied, use 0xff for default profile
 *
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret set_package_profile_by_id(mt_uint8 portid, mt_uint16 profileid)
{
	//read/write
	int fh;
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 loid[128];
	mt_uint8 tmp[DMT_WORD];
	mt_uint32 length;
	mt_uint32 start, count, bytesread;

	fh = open(PACKAGE_PATH, O_RDONLY);

	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "firmware package file \"%s\" was not found.", PACKAGE_PATH);
		return DSL_ERR_NOTOK;
	}

	if(profileid == 0xff) { // user didn't input profile id or user inputted 0xff
		// obtain default profile id
		lseek(fh, DEFAULT_PROFILE, SEEK_SET);
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		profileid = GET_DMT_WORD(tmp);	// get profile id
		//printf("default profile: %d\n", profileid);
	}

	// obtain profile count
	lseek(fh, PROFILE_COUNT, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get profile count
	//printf("profile count: %d\n", count);

	// profile id out of range, skip applying
	if(profileid == 0 || profileid > count) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "No such Profile ID: %d", profileid);
		ret = DSL_ERR_NOTOK;
		goto end_sppid;
	}

	// obtain profile start
	lseek(fh, PROFILE_START, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	start = GET_DMT_WORD(tmp);	// get start

	// obtain profile starting offset
	lseek(fh, start+((profileid - 1)*PROFILE_HDR_LEN), SEEK_SET);
	//printf("profile start offset: %d\n", start+((profileid - 1)*PROFILE_HDR_LEN));
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	start = GET_DMT_WORD(tmp);	// get start address
	//printf("profile start: 0x%08x\n", start);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	length = GET_DMT_WORD(tmp);	// get profile name string length
	//printf("profile name string length: 0x%08x\n", length);
	while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	loid[length] = 0;	// add NULL terminator in the string end
	PRINTL(DBG_LVL_CLIDBG, "profile name: %s\n", loid);

	// change file descriptor to profile starting offset
	lseek(fh, start, SEEK_SET);
	// obtain profile settings
	while(1) {
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get profile id size
		//printf("oid string length: 0x%08x\n", length);
		while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		loid[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "LOID: %s", loid);
		if(strcmp(loid, END_OID) == 0)
			break;

		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get length
		PRINTL(DBG_LVL_CLIDBG, "length: %d", length);
		while (((bytesread = read(fh, pbuffer, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		pbuffer[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "value: %s", pbuffer);

		ret = set_port_mib(portid, loid, pbuffer);
		if(ret == DSL_ERR_OK)
			PRINTL(DBG_LVL_CLI, "MIB value has set.");
		else
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
	}

end_sppid:
	close(fh);

	return ret;
}


/**
 * set_package_profile_by_name applys profile into DMT from Firmware package by profile name
 * \param portid
 * 		port ID number
 * \param prefix
 * 		prefix for identifying profile type
 * \param profile_name
 * 		profile name to be applied, profile name must be obainted by get_package_profile_name first
 *
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret set_package_profile_by_name(mt_uint8 portid, mt_uint8 *prefix, mt_uint8 *profile_name)
{
	//read/write
	int fh;
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 loid[128];
	mt_uint8 pname[PNAMETMP];
	mt_uint8 tmp[DMT_WORD];
	mt_uint32 i;
	mt_uint32 length, offset;
	mt_uint32 start, count, bytesread;

	fh = open(PACKAGE_PATH, O_RDONLY);

	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "firmware package file \"%s\" was not found.", PACKAGE_PATH);
		return DSL_ERR_NOTOK;
	}

	sprintf(pname, "%s%s", prefix, profile_name);

	// obtain profile count
	lseek(fh, PROFILE_COUNT, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get profile count
	//printf("profile count: %d\n", count);

	// obtain profile start
	lseek(fh, PROFILE_START, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	offset = GET_DMT_WORD(tmp);	// get start offset

	for(i=0;i<count;i++) {
		// obtain profile starting offset
		lseek(fh, offset+(i*PROFILE_HDR_LEN), SEEK_SET);
		//printf("profile start offset: %d\n", start+(i*PROFILE_HDR_LEN));
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		start = GET_DMT_WORD(tmp);	// get start address
		//printf("profile start: 0x%08x\n", start);
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get profile name string length
		//printf("profile name string length: 0x%08x\n", length);
		while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		loid[length] = 0;	// add NULL terminator in the string end
		//printf("profile name: %s\n", loid);
		if(strcmp(pname, loid) == 0) {
			//printf("profile hit!!!\n");
			//printf("profile name: %s\n", loid);
			break;
		}

		if(i+1 == count) {
			PRINTL(DBG_LVL_ALWAYS, "No such Profile!!");
			ret = DSL_ERR_NOTOK;
			goto end_sppname;
		}
	}

	// change file descriptor to profile starting offset
	lseek(fh, start, SEEK_SET);
	// obtain profile settings
	while(1) {
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get profile id size
		//printf("oid string length: 0x%08x\n", length);
		while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		loid[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "LOID: %s", loid);
		if(strcmp(loid, END_OID) == 0)
			break;

		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get length
		PRINTL(DBG_LVL_CLIDBG, "length: %d", length);
		while (((bytesread = read(fh, pbuffer, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		pbuffer[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "value: %s", pbuffer);

		ret = set_port_mib(portid, loid, pbuffer);
		if(ret == DSL_ERR_OK)
			PRINTL(DBG_LVL_CLI, "MIB value has set.");
		else
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting MIB value.");
	}

end_sppname:
	close(fh);

	return ret;
}


/**
 * get_package_profile_oid_value gets specified OID value from Firmware package profile and return to user
 * \param prefix
 * 		prefix for identifying profile type
 * \param profile_name
 * 		profile name to be used, profile name must be obainted by get_package_profile_name first
 * \param oid_str
 * 		OID in string format, for example: "0.1.2.3.4"
 * \param value_str
 * 		buffer to receive the returned OID value, value is in hex-string format, for example: "123def"
 *
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret get_package_profile_oid_value(mt_uint8 *prefix, mt_uint8 *profile_name, mt_uint8 *oid_str, mt_uint8 *value_str)
{
	//read/write
	int fh;
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 loid[128];
	mt_uint8 pname[PNAMETMP];
	mt_uint8 tmp[DMT_WORD];
	mt_uint32 i;
	mt_uint32 length, offset;
	mt_uint32 start, count, bytesread;

	fh = open(PACKAGE_PATH, O_RDONLY);

	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "firmware package file \"%s\" was not found.", PACKAGE_PATH);
		return DSL_ERR_NOTOK;
	}

	sprintf(pname, "%s%s", prefix, profile_name);

	// obtain profile count
	lseek(fh, PROFILE_COUNT, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get profile count
	//printf("profile count: %d\n", count);

	// obtain profile start
	lseek(fh, PROFILE_START, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	offset = GET_DMT_WORD(tmp);	// get start offset

	for(i=0;i<count;i++) {
		// obtain profile starting offset
		lseek(fh, offset+(i*PROFILE_HDR_LEN), SEEK_SET);
		//printf("profile start offset: %d\n", start+(i*PROFILE_HDR_LEN));
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		start = GET_DMT_WORD(tmp);	// get start address
		//printf("profile start: 0x%08x\n", start);
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get profile name string length
		//printf("profile name string length: 0x%08x\n", length);
		while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		loid[length] = 0;	// add NULL terminator in the string end
		//printf("profile name: %s\n", loid);
		if(strcmp(pname, loid) == 0) {
			//printf("profile hit!!!\n");
			//printf("profile name: %s\n", loid);
			break;
		}

		if(i+1 == count) {
			PRINTL(DBG_LVL_ALWAYS, "No such Profile!!");
			ret = DSL_ERR_NOTOK;
			goto end_gppo;
		}
	}

	// change file descriptor to profile starting offset
	lseek(fh, start, SEEK_SET);
	// obtain profile settings
	while(1) {
		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get profile id size
		//printf("oid string length: 0x%08x\n", length);
		while (((bytesread = read(fh, loid, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		loid[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "LOID: %s", loid);

		if(strcmp(loid, END_OID) == 0) {
			PRINTL(DBG_LVL_ALWAYS, "No such OID!!");
			value_str[0] = 0;
			ret = DSL_ERR_NOTOK;
			break;
		}

		while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		length = GET_DMT_WORD(tmp);	// get length
		PRINTL(DBG_LVL_CLIDBG, "length: %d", length);
		while (((bytesread = read(fh, pbuffer, length)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
		pbuffer[length] = 0;	// add NULL terminator in the string end
		PRINTL(DBG_LVL_CLIDBG, "value: %s", pbuffer);

		if(strcmp(loid, oid_str) == 0) {
			strcpy(value_str, pbuffer);
			break;
		}
	}

end_gppo:
	close(fh);

	return ret;
}


//#if defined (SPI_ACCESS)
/**
 * Flash DMT Firmware to SPI flash memory
 */
mt_ret test_flash(unsigned char * pbuf, cmdt * pcmd)
{
	//read/write
	int kk;
	int fh;
	mt_ret ret;
	mt_uint8 buf[BLKSIZE];
	mt_uint8 fname[64];
	mt_uint32 offset = 0;	// number of offset to skip from the beginning
	mt_uint32 bytesread;

	ret = sscanf(pbuf, "%s %i", fname, &offset);
	if( ret != 1 && ret != 2 ){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <firmware_path> <offset>");

		return -DSL_ERR_NOPASS;
	}

	PRINTL(DBG_LVL_ALWAYS, "Flashing firmware %s...", fname);

	fh = open(fname, O_RDONLY);
	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS, "firmware file \"%s\" was not found!!",fname);
		return DSL_ERR_NOTOK;
	}

	PRINTL(DBG_LVL_ALWAYS, "Erasing Flash Memory...");

	ret = flash_firmware(MT_FIRMWARE_START, 0, NULL);
	if(ret != DSL_ERR_OK){                       /* real error on tofd */
		close(fh);
		return ret;
	}

	PRINTL(DBG_LVL_ALWAYS, "\nStart Flashing Firmware...");

	if(offset != 0) lseek(fh, offset, SEEK_SET);

	while(1) {
		while (((bytesread = read(fh, buf, BLKSIZE)) == -1) && (errno == EINTR) ) ;         /* handle interruption by signal */

		if (bytesread <= 0)          /* real error or end-of-file on fromfd */
			break;

		ret = flash_firmware(MT_FIRMWARE_CONT, bytesread, (mt_uint8 *)buf);

		if(ret != DSL_ERR_OK){                       /* real error on tofd */
			close(fh);
			PRINTL(DBG_LVL_ALWAYS, "\nFlashing Firmware Failed!!");
			return ret;
		}

		//PRINTL(DBG_LVL_ALWAYS, ".");
	}

	ret = flash_firmware(MT_FIRMWARE_END, 0, (mt_uint8 *)buf);

	//delay for waiting DMT boot up
	sleep(2);

	if (ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "\nFlashing Firmware Complete.");

	close(fh);

	return ret;
}
//#endif //defined (SPI_ACCESS)


/**
 * Download Firmware from File
 */
mt_ret test_df(unsigned char * pbuf, cmdt * pcmd)
{
	//read/write
	int fh;
	mt_ret ret;
	mt_uint8 buf[BLKSIZE];
	mt_uint8 fname[64];
	mt_uint32 i;
	mt_uint32 port;
	mt_uint32 portid;
	mt_uint32 bytesread;

	ret = sscanf(pbuf, "%i %s", &portid, fname);
	if( ret != 2 ){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> <firmware_path>");

		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	//PRINTL(DBG_LVL_ALWAYS, "portid=%i,fname=\"%s\"", portid, fname);

	PRINTL(DBG_LVL_ALWAYS, "Downloading firmware %s for port %i...", fname, portid+1);

	fh = open(fname, O_RDONLY);
	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS, "firmware file \"%s\" was not found!!",fname);
		return DSL_ERR_NOTOK;
	}

	ret = download_firmware(portid, MT_FIRMWARE_START, 0, NULL);
	if(ret != DSL_ERR_OK){                       /* real error on tofd */
		close(fh);
		return ret;
	}

	PRINTL(DBG_LVL_ALWAYS, "Start Downloading Firmware...");

	while(1) {
		while (((bytesread = read(fh, buf, BLKSIZE)) == -1) && (errno == EINTR) ) ;         /* handle interruption by signal */

		if (bytesread <= 0)          /* real error or end-of-file on fromfd */
			break;

		ret = download_firmware(portid, MT_FIRMWARE_CONT, bytesread, (mt_uint8 *)buf);

		if(ret != DSL_ERR_OK){                       /* real error on tofd */
			close(fh);
			PRINTL(DBG_LVL_ALWAYS, "\nDownloading Firmware Failed!!");
			return ret;
		}

		//PRINTL(DBG_LVL_ALWAYS, ".");
	}

	ret = download_firmware(portid, MT_FIRMWARE_END, 0, (mt_uint8 *)buf);

	if (ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "\nDownloading Firmware Complete.");

	close(fh);

	//delay for gap of two commands to ensure following MIB can be set correctly
	for(i=0;i<10000000;i++);

	return ret;
}


/**
 * download_package_firmware download Firmware for DMT and xDSL binary driver for xDSL device driver from Firmware Package File
 * \param portid
 * 		port ID number
 * \param firmwareid
 * 		id of firmware to be downloaded
 *
 * \return
 * 	- 0 : success
 * 	- not 0 : error code
 */
mt_ret download_package_firmware(mt_uint8 portid, mt_uint16 firmwareid)
{
	int fh;
	FILE *streamout;
	mt_ret ret;
	mt_uint8 buf[BLKSIZE];
	mt_uint8 command[64];
	mt_uint8 line[256];
	mt_uint8 tmp[DMT_WORD];
	mt_uint32 i, bytesread;
	mt_uint32 start, count;

	fh = open(PACKAGE_PATH, O_RDONLY);

	if( fh == -1){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "firmware package file \"%s\" was not found.", PACKAGE_PATH);
		return DSL_ERR_NOTOK;
	}

	// obtain firmware count
	lseek(fh, FW_COUNT, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get firmware dip count
	//printf("firmware count: %d\n", count);

	// firmware id out of range, skip downloading
	if(firmwareid == 0 || firmwareid > count) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "No such Firmware ID: %d", firmwareid);
		ret = DSL_ERR_NOTOK;
		goto end_dpf;
	}

	PRINTL(DBG_LVL_ALWAYS, "Downloading firmware for port %i..", portid+1);

	// obtain firmware starting offset
	lseek(fh, FW_START_OFFSET+((firmwareid - 1)*DMT_WORD), SEEK_SET);
	//printf("firmware start offset: %d\n", FW_START_OFFSET+((firmwareid - 1)*DMT_WORD));
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	start = GET_DMT_WORD(tmp);	// get start address
	//printf("firmware start: 0x%08x\n", start);
	// obtain firmware size
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get firmware size
	count -= start;	// use difference of two firmware start to calculate firmware size
	PRINTL(DBG_LVL_ALWAYS, "Firmware size is %d bytes", count);

	// change file descriptor to firmware starting offset
	lseek(fh, start, SEEK_SET);

	ret = download_firmware(portid, MT_FIRMWARE_START, firmwareid, NULL);
	if(ret != DSL_ERR_OK){                       // real error on tofd
		goto end_dpf;
	}
	PRINTL(DBG_LVL_ALWAYS, "Start Downloading Firmware...");

	while(1){
		if (count < BLKSIZE){
			while (((bytesread = read(fh, buf, count)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			count = 0;
		}
		else{
			while (((bytesread = read(fh, buf, BLKSIZE)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			count -= BLKSIZE;
		}

		ret = download_firmware(portid, MT_FIRMWARE_CONT, bytesread, (mt_uint8 *)buf);

		if(ret != DSL_ERR_OK){                       // real error on tofd
			PRINTL(DBG_LVL_ALWAYS, "\nDownloading Firmware Failed!!");
			goto end_dpf;
		}

		//PRINTL(DBG_LVL_ALWAYS, ".");

		if (!count)          // real error or end-of-file on fromfd
			break;
	}

	ret = download_firmware(portid, MT_FIRMWARE_END, 0, NULL);

	if (ret == DSL_ERR_OK)
		PRINTL(DBG_LVL_ALWAYS, "\nDownloading Firmware Complete.");

	/*if(portid == MT_PORT_BroadCast) {
		for(i=0;i<MAX_PORTS;i++) profile_status[i] = 0;
	}
	else {
		profile_status[portid] = 0;
	}*/

	//reset disabled port by board.conf
	for(i=1;i<=MAX_PORTS;i++) {
		sprintf(command, ""CFGMGR_PATH" -r V_POWEROFF_P%d", i);
		if((streamout = popen(command, "r")) != NULL) {
			if(fgets(line, 256, streamout) != NULL) {
				stripwhite(line);
				if(strcmp("\n", line) == 0) {
					printf("Port%d Power Off!!\n", i);
					portid = i - 1;
					ret = reset_chip_low(portid);
					if(ret != DSL_ERR_OK)
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i reset low failed.", portid);
				}
			}
			pclose(streamout);
		}
	}

	//delay for gap of two commands to ensure following MIB can be set correctly
	sleep(1);

	// loading firmware binary driver into vdsl.ko by ioctl
	// obtain firmware binary driver starting offset
	lseek(fh, FW_BINDRV_START_OFFSET, SEEK_SET);
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	start = GET_DMT_WORD(tmp);	// get firmware size
	//printf("firmware binary driver start: 0x%08x\n", start);
	// obtain firmware binary driver size
	while (((bytesread = read(fh, tmp, DMT_WORD)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
	count = GET_DMT_WORD(tmp);	// get firmware size
	//printf("firmware binary driver size: %d bytes", count);

	if(count == 0) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "xDSL binary driver does not exist!!");
		goto end_dpf;
	}

	// change file descriptor to firmware binary driver starting offset
	lseek(fh, start, SEEK_SET);

	ret = load_firmware_bindrv(MT_FIRMWARE_START, 0, NULL);
	if(ret != DSL_ERR_OK) {                       // real error on tofd
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "xDSL binary driver loading failed!!");
		goto end_dpf;
	}

	while(1) {
		if (count < BLKSIZE){
			while (((bytesread = read(fh, buf, count)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			count = 0;
		}
		else{
			while (((bytesread = read(fh, buf, BLKSIZE)) == -1) && (errno == EINTR) ) ;         // handle interruption by signal
			count -= BLKSIZE;
		}

		ret = load_firmware_bindrv(MT_FIRMWARE_CONT, bytesread, (mt_uint8 *)buf);

		if(ret != DSL_ERR_OK){                       // real error on tofd
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "xDSL binary driver loading failed!!");
			goto end_dpf;
		}

		if (!count)          // real error or end-of-file on fromfd
			break;
	}

	ret = load_firmware_bindrv(MT_FIRMWARE_END, 0, NULL);

end_dpf:
	close(fh);

	return ret;
}


/**
 * Show ADSL VPI/VCI VLAN mapping
 * \par Usage:
 * 		cmd <pvc>
 * \param pvc: Permanent Virtual Circuit to show
 * \return
 *
 */
mt_ret parse_pvc_rule(void)
{
	mt_ret ret = DSL_ERR_OK;
	mt_uint8 i;
	mt_uint8 *token;
	FILE *streamout;
	char line[64], command[256];

	for(i=0;i<NUM_OF_PVC;i++) {
		sprintf(command, ""CFGMGR_PATH" -r ADSL_PVC%d", i);
		if((streamout = popen(command, "r")) != NULL) {
			if(fgets(line, 64, streamout) != NULL) {
				stripwhite(line);
			}
			pclose(streamout);
		}

		token = strtok(line, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].vpi = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].vci = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].vlanid = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].prio = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].encap = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].atmqos = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].pcr = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].scr = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].mbs = strtoul(token, NULL, 16);

		token = strtok(NULL, "@");
		if(token == NULL) return DSL_ERR_NOTOK;
		atm_pvc[i].active = strtoul(token, NULL, 16);

		//printf("pvc%d: %d %d %d %d %d %d %d %d %d %d\n", i, atm_pvc[i].vpi, atm_pvc[i].vci, atm_pvc[i].vlanid, atm_pvc[i].prio, atm_pvc[i].encap, atm_pvc[i].atmqos, atm_pvc[i].pcr, atm_pvc[i].scr, atm_pvc[i].mbs, atm_pvc[i].active);
	}

	return ret;
}


/**
 * Download Firmware and Profile from Firmware Package File
 */
mt_ret test_package(unsigned char * pbuf, cmdt * pcmd)
{
	//read/write
	mt_ret ret;
	mt_uint32 portid, firmwareid, profileid = 0xff;

	ret = sscanf(pbuf, "%i %i %i", &portid, &firmwareid, &profileid);
	if( ret != 2 && ret != 3 ){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <portid> <firmwareid> [<profileid>]");
		PRINTL(DBG_LVL_ALWAYS, "       [portid: 1 - %d]", MAX_PORTS);
		PRINTL(DBG_LVL_ALWAYS, "       [firmwareid: OT=1, RT=2, 0 to skip]");
		PRINTL(DBG_LVL_ALWAYS, "       [profileid: 1 - 8, 0 to skip]");

		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	//PRINTL(DBG_LVL_ALWAYS, "firmwareid=%i,portid=%i", firmwareid, portid);

	// user didn't want to skip downloading firmware
	if(firmwareid != 0) {
		ret = download_package_firmware(portid, firmwareid);
	}

	// user didn't want to skip setting profile
	if(profileid != 0) {
		// apply profile settings to DMT
		ret = set_package_profile_by_id(portid, profileid);

		// apply ATM PVC settings to DMT
		ret = set_port_atm_pvc(portid);
	}

	return ret;
}


/**
 * Show running firmware version and OT/RT indicatior
 */
int show_firmver(unsigned char * pbuf, cmdt * pcmd)
{
	char buf[MAX_BUFFER];
	char *ptr;
	int lvlold;
	int value1 = -1;
	mt_ret ret;
	mt_uint8 command[64];
	mt_uint8 line[256];
	mt_uint8 phy_status;
	mt_uint32 portid;
	FILE *streamout;

	ret = sscanf(pbuf, "%i", &portid);

	if( ret != 1 )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage:cmd <portid>");
		return -DSL_ERR_NOPASS;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		return DSL_ERR_NOTOK;
	}

	get_debug_level(&lvlold);
	if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
	else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
	else set_debug_level(DBG_LVL_NONE);

	bzero(buf, sizeof(buf));
	get_version(buf);
	PRINTL(DBG_LVL_ALWAYS, "xDSLDriverVersion=%s - %s", strtok(buf, ";"), strtok(NULL, ";"));
	PRINTL(DBG_LVL_XML, "xDSLDriverVersion %s|", buf);

	ret = get_port_mib(portid, xdslFirmwareVer, pbuffer);
	if(ret == DSL_ERR_OK) {
		value1 = strtoul(pbuffer+2, NULL, 16);
		PRINTL(DBG_LVL_ALWAYS, "xDSLFirmwareVersion=%06x", value1);
		PRINTL(DBG_LVL_XML, "xDSLFirmwareVersion %06x|", value1);
	}
	else {
		sprintf(command, ""CFGMGR_PATH" -r FW_VERSION");
		if((streamout = popen(command, "r")) != NULL) {
			if(fgets(line, 256, streamout) != NULL) {
				stripwhite(line);
			}
			pclose(streamout);
		}
		PRINTL(DBG_LVL_ALWAYS, "xDSLFirmwareVersion=%s", line);
		PRINTL(DBG_LVL_XML, "xDSLFirmwareVersion %s|", line);
	}

	ret = get_port_mib(portid, xdslFirmwareDate, pbuffer);
	value1 = strtoul(pbuffer+2, NULL, 16);
	//PRINTL(DBG_LVL_XML, "%06x ",value1);
	PRINTL(DBG_LVL_ALWAYS, "xDSLFirmwareDate=%06x", value1);

	ret = get_port_mib(portid, xdslFirmwareTime, pbuffer);
	value1 = strtoul(pbuffer+2, NULL, 16);
	//PRINTL(DBG_LVL_XML, "%06x", value1);
	PRINTL(DBG_LVL_ALWAYS, "xDSLFirmwareTime=%06x", value1);

	//ret = get_port_mib(portid, xdslBoardType, pbuffer);
	//value1 = strtoul(pbuffer+2, NULL, 16);
	//PRINTL(DBG_LVL_ALWAYS, "BoardType=%06x",v alue1);

	ret = get_port_mib(portid, xdslTerminal, pbuffer);
	value1 = strtoul(pbuffer+2, NULL, 16);
	//PRINTL(DBG_LVL_ALWAYS, "Terminal=%06x\nBit0(1-RT,0-OT)\nBit1(1-Xenos/SMII,0-Award/MII)\nBit2(1-HDLC,0-NOHDLC)\nBit3(1-NeedConnect,0-NoNeedConnect)", value1);

	bzero(buf, sizeof(buf));
	if( value1 & 0x00000001 ) {
		strcat(buf, "xDSLFirmwareMode=RT");
	}
	else {
		strcat(buf, "xDSLFirmwareMode=OT");
	}
	/*if( value1 & 0x00000002 ) {
		strcat(buf, " XENOS/SMII,");
	}
	else {
		strcat(buf, " AWARD/MII,");
	}
	if( value1 & 0x00000004 ) {
		strcat(buf, " Have HDLC Framming,");
	}
	else {
		strcat(buf, " No HDCL Framming,");
	}
	if( value1 & 0x00000008 ) {
		strcat(buf, " Need Connect Command,");
	}
	else {
		strcat(buf, " NoNeed Connect Command,");
	}
	if( value1 & 0x00000010 ) {
		strcat(buf, " Pacing Signal Enabled");
	}
	else {
		strcat(buf, " Pacing Signal Disabled");
	}*/
	PRINTL(DBG_LVL_ALWAYS, "%s", buf);

	set_debug_level(lvlold);

	return ret;
}


/**
 * Test Hardware Settings
 */
mt_ret test_board_set(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint32 id;
	mt_uint32 portid;
	mt_uint32 count;
	mt_uint32 action;
	mt_uint32 port, cond;
	int i, j;
	int lvlold;

	count = 0;

	ret = sscanf(pbuf, "%i %i %i", &id, &portid, &count);

	if( ret != 1 && ret != 2 && ret != 3 ){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> [<portid> [<count>]], port id start from 1");
		PRINTL(DBG_LVL_ALWAYS, "  1: Random Connect/Disconnect Test");
		PRINTL(DBG_LVL_ALWAYS, "  2: Temporary CRC Test");
		PRINTL(DBG_LVL_ALWAYS, "  3: ResetPort Always High");
		PRINTL(DBG_LVL_ALWAYS, "  4: ResetPort Always Low");
		PRINTL(DBG_LVL_ALWAYS, "  5: ResetPort");
		return DSL_ERR_OK;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));
	PRINTL(DBG_LVL_ALWAYS, "test id=%i,portid=%i,count=%i", id, portid, count);

	switch(id)
	{
		case 1:
			portid++;

			get_debug_level(&lvlold);
			if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
			else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
			else set_debug_level(DBG_LVL_NONE);

			for(i=0;i<count;i++) {
				while(!(MT_AVAILABLE_PORT & (1<<(port=random()%portid))));	//make sure the random port is available

				action=random()%2;
				if(action) {
					PRINTL(DBG_LVL_ALWAYS, "connect_port() on Port%i", port+1);
					ret = connect_port(port);
				}
				else {
					PRINTL(DBG_LVL_ALWAYS, "disconnect_port() on Port%i", port+1);
					ret = disconnect_port(port);
				}

				sleep(10);
			}

			set_debug_level(lvlold);
			break;

		case 2:
			get_debug_level(&lvlold);
			if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
			else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
			else set_debug_level(DBG_LVL_NONE);

			for(i=0;i<count;i++) {
				ret = execute("vdsl init 2 1", commands);
				sleep(50);
				ret = execute("vdsl show linerate 1", commands);
				ret = execute("vdsl show firmver 1", commands);

				for(j=0;j<900;j++) sleep(1);

				ret = execute("vdsl show linerate 1", commands);
				ret = execute("vdsl show firmver 1", commands);
			}

			set_debug_level(lvlold);
			break;

		case 3:
			ret = reset_chip_high(portid);
			if(ret != DSL_ERR_OK){
				if(portid == MT_PORT_BroadCast)
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d reset high failed.", MT_PORT_BroadCast);
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i reset high failed.", portid+1);

				return ret;
			}
			else{
				if(portid == MT_PORT_BroadCast) {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d always high.", MT_PORT_BroadCast);
				}
				else {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i always high.", portid+1);
				}
			}
			break;

		case 4:
			ret = reset_chip_low(portid);
			if(ret != DSL_ERR_OK) {
				if(portid == MT_PORT_BroadCast)
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d reset low failed.", MT_PORT_BroadCast);
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i reset low failed.", portid+1);

				return ret;
			}
			else {
				if(portid == MT_PORT_BroadCast) {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d always low.", MT_PORT_BroadCast);
				}
				else {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i always low.", portid+1);
				}
			}
			break;

		case 5:
			ret = reset_chip(portid);
			if(ret != DSL_ERR_OK) {
				if(portid == MT_PORT_BroadCast)
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d reset failed.", MT_PORT_BroadCast);
				else
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i reset failed.", portid+1);

				return ret;
			}
			else {
				if(portid == MT_PORT_BroadCast) {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port 1-%d have reset.", MT_PORT_BroadCast);
				}
				else {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Port %i has reset.", portid+1);
				}
			}
			break;

		default:
			break;
	}

	return ret;
}


/**
 * Test Port Operations
 */
mt_ret test_portop(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 command[64];
	mt_uint8 line[256];
	mt_uint8 state, port_mode;
	mt_uint32 id;
	mt_uint32 portid;
	mt_uint32 port, group, table, idx, param;
	mt_uint32 cond;
	mt_uint32 i, value, value1, value2;
	cmdt *cmdtmp;
	FILE *streamout;
	int lvlold;
	int vdslfd;

	ret = sscanf(pbuf, "%i %s", &id, portbuf);

	if( ret != 2 ) {
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> <portid>, port id start from 1");
		PRINTL(DBG_LVL_ALWAYS, "  1: Enable Port");
		PRINTL(DBG_LVL_ALWAYS, "  2: Disable Port");
		PRINTL(DBG_LVL_ALWAYS, "  3: Enable Pacing Signal");
		PRINTL(DBG_LVL_ALWAYS, "  4: Disable Pacing Signal");
		PRINTL(DBG_LVL_ALWAYS, "  5: Power on Port");
		PRINTL(DBG_LVL_ALWAYS, "  6: Power off Port");
#ifdef WITH_ADV
		PRINTL(DBG_LVL_ALWAYS, "  7: Init Modem");
		PRINTL(DBG_LVL_ALWAYS, "  8: Shutdown Modem(Shutdown Chip)");
		PRINTL(DBG_LVL_ALWAYS, "  9: Reset Modem");
#endif	//WITH_ADV

		return -DSL_ERR_NOPASS;
	}

	//portid--;
	//dsl_er(check_param(MT_PAR_PORT, 0, portid));
	//PRINTL(DBG_LVL_ALWAYS, "test id=%i,portid=%i", id, portid);

	ret = process_port();
	if(ret != DSL_ERR_OK) return ret;

	for(i=0;i<portcnt;i++) {
		portid = portnum[i] - 1;
		dsl_er(check_param(MT_PAR_PORT, 0, portid));

		switch(id)
		{
			case 1: //Enable Port
				if(portid == MT_PORT_BroadCast) {
					for(port=0; port<MAX_PORTS; port++) {
						sprintf(command, ""CFGMGR_PATH" -w V_DISABLE_P%d=#", port+1);
						ret = system(command);

						PRINTL(DBG_LVL_CLI, "Enabling Port%d...", port+1);
						ret = connect_port(port);
					}
				}
				else {
					sprintf(command, ""CFGMGR_PATH" -w V_DISABLE_P%d=#", portid+1);
					ret = system(command);

					PRINTL(DBG_LVL_CLI, "Enabling Port%d...", portid+1);
					ret = connect_port(portid);
				}
				break;

			case 2: //Disable Port
				if(portid == MT_PORT_BroadCast) {
					for(port=0; port<MAX_PORTS; port++) {
						sprintf(command, ""CFGMGR_PATH" -w V_DISABLE_P%d=", port+1);
						ret = system(command);

						PRINTL(DBG_LVL_CLI, "Disabling Port%d...", port+1);
						ret = disconnect_port(port);
					}
				}
				else {
					sprintf(command, ""CFGMGR_PATH" -w V_DISABLE_P%d=", portid+1);
					ret = system(command);

					PRINTL(DBG_LVL_CLI, "Disabling Port%d...", portid+1);
					ret = disconnect_port(portid);
				}
				break;

			case 3: //Enable Pacing Signal
				vdslfd=open("/dev/vdsl", O_RDWR);
				if( vdslfd == -1) {
					PRINTL(DBG_LVL_CLI,"vdsl device open failed.\n");
					return DSL_ERR_NOTOK;
				}

				ret = ioctl(vdslfd, VDSL_ENABLE_PACING, NULL);
				if(ret != DSL_ERR_OK) {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Can't enable pacing signal.");
					close(vdslfd);
					return ret;
				}

				close(vdslfd);

				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Pacing signal enabled.");
				break;

			case 4: //Disable Pacing Signal
				vdslfd=open("/dev/vdsl", O_RDWR);
				if( vdslfd == -1) {
					PRINTL(DBG_LVL_CLI,"vdsl device open failed.\n");
					return DSL_ERR_NOTOK;
				}

				ret = ioctl(vdslfd, VDSL_DISABLE_PACING, NULL);
				if(ret != DSL_ERR_OK) {
					PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Can't disable pacing signal.");
					close(vdslfd);
					return ret;
				}

				close(vdslfd);

				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Pacing signal disabled.");
				break;

			case 5: //Power on Port
				if(portid == MT_PORT_BroadCast) {
					for(port=0; port<MAX_PORTS; port++) {
						bzero(command, sizeof(command));
						cmdtmp = commands;

						sprintf(command, ""CFGMGR_PATH" -w V_POWEROFF_P%d=#", port+1);
						ret = system(command);

						sprintf(command, ""CFGMGR_PATH" -r PORT%d_MODE", port+1);
						if((streamout = popen(command, "r")) != NULL) {
							if(fgets(line, 256, streamout) != NULL) {
								stripwhite(line);
							}
							pclose(streamout);
						}
						if(strcmp("CO", line) == 0) {
							port_mode = 0;	 //CO Mode
							sprintf(command, "util dpf %d 1", port+1);
						}
						else if(strcmp("CPE", line) == 0) {
							port_mode = 1;	//CPE Mode
							sprintf(command, "util dpf %d 2", port+1);
						}
						else {	//Undefined Mode
							PRINTL(DBG_LVL_ALWAYS, "Port%d Mode Missing!!\n", port);
							port_mode = 2;
							sprintf(command, "");
						}
						commands = root_cmd;
						execute_line(command, commands);

#ifdef VIA_SAMSUNG
						sprintf(command, ""CFGMGR_PATH" -r VDSL_SMII_MODE");
						if((streamout = popen(command, "r")) != NULL) {
							if(fgets(line, 256, streamout) != NULL) {
								stripwhite(line);
							}
							pclose(streamout);
						}
						sprintf(command, "interface xdsl set oid %d 2.1.30 %s", port+1, line);
						commands = root_cmd;
						execute_line(command, commands);

						sprintf(command, ""CFGMGR_PATH" -r VDSL_SMII_DELAY");
						if((streamout = popen(command, "r")) != NULL) {
							if(fgets(line, 256, streamout) != NULL) {
								stripwhite(line);
							}
							pclose(streamout);
						}
						sprintf(command, "interface xdsl set oid %d 2.1.32 %s", port+1, line);
						commands = root_cmd;
						execute_line(command, commands);
#endif	//VIA_SAMSUNG

						if(port_mode == 0) {	//CO Mode
#if defined (CO_API) && defined (TEMPLATE_PROFILE)
							if(init_profile_linklist != 2){
								//init all profile space
								profiles_main_vdslprofile_init(VDSL_PROFILE_END_ID);
							}
							sprintf(command,
									"profiles port set line-alarm-template %d %s %s",
									port+1,
									port_map_linetemplate[port],
									port_map_alarmtemplate[port]);
							commands = root_cmd;
							execute_line(command, commands);
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)
						}
						else if(port_mode == 1) {	//CPE Mode
							sprintf(command, "util portop 1 %d", port+1);
							commands = root_cmd;
							execute_line(command, commands);
						}
						else	//Undefined Mode
							PRINTL(DBG_LVL_ALWAYS, "Port%d Mode Missing!!\n", port);

						commands = cmdtmp;
					}
				}
				else {
					bzero(command, sizeof(command));
					cmdtmp = commands;

					sprintf(command, ""CFGMGR_PATH" -w V_POWEROFF_P%d=#", portid+1);
					ret = system(command);

					sprintf(command, ""CFGMGR_PATH" -r PORT%d_MODE", portid+1);
					if((streamout = popen(command, "r")) != NULL) {
						if(fgets(line, 256, streamout) != NULL) {
							stripwhite(line);
						}
						pclose(streamout);
					}
					if(strcmp("CO", line) == 0) {
						port_mode = 0;	 //CO Mode
						sprintf(command, "util dpf %d 1", portid+1);
					}
					else if(strcmp("CPE", line) == 0) {
						port_mode = 1;	//CPE Mode
						sprintf(command, "util dpf %d 2", portid+1);
					}
					else {	//Undefined Mode
						PRINTL(DBG_LVL_ALWAYS, "Port%d Mode Missing!!\n", portid);
						port_mode = 2;
						sprintf(command, "");
					}
					commands = root_cmd;
					execute_line(command, commands);

#ifdef VIA_SAMSUNG
					sprintf(command, ""CFGMGR_PATH" -r VDSL_SMII_MODE");
					if((streamout = popen(command, "r")) != NULL) {
						if(fgets(line, 256, streamout) != NULL) {
							stripwhite(line);
						}
						pclose(streamout);
					}
					sprintf(command, "interface xdsl set oid %d 2.1.30 %s", portid+1, line);
					commands = root_cmd;
					execute_line(command, commands);

					sprintf(command, ""CFGMGR_PATH" -r VDSL_SMII_DELAY");
					if((streamout = popen(command, "r")) != NULL) {
						if(fgets(line, 256, streamout) != NULL) {
							stripwhite(line);
						}
						pclose(streamout);
					}
					sprintf(command, "interface xdsl set oid %d 2.1.32 %s", portid+1, line);
					commands = root_cmd;
					execute_line(command, commands);
#endif	//VIA_SAMSUNG

					if(port_mode == 0) {	//CO Mode
#if defined (CO_API) && defined (TEMPLATE_PROFILE)
						if(init_profile_linklist != 2){
							//init all profile space
							profiles_main_vdslprofile_init(VDSL_PROFILE_END_ID);
						}
						sprintf(command,
								"profiles port set line-alarm-template %d %s %s",
								portid+1,
								port_map_linetemplate[portid],
								port_map_alarmtemplate[portid]);
						commands = root_cmd;
						execute_line(command, commands);
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)
					}
					else if(port_mode == 1) {	//CPE Mode
						sprintf(command, "util portop 1 %d", portid+1);
						commands = root_cmd;
						execute_line(command, commands);
					}
					else	//Undefined Mode
						PRINTL(DBG_LVL_ALWAYS, "Port%d Mode Missing!!\n", portid);

					commands = cmdtmp;
				}
				break;

			case 6: //Power off Port
				if(portid == MT_PORT_BroadCast) {
					for(port=0; port<MAX_PORTS; port++) {
						bzero(command, sizeof(command));
						cmdtmp = commands;

						sprintf(command, ""CFGMGR_PATH" -w V_POWEROFF_P%d=", port+1);
						ret = system(command);

						sprintf(command, "util bdset 5 %d", port+1);
						commands = root_cmd;
						execute_line(command, commands);

						commands = cmdtmp;
					}
				}
				else {
					bzero(command, sizeof(command));
					cmdtmp = commands;

					sprintf(command, ""CFGMGR_PATH" -w V_POWEROFF_P%d=", portid+1);
					ret = system(command);

					sprintf(command, "util bdset 5 %d", portid+1);
					commands = root_cmd;
					execute_line(command, commands);

					commands = cmdtmp;
				}
				break;

#ifdef WITH_ADV
			case 7: //Init Modem
				PRINTL(DBG_LVL_CLI, "Run InitModem Test\n");
				ret = init_modem(portid);
				break;

			case 8: //Shutdown Modem
				PRINTL(DBG_LVL_CLI, "Run Shutdown Modem Test\n");
				ret = shutdown_chip(portid);
				break;

			case 9: //Reset Modem
				PRINTL(DBG_LVL_CLI, "Run ResetModem Test\n");
				ret = reset_modem(portid);
				break;
#endif //WITH_ADV

			default:
				ret = DSL_ERR_OK;
				break;
		}
	}

	if(id == 2) sleep(1);	//delay for disconnect_port()

	return ret;
}


/**
 * Test PHY Status: Test GetPhyStatus 10 times to monitor status
 */
mt_ret test_phy_status(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 phy_status = MT_OPSTATE_OTHER;
	mt_uint32 portid, i;
	int lvlold;

	ret = sscanf(pbuf, "%i", &portid);

	if( ret != 1 ) {
		PRINTL(DBG_LVL_ALWAYS, "Usage:cmd <portid>");
		return DSL_ERR_OK;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	get_debug_level(&lvlold);
	if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
	else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
	else set_debug_level(DBG_LVL_NONE);

	//for(i=0;i<10;i++)
	//{
		ret = get_phy_status(portid, &phy_status, NULL);
		switch(phy_status) {
			case MT_OPSTATE_DISABLE:
				PRINTL(DBG_LVL_XML, "1");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is DISABLE");
				break;

			case MT_OPSTATE_IDLE:
				PRINTL(DBG_LVL_XML, "2");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is IDLE");
				break;

			case MT_OPSTATE_HANDSHAKE:
				PRINTL(DBG_LVL_XML, "3");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is HANDSHAKE");
				break;

			case MT_OPSTATE_TRAINING:
				PRINTL(DBG_LVL_XML, "4");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is TRAINING");
				break;

			case MT_OPSTATE_SHOWTIME:
				PRINTL(DBG_LVL_XML, "5");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is SHOWTIME");
				break;

			case MT_OPSTATE_ERROR:
				PRINTL(DBG_LVL_XML, "6");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is ERROR");
				break;

			default:
				PRINTL(DBG_LVL_XML, "7");
				PRINTL(DBG_LVL_ALWAYS, "OPSTATE is OTHER(Impossible!!)");
		}

		//delay
		//dsldelay(1,1);
	//}

	set_debug_level(lvlold);

	return ret;
}


/**
 * Clear Port Statistics: Clear Statistics of specified Port
 */
mt_ret clear_statistic(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint32 portid;

	ret = sscanf(pbuf, "%i", &portid);

	if( ret != 1 ) {
		PRINTL(DBG_LVL_ALWAYS, "Usage:cmd <portid>");
		return DSL_ERR_OK;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	ret = clear_port_stats(portid);

	return ret;
}


/**
 * Setup debug level
 */
mt_ret debug_change_level(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint32 level;

	ret = sscanf(pbuf, "%x", &level);
	if( ret != 1 )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage:cmd <debug_level> ex: cmd 0x00000020");
		return -DSL_ERR_NOPASS;
	}

	ret = set_debug_level(level);

	if(ret != DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error setting debug level!!");
	}
	else
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "set_debug_level(0x%x)",level);

	return ret;
}


#ifdef MT_DEBUG
/**
 * Test Host Port Read/Write
*/
mt_ret test_hpi(unsigned char * pbuf, cmdt * pcmd)
{
	int vdslfd;
	char ioctl_buf[64];
	int i,value;
	mt_ret ret;
	mt_uint32 id, portid;

	ret = sscanf(pbuf, "%i %i", &id, &portid);
	if( ret != 2 )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> <portid>");
		PRINTL(DBG_LVL_ALWAYS, "  1: HPI R/W Test on Specified Port");
		PRINTL(DBG_LVL_ALWAYS, "  2: HPI Write Test on Specified Port");
		PRINTL(DBG_LVL_ALWAYS, "  3: HPI Read Test on Specified Port");
		PRINTL(DBG_LVL_ALWAYS, "  4: HPI Read always 50 bytes on Specified Port");

		return -DSL_ERR_NOPASS;
	}

	PRINTL(DBG_LVL_ALWAYS, "test id=%i,portid=%i,delay=%i", id, portid);

	vdslfd=open("/dev/vdsl", O_RDWR);

	if( vdslfd == -1){
		PRINTL(DBG_LVL_CLI,"vdsl device open failed.\n");
		return DSL_ERR_NOTOK;
	}

	portid--;

	ret = DSL_ERR_OK;
	switch(id)
	{
		case 1:
			ioctl_buf[0] = portid;	//portid
			ret = ioctl(vdslfd, VDSL_HPI_TEST, ioctl_buf);
			break;

		case 2:
			ioctl_buf[0] = portid;	//portid
			ret = ioctl(vdslfd, VDSL_HPI_TEST_WT, ioctl_buf);
			break;

		case 3:
			ioctl_buf[0] = portid;	//portid
			ret = ioctl(vdslfd, VDSL_HPI_TEST_RD, ioctl_buf);
			break;

		case 4:
			ioctl_buf[0] = portid;	//portid
			ret = ioctl(vdslfd, VDSL_HPI_TEST_RD_FIXSIZE, ioctl_buf);
			break;

		default:
			break;
	}

	close(vdslfd);

	return ret;
}


/**
 * Memory Read/Write and random port random memory reading test
 */
mt_ret test_mem(unsigned char * pbuf, cmdt * pcmd)
{
	unsigned short* pBase;
	mt_ret ret;
	mt_uint8 pdata[3], temp[512];
	mt_uint8 phy_status;
	mt_uint32 i, port;
	mt_uint32 id, portid, memtype, address, count, data;
	volatile mt_uint32* p_gpio_addr;
	volatile mt_uint32* p_gpio_data = (mt_uint32*) GPIO_DATA;
	volatile mt_uint16* flash_addr;
	mt_uint8 *p;
	int lvlold;
	int NumRegion;
	FILE *fp, *fp1;
	mt_uint8 bytedata, bytesread;
	mt_uint16 fdata, romdata, active_firmware, crc_config;

	address=count=0;

	ret = sscanf(pbuf, "%i %i %x %x %x %x", &id, &portid, &memtype, &address, &count, &data);
	if( (id == 3 && ret != 3) || (id != 3 && ret != 5 && ret != 6)){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> <portid> <memtype> [<address-hex> <count-hex> [<data-hex>]]");
		PRINTL(DBG_LVL_ALWAYS, "  1: Read Memory Test. ex: ut mem 1 1 0 ff 0a");
		PRINTL(DBG_LVL_ALWAYS, "  2: Write Memory Test. ex: ut mem 2 1 0 ff 0a 1f");
		PRINTL(DBG_LVL_ALWAYS, "  3: Memory Random Reading Test. <testid> <maxport> <count-hex>.");
		PRINTL(DBG_LVL_ALWAYS, "     ex: ut mem 3 1 100000");
		PRINTL(DBG_LVL_ALWAYS, "  4: Read Award Memory Test(<memtype> will be ignored here).");
		PRINTL(DBG_LVL_ALWAYS, "     ex: ut mem 4 1 0 ff 0a");
		PRINTL(DBG_LVL_ALWAYS, "  5: Write Award Memory Test(<memtype> will be ignored here).");
		PRINTL(DBG_LVL_ALWAYS, "     ex: ut mem 5 1 0 ff 0a 1f");
		PRINTL(DBG_LVL_ALWAYS, "  6 Read Remote Memory Test. ex: ut mem 1 1 0 ff 0a");
		PRINTL(DBG_LVL_ALWAYS, "  [memtype: x=0, y=1, p=2]");

		return -DSL_ERR_NOPASS;
	}

	dsl_er(check_param(MT_PAR_PORT,0,portid-1));

/*#ifdef IOCTL_INTRFCE
	if(id==3){
		for(port=0;port<portid;port++){
			get_phy_status(port, &phy_status, NULL);
			if((MT_AVAILABLE_PORT & (1<<port)) && phy_status < MT_OPSTATE_IDLE){
				PRINTL(DBG_LVL_ALWAYS, "Please download firmware for Port%i first!!", port+1);
				return DSL_ERR_OK;
			}
		}
	}
	else{
		get_phy_status(portid-1, &phy_status, NULL);
		if(phy_status < MT_OPSTATE_IDLE){
			PRINTL(DBG_LVL_ALWAYS, "Please download firmware for Port%i first!!", portid);
			return DSL_ERR_OK;
		}
	}
#endif*/

	portid--;

	//if((id==1 || id==2) && address>MT_X_MEM_BOUND)
	if(id==2 && address>MT_X_MEM_BOUND){
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory address exceed boundary.");
		return DSL_ERR_NOTOK;
	}

	get_debug_level(&lvlold);
	if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
	else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
	else set_debug_level(DBG_LVL_NONE);

	PRINTL(DBG_LVL_ALWAYS, "test id=%i,portid=%i,memtype=%i,address=0x%06x,count=0x%06x", id, portid, memtype, address ,count);
	if(id==2) PRINTL(DBG_LVL_ALWAYS, ",data=0x%06x", data);

	ret = DSL_ERR_OK;
	switch(id){
		case 1:
			//for(i=0; address<=MT_X_MEM_BOUND && i<count; address++, i++)
			for(i=0; i<count; address++, i++){
					if(get_port_mem(portid,address,memtype,pdata)!=DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}
//yalee980715 for dump soc format
#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
					PRINTL(DBG_LVL_ALWAYS, "%06x,%02x%02x%02x", address, pdata[0], pdata[1], pdata[2]);
#else
					PRINTL(DBG_LVL_ALWAYS, "%06i == Address:%06x Data:%02x%02x%02x ==", i, address, pdata[0], pdata[1], pdata[2]);
#endif
			}

			break;

		case 2:
			pdata[0] = (data & 0xff0000)>>16;
			pdata[1] = (data & 0x00ff00)>>8;
			pdata[2] = data & 0x0000ff;

			for(i=0; address<=MT_X_MEM_BOUND && i<count; address++, i++){
					if(set_port_mem(portid,address,memtype,pdata)!=DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory writing error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					PRINTL(DBG_LVL_ALWAYS, "%06i == Address:%06x has written ==", i, address);
			}

			break;

		case 3:
			if( portid==MAX_PORTS ){
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "BroadCast port not supported!!");
				ret = DSL_ERR_NOTOK;
				break;
			}
			portid++;

			for(count=memtype,i=0;i<count;i++){
				while(!(MT_AVAILABLE_PORT & (1<<(port=random()%portid))));	//make sure the random port is available
				address=random()%0x4000;	//X, Y memory boundary is 3fff
				memtype=random()%2;	//it should be 3 here, used 2 for now is preventing from accessing P memory

					if(get_port_mem(port,address,memtype,pdata)!=DSL_ERR_OK){
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error on Port%i!!", port+1);
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(memtype==0)
						PRINTL(DBG_LVL_ALWAYS, "%06i == Port%i X Address:%06x Data:%02x%02x%02x ==", i, port+1, address, pdata[0], pdata[1], pdata[2]);
					else
						PRINTL(DBG_LVL_ALWAYS, "%06i == Port%i Y Address:%06x Data:%02x%02x%02x ==", i, port+1, address, pdata[0], pdata[1], pdata[2]);
			}

			break;

		case 4:
			p_gpio_addr=(mt_uint32*)address;

			for(i=0; i<count; p_gpio_addr++, i++)
				PRINTL(DBG_LVL_ALWAYS, "%06i == Address:%08x Data:%08x ==", i, p_gpio_addr, *p_gpio_addr);

			break;

		case 5:
			p_gpio_addr=(mt_uint32*)address;

			for(i=0; i<count; p_gpio_addr++, i++){
				*p_gpio_addr=data;
				PRINTL(DBG_LVL_ALWAYS, "%06i == Address:%08x has written ==", i, p_gpio_addr);
			}

			break;

		case 6:
			//for(i=0; address<=MT_X_MEM_BOUND && i<count; address++, i++)
			for(i=0; i<count; address++, i++){
					if(get_port_mem_remote(portid, address, memtype, pdata)!=DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					PRINTL(DBG_LVL_ALWAYS, "%06i == Address:%06x Data:%02x%02x%02x ==", i, address, pdata[0], pdata[1], pdata[2]);
			}

			break;

		default:
			break;
	}

	set_debug_level(lvlold);

	return ret;
}

/*
**	dump_soc: Dump SOC data
*/
mt_ret dump_soc(unsigned char * pbuf, cmdt * pcmd)
{
	int ret, i;
	int portid = -1;
	char cmd[64];
	mt_uint8 pdata[3];
	mt_uint8 phy_status, strvalue[BUF_SIZE];
	int fd, saved_fd;
	FILE *fp;

	ret = sscanf(pbuf, "%i", &portid);

	if( ret != 1 )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage:cmd <portid>");
		return DSL_ERR_OK;
	}

	portid--;
	dsl_er(check_param(MT_PAR_PORT, 0, portid));

	printf("dumpsoc...\n");
	printf("please wait for a while...\n");

	saved_fd = dup(STDOUT_FILENO);
	if((fp = fopen("/tmp/dumpsoc.txt", "w")) == NULL) {
		printf("fopen failed\n");
		return ret;
	}
	fd = fileno(fp);

	if (dup2(fd, STDOUT_FILENO) == -1) {
		printf("dup2 failed\n");
		return ret;
	}

	printf("Disconnect vdsl :\n");
	sprintf(cmd, "util portop 2 %d", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nclear ModemErrBuf :\n");
	sprintf(cmd, "interface xdsl set oid %d 2.1.7.1 00001a", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	sprintf(cmd, "interface xdsl set oid %d 2.1.8.1 000001", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nConnect vdsl :\n");
	sprintf(cmd, "util portop 1 %d", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nRead mib :(read 90 tims, once per second)\n\n");
	for (i = 0; i < 90; i++) {
		if(get_port_mem(portid, 0x001b, 0, pdata) != DSL_ERR_OK) {
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error!!");
			ret = DSL_ERR_NOTOK;
			break;
		}
#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
		PRINTL(DBG_LVL_ALWAYS, "MedemStatus : %02x%02x%02x", pdata[0], pdata[1], pdata[2]);
#else
		PRINTL(DBG_LVL_ALWAYS, "%06i == Address:00001b Data:%02x%02x%02x ==", i, pdata[0], pdata[1], pdata[2]);
#endif

		get_phy_status(portid, &phy_status, NULL);
		if(phy_status == MT_OPSTATE_DISABLE){
			ret = DSL_ERR_NOTOK;
		} else {
			ret = get_port_mib(portid, ModemErrBuf, pbuffer);
			if(ret == DSL_ERR_OK) {
				sprintf(strvalue, "ModemErrBuf : ");
				strcat(strvalue, pbuffer+2);
				PRINTL(DBG_LVL_ALWAYS, "%s", strvalue);
//				value = strtoul(pbuffer+2, NULL, 16);
//				PRINTL(DBG_LVL_XML, "<MIB idx=\"0\" loid=\"%s\" value=\"%d\"></MIB>", loid, value);
			}
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting MIB value.");

			printf("xdslLineDiagTrn: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.57.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

			printf("xdslLineDiagTrnD: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.58.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

			printf("xdslLineDiagTrnDD: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.59.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrn, "xdslLineDiagTrn :", NULL, 1);
//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrnD, "xdslLineDiagTrnD :", NULL, 1);
//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrnDD, "xdslLineDiagTrnDD :", NULL, 1);
		}
		printf("\n");
		sleep(1);
	}

	printf("Disconnect vdsl :\n");
	sprintf(cmd, "util portop 2 %d", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nSet CfgDisableRecovery = 0000fd...\n");
	sprintf(cmd, "interface xdsl set oid %d 2.1.143.1 0000fd", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nConnect vdsl :\n");
	sprintf(cmd, "util portop 1 %d", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nRead mib :(read 60 tims, once per second)\n");
	for (i = 0; i < 60; i++) {
		if(get_port_mem(portid, 0x001b, 0, pdata) != DSL_ERR_OK) {
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error!!");
			ret = DSL_ERR_NOTOK;
			break;
		}
#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
		PRINTL(DBG_LVL_ALWAYS, "MedemStatus : %02x%02x%02x", pdata[0], pdata[1], pdata[2]);
#else
		PRINTL(DBG_LVL_ALWAYS, "%06i == Address:00001b Data:%02x%02x%02x ==", i, pdata[0], pdata[1], pdata[2]);
#endif

		get_phy_status(portid, &phy_status, NULL);
		if(phy_status == MT_OPSTATE_DISABLE){
			ret = DSL_ERR_NOTOK;
		} else {
			ret = get_port_mib(portid, ModemErrBuf, pbuffer);
			if(ret == DSL_ERR_OK) {
				sprintf(strvalue, "ModemErrBuf : ");
				strcat(strvalue, pbuffer+2);
				PRINTL(DBG_LVL_ALWAYS, "%s", strvalue);
//				value = strtoul(pbuffer+2, NULL, 16);
//				PRINTL(DBG_LVL_XML, "<MIB idx=\"0\" loid=\"%s\" value=\"%d\"></MIB>", loid, value);
			}
			else
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting MIB value.");

			printf("xdslLineDiagTrn: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.57.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

			printf("xdslLineDiagTrnD: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.58.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

			printf("xdslLineDiagTrnDD: ");
			sprintf(cmd, "interface xdsl show oid %d 2.1.59.1", portid+1);
			commands = root_cmd;
			execute_line(cmd, commands);

//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrn, "xdslLineDiagTrn :", NULL, 1);
//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrnD, "xdslLineDiagTrnD :", NULL, 1);
//			print_mib_uint32(portid, "%s %s", xdslLineDiagTrnDD, "xdslLineDiagTrnDD :", NULL, 1);
		}
		printf("\n");
		sleep(1);
	}

	printf("Dump Soc (need 10~15 seconds) :\n");
	sprintf(cmd, "util mem 1 %d 0 4c00 000800", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	get_phy_status(portid, &phy_status, NULL);
	if(phy_status == MT_OPSTATE_DISABLE){
		ret = DSL_ERR_NOTOK;
	} else {
//		print_mib_uint32(portid, "%s %s", xdslGhsOtVendorId, "xdslGhsOtVendorId :", NULL, 1);
		printf("\nxdslGhsOtVendorId(MIB2 value):\n");
		sprintf(cmd, "interface xdsl show oid %d 1.2.10.1", portid+1);
		commands = root_cmd;
		execute_line(cmd, commands);
		printf("\n");
		// version number is more than 6bits.
		ret = get_port_mib(portid, xdsl2LInvVersionNumber, pbuffer);
		if(ret == DSL_ERR_OK) {
			sprintf(strvalue, "xdsl2LInvVersionNumber : ");
			strcat(strvalue, pbuffer+2);
			PRINTL(DBG_LVL_ALWAYS, "%s", strvalue);
//			value = strtoul(pbuffer+2, NULL, 16);
//			PRINTL(DBG_LVL_XML, "<MIB idx=\"0\" loid=\"%s\" value=\"%d\"></MIB>", loid, value);
		}
		else
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Error getting MIB value.");
		print_mib_uint32(portid, "%s %s", xdsl2LineStatusAttainableRateUs, "xdsl2LineStatusAttainableRateUs :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2LineStatusAttainableRateDs, "xdsl2LineStatusAttainableRateDs :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActDataRate, "xdsl2ChStatusActDataRate0 :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActDataRate, "xdsl2ChStatusActDataRate3 :", NULL, 3);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActDelay, "xdsl2ChStatusActDelay0 :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActDelay, "xdsl2ChStatusActDelay3 :", NULL, 3);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActInp, "xdsl2ChStatusActInp0 :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2ChStatusActInp, "xdsl2ChStatusActInp3 :", NULL, 3);
		print_mib_uint32(portid, "%s %s", xdsl2LineStatusElectricalLength, "xdsl2LineStatusElectricalLength :", NULL, 1);
		print_mib_uint32(portid, "%s %s", xdsl2LineStatusActProfile, "xdsl2LineStatusActProfile :", NULL, 1);
	}

	printf("\nfirmware version :\n");
	sprintf(cmd, "util firmver %d", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	printf("\nSet CfgDisableRecovery = 000000...\n");
	sprintf(cmd, "interface xdsl set oid %d 2.1.143.1 000000", portid+1);
	commands = root_cmd;
	execute_line(cmd, commands);

	sleep(1);

	printf("\nRead PM count :(read 90 tims, once per second)\n");
	for (i = 0; i < 90; i++) {
		if(get_port_mem(portid, 0x001b, 0, pdata) != DSL_ERR_OK) {
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Memory reading error!!");
			ret = DSL_ERR_NOTOK;
			break;
		}
#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
		PRINTL(DBG_LVL_ALWAYS, "MedemStatus : %02x%02x%02x", pdata[0], pdata[1], pdata[2]);
#else
		PRINTL(DBG_LVL_ALWAYS, "%06i == Address:00001b Data:%02x%02x%02x ==", i, pdata[0], pdata[1], pdata[2]);
#endif

		if(phy_status == MT_OPSTATE_DISABLE){
			ret = DSL_ERR_NOTOK;
		} else {
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin1 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin2 :", NULL, 3);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin3 :", NULL, 4);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin3 :", NULL, 5);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin5 :", NULL, 6);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin6 :", NULL, 7);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin7 :", NULL, 8);
			print_mib_uint32(portid, "%s %s", xdsl2LineBandStatusSnrMargin, "xdsl2LineBandStatusSnrMargin3 :", NULL, 9);
			print_mib_uint32(portid, "%s %s", xdslMetanoiaOtPhysCurrStatus, "xdslMetanoiaOtPhysCurrStatus :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdslMetanoiaOtPhysCurrStatus, "xdslMetanoiaRtPhysCurrStatus :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrCorrectedBlocks0, "xdslMetanoiaOtCurrCorrectedBlocks :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrCorrectedBlocks0, "xdslMetanoiaRtCurrCorrectedBlocks :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrCodingViolation0, "xdslMetanoiaOtCurrCodingViolations :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrCodingViolation0, "xdslMetanoiaRtCurrCodingViolations :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrFecs0, "xdslMetanoiaOtCurrFecs :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrFecs0, "xdslMetanoiaRtCurrFecs :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrEs0, "xdslMetanoiaOtCurrEs :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrEs0, "xdslMetanoiaRtCurrEs :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrSes0, "xdslMetanoiaOtCurrSes :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrSes0, "xdslMetanoiaRtCurrSes :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrLoss0, "xdslMetanoiaOtCurrLoss :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrLoss0, "xdslMetanoiaRtCurrLoss :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrUas0, "xdslMetanoiaOtCurrUas :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCurrUas0, "xdslMetanoiaRtCurrUas :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCorrectedBlocks0, "xdslMetanoiaOtCorrectedBlocks0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCorrectedBlocks0, "xdslMetanoiaRtCorrectedBlocks0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCodingViolations0, "xdslMetanoiaOtCodingViolations0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtCodingViolations0, "xdslMetanoiaRtCodingViolations0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtFecs0, "xdslMetanoiaOtFecs0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtFecs0, "xdslMetanoiaRtFecs0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtEs0, "xdslMetanoiaOtEs0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtEs0, "xdslMetanoiaRtEs0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtSes0, "xdslMetanoiaOtSes0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtSes0, "xdslMetanoiaRtSes0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtLoss0, "xdslMetanoiaOtLoss0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtLoss0, "xdslMetanoiaRtLoss0 :", NULL, 2);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtUas0, "xdslMetanoiaOtUas0 :", NULL, 1);
			print_mib_uint32(portid, "%s %s", xdsl2MetanoiaOtUas0, "xdslMetanoiaRtUas0 :", NULL, 2);
		}
		printf("\n");
		sleep(1);
	}

	printf("\n");

	fclose(fp);
	fflush(stdout);
	dup2(saved_fd, STDOUT_FILENO);

	printf("\nDump soc successfully.\n");

	return ret;
}

#ifdef DEBUG_API
/**
 * Dump memory content
 */
mt_ret debug_dump_mem(unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	mt_uint8 pdata[3];
	mt_uint32 id, portid, memtype, address, count, i;
	int lvlold;
	FILE *fdump, *fdump2;

	address = count = 0;

	ret = sscanf(pbuf, "%i %i %i %i", &id, &portid, &address, &count);
	if( ret != 2 && ret != 4 )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <testid> <portid> <address> <count>");
		PRINTL(DBG_LVL_ALWAYS, "  1: Dump whole X Memory(<address>,<count> are not used)");
		PRINTL(DBG_LVL_ALWAYS, "  2: Dump whole Y Memory(<address>,<count> are not used)");
		PRINTL(DBG_LVL_ALWAYS, "  3: Dump whole P Memory(<address>,<count> are not used)");
		PRINTL(DBG_LVL_ALWAYS, "  4: Dump whole X and Y Memory(<address>,<count> are not used)");
		PRINTL(DBG_LVL_ALWAYS, "  5: Dump X Memory with specified address and count");
		PRINTL(DBG_LVL_ALWAYS, "  6: Dump Y Memory with specified address and count");
		PRINTL(DBG_LVL_ALWAYS, "  7: Dump P Memory with specified address and count");

		return -DSL_ERR_NOPASS;
	}
	PRINTL(DBG_LVL_ALWAYS, "portid=%i,address=%i,count=%i",portid,address,count);

	get_debug_level(&lvlold);
	if(lvlold & DBG_LVL_CLI) set_debug_level(DBG_LVL_CLI);
	else if(lvlold & DBG_LVL_XML) set_debug_level(DBG_LVL_XML);
	else set_debug_level(DBG_LVL_NONE);

	fdump=fopen("/tmp/memdumpt.txt", "w");
	fdump2=fopen("/tmp/memdumpb.txt", "wb");

	portid--;

	ret = DSL_ERR_OK;
	switch(id)
	{
		case 1:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_X;
			for(address=0; address<=MT_X_MEM_BOUND; address++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "X memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(address != 0 && ((address + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump, "%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fwrite(pdata, sizeof(pdata), 1, fdump2);
			}

			PRINTL(DBG_LVL_ALWAYS, "Complete!");
			break;

		case 2:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_Y;
			for(address=0; address<=MT_Y_MEM_BOUND; address++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Y memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(address != 0 && ((address + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump, "%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fwrite(pdata, sizeof(pdata), 1, fdump2);
			}

			PRINTL(DBG_LVL_ALWAYS, "\nComplete!");
			break;

		case 3:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_P;
			for(address=0; address<=MT_P_MEM_BOUND; address++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "P memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(address != 0 && ((address + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump,"%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fprintf(fdump2,"%c%c%c", pdata[0], pdata[1], pdata[2]);
			}

			PRINTL(DBG_LVL_ALWAYS, "\nComplete!");
			break;

		case 4:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_X;
			for(address=0; address<=MT_X_MEM_BOUND; address++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "X memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(address != 0 && ((address + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump, "%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fwrite(pdata, sizeof(pdata), 1, fdump2);
			}

			fprintf(fdump,"\n\n\n");

			memtype=MT_MEMTYPE_Y;
			for(address=0; address<=MT_Y_MEM_BOUND; address++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Y memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(address != 0 && ((address + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump, "%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fwrite(pdata, sizeof(pdata), 1, fdump2);
			}

			PRINTL(DBG_LVL_ALWAYS, "\nComplete!");
			break;

		case 5:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_X;
			for(i=0; address<=MT_X_MEM_BOUND && i<count; address++, i++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "X memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(i != 0 && ((i + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump,"%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fprintf(fdump2,"%c%c%c", pdata[0], pdata[1], pdata[2]);
			}

			PRINTL(DBG_LVL_ALWAYS, "Complete!");
			break;

		case 6:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_Y;
			for(i=0; address<=MT_Y_MEM_BOUND && i<count; address++, i++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Y memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(i != 0 && ((i + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump,"%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fprintf(fdump2,"%c%c%c", pdata[0], pdata[1], pdata[2]);
			}

			PRINTL(DBG_LVL_ALWAYS, "Complete!");
			break;

		case 7:
			PRINTL(DBG_LVL_ALWAYS, "Dumping...");

			memtype=MT_MEMTYPE_P;
			for(i=0; address<=MT_P_MEM_BOUND && i<count; address++, i++)
			{
					if(get_port_mem(portid,address,memtype,pdata) != DSL_ERR_OK) {
						PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "P memory dumping error!!");
						ret = DSL_ERR_NOTOK;
						break;
					}

					if(i != 0 && ((i + 1) % 1000) == 0)
						PRINTL(DBG_LVL_ALWAYS, ".");

					fprintf(fdump,"%02x%02x%02x\n", pdata[0], pdata[1], pdata[2]);
					fprintf(fdump2,"%c%c%c", pdata[0], pdata[1], pdata[2]);
			}

			PRINTL(DBG_LVL_ALWAYS, "Complete!");
			break;

		default:
			break;
	}

	fclose(fdump);
	fclose(fdump2);

	set_debug_level(lvlold);

	return ret;
}


mt_ret test_reg(unsigned char * pbuf, cmdt * pcmd)
{
	int fd;
	void *gpio_addr,*pgpio_addr;
	mt_uint32 id, base_address, offset, value, u32tmp, ret;

	/*id = 1 is read , id = 2 is write*/
	ret = sscanf(pbuf, "%i %x %x %x", &id, &base_address, &offset, &value);

	if ( (id == 1 && ret != 3) || (id == 2 && ret != 4) )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage: id = 1 is read, = 2 is write");
		PRINTL(DBG_LVL_ALWAYS, "       cmd <id> <base_addr-hex> <offset-hex> <value-hex>");
		 return -DSL_ERR_NOPASS;
	}

	/* Must be O_SYNC, non cache */
	if((fd = open("/dev/mem", O_RDWR | O_SYNC)) == -1) FATAL;
	printf("/dev/mem opened.\n");
	fflush(stdout);

	pgpio_addr = mmap(0, MAP_SIZE, PROT_READ | PROT_WRITE, MAP_SHARED,fd,base_address & ~MAP_MASK);
	if(pgpio_addr == (void *) -1) {
			printf("map_base:%x \n",pgpio_addr); FATAL;}
	printf("Memory mapped at address %p.\n", pgpio_addr);

	fflush(stdout);
	gpio_addr = pgpio_addr + (base_address & MAP_MASK);


	switch(id)
	{
		case 1:
			u32tmp = (*(volatile mt_uint32 *)(gpio_addr+offset));
			printf("0x%08x=%08x\n",base_address+offset,u32tmp);
			break;
		case 2:
			(*(volatile mt_uint32 *)(gpio_addr+offset))=value;
			u32tmp = (*(volatile mt_uint32 *)(gpio_addr+offset));
			printf("0x%08x=%08x\n",base_address+offset,u32tmp);
			break;
	}
	close(fd);
	return DSL_ERR_OK;
}


mt_ret test_regwbit(unsigned char * pbuf, cmdt * pcmd)
{

	int fd;
	void *gpio_addr,*pgpio_addr;
	mt_uint32 base_address, offset, ret;

        char vname_s[256], value_s[256], *cp;
        unsigned int i, cnt, mask, base;
        unsigned int value_r, value, sbit, ebit;
	char command[256];

	ret = sscanf(pbuf, "%x %x %d %d %s %s", &base_address, &offset, &sbit, &ebit, value_s, vname_s);

	if ( (ret != 5) && (ret != 6) )
	{
		PRINTL(DBG_LVL_ALWAYS, "Usage: id = 1 is read, = 2 is write");
		PRINTL(DBG_LVL_ALWAYS, "       cmd <id> <base_addr-hex> <offset-hex> <value-hex>");
		 return -DSL_ERR_NOPASS;
	}

	/* Must be O_SYNC, non cache */
	if((fd = open("/dev/mem", O_RDWR | O_SYNC)) == -1) FATAL;
	printf("/dev/mem opened.\n");
	fflush(stdout);

	pgpio_addr = mmap(0, MAP_SIZE, PROT_READ | PROT_WRITE, MAP_SHARED,fd,base_address & ~MAP_MASK);
	if(pgpio_addr == (void *) -1) {
			printf("map_base:%x \n",pgpio_addr); FATAL;}
	printf("Memory mapped at address %p.\n", pgpio_addr);

	fflush(stdout);
	gpio_addr = pgpio_addr + (base_address & MAP_MASK);

//yalee961224 start:tony prog **//
        cp = &value_s[0];
        if (*cp == '0') {
                cp++;
                if ((*cp == 'x')) {
                        base = 16;
                        cp++;
                }
                else {
                        base = 8;
                }
        }
        else {
                base = 10;
        }
        value = strtoul(cp, NULL, base);
	value_r = (*(volatile mt_uint32 *)(gpio_addr+offset));

//yalee961224:tony prog
        if (sbit > ebit){
                cnt = ebit;
                ebit = sbit;
                sbit = cnt;
        }
        cnt = ebit-sbit;
        mask = 0;
        for (i=0; i<(cnt+1); i++) {
                mask |= (0x1<<i);
        }
        value_r &= ~(mask<<sbit);
        value &= mask;
        value_r |= (value<<sbit);

        if (ret==6) { //yalee961224:modify ret = 6
                sprintf(command, ""CFGMGR_PATH" -w %s=%d", vname_s, value);
                ret = system(command);
        }




	(*(volatile mt_uint32 *)(gpio_addr+offset)) = value_r;
	close(fd);
	return 0;
}


mt_ret test_regrbit(unsigned char * pbuf, cmdt * pcmd)
{

	int fd;
	void *gpio_addr,*pgpio_addr;
	mt_uint32 base_address, offset, ret;

        char vname_s[256];
        unsigned int i, cnt, mask;
        unsigned int value_r, value, sbit, ebit;
	char command[256];


	ret = sscanf(pbuf, "%x %x %d %d %s", &base_address, &offset, &sbit, &ebit, vname_s);

	if( (ret != 4) && (ret !=5) )
	{
		PRINTL(DBG_LVL_ALWAYS, "Read switch register bits, <base_address> <offset> <start bit> <end bit> [variable name]");
		PRINTL(DBG_LVL_XML, "argument of command is fail");
		return -DSL_ERR_NOPASS;
	}


	/* Must be O_SYNC, non cache */
	if((fd = open("/dev/mem", O_RDWR | O_SYNC)) == -1) FATAL;
	printf("/dev/mem opened.\n");
	fflush(stdout);

	pgpio_addr = mmap(0, MAP_SIZE, PROT_READ | PROT_WRITE, MAP_SHARED,fd,base_address & ~MAP_MASK);
	if(pgpio_addr == (void *) -1) {
			printf("map_base:%x \n",pgpio_addr); FATAL;}
	printf("Memory mapped at address %p.\n", pgpio_addr);

	fflush(stdout);
	gpio_addr = pgpio_addr + (base_address & MAP_MASK);

	value_r = (*(volatile mt_uint32 *)(gpio_addr+offset));


//tony prog
	if (sbit > ebit){
		cnt = ebit;
		ebit = sbit;
		sbit = cnt;
	}

	cnt = ebit-sbit;
	mask = 0;
	for (i=0; i<(cnt+1); i++) {
		mask |= (0x1<<i);
	}
	value = (value_r >> sbit) & mask;

	if (ret==5) {//yalee961224: modify ret = 5
		sprintf(command, ""CFGMGR_PATH" -w %s=%d", vname_s, value);
		ret = system(command);
	}
	PRINTL(DBG_LVL_ALWAYS, "value = %d\n", value);
	PRINTL(DBG_LVL_XML, "%d", value);

	return 0;

}


/** change to interactive mode */
int com_interact (unsigned char * pbuf, cmdt * pcmd)
{
	statical = 0;
	return DSL_ERR_OK;
}


/** change to static mode */
int com_static (unsigned char * pbuf, cmdt * pcmd)
{
	statical = 1;
	return DSL_ERR_OK;
}



#endif //DEBUG_API
#endif	//MT_DEBUG

#if defined (CO_API) && defined (TEMPLATE_PROFILE)
int com_get_bnd_psd (unsigned char * pbuf, cmdt * pcmd)
{
	mt_ret ret;
	char BndProfileName[256];
	char mibsBuff[256];

	ret = sscanf(pbuf, "%s", BndProfileName);
	if (ret != 1){
		PRINTL(DBG_LVL_ALWAYS, "Usage:	cmd <BND profile name>");
		return -DSL_ERR_NOPASS;
	}

	ret = syspackage_checkid_fun(BND_PREFIX, BndProfileName);
	if (ret != DSL_ERR_OK){
		PRINTL(DBG_LVL_ALWAYS, "%s BAD profile name isn't exist in system package",BndProfileName);
		return -DSL_ERR_NOPASS;
	}

	//PRINTL(DBG_LVL_ALWAYS, "%s BND profile transmit power and psd default value",BndProfileName);
	get_package_profile_oid_value(BND_PREFIX , BndProfileName, xdsl2LConfProfMaxNomAtpDs, mibsBuff);
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "maxNomAtpDs:%s|",mibsBuff);
	get_package_profile_oid_value(BND_PREFIX , BndProfileName, xdsl2LConfProfMaxNomAtpUs, mibsBuff);
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "maxNomAtpUs:%s|",mibsBuff);
	get_package_profile_oid_value(BND_PREFIX , BndProfileName, xdsl2LConfProfMaxNomPsdDs, mibsBuff);
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "maxNomPsdDs:%s|",mibsBuff);
	get_package_profile_oid_value(BND_PREFIX , BndProfileName, xdsl2LConfProfMaxNomPsdUs, mibsBuff);
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "maxNomPsdUs:%s|",mibsBuff);
	//yalee990706:for web used
	get_package_profile_oid_value(BND_PREFIX , BndProfileName, xdsl2LConfProfProfiles, mibsBuff);
	PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "profiles:%s|",mibsBuff);

	return DSL_ERR_OK;
}
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)


//=======================util cmd tables=============
cmdt util_cmd[] = {
	{"df","Download firmware from file, <portid> <firmware_path>", NULL, NULL, test_df},
	{"dpf", "Download firmware from package file, <portid> <firmwareid> [<profileid>]", NULL, NULL, test_package},
	{"ff","Flash firmware to SPI flash memory, <firmware_path> <offset>", NULL, NULL, test_flash},
	{"bdset", "Board settings test, <testid> <portid> [<count>]", NULL, NULL, test_board_set},
	{"portop", "Port operations, <testid> <portid>", NULL, NULL, test_portop},
	{"level", "Set debug level, <debug_level>", NULL, NULL, debug_change_level},
	{"firmver","Show running firmware version and OT/RT indicatior, <portid>", NULL, NULL, show_firmver},
	{"phystatus", "Get MT_OPSTATE_* status, <portid>", NULL, NULL, test_phy_status},
	{"clrstats", "Clear Port statistic information, <portid>", NULL, NULL, clear_statistic},
#if defined (CO_API) && defined (TEMPLATE_PROFILE)
	{"getbndpsd", "Get BND profile transmit power and psd default value, <BND profile name>", NULL, NULL, com_get_bnd_psd},
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)
#ifdef MT_DEBUG
	{"hpi", "HPI test, <testid> <portid>", NULL, NULL, test_hpi},
	{"mem", "Memory read/write test, \n\t\t   <testid> <portid> <memtype> [<address-hex> <count-hex> [<data-hex>]]", NULL, NULL, test_mem},
	{"dumpsoc", "Dump entire SOC data, <portid>", NULL, NULL, dump_soc},
#ifdef DEBUG_API
	{"dump", "Dump memory content, <testid> <portid> [<address> <count>]", NULL, NULL, debug_dump_mem},
	{"reg", "star cpu reg read.write, \n\t\t   <testid> <base_addr-hex> <offset-hex> <data-hex>", NULL,NULL, test_reg},
	{"regwbit", "star cpu reg write bit, \n\t\t   <testid> <base_addr-hex> <offset-hex> <data-hex>", NULL,NULL, test_regwbit},
	{"regrbit", "star cpu reg read bit, \n\t\t   <testid> <base_addr-hex> <offset-hex> <data-hex>", NULL,NULL, test_regrbit},
	{"tab", "Toggle tab completion", NULL, NULL, toggle_tab},
	{"interactive", "Run CLI in interactive mode", NULL, NULL, com_interact},
	{"static", "Run CLI in static mode", NULL, NULL, com_static},
#endif	//DEBUG_API
#endif	//MT_DEBUG
	{"help", "Help display an overview of the CLI syntax", NULL, NULL,com_help},
	{"exit", "Change to upper directory", root_cmd, NULL, com_home},
	{"home", "Return to the root directory", root_cmd, NULL, com_home},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};


/** Strip whitespace from the start and end of STRING.  Return a pointer
   into STRING. */
char * stripwhite (char *string)
{
	register char *s, *t;

	for (s = string; isspace (*s); s++);	//skip leading spaces

	if (*s == 0) return (s);

	t = s + strlen (s) - 1;		//point t to end of string

	while ((t > s && isspace (*t)) || (*t == '?')) t--;		//skip tail spaces backward

	*++t = '\0';	//fill-in end of string

	return s;
}


/** Strip inline comment from STRING. */
void * stripcomment (char *string)
{
	register char *s, *t;

	t = string;
	s = t + strlen (t) - 1;	//point s to end of string

	for (;s > t;s--) {
		if (*s == '#') *s = '\0';	//strip inline comment
	}
}


#ifdef DEBUG_API
/** Get system config */
int com_getcfg (unsigned char * pbuf, cmdt * pcmd)
{
	char confname[64];
	char command[256];
	char line[256];
	mt_ret ret;
	FILE *streamout;

	ret = sscanf(pbuf, "%s", confname);
	if( ret != 1 ){
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <config-name>");

		return -DSL_ERR_NOPASS;
	}

	sprintf(command, ""CFGMGR_PATH" -r %s", confname);
	if((streamout = popen(command, "r")) != NULL) {
		if(fgets(line, 256, streamout) != NULL) {
			stripwhite(line);
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "%s", line);
		}
		pclose(streamout);
	}

	return DSL_ERR_OK;
}


/** Set system config */
int com_setcfg (unsigned char * pbuf, cmdt * pcmd)
{
	char confname[256];
	char param[256];
	char command[256];
	mt_ret ret;

	ret = sscanf(pbuf, "%s %s", confname, param);
	if (ret == 2) {
		sprintf(command, ""CFGMGR_PATH" -w %s=%s", confname, param);
	}
	else if (ret == 1) {
		sprintf(command, ""CFGMGR_PATH" -w %s=", confname);
	}
	else {
		PRINTL(DBG_LVL_ALWAYS, "Usage: cmd <config-name> <param>");
		return -DSL_ERR_NOPASS;
	}

	ret = system(command);

	return DSL_ERR_OK;
}
#endif	//DEBUG_API


/**
 * dslcli script file parser
 */
int script(unsigned char * pbuf, cmdt * pcmd)
{
	int ret, chipid, cliexit;
	unsigned char buffer[MAX_BUFFER], *tp;
	FILE *sp;

	ret = sscanf(pbuf, "%s", buffer);
	if(ret != 1) {
		PRINTL(DBG_LVL_ALWAYS, "Usage: script <file_name>");
		return -DSL_ERR_NOPASS;
	}

	sp = fopen(buffer, "r");
	if(sp == NULL) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Script file %s was not found.", buffer);
		cliexit = DSL_ERR_NOTOK;
	}
    else {
		PRINTL(DBG_LVL_ALWAYS, "Running script %s:", buffer);
		cliexit = DSL_ERR_OK;
		while(cliexit != DSL_ERR_EXITCLI) {
			commands = root_cmd;
			bzero(buffer, sizeof(char) * CMD_BUFFER_SIZE);
			tp = fgets(buffer, CMD_BUFFER_SIZE, sp);
			if(tp == NULL)
				break;
			else {
				PRINTL(DBG_LVL_ALWAYS, ">>%s", buffer);
				cliexit = execute_line(buffer, commands);
			}
		}
		fclose(sp);
	}

	//if in interactive mode, change back to root_cmd once
	if(!statical) com_root(NULL, NULL);
	return cliexit;
}


/**
 * dslcli shell script file parser
 */
int shscript(unsigned char * pbuf, cmdt * pcmd)
{
	int ret, chipid, status, buf_len;
	unsigned char *tp, buffer[MAX_BUFFER], tmp[MAX_BUFFER+3];
	unsigned char line[XML_CMD_BUFFER_SIZE];
	FILE *sp;
	FILE *streamout;

	ret = sscanf(pbuf, "%s", buffer);
	if(ret != 1) {
		PRINTL(DBG_LVL_ALWAYS, "Usage: shellscript <file_name>");
		return -DSL_ERR_NOPASS;
	}

	buf_len = strlen(buffer);
	memcpy(tmp, buffer, MAX_BUFFER);
	if(tmp[buf_len-1]=='&') {
		tmp[buf_len-1]='\0';
	}
	sp = fopen(tmp, "r");
	if(sp == NULL) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Script file %s was not found.", buffer);
		ret = DSL_ERR_NOTOK;
	}
    else {
		PRINTL(DBG_LVL_ALWAYS, "Running shell script %s:", buffer);
		ret = DSL_ERR_OK;

		sprintf(tmp, "sh %s", pbuf);
		status = system(tmp);
		if(status < 0) {
				PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML,"system() error!!");
				ret = DSL_ERR_NOTOK;
		}

		fclose(sp);
    }
	return ret;
}


/** Print usage information and exit */
void usage(char * name) {

  PRINTL(DBG_LVL_ALWAYS, "DSLCLI - The xDSL Configuration Shell utility.");
  PRINTL(DBG_LVL_ALWAYS, "Copyright (c) 2005 Metanoia, Inc. All rights reserved.\n");
  PRINTL(DBG_LVL_ALWAYS, "Usage: %s [option]\n", name);
  PRINTL(DBG_LVL_ALWAYS, "Options:");
  PRINTL(DBG_LVL_ALWAYS, "  clean    - Reset internal state to default values.\n             Does not change the system state.");
  PRINTL(DBG_LVL_ALWAYS, "  setup    - Setup system on boot from configuration file.");
  PRINTL(DBG_LVL_ALWAYS, "  silent   - Work normally, but don't display prompt.\n             Good for use from scripts.");
  PRINTL(DBG_LVL_ALWAYS, "  static   - Work statically, and don't change command level.\n             Classical dslman mode.");
  PRINTL(DBG_LVL_ALWAYS, "  resident - Work residently, and don't display prompt.\n             Can be used for background residence.");
  PRINTL(DBG_LVL_ALWAYS, "  script_filename - Execute scripted commands one by one.\n             Good for running batch commands.");

  exit(0);

}


/** dslcli main program */
int main(int argc, char * argv[])
{
	unsigned char buffer[CMD_BUFFER_SIZE];
	unsigned char *tp;
	int ret;
	int setup = 0;
	int resident = 0;
	int script = 0;
	int cliexit = 0;
	int done;
	FILE *sp = NULL;
	int j = 0;

	progname = argv[0];

	if((argc == 2) && (!strncmp(argv[1], "setup", 5)) ) {
		setup = 1;
		show_prompt = 0;
		//set_debug_level(DBG_LVL_ALWAYS);
		set_debug_level(DBG_LVL_SETUP);

#ifdef VIA_SAMSUNG
		if(sp = fopen("/mnt/meta_hack.sh", "r")){
			fclose(sp);
			shscript("/mnt/meta_hack.sh", NULL);
		}
#endif

		if(!( sp = fopen(CONFIG_PATH, "r"))){
			PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Failed to open config file %s", strerror(errno));
			exit(1);
		}

		//dup2(fileno(cfg_file), fileno(stdin));
		script = 1;
	}
	else if ((argc == 2) && (!strncmp(argv[1], "clean", 5)) ) {
		setup = 1;
		show_prompt = 1;
		set_debug_level(DBG_LVL_ALWAYS);
	}
	else if ((argc == 2) && (!strncmp(argv[1], "silent", 6)) ) {
		setup = 0;
		show_prompt = 0;
		set_debug_level(DBG_LVL_XML);

		silent = 1;
	}
	else if ((argc == 2) && (!strncmp(argv[1], "resident", 8)) ) {
		setup = 0;
		show_prompt = 0;
		set_debug_level(DBG_LVL_NONE);

		resident = 1;
	}
	else if ((argc == 2) && (!strncmp(argv[1], "static", 6)) ) {
		setup = 0;
		show_prompt = 1;
		set_debug_level(DBG_LVL_CLI);

		statical = 1;
	}
	else if (argc > 1) {
		setup = 0;
		show_prompt = 0;
		//set_debug_level(DBG_LVL_CLI);

		if(!( sp = fopen(argv[1], "r"))) {
			set_debug_level(DBG_LVL_OPTION);
			//we execute the possible command once and exit dslcli so we must add a extra config init here
			init_config(setup);
			cliexit = execute_line(argv[1], commands);
			if(cliexit != DSL_ERR_OK) {
				PRINTL(DBG_LVL_ALWAYS, "\nInvalid option or command or script file \"%s\" was not found.\n", argv[1]);
				usage(argv[0]);
				exit(1);
			}
			else
				exit(0);
		}
		else {	//file open successed, mark it as script mode
			script = 1;
		}
	}
	else {	//argc == 1
		setup = 0;
		show_prompt = 1;
		set_debug_level(DBG_LVL_CLI);
	}

#ifdef MT_DEBUG
	//set_debug_level(DBG_LVL_MAPPER | DBG_LVL_CHIP | DBG_LVL_BOARD | DBG_LVL_PORT | DBG_LVL_IF | DBG_LVL_LIB | DBG_LVL_MISC);
	//set_debug_level(DBG_LVL_CLI);
	//set_debug_level(DBG_LVL_ALWAYS);
	//PRINTL(DBG_LVL_ALWAYS, "debug_level has set to DBG_LVL_ALWAYS\n");
#endif

	PRINTL(DBG_LVL_ALWAYS, "%s for %s - %s %s, Make time:%s", argv[0], BOARD_NAME, DSL2VERSION, "$Revision: 1.238 $", MAKETIME);
	PRINTL(DBG_LVL_ALWAYS, "argc = %i, argv[0] = %s, argv[1] = %s", argc, argv[0], argv[1]);

	if((ret = init_config(setup))) {
		PRINTL(DBG_LVL_ALWAYS, "Internal error: %s", strerror(ret));
	}

	umask(022);

#ifdef XML_MIN_SIZE
	//silent mode can execute batch actions in one shot
	if(silent) {
		unsigned char c;
		int ci = 0;
		int depth = 0;
		unsigned char *token;
		unsigned char xml_cmd_buf[XML_CMD_BUFFER_SIZE];
		XML_Parser parser;
#ifdef VIA_SAMSUNG
		unsigned char *pxml;
#endif

		syslog(1, "read from stdin");
		while(1){
			fread(&c, 1, 1, stdin);
			xml_cmd_buf[ci++] = c;
			if(c == 0x0 || ci == XML_CMD_BUFFER_SIZE) break;
		}
#ifdef OPEN_LOG
		syslog(1, xml_cmd_buf);
#endif

		if(ci == XML_CMD_BUFFER_SIZE && xml_cmd_buf[ci-1] != 0x0) {
			PRINTD("<REPLY id=\"\" status=\"0\" msg=\"Buffer overflow!!\"></REPLY>");
#ifdef OPEN_LOG
			syslog(1, "Buffer overflow!!");
#endif
			goto end_silent;
		}

#ifdef VIA_SAMSUNG
		pxml = xml_cmd_buf;
		token = (unsigned char *)strcasestr(pxml, "</ACTION>");
		while (token != NULL) {
			bzero(buffer, sizeof(char) * CMD_BUFFER_SIZE);
			*token = '\0';
			strcat(buffer, pxml);
			strcat(buffer, "</ACTION>");
			*token = '<';
			pxml = token + 9; // 9 is the length of "</ACTION>";

			depth = 0;
			//execute_line("profiles xdsl-line init", root_cmd);
			parser = XML_ParserCreate(NULL);
			XML_SetUserData(parser, &depth);
			XML_SetElementHandler(parser, startElement, endElement);
			//printf("buffer:%s, strlen(buffer)=%d\n", buffer, strlen(buffer));

			if (!XML_Parse(parser, buffer, strlen(buffer), 1)) {
				sprintf(buffer,
							"%s at line %d\n",
							XML_ErrorString(XML_GetErrorCode(parser)),
							XML_GetCurrentLineNumber(parser));
				PRINTD("%s", buffer);

				XML_ParserFree(parser);
				goto end_silent;
			}
  			XML_ParserFree(parser);
			token = (unsigned char *)strcasestr(pxml, "</ACTION>");
		}
#else //not VIA_SAMSUNG
		token = strtok(xml_cmd_buf, ">");
		while(token != NULL) {
			bzero(buffer, sizeof(char) * MAX_BUFFER);
			for(ci=0; ci<2; ci++) {
				strcat(buffer, token);
				buffer[strlen(buffer)+1] = buffer[strlen(buffer)];
				buffer[strlen(buffer)] = '>';
				token = strtok(NULL, ">");
				if(token == NULL) break;
			}
#ifdef OPEN_LOG
			syslog(1, buffer);
#endif

			/*if(!strcmp(buffer, "<ACTION id=\"cli\" cmd=\"seq1\"></ACTION>")) {
				PRINTD("<REPLY id=\"\" status=\"0\" msg=\"reply seq1\"></REPLY>");
				syslog(1, "seq1");
				goto end_silent;
			}
			if(!strcmp(buffer, "<ACTION id=\"cli\" cmd=\"seq2\"></ACTION>")) {
				PRINTD("<REPLY id=\"\" status=\"0\" msg=\"reply seq2\"></REPLY>");
				syslog(1, "seq2");
				strcpy(buffer, NULL);
				goto end_silent;
			}
			if(!strcmp(buffer, "<ACTION id=\"cli\" cmd=\"seq3\"></ACTION>")) {
				PRINTD("<REPLY id=\"\" status=\"0\" msg=\"reply seq3\"></REPLY>");
				syslog(1, "seq3");
				goto out_sh;
			}
			if(!strcmp(buffer, "<ACTION id=\"cli\" cmd=\"seq4\"></ACTION>")) {
				syslog(1, "seq4");
				goto end_silent;
			}*/

			parser = XML_ParserCreate(NULL);
			XML_SetUserData(parser, &depth);
			XML_SetElementHandler(parser, startElement, endElement);
			if (!XML_Parse(parser, buffer, strlen(buffer), 1)) {
				sprintf(buffer,
						"<REPLY id=\"\" status=\"0\" msg=\"%s at line %d\"></REPLY>",
						XML_ErrorString(XML_GetErrorCode(parser)),
						XML_GetCurrentLineNumber(parser));
				PRINTD("%s", buffer);

				XML_ParserFree(parser);
				goto end_silent;
			}
			XML_ParserFree(parser);
		}
#endif //VIA_SAMSUNG

end_silent:
		//send appWeb the output ending character
		c = 0x0;
		ci = fwrite(&c, sizeof(char), 1, stdout);
		sprintf(xml_cmd_buf, "%d", ci);
#ifdef OPEN_LOG
		syslog(1, xml_cmd_buf);
#endif
		syslog(1, "End of Command");
		goto out_sh;
	}
#endif	//XML_MIN_SIZE

	//resident mode will keep running in debug mode
	if(resident) {
		PRINTL(DBG_LVL_ALWAYS, "Running remote tool mode...");
		signal(SIGCHLD, SIG_IGN);
		//daemon(0, 0);	//daemon(nochdir, noclose);
#ifdef __USE_SOCKET__
		open_socket();
#else
		debug_enable(NULL, NULL);
#endif

		goto out_sh;
	}

	if(script) {
		if(setup) {
			PRINTL(DBG_LVL_ALWAYS, "Setup executing...");
		}
		else {
			PRINTL(DBG_LVL_ALWAYS, "Running script \"%s\":", argv[1]);
		}

		while(cliexit != DSL_ERR_EXITCLI) {
			commands = root_cmd;
			bzero(buffer, sizeof(char) * CMD_BUFFER_SIZE);
			tp = fgets(buffer, CMD_BUFFER_SIZE, sp);
			if(tp == NULL)
				break;
			else {
				if(!setup) PRINTD(">>%s",buffer);
				cliexit = execute_line(buffer, commands);
			}
		}
		fclose(sp);

		goto out_sh;
	}

	if(show_prompt) {
		set_prompt(NULL);

		init_tab_term();		/* Set up terminal for TAB completion */

		if (signal(SIGBUS, sig_term) == SIG_ERR)
			fprintf(stderr, "can't catch signal SIGBUS\n");

		if (signal(SIGINT, sig_term) == SIG_ERR)
			fprintf(stderr, "can't catch signal SIGINT\n");
	}

	/* Start command loop */
	while(cliexit != DSL_ERR_EXITCLI) {
		if(show_prompt) PRINTD(prompt);
		get_line(buffer);
		cliexit = execute_line(stripwhite(buffer), commands);
		if(statical) com_root(NULL, NULL);
	}

	if(show_prompt)
		restore_tab_term();	/* Restore terminal settings */

out_sh:
#if defined (CO_API) && defined (TEMPLATE_PROFILE)
	profiles_main_vdslprofile_save();
	free_vdsl_profiles_fun();
#endif	//defined (CO_API) && defined (TEMPLATE_PROFILE)

	if(show_prompt)
		exit_sh(DSL_ERR_OK);

	return 0;
}


#ifdef DEBUG_API
/** Reboot the system */
int com_reboot (unsigned char * pbuf, cmdt * pcmd)
{
	kill(1, SIGTERM);
	return DSL_ERR_OK;
}
#endif	//DEBUG_API


#if defined (CO_API)
/** dslcli user config saving entry point */
int com_save (unsigned char * pbuf, cmdt * pcmd)
{
	//int ret=0;
	int ret = DSL_ERR_OK;
	FILE * file;

#if defined (RALINK_RT3052) || defined (RALINK_RT3352)
	ret = system("sh /mtd/vdsl_control.sh save 1>/dev/null 2>/dev/null");
	if(ret != 0) ret = DSL_ERR_NOTOK;	//0: success 256: fail
#elif defined (VIA_SAMSUNG)
	//more save stuff to do through save.sh
	profiles_main_vdslprofile_save();
	ret = system("sh /mtd/save.sh 1>/dev/null 2>/dev/null");
	if(ret != 0) ret = DSL_ERR_NOTOK;	//0: success 256: fail
#endif

	if(ret != DSL_ERR_OK) {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Save failed! - %s", strerror(ret));
	}
	else {
		PRINTL(DBG_LVL_ALWAYS|DBG_LVL_XML, "Configuration saved");
	}

	return ret;
}
#endif


#ifdef WITH_ADV
int com_diag (unsigned char * pbuf, cmdt * pcmd)
{
	commands = diag_cmd;
	set_prompt("(diag)");
	return DSL_ERR_OK;
}
#endif //WITH_ADV


int com_interface (unsigned char * pbuf, cmdt * pcmd)
{
	commands = interface1_cmd;
	set_prompt("(interface)");
	return DSL_ERR_OK;
}


#ifdef CO_API
int com_profiles (unsigned char * pbuf, cmdt * pcmd)
{
	commands = profiles_cmd;
	set_prompt("(profiles)");
	return DSL_ERR_OK;
}
#endif //CO_API


int com_show_xdsl (unsigned char * pbuf, cmdt * pcmd)
{
	commands = show_xdsl_cmd;
	set_prompt("(show)");
	return DSL_ERR_OK;
}


int com_util (unsigned char * pbuf, cmdt * pcmd)
{
	commands = util_cmd;
	set_prompt("(util)");
	return DSL_ERR_OK;
}


//=======================root cmd tables===============
cmdt root_cmd[] = {
#ifdef	WITH_ADV
	{"diag", "Change to diagnostic directory",diag_cmd, NULL, com_diag},
#endif //WITH_ADV
	{"interface", "Commands for interfaces", interface1_cmd, NULL, com_interface},
#ifdef CO_API
	{"profiles", "Commands for profiles", profiles_cmd, NULL, com_profiles},
#endif //CO_API
	{"show", "Display xDSL informatons", show_xdsl_cmd, NULL, com_show_xdsl},
#ifdef TR069
	{"tr069",  "Change to tr069 directory", tr069_cmd, NULL, com_tr069},
#endif
	{"util", "Change to utility directory", util_cmd, NULL, com_util},
	{"script", "Executes a CLI script file", NULL, NULL, script},
	{"shellscript", "Executes a SHELL script file", NULL, NULL, shscript},
#ifdef DEBUG_API
	{"getcfg", "Get system config: getcfg <config-name>", NULL, NULL, com_getcfg},
	{"setcfg", "Set system config: setcfg <config-name> <param>", NULL, NULL, com_setcfg},
	{"reboot", "Reboot", NULL, NULL,com_reboot},
#endif // DEBUG_API
#if defined (CO_API)
	{"save", "Save configuration", NULL, NULL, com_save},
#endif
	{"help", "Help display an overview of the CLI syntax", NULL, NULL, com_help},
	{"exit", "Change to upper directory", NULL, NULL, com_exitf},
	{"quit", "Exit this CLI session", NULL, NULL, com_exitf},
	{"?", "Help", NULL, NULL, com_help},
	{NULL, NULL, NULL, NULL, NULL}
};
