#ifndef sualib_sualib_api_h
#define sualib_sualib_api_h

#include <stdint.h>
#include <unistd.h>

#define SUALIB_VERSION			"0.4"

struct sualib_string_t {
	char*   ptr;
	ssize_t slen;
};

typedef struct sualib_string_t sualib_string_t;

typedef struct sualib_acc_cfg {
	char* id;
	char* registrator;
	char* auth_realm;
	char* user;
	char* pass;
	char* cookies;
} sualib_acc_cfg_t;

typedef enum sualib_result {
	SUALIB_OK,
	SUALIB_ERR_NOT_INITILIZED,
	SUALIB_ERR_NOT_STARTED,
	SUALIB_ERR_NO_ACTIVE_CALL,
	SUALIB_ERR_ANOTHER_CALL_IN_PROGRESS,
	SUALIB_ERR_MEDIA_NOT_INITILIZED,
	SUALIB_ERR_INVALID_INPUT,
	SUALIB_ERR_EXECUTION,
}sualib_status_t;

typedef enum sualib_call_state {
	S_CONN_TRYING,
	S_CONN_RINGING,
	S_CONN_CONNECTED_OK,
	S_CONN_DISCONNECTED_OK,
	S_CONN_DISCONNECTED_ELSEWHERE,
	S_CONN_REJECTED,
	S_CONN_BUSY,
	S_CONN_ERROR,
}sualib_call_state_t;

typedef enum sualib_audio {
	S_DEFAULT_MIC,
	S_LOUDSPEAKER,
	S_ERAPIECE,
	S_BLUETOOTH,
}sualib_audio_t;

typedef struct sualib_conn_state {
	sualib_call_state_t code;
	unsigned int        internal_code;
	sualib_string_t     msg;
	sualib_string_t     codec;
} sualib_conn_state_t;

typedef struct sualib_app_cfg {
	/* REQUIRED: Account configuration for sualib to register */
	sualib_acc_cfg_t acc;

	/* OPTIONAL: 0 - RTP logs disabled , setting to 1 (enabled) my slow down library */
	unsigned int enable_rtp_logs;

	/* OPTIONAL: 0 - using TCP transport , 1 - using UDP transport (WARNING: only for ANDROID!) */
	unsigned int use_udp_transport;

	unsigned int use_tls_transport;

	/* REQUIRED: Tell app that pjsua is (re)started, msg will contain start error message */
	void (*on_started)(int status, sualib_string_t title);

	/**
	 * REQUIRED: Tell app that we are receiving incoming call, so GUI app can accept it or decline
	 * based on caller id given.
	 */
	void (*on_call_incoming)(sualib_string_t remote);

	/* REQUIRED: Tell app that the caller has disconnected */
	void (*on_call_disconnected)(sualib_string_t remote, sualib_conn_state_t state);

	/* REQUIRED: Tell app what is the state of outgoing session */
	void (*on_call_connecting)(sualib_string_t remote, sualib_conn_state_t state);

	/* OPTIONAL: Tell app that we are receiving DTMF digit */
	void (*on_dtmf_digit_received)(int digit);

	/* OPTIONAL: Inform about registration status code */
	void (*on_reg_state)(int code);

	/* OPTIONAL: Override existing (pjsib-based) log function */
	void (*sualib_log_func)(int level, const char* buffer, int len);

} sualib_app_cfg_t;

/**
 * All of the REQUIRED callbacks from sualib_app_cfg_t MUST be implemented (at least assigned to
 * stub/mock functions) to proper initialize the sualib app.
 */
sualib_status_t sualib_app_init(sualib_app_cfg_t *);
sualib_status_t sualib_app_run(void);
sualib_status_t sualib_app_destroy(void);

void sualib_set_logging_level(int level);

#ifdef SUALIB_ANDROID_API
/* for thread safe account re-registration*/
void sualib_app_reregister(void);

#else
/**
 * As the process is normally suspended when application is in the background (like in iOS environment),
 * the worker thread that handles TCP keepalive timer is also suspended. So basically application needs to
 * schedule periodic wakeup to allow the library send TCP keep-alive
 */
void sualib_app_keepalive(void);

#endif

/* update account information (for example after changing contact number
 * REMEBMER to wait for registration status code afterwars
 */
sualib_status_t update_account(sualib_acc_cfg_t acc);

/**
 * Following may be called only after app_init & app_run, and before app_destroy.
 */

/* Basic call control: make, answer(ok/reject), end */
sualib_status_t sualib_make_call(sualib_acc_cfg_t acc, char *);
sualib_status_t sualib_answer_call(int answer_ok);
sualib_status_t sualib_end_call(void);

/* Microphone mute toggle. Set m = 0 to mute microphone and !=0 to bring back old volume level. */
sualib_status_t sualib_mute_mic(float m);
sualib_status_t sualib_set_tx(float m);

/* Chose audio output route: DEFAULT_MIC, LOUDSPEAKER, EARPIECE, BLUETOOTH */
sualib_status_t sualib_audio_route(sualib_audio_t);

/**
 * sualib_send_dtmf_digits sends DTMF digits, making two attempts:
 * 1. using RFC 2833 if remote accepts RFC 2833 DTMF or when remote fails to accept this
 * 2. using RFC INFO header (content-type application/dtmf-relay)
 * The second attempt is made only if the first one fails. Accepted DTMF digits are from range:
 * 0-9#*A-D (or lowercase letters a-d)
 */
sualib_status_t sualib_send_dtmf_digits(char *digits);

#endif
