/*
 * speed_test.h
 *
 *  Created on: Jan 16, 2014
 *      Author: mateuszslupny
 */

#ifndef SPEED_TEST_H_
#define SPEED_TEST_H_

#include "common/queue/queue.h"

#include "common/cutils/uthash.h"
#include "common/cutils/utlist.h"

#define member_size(type, member) sizeof(((type *)0)->member)


struct event_base;

typedef enum {
	CONNECTION_TYPE_UNRECOGINZE = 0,
	CONNECTION_TYPE_VDSL = 1,
	CONNECTION_TYPE_FIBER = 2,
	CONNECTION_TYPE_ADSL = 3,
	CONNECTION_TYPE_3G = 4,
	CONNECTION_TYPE_LTE = 5,
	CONNECTION_TYPE_BONDING = 6
} speed_test_connection_type_t;

typedef struct speed_test speed_test_t;

typedef struct {
	enum {
		SPEED_TEST_EVENT_UNDEFINE,
		SPEED_TEST_EVENT_PRETEST_DONE,
		SPEED_TEST_EVENT_PROGRESS_PING,
		SPEED_TEST_EVENT_PROGRESS_DOWNLOAD,
		SPEED_TEST_EVENT_PROGRESS_UPLOAD,
		SPEED_TEST_EVENT_DONE,
		SPEED_TEST_EVENT_DOWNLOAD_DONE,
		SPEED_TEST_EVENT_UPLOAD_DONE,
		SPEED_TEST_EVENT_ERROR,
		SPEED_TEST_EVENT_FINISHED,
	} type;

	unsigned char percent_complete;
	float time_remaining;

	struct {
		char server[16];
	} parties;

	union {
		double current_value;

		struct {
			double max_speed;
			double avg_speed;
		} result;

		char error_string[128];
	};
	char origin[16];
} speed_test_event_t;

typedef void (*speed_test_http_callback_t)(speed_test_t *test, const char* response);

typedef struct {
	unsigned char (*make_http_request)(speed_test_t *test, const char* path, const char* server,
			unsigned int port, char* json, speed_test_http_callback_t http_callback);
	void (*event)(speed_test_t *test, speed_test_event_t* event);
} speed_test_inh_t;

typedef struct {
	double ping;
	double max_download_speed;
	double avg_download_speed;
	double max_upload_speed;
	double avg_upload_speed;
	char protocol[32];
} speed_test_results_t;

typedef enum {
	NOT_EXECUTED,
	SUCCESS,
	TIMEOUT,
	CANCELED,
} lan_test_status_t;

typedef struct {
	lan_test_status_t status;
	char connection_type[32];
	double download_speed;
	double upload_speed;
	char protocol[32];
} speed_test_lan_results_t;

typedef enum {
	SPEED_TEST_TYPE_NONE,
	SPEED_TEST_TYPE_VESTIACOM,
	SPEED_TEST_TYPE_BCM_SPEEDSVC,
	SPEED_TEST_TYPE_SAH_SPEEDSVC,
} speed_test_type_t;

typedef struct {
	speed_test_connection_type_t type; /*key*/
	unsigned int max_speed_download;
	unsigned int max_speed_upload;
	unsigned int download_connections;
	unsigned int upload_connections;
	speed_test_type_t test_type_override;
	UT_hash_handle hh;         /* makes this structure hashable */
} connection_parameters_t;

typedef enum {
	SPEEDSVC_STATE_PERIODIC_RESULT,
	SPEEDSVC_STATE_DONE,
	SPEEDSVC_STATE_ERROR
} speedsvc_state_t;

typedef struct {
	void *context;
	speedsvc_state_t state;
	float value;
	float current_value;
} speedsvc_event_t;

typedef void (*speedsvc_cb_t)(speedsvc_event_t *event);
typedef unsigned char (*speed_test_event_cb_t)(speed_test_event_t*);
typedef int (*reset_config_callback_t)(void *);
typedef void (*on_expspeed_callback_t)();
typedef void (*on_send_result_callback_t)(int, speed_test_results_t*, speed_test_lan_results_t*, char*);

speed_test_t* speed_test_new(const speed_test_inh_t* inh, void* owner, const char* origin, uint8_t wait_lan_result);
void speed_test_free(speed_test_t **test);
int speed_test_start(speed_test_t *test);
void speed_test_stop(speed_test_t *test);
unsigned char speed_test_report_lan_result(speed_test_t *test,
                                           const char* connection_type,
                                           double download,
                                           double upload,
                                           const char *protocol);

const char* speed_test_event_to_string(int event);
void* speed_test_get_owner(speed_test_t *test);
char *speed_test_get_origin(speed_test_t *test);
unsigned char speed_test_finished(speed_test_t *test);

unsigned char speed_test_set_management_server_address(const char* name, const char *value);
unsigned char speed_test_set_management_server_port(const char* name, const char *value);
unsigned char speed_test_set_download_duration(const char* name, const char* value);
unsigned char speed_test_set_upload_duration(const char* name, const char* value);
unsigned char speed_test_set_download_period(const char* name, const char* value);
unsigned char speed_test_set_upload_period(const char* name, const char* value);
unsigned char speed_test_set_local_test(speed_test_t *test, const char* address);
unsigned char speed_test_set_mode(speed_test_t* test, const char* mode); // currently only for local test
unsigned char speed_test_download_duration_integrity_check(void);
unsigned char speed_test_upload_duration_integrity_check(void);
unsigned char speed_test_set_socket_timeout(const char* name, const char* value);
unsigned char speed_test_set_window_size(const char* name, const char* value);
unsigned char speed_test_set_read_buffer_size(const char* name, const char* value);
unsigned char speed_test_set_poll_timeout_ms(const char* name, const char* value);
unsigned char speed_test_set_send_buffer_size(const char* name, const char* value);
unsigned char speed_test_set_download_avg_low_offset(const char* name, const char* value);
unsigned char speed_test_set_download_avg_top_offset(const char* name, const char* value);
unsigned char speed_test_set_upload_avg_top_offset(const char* name, const char* value);
unsigned char speed_test_set_upload_avg_low_offset(const char* name, const char* value);
unsigned char speed_test_download_avg_offset_integrity_check(void);
unsigned char speed_test_upload_avg_offset_integrity_check(void);
unsigned char speed_test_set_typeof_connection_command(const char* name, const char* value);
unsigned char speed_test_set_default_connection_type(const char* name, const char* value);
unsigned char speed_test_add_connection_definition(const char* name, const char* value);
unsigned char speed_test_set_count_proto_overhead(const char* name, const char* value);
unsigned char speed_test_add_upload_result_fine_tune(const char* name, const char* value);
unsigned char speed_test_add_download_result_fine_tune(const char* name, const char* value);
unsigned char speed_test_set_test_type(const char* name, const char* value);
unsigned char speed_test_set_lan_test_timeout(const char* name, const char* value);
unsigned char speed_test_set_send_lan_result_field(const char* name, const char* value);
unsigned char speed_test_set_max_kbps(const char *name, const char *value);
unsigned char speed_test_set_device_description(const char *name, const char *value);
unsigned char speed_test_set_print_summary(const char *name, const char *value);

void speed_test_set_reset_config_callback(reset_config_callback_t c);
void speed_test_set_on_expspeed_callback(on_expspeed_callback_t c);
void speed_test_set_on_send_result_callback(on_send_result_callback_t c);

typedef enum {
	download_duration,
	upload_duration,
} speed_test_param_t;
unsigned int speed_test_get_param_uint(speed_test_param_t p);

unsigned char bcm_speedsvc_set_ds_duration(const char *name, const char *value);
unsigned char bcm_speedsvc_set_loss_percentage(const char *name, const char *value);
unsigned char bcm_speedsvc_set_packet_len(const char *name, const char *value);
unsigned char bcm_speedsvc_set_pretest_duration(const char *name, const char *value);
unsigned char bcm_speedsvc_set_us_duration(const char *name, const char *value);
unsigned char bcm_speedsvc_set_version(const char *name, const char *value);
unsigned char bcm_speedsvc_set_port_offset(const char *name, const char *value);
unsigned char bcm_speedsvc_set_test_method(const char *name, const char *value);
unsigned char bcm_speedsvc_set_test_algorithm(const char *name, const char *value);
unsigned char bcm_speedsvc_set_stop_on_converge(const char *name, const char *value);

const char* bcm_speedsvc_get_test_method();

connection_parameters_t* find_connection_parameter(speed_test_connection_type_t type);
void log_connection_params(connection_parameters_t *cp);

#endif /* SPEED_TEST_H_ */
