//
//  NHClient.h
//  HomeNGTI
//
//  Copyright © 2017 Swisscom. All rights reserved.
//

#import "NHClientListener.h"
#import "NHClientLogger.h"
#import "NHClientTypes.h"

NS_ASSUME_NONNULL_BEGIN

/** Completion block for checkIBStatusWithCompletion */
typedef void(^NHRetrieveBoosterStatusCompletion)(NHBoosterStatus status);

/**
 * Main interface to the HomeNGTI framework
 */

@interface NHClient : NSObject

///--------------------------------------------------------------
/// @name Class properties
///--------------------------------------------------------------

/**
 * Minimum firmware level for Internet-Box 2 to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box 2 does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBox2FirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Plus to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box Plus does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxPlusFirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Standard to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box standard does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxStandardFirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Light to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box Light does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxLightFirmwareLevelForWLANBox;

///--------------------------------------------------------------
/// @name Initialization
///--------------------------------------------------------------

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

/**
 * @brief Initialize the framework
 * @note Please make sure the resulting object is retained in a property and does not go out of scope
 *
 * @param clientListener Object implementing the NHClientListener protocol, use this to listen for events from the framework
 */
- (instancetype)initWithClientListener:(NSObject<NHClientListener> *)clientListener;

///--------------------------------------------------------------
/// @name Configuration
///--------------------------------------------------------------

/**
 * @brief Update gateway credentials. This should be the SmartVoice account credentials.
 *
 * @param username Context ID to use in Softathome API calls
 * @param password Cookie to use in Softathome API calls
 */
- (void)updateGatewayCredentialsWithUserName:(NSString *)username password:(NSString *)password;

///--------------------------------------------------------------
/// @name WLAN setup wizard
///--------------------------------------------------------------

/**
 *  @brief Launch the WLAN setup wizard
 *
 * @param navigationController The existing UINavigationController of your app
 */
- (void)launchWLANSetupWithNavigationController:(UINavigationController *)navigationController;

/**
 * @brief Abort the WLAN setup wizard
 *        Important: It will be up to the calling application to return to the desired UI location
 */
- (void)cancelWLANSetup;

///--------------------------------------------------------------
/// @name Internet Booster setup wizard
///--------------------------------------------------------------

/**
 * @brief Launch the Internet-Booster setup wizard
 *
 * @param navigationController The existing UINavigationController of your app
 */
- (void)launchBoosterSetupWithNavigationController:(UINavigationController *)navigationController;

/**
 * @brief Abort the Internet Booster setup wizard
 *        Important: It will be up to the calling application to return to the desired UI location
 */
- (void)cancelBoosterSetup;

/**
 * @brief Retrieve current Internet-Booster status
 *
 * @param completion Completion block that will be called when status is retrieved
 */
- (void)checkBoosterStatusWithCompletion:(NHRetrieveBoosterStatusCompletion)completion;

///--------------------------------------------------------------
/// @name External logging
///--------------------------------------------------------------

/**
 * @brief Register external logging listener
 *
 * @param logger Object conforming to NHClientLogger, set to nil if no longer desired
 */
- (void)registerClientLogger:(NSObject<NHClientLogger> * _Nullable)logger;

NS_ASSUME_NONNULL_END

@end
