//
//  NHClient.h
//  HomeNGTI
//
//  Copyright © 2017 Swisscom. All rights reserved.
//

#import "NHClientListener.h"
#import "NHClientDelegate.h"
#import "NHClientLogger.h"
#import "NHClientTypes.h"
#import "NHWLANBoxInfo.h"

NS_ASSUME_NONNULL_BEGIN

/** Completion block for checkBoosterStatusWithCompletion */
typedef void(^NHCheckBoosterStatusCompletion)(NHBoosterStatus status);

/**
 * Main interface to the HomeNGTI framework
 */

@interface NHClient : NSObject

///--------------------------------------------------------------
/// @name Class properties
///--------------------------------------------------------------

/**
 * HomeNGTI framework version
 */
@property (readonly, class) NSString *frameworkVersion;

///--------------------------------------------------------------
/// @name Initialization
///--------------------------------------------------------------

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

/**
 * Shared NHClient instance
 */
+ (instancetype)sharedInstance;

///--------------------------------------------------------------
/// @name Configuration
///--------------------------------------------------------------

/**
 * Set external logging listener. Object should conform to NHClientLogger protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientLogger> *clientLogger;

/**
 * Set framework events listener. Object should conform to NHClientListener protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientListener> *clientListener;

/**
 * Set framework delegate. Object should conform to NHClientDelegate protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientDelegate> *clientDelegate;

/**
 * @brief Update gateway credentials. This should be the SmartVoice account credentials.
 *
 * @param username Context ID to use in Softathome API calls
 * @param password Cookie to use in Softathome API calls
 * @param gatewayType Type of Internet-Box that is connected
 * @param gatewayVersion NP version of Internet-Box, for example "10.00.44"
 */
- (void)updateGatewayCredentialsWithUserName:(NSString *)username password:(NSString *)password gatewayType:(NHGatewayType)gatewayType gatewayVersion:(NSString *)gatewayVersion;

///--------------------------------------------------------------
/// @name WLAN setup wizard
///--------------------------------------------------------------

/**
 * Indicates if the WLAN setup feature should be enabled in HomeApp.
 * Based on Gateway type and NP firmware version level configured in updateGatewayCredentialsWithUserName
 */
@property (atomic, readonly) BOOL wlanSetupSupported;

/**
 * @brief Launch the WLAN setup wizard. 
 * Note: the property wlanSetupSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after WLAN-Box setup flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchWLANSetupWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Abort the WLAN setup wizard
 */
- (void)cancelWLANSetup;

/**
 * @brief Retrieve information about available WLAN-Boxes (for Live tab). 
 * Note: the property wlanSetupSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param completionHandler Completion block that will be called when information is retrieved.
 */
- (void)checkWLANBoxInfoWithCompletionHandler:(void(^)(NSArray<NHWLANBoxInfo *> *wlanBoxesInfo))completionHandler;

///--------------------------------------------------------------
/// @name Internet Booster setup wizard
///--------------------------------------------------------------

/**
 * Indicates if Internet Booster setup feature should be enabled in HomeApp.
 * Based on Gateway type and NP firmware version level configured in updateGatewayCredentialsWithUserName
 */
@property (atomic, readonly) BOOL boosterSetupSupported;

/**
 * @brief Launch the Internet-Booster setup wizard. 
 * Note: the property boosterSetupSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after Internet-Booster setup flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchBoosterSetupWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Abort the Internet Booster setup wizard
 */
- (void)cancelBoosterSetup;

/**
 * @brief Retrieve current Internet-Booster status. 
 * Note: the property boosterSetupSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param completion Completion block that will be called when status is retrieved
 */
- (void)checkBoosterStatusWithCompletion:(NHCheckBoosterStatusCompletion)completion;

///--------------------------------------------------------------
/// @name List of Devices
///--------------------------------------------------------------

/**
 * Indicates if Device List feature should be enabled in HomeApp.
 * Based on Gateway type and NP firmware version level configured in updateGatewayCredentialsWithUserName
 */
@property (atomic, readonly) BOOL deviceListFlowSupported;

/**
 * @brief Launch the List of Devices flow. 
 * Note: the property deviceListFlowSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after Device List flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchDeviceListFlowWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Force the List of Devices flow to finish.
 * Usually you don't have to call this method as the device list will be hidden when user navigates back in the UI.
 */
- (void)cancelDeviceListFlow;

///--------------------------------------------------------------
/// @name WLAN Test
///--------------------------------------------------------------

/**
 * Indicates if WLAN Test feature should be enabled in HomeApp.
 * Based on Gateway type and NP firmware version level configured in updateGatewayCredentialsWithUserName
 */
@property (atomic, readonly) BOOL wlanTestFlowSupported;

/**
 * @brief Launch the WLAN-Test flow. 
 * Note: the property wlanTestFlowSupported will indicate if this flow is supported for the currently configured gateway type and firmware version
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after WLAN-Test flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchWLANTestFlowWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Force the WLAN-Test flow to finish.
 * Usually you don't have to call this method as the WLAN-Test screens will be hidden when user navigates back in the UI.
 */
- (void)cancelWLANTestFlow;

///--------------------------------------------------------------
/// @name Remote access support
///--------------------------------------------------------------

/**
 * Called from Homeapp when remote access connectivity state has changed.
 * The HomeNGTI framework will then query NHClientDelegate::getRemoteTunnelPort to get the current status.
 */
- (void)remoteAccessStateChanged;

@end

NS_ASSUME_NONNULL_END
