//
//  NHClient.h
//  HomeNGTI
//
//  Copyright © 2017 Swisscom. All rights reserved.
//

#import "NHClientListener.h"
#import "NHClientDelegate.h"
#import "NHClientLogger.h"
#import "NHClientTypes.h"
#import "NHWLANBoxInfo.h"

NS_ASSUME_NONNULL_BEGIN

/** Completion block for checkBoosterStatusWithCompletion */
typedef void(^NHCheckBoosterStatusCompletion)(NHBoosterStatus status);

/**
 * Main interface to the HomeNGTI framework
 */

@interface NHClient : NSObject

///--------------------------------------------------------------
/// @name Class properties
///--------------------------------------------------------------

/**
 * HomeNGTI framework version
 */
@property (readonly, class) NSString *frameworkVersion;

///--------------------------------------------------------------
/// @name Minimum firmware levels for WLAN-Box
///--------------------------------------------------------------

/**
 * Minimum firmware level for Internet-Box 2 to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box 2 does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBox2FirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Plus to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box Plus does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxPlusFirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Standard to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box standard does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxStandardFirmwareLevelForWLANBox;

/**
 * Minimum firmware level for Internet-Box Light to enable WLAN-Box item in HomeApp
 * Will be 'nil' if Internet-Box Light does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxLightFirmwareLevelForWLANBox;

///--------------------------------------------------------------
/// @name Minimum firmware levels for Internet-Booster
///--------------------------------------------------------------

/**
 * Minimum firmware level for Internet-Box 2 to enable Internet-Booster item in HomeApp
 * Will be 'nil' if Internet-Box 2 does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBox2FirmwareLevelForInternetBooster;

/**
 * Minimum firmware level for Internet-Box Plus to enable Internet-Booster item in HomeApp
 * Will be 'nil' if Internet-Box Plus does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxPlusFirmwareLevelForInternetBooster;

/**
 * Minimum firmware level for Internet-Box Standard to enable Internet-Booster item in HomeApp
 * Will be 'nil' if Internet-Box standard does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxStandardFirmwareLevelForInternetBooster;

/**
 * Minimum firmware level for Internet-Box Light to enable Internet-Booster item in HomeApp
 * Will be 'nil' if Internet-Box Light does not support this functionality
 */
@property (readonly, nullable, class) NSString *minimumInternetBoxLightFirmwareLevelForInternetBooster;


///--------------------------------------------------------------
/// @name Initialization
///--------------------------------------------------------------

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

/**
 * Shared NHClient instance
 */
+ (instancetype)sharedInstance;

///--------------------------------------------------------------
/// @name Configuration
///--------------------------------------------------------------

/**
 * Set external logging listener. Object should conform to NHClientLogger protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientLogger> *clientLogger;

/**
 * Set framework events listener. Object should conform to NHClientListener protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientListener> *clientListener;

/**
 * Set framework delegate. Object should conform to NHClientDelegate protocol, set to nil if no longer desired
 */
@property (atomic, weak, nullable) NSObject<NHClientDelegate> *clientDelegate;

/**
 * @brief Update gateway credentials. This should be the SmartVoice account credentials.
 *
 * @param username Context ID to use in Softathome API calls
 * @param password Cookie to use in Softathome API calls
 * @param gatewayType Type of Internet-Box that is connected
 */
- (void)updateGatewayCredentialsWithUserName:(NSString *)username password:(NSString *)password gatewayType:(NHGatewayType)gatewayType;

/**
 * @brief Configure framework with parameters from host application.
 * Note: At the moment there are no public parameters to change.
 * This can be changed in future.
 *
 * @param parameters Dictionary of required parameters to set or change and their values
 */
- (void)configureWithParameters:(NSDictionary<NSString *, id> *)parameters;

///--------------------------------------------------------------
/// @name WLAN setup wizard
///--------------------------------------------------------------

/**
 *  @brief Launch the WLAN setup wizard
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after WLAN-Box setup flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchWLANSetupWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Abort the WLAN setup wizard
 */
- (void)cancelWLANSetup;

/**
 * @brief Retrieve information about available WLAN-Boxes (for Live tab)
 *
 * @param completionHandler Completion block that will be called when information is retrieved.
 */
- (void)checkWLANBoxInfoWithCompletionHandler:(void(^)(NSArray<NHWLANBoxInfo *> *wlanBoxesInfo))completionHandler;

///--------------------------------------------------------------
/// @name Internet Booster setup wizard
///--------------------------------------------------------------

/**
 * @brief Launch the Internet-Booster setup wizard
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after Internet-Booster setup flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchBoosterSetupWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Abort the Internet Booster setup wizard
 */
- (void)cancelBoosterSetup;

/**
 * @brief Retrieve current Internet-Booster status
 *
 * @param completion Completion block that will be called when status is retrieved
 */
- (void)checkBoosterStatusWithCompletion:(NHCheckBoosterStatusCompletion)completion;

///--------------------------------------------------------------
/// @name List of Devices
///--------------------------------------------------------------

/**
 * @brief Launch the List of Devices flow
 *
 * @param navigationController The existing UINavigationController of your app
 * @param completionHandler Completion handler (if set) will be called after Device List flow is finished,
 *        resources are cleaned and UI unwinded to the initial screen.
 */
- (void)launchDeviceListFlowWithNavigationController:(UINavigationController *)navigationController completionHandler:(nullable void(^)(void))completionHandler;

/**
 * @brief Force the List of Devices flow to finish.
 * Usually you don't have to call this method as the device list will be hidden when user navigates back in the UI.
 */
- (void)cancelDeviceListFlow;

@end

NS_ASSUME_NONNULL_END
