#ifndef ID_H
#define ID_H

#include <string>
#include <sstream>
#include <ostream>
#include <tuple>
#include <cassert>
#include <cstdint>

class real_id_type {
public:
	using type = std::uint32_t;
private:
	type id;
public:
	explicit real_id_type(type id = 0): id(id) {}
	bool operator == (const real_id_type& other) const { return id == other.id; }
	bool operator <  (const real_id_type& other) const { return id <  other.id; }
	operator type () const { return id; }
};

class server_id_type {
public:
	using type = std::uint8_t;
private:
	type id;
public:
	explicit server_id_type(type id = 0): id(id) {}
	bool operator == (const server_id_type& other) const { return id == other.id; }
	bool operator <  (const server_id_type& other) const { return id <  other.id; }
	operator real_id_type::type () const { return real_id_type::type(id); }
};



namespace tag {
// designates types of ids so they are not mixed anywhere
struct GatewayClient {};
struct LanDevice {};
}

using decoded_id_tuple = std::tuple<server_id_type,real_id_type>;

class base_id_type {
protected:
	real_id_type compound_id;
public:
	static constexpr real_id_type::type server_bits_count = 3;
	static constexpr real_id_type::type real_bits_count   = (sizeof(real_id_type::type) * 8) - server_bits_count;
	static constexpr real_id_type::type real_bitmask      = (1 << real_bits_count) - 1;
	static constexpr real_id_type::type server_bitmask    = ((1 << server_bits_count) - 1) << real_bits_count;
protected:
	static real_id_type encode(server_id_type server) {
		assert((server & 7) == server);
		auto encoded = (real_id_type::type(server) & 7) << real_bits_count;
		return real_id_type(encoded);
	}

	static real_id_type encode(server_id_type server, real_id_type device) {
		return real_id_type(encode(server) | device);
	}

	static decoded_id_tuple decode(real_id_type device) {
		server_id_type s_id((device & server_bitmask) >> real_bits_count);
		real_id_type r_id(device & real_bitmask);
		return std::make_tuple(s_id,r_id);
	}

	explicit base_id_type(real_id_type::type compound_id): compound_id(compound_id) {}
public:
	explicit base_id_type(server_id_type server, real_id_type device)
	:	compound_id(encode(server,device))
	{}
	explicit base_id_type(server_id_type server, real_id_type::type device)
	:	compound_id(encode(server,real_id_type(device)))
	{}
	explicit base_id_type() {}

	bool operator! () const { return compound_id == 0; }
	explicit operator bool () const { return compound_id != 0; }
//	template <typename Any> operator Any () const = delete;

	std::string log_string() const {
		std::ostringstream out;
		decoded_id_tuple ids = decode(compound_id);
		out << "[server:" << std::get<server_id_type>(ids) << " id:" << std::get<real_id_type>(ids) << "]";
		return out.str();
	}
};

template <class Tag>
class id_type: public base_id_type {
private:
	explicit id_type(real_id_type::type compound_id): base_id_type(compound_id) {}
public:
	using encoded_id_type = real_id_type::type;
	explicit id_type(server_id_type server, real_id_type       device): base_id_type(server,device) {}
	explicit id_type(server_id_type server, real_id_type::type device): base_id_type(server,device) {}
	explicit id_type() {}
	bool operator == (const id_type& other) const { return compound_id == other.compound_id; }
	bool operator <  (const id_type& other) const { return compound_id <  other.compound_id; }
	bool operator != (const id_type& other) const { return !(*this == other); }
	static encoded_id_type  encoded(const id_type& id) { return id.compound_id; }
	static decoded_id_tuple decode(const id_type& id)  { return base_id_type::decode(id.compound_id); }
	static id_type          from_encoded(encoded_id_type compound_id) { return id_type(compound_id); }
};

#endif // ID_H
