//-----------------------------------------------------------------------------
// F50x_Temp_Compensated_Oscillator_Example.c
//-----------------------------------------------------------------------------
// Copyright 2009 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This program shows the use of the internal temperature sensor in adjusting
//  the oscillator calibration to account for temperature changes. This will
//  improve the 0.5% accuracy of the internal oscillator to 0.25% or better.
//
// How To Test:
//
// 1) Open a project containing F50x_Temp_Compensated_Oscillator_Example.c
//
// 2) Open the Tools menu and select Erase Code Space
//
// 3) Compile, link and download the code. 
//
// 4) Run the code 
//
// 5) The designer can see the temperature by attaching a serial cable and 
//    opening a terminal type program with the following setup:
//    
//    115200 baud, no parity, 8 bits, 1 stop bit, no control
//
// 6) The format of the message is as follows:
//
//    T = 24.3(C)    OSCIFIN = 21   temp_distance = 700
// 
//    T - the temperature in Celcius
//    OSCIFIN is the current value of OSCIFIN in decimal
//    temp_dist is the delta from room temperature multiplied by 1000
//      a temp_dist value of 700 means we are .7 degrees away from 25 degrees
//      NOTE:This number is unsigned
//
// 7) The designer can see the temperature change by simply touching the 
//    IC with any warm object including a finger (please be careful to
//    avoid ESD issues!)
//
// 8) The designer can also change the calibration (offset compensation) of the
//    temperature sensor by simply sending a string to the part using the 
//    terminal program. The string must have the following format followed by
//    a carriage return:
//
//    245
//
//    Where 245 means that the current ambient temperature is of 24.5C. The 
//    program will automatically calculate and program the proper offset so
//    that the temperature sensor will be adjusted.
//
// Target:         C8051F50x
// Tool chain:     Keil C51 8 / Keil EVAL C51
// Command Line:   None
//
// Revision History:
//
// Release 1.0 / 28 Jan 2009 (AS)
//    -Initial Revision
//
// Obs1: This program uses the following equation from the datasheet, chapter
//       5.10:  Vtemp = 3.33T(C) + 856mV
//
//       Moving T(C) to the left-hand side we have:
//       
//          T(C) = (1 / 3.33)Vtemp - (856mV / 3.33)
//
//       In this instance the equivalent linear equation is given by:
//
//          T(C) = 0.3003*Vtemp - 257.06 (V in millivolts)
//
//       Converting from V to ADC would give us:
//
//          T(C) = (0.3003*VDD*ADC) - 257.06; 
//
//       Sampling 256 times:
//
//          T(C) = (0.3003*VDD)*(ADC_sum/256) - 257.06;
//
//       Assuming a VDD = 1500 mV it leads to:
//
//          T(C) = 1.760*ADC_sum - 257.06;
//
//       To allow for precise calculations using fixed point math these
//       factors are going to be scaled up 1000x
//
// Remarks:
//
//       SLOPE - coefficient of the line that multiplies the ADC_sum value
//       OFFSET - the coefficient to be added or subtracted to the line
//       T(C) - Temperature in Celcius
//       ADC_sum - Sum of the ADC conversions
//       AVG_COUNT - This variable is how much the ADC_sum needs to be divided
//                   by in order to achieve the average ADC reading
//       Oversampling - For more information on oversampling to improve ADC
//                      resolution, please refer to Application Note AN118.
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <compiler_defs.h>
#include <c8051f500_defs.h>                 
#include <stdio.h>
#include <stdlib.h>

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

#define TRUE 1
#define FALSE 0
#define SCALE 1000L                    // Scale for temp calculations
#define SAMPLING_NUMBER 256L           // Number of samples per calculation
#define AVG_COUNT 12                   // Number of shifts (>>N) to reach the 
                                       // average value of the ADC
#define SLOPE 1760                     // Slope of the temp transfer function
#define OFFSET 257057L                 // Offset for the temp transfer function
#define SYSCLK 24000000                // SYSCLK frequency in Hz
#define BAUDRATE 115200                // Baud rate of UART in bps
#define COMP_ADDRESS 0x5000            // Location of compensation
#define TIMER0_RELOAD_HIGH    0        // Timer0 High register for Timer0_wait
#define TIMER0_RELOAD_LOW   255        // Timer0 Low register for Timer0_wait

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

unsigned long ADC_SUM = 0;             // Accumulates the ADC samples
bit CONV_COMPLETE = FALSE;             // ADC_SUM result ready flag
unsigned long code COMPENSATION _at_ COMP_ADDRESS; // Flash stored compensation
long temp_scaled;                      // Scaled noncompensated temperature value
long temp_whole;                       // Stores integer portion for output
long temp_distance;                    // Stores distance in degrees from room
int temp_frac = 0;                     // Stores fractional portion of temp

// Array of temp_dist values that trigger a change in OSCIFIN
unsigned long code TEMP_ARRAY[9] = {24296 , 42097, 54367, 64352, 72993, 
                              80729, 87794, 94341, 100470};
//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void PORT_Init (void);
void Timer2_Init(void);
void UART0_Init (void);
void ADC0_Init (void);
void Timer0_wait(int ms);
void Calibrate_Temp(int cal_input, long temp);
void Calc_Temp();
void Calibrate_Oscillator();
INTERRUPT_PROTO (ADC0_ISR, INTERRUPT_ADC0_EOC);

//-----------------------------------------------------------------------------
// main() Routine
//-----------------------------------------------------------------------------

void main (void) 
{
   bit New_Cal;                        // Flag - new temp calib value available
   int Cal_Val;                        // Calibration value for temp

   U8 SFRPAGE_save = SFRPAGE; 
   SFRPAGE = ACTIVE_PAGE;              // Set for PCA0MD

   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer 
                                       // enable)
   VDM0CN = 0x80;                      // Vdd monitor enabled

   SYSCLK_Init ();                     // Initialize Oscillator
   Timer2_Init ();                     // Initialize timer2
   PORT_Init ();                       // Initialize crossbar and GPIO
   ADC0_Init ();                       // Initialize ADC0
   UART0_Init ();                      // Initialize UART

   RSTSRC = 0x02;                      // Vdd monitor enabled as reset source

   Timer0_wait(5);                     // Wait for VREF to settle

   EA = 1;                             // Enable all interrupts
   
   printf("\f");                       // Clear screen before output begins

   SFRPAGE_save = SFRPAGE;

   // OSCIFIN is located on the CONFIG_PAGE
   SFRPAGE = CONFIG_PAGE;     
   // Check if OSCIFIN is too big for the frequency modifier
   //  algorithm to cover the full temperature spectrum
   if( OSCIFIN >= 54)
   {
      // If it is too large, lower the starting value to allow
      //  algorithm to run properly. Note that this will lower
      //  the average frequency of the oscillator
      printf("OSCIFIN out of bounds, resetting value to 54d");
      OSCIFIN = 54;
   }

   SFRPAGE = SFRPAGE_save;
    
   while (1)                           // Spin forever
   {
      if ( CONV_COMPLETE )             // If ADC completed
      {
         Calc_Temp();                  // Calculate the current temperature

         printf ("\n T =%3ld.%02d (C)   ",temp_whole,temp_frac);
         
         Calibrate_Oscillator();       // Calibrate oscillator according to temp           
         
         CONV_COMPLETE = FALSE;        // Reset ADC_sum conversion complete flag

         if ( New_Cal )                // If new calibration data
         {
            Calibrate_Temp(Cal_Val,temp_scaled);  // Calibrate temp sensor
            New_Cal = FALSE;           // Reset new temp calibration flag
         }
      }

      if( RI0 ) // Check if any key has been pressed to enter new calibration
      {
         scanf( "%d", &Cal_Val );
         printf( "You entered %d as the new calibration value \n", Cal_Val); 
         New_Cal = TRUE;               // Set flag for new calibration
      }      
   }
}

//-----------------------------------------------------------------------------
// Calibrate_Oscillator
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This routine checks what the temperature delta is between 25 degrees and the
// current temperature. Then this routine will modify OSCIFIN according to what
// the temperature delta is.
//
//-----------------------------------------------------------------------------
void Calibrate_Oscillator()
{
   static int temp_array_counter = 0;  // Array counter for OSCIFIN algorithm
   unsigned char oscifin_value;        // Temporary value holder for OSCIFIN

   U8 SFRPAGE_save = SFRPAGE;          // Save the current SFRPAGE
   SFRPAGE_save = SFRPAGE;

   SFRPAGE = CONFIG_PAGE;              // Change to CONFIG_PAGE for OSCIFIN

   oscifin_value = OSCIFIN;

   // Compare temp_distance to the TEMP_ARRAY values and adjusts accordingly
   if (temp_distance >= TEMP_ARRAY[temp_array_counter] 
         && temp_array_counter < 9)
   {
      oscifin_value++;
      OSCIFIN = oscifin_value;
      temp_array_counter++;            // Increment counter
   } else if ((temp_distance + 500) < TEMP_ARRAY[temp_array_counter - 1] 
                        && temp_array_counter > 0)
   {
      oscifin_value--;
      OSCIFIN = oscifin_value;
      temp_array_counter--;            // Decrement counter
   }

   SFRPAGE = SFRPAGE_save;             // Return SFRPAGE to original state

   // Print out the current OSCIFIN value and the current temp_distance
   printf("OSCIFIN = %d\ttemp_distance =%5lu",(int)oscifin_value,temp_distance); 
}


//-----------------------------------------------------------------------------
// Calc_Temp
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
// Globals used : temp_scaled   - converted temperature right after applying 
//                                slope and offset, used in Calibrate_Temp()
//                temp_whole    - whole number portion of the temp reading
//                temp_distance - delta value of temperature from 25 degrees C
//                temp_frac     - fraction potion of the temperature reading
//
// This routine converts the ADC readings to temperature values by applying the 
// Temperature Sensor slope and offset values from the datasheet. The math is
// explained in more detail at the top of this program. 
//
//-----------------------------------------------------------------------------
void Calc_Temp()
{
   long temp_comp;                     // Scaled and compensated temp

   // Calculate the temperature with rounding
   temp_scaled = (long) ADC_SUM;

   temp_scaled *= SLOPE;               // Calculate rounded temperature
            
   // Shift-right by AVG_COUNT to divide by the sample count,
   //  shift the decimal place into the correct position, 
   //  and get the average value of the ADC*SLOPE.
   temp_scaled = temp_scaled >> AVG_COUNT;

   temp_scaled -= OFFSET;              // Apply offset to temp      
   temp_comp = temp_scaled - COMPENSATION;   //Apply compensation

   // Calculate temp_distance from 25 degrees
   temp_distance = temp_comp - 25000;

   // Take whole number portion unscaled for UART output
   temp_whole = temp_comp/SCALE; 
            
   // temp_frac is the unscaled decimal portion of temperature
   temp_frac = (int)(( temp_comp - temp_whole * SCALE ) / (SCALE/100));
            
   // If negative, remove negative from temp_frac portion for output.
   if( temp_frac < 0 )
   {
      temp_frac *= -1;        
   }

   // Take the absolute value of temp_distance because it is a delta value
   if (temp_distance < 0)
   {
      temp_distance *= -1;
   }
}


//-----------------------------------------------------------------------------
// Calibrate_Temp
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : cal_input - the value used as reference temperature
//                temp      - the temperature measured at calibration command
// This function rewrites the value of the compensation temperature, the 
// difference between the reference temperature and the one being measured at
// that moment.
// This difference is saved in flash to be added to further measurements.
//
// REMARK: This function erases the code page where the compensation variable
//         is stored. This is done to enable the variable to be correctly 
//         modified.
//         The content of the page is not saved because this is the only info
//         stored in this page and there is no code in this page. 
//         A program that has code and/or data in this page must first save the
//         content of the page before erasing it.
//
//-----------------------------------------------------------------------------

void Calibrate_Temp(int cal_input, long temp)
{
   bit EA_state=EA;                    // Preserves EA state
   unsigned char xdata * codePtr;      // Used to write calibration
                                       // Value into FLASH memory
   UU32 COMP_VALUE;                    // Byte-addressable long variable

   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = ACTIVE_PAGE;

   VDM0CN = 0xA0;                      // Vdd monitor enabled at high threshold
   // Point to the compensation register that will contain the temperature 
   // offset to be used with the temperature sensor
   codePtr=(unsigned char xdata*) &COMPENSATION;               

   COMP_VALUE.S32 = (long) cal_input;
   COMP_VALUE.S32 *= 100;
   COMP_VALUE.S32 = temp - (long) COMP_VALUE.S32;

   EA = 0;                             // Disable interrupts
   PSCTL |= 0x02;                      // Enable page erase
   PSCTL |= 0x01;

   FLKEY=0xA5;                         // Input first flash unlock code
   FLKEY=0xF1;                         // Input second flash unlock code,

   *codePtr = 0xFF;                    // Write something in the page, which
                                       // initiates the erase process   
   PSCTL = 0;                          // Disable all, flash locked again

   FLKEY=0xA5;                         // Input first flash unlock code
   FLKEY=0xF1;                         // Input second flash unlock code,
                                       // FLASH is now unlocked
   PSCTL |= 0x01;                      // Enable FLASH Writes

   // Write high byte of compensation
   *(codePtr++) = (unsigned char)(COMP_VALUE.U8[0]);         

   PSCTL &= ~0x03;                     // Disable FLASH Writes, flash locked
 
   FLKEY=0xA5;                         // Input first flash unlock code
   FLKEY=0xF1;                         // Input second flash unlock code,
                                       // FLASH is now unlocked
 
   PSCTL |= 0x01;                      // Enable FLASH Writes

   // Write 2nd byte of compensation
   *(codePtr++) = (unsigned char)(COMP_VALUE.U8[1]);

   PSCTL &= ~0x03;                     // Disable FLASH Writes, flash locked

   FLKEY=0xA5;                         // Input first flash unlock code
   FLKEY=0xF1;                         // Input second flash unlock code,
                                       // FLASH is now unlocked
 
   PSCTL |= 0x01;                      // Enable FLASH Writes

   // Write 3rd byte of compensation
   *(codePtr++) = (unsigned char)(COMP_VALUE.U8[2]);

   PSCTL &= ~0x03;                     // Disable FLASH Writes

   FLKEY=0xA5;                         // Input first flash unlock code
   FLKEY=0xF1;                         // Input second flash unlock code,
                                       // FLASH is now unlocked
 
   PSCTL |= 0x01;                      // Enable FLASH Writes

   // Write last byte of compensation
   *codePtr = (unsigned char)(COMP_VALUE.U8[3]);

     
   PSCTL = 0;                          // Flash locked
   EA = EA_state;                      // Restore interrupt state

   VDM0CN = 0x80;                      // Vdd monitor enabled at low threshold

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This routine initializes the system clock to use the internal oscillator
// at its maximum frequency.
// Also enables the Missing Clock Detector.
//
//-----------------------------------------------------------------------------

void SYSCLK_Init (void)
{
   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = CONFIG_PAGE;

   OSCICN = 0x87;                      // Configure internal oscillator for
                                       // maximum frequency
   RSTSRC = 0x04;                      // enable missing clock detector

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This function initializes the GPIO and the Crossbar
//
// P0^2   digital   push-pull    SYSCLK
// P0^4   digital   push-pull    UART TX
// P0^5   digital   open-drain   UART RX
//
//-----------------------------------------------------------------------------

void PORT_Init (void)
{
   U8 SFRPAGE_save = SFRPAGE;

   SFRPAGE = CONFIG_PAGE;
   
   P0SKIP |= 0x03;                     // Skip P0.0 (VREF)
   P0MDIN &= ~0x01;                    // Configure VREF pin to analog
   P0MDOUT |= 0x54;                    // Set TX and SYSCLK to push-pull

   XBR0     = 0x01;                    // Enable UART0
   XBR1     = 0x02;                    // SYSCLK on pin 0.2
   XBR2     = 0x40;                    // Enable crossbar and weak pull-ups

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure Timer2 to 16-bit auto-reload and generate an interrupt at 390uS 
// intervals. A Timer 2 overflow automatically triggers an ADC0 conversion.
// 
//-----------------------------------------------------------------------------

void Timer2_Init (void)
{
   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = ACTIVE_PAGE;

   TMR2CN  = 0x00;                     // Stop Timer2; Clear TF2;
                                       // Use SYSCLK as timebase, 16-bit 
                                       //  auto-reload
   CKCON  |= 0x10;                     // Select SYSCLK for timer 2 source
   TMR2RL  = 65535 - (SYSCLK / 2560);  // Init reload value 
   TMR2    = 0xffff;                   // Set to reload immediately
   TR2     = 1;                        // Start Timer2

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure UART0 to use Timer1, for <BAUDRATE> and 8-N-1.
//
//-----------------------------------------------------------------------------

void UART0_Init (void)
{
   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = CONFIG_PAGE;   

 // Baud Rate = [BRG Clock / (65536 - (SBRLH0:SBRLL0))] x 1/2 x 1 / Prescaler

#if   ((SYSCLK / BAUDRATE / 2 / 0xFFFF) < 1)
      SBRL0 = -(SYSCLK / BAUDRATE / 2);
      SBCON0 |= 0x03;                  // Set prescaler to 1
#elif ((SYSCLK / BAUDRATE / 2 / 0xFFFF) < 4)
      SBRL0 = -(SYSCLK / BAUDRATE / 2 / 4);
      SBCON0 &= ~0x03;
      SBCON0 |= 0x01;                  // Set prescaler to 4
#elif ((SYSCLK / BAUDRATE / 2 / 0xFFFF) < 12)
      SBRL0 = -(SYSCLK / BAUDRATE / 2 / 12);
      SBCON0 &= ~0x03;                 // Set prescaler to 12
#else
      SBRL0 = -(SYSCLK / BAUDRATE / 2 / 48);
      SBCON0 &= ~0x03;
      SBCON0 |= 0x02;                  // Set prescaler to 48
#endif

   SBCON0 |= 0x40;                     // Enable baud rate generator

   SFRPAGE = ACTIVE_PAGE;

   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       // clear RI0 and TI0 bits
   TI0 = 1;                            // Indicate TX0 ready

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Initialize the ADC to use the temperature sensor.
//
//-----------------------------------------------------------------------------

void ADC0_Init (void)
{
   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = ACTIVE_PAGE;

   REF0CN = 0x27;                      // Vref == 1.5v, Temp. Sensor ON, Bias ON
   ADC0MX = 0x30;                      // Selects Temp. Sensor  
   ADC0CF = 0x38;                      // AD0SC = 7, right justified
   ADC0CN = 0x83;                      // ADC ON, starts on TMR2 overflow
   EIE1 |= 0x04;                       // enable ADC0 conversion complete int.

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This ISR averages 256 samples then copies the result to ADC_SUM.
//
//-----------------------------------------------------------------------------

INTERRUPT (ADC_ISR, INTERRUPT_ADC0_EOC)
{
   static unsigned long accumulator = 0;     // accumulator for averaging
   static unsigned int measurements = SAMPLING_NUMBER;   // measurement counter
   
   AD0INT = 0;                         // clear ADC0 conv. complete flag
  
   // Checks if obtained the necessary number of measurements
   if(measurements == 0 )
   {  
      ADC_SUM = accumulator;           // Copy total into ADC_SUM
      measurements = SAMPLING_NUMBER;  // Reset counter
      accumulator = 0;                 // Reset accumulator
      CONV_COMPLETE = TRUE;            // Set result ready flag
   } 
   else
   {     
	  accumulator += ADC0;             // If not, keep adding while shifting
                                       // out unused bits in ADC0
      measurements--;
   }      
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Timer0_wait
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1) int ms - number of milliseconds to wait
//                        range is positive range of an int: 0 to 32767
//
// This function configures the Timer0 as a 16-bit timer, interrupt enabled.
// Using the internal osc. at 24.5MHz with a prescaler of 1:8 and reloading the
// T0 registers with TIMER0_RELOAD_HIGH/LOW, it will wait for <ms>
// milliseconds.
// Note: The Timer0 uses a 1:12 prescaler
//-----------------------------------------------------------------------------
void Timer0_wait(int ms)
{
   U8 SFRPAGE_save = SFRPAGE;
   SFRPAGE = ACTIVE_PAGE;

   TH0 = TIMER0_RELOAD_HIGH;           // Init Timer0 High register
   TL0 = TIMER0_RELOAD_LOW ;           // Init Timer0 Low register
   TMOD |= 0x01;                       // Timer0 in 16-bit mode
   CKCON &= 0xFC;                      // Timer0 uses a 1:12 prescaler
   TR0  = 1;                           // Timer0 ON

   while(ms)
   {
      TF0 = 0;                         // Clear flag to initialize
      while(!TF0);                     // Wait until timer overflows
      ms--;                            // Decrement ms
   }

   TR0 = 0;                            // Timer0 OFF

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------