
//-----------------------------------------------------------------------------
//
// Copyright 2003 Cygnal Integrated Products, Inc.
// 
// Filename:      F300_HighSpeed_BC.c
// Target Device: 8051F300
// Created:       1 March 2003
// Created By:    DKC
// Tool chain:    KEIL Eval C51
//
// This is a stand alone battery charger for a Lithium ION battery.
// It utilizes a buck converter, controlled by the on-chip 8-bit PWM,
// to provide constant current followed by constant voltage battery charge.
// The High Frequency Output Mode is used to generate the switch rate. 
// The default rate is 510 kHz.
//
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f300.h>
#include "F300_HighSpeed_BC.h"          // Battery Hearder File
                   
//-----------------------------------------------------------------------------
// Functions
//-----------------------------------------------------------------------------

void Config_F300(void)
{ RSTSRC   = 0x02;                      // Enable VDD Monitor  	
  XBR0     = 0x37;                      // Skip P0.0,1,2,4,5; they're analog In
  XBR1     = 0x90;                      // Enable P0.6, P0.7, as CEX0 and CEX1
  XBR2     = 0x40;                      // Make CEX0 an 8-Bit PWM 
                                        // and CEX1 Frequency Output Mode
                                        // Also, Enable crossbar and weak pull-ups

  CMPIN2   = 1;                         // Make Comparator Output Initially low
  CMPIN1   = 0;                         //  to minimize current spikes on start-up 

  P0MDOUT  = 0xC8;                      // Set P0.3,6,7 output to push-pull 
  P0MDIN   = 0xC8;                      // Configure P0.0,1,2,4,5 as Analog Inputs

  OSCICN   = 0x07;                      // Set SYSCLK to 24.5MHz, internal osc.	

  ADC0CN   = 0xC0;                      // Turn on the ADC Module; 
                                        //  enable low power mode for settling
		
  REF0CN   = 0x0C;                      // Configure ADC's to use VDD for 
                                        //  Voltage Reference,
                                        //  Enable On-chip Temperature Sensor

//----------------------------------------------------------------
// Comparator Register Configuration
//----------------------------------------------------------------

  CPT0MX = 0x22;                        // Comparator 0 MUX Selection Register
                                        // P0.4,5 Input to Comparator
                                        // P0.3 Output of Comparator
  CPT0MD = 0x00;                        // Comparator 0 Mode Selection Register
  CPT0CN = 0x80;                        // Comparator 0 Control Register, Turn on
  
//-----------------------------------------------------------------------------
// PCA Configuration
//-----------------------------------------------------------------------------
  PCA0MD   = 0x00;                      // Disable WDT
  PCA0MD   = 0x08;                      // Set PWM Time base = SYSCLK

  PCA0L    = 0x00;                      // Initialize PCA Counter to Zero
  PCA0H    = 0x00;	
	
  PCA0CN   = 0x40;                      // Enable PCA Counter
                                        // Clear PCA Counter Overflow flag
  //Module 0
  PCA0CPM0 = 0x00;                      // Configure CCM0 to Frequency Output Mode
  PCA0CPL0 = 0x28;                      // Initialize PCA PWM to small duty cycle
  PCA0CPH0 = 0x28;                      // 0x18 makes output frequency ~510kHz
                                        // 0x28 makes output frequency ~306kHz
  
  //Module 1
  PCA0CPM1 = 0x42;                      // Configure CCM0 to 8-bit PWM mode
  PCA0CPL1 = 0xE0;                      // Initialize PCA PWM to small duty cycle
  PCA0CPH1 = 0xE0;                      // 0xB9 Ensures a Soft Initial Charge
	
  //Module 2
  PCA0CPM2 = 0x49;                      // Configure Module 1 as software timer
  PCA0CPL2 = 0xFF;                      // Initialize to 255 so that Interrupt
                                        //     is generated when PCA ends 
                                        // 8-bit PWM Cycle
  PCA0CPH2 = 0x00;                      // PCA0CPH is the high byte of the 
                                        //    Output Compare Module

  EIE1     = 0x08;                      // Enable PCA Overflow Interrupt
}

//-----------------------------------------------------------------------------
// Reset_Time_Base - Resets all Time Counting Values
//-----------------------------------------------------------------------------
void Reset_Time_Base()
{
  TIME.sec     = 0x00;
  TIME.min     = 0x00;
  TIME.hour    = 0x00;
  TIME.t_count = PWM_CLOCK; 
}

//-----------------------------------------------------------------------------
// Initialize CalibrateADCforVoltageMeasurement
//-----------------------------------------------------------------------------
// This function calibrates the voltage channel and stores the calibration
// coefficients in the parameters volt_slope and volt_offset.
//
void CalibrateADCforMeasurement()
// This calibration routine uses a 2 point cal.  
{ unsigned char xdata *pwrite;          // FLASH write pointer
  long i=0;
	
  EA = 0;                               // Disable All Interrupts

  // Wait until 1st calibration voltage is ready for cal
  //while (SW0 == 1);                     // Wait until SW0 pushed
  for (i=0;i<100000;i++);               // Wait for Switch Bounce
  
  // Once ready, Get the first calibration voltage
  AMX0SL = VBAT;                        // Select appropriate input for AMUX
  ADC0CF = (SYSCLK/5000000) << 3;       // ADC conversion clock = 5.0MHz
  ADC0CF &=0xF8;                        // Clear any Previous Gain Settings
  ADC0CF |= 0x01;                       // PGA gain = 1
  temp_INT_1.i = Measure(); 
	
  // Wait until 2nd calibration voltage is ready for cal
  //while (SW0 == 1);                     // Wait until SW0 pushed
  //for (i=0;i<100000;i++);               // Wait for Switch Bounce

  // Once ready, Get the 2nd calibration voltage
  AMX0SL = VBAT;                        //	Change Mux for second point
  temp_INT_2.i = Measure();
     
  // Calculate the SLOPE                // V1 and V2 are in tenth of a degree
  temp_LONG_1.l = (unsigned)(temp_INT_2.i-temp_INT_1.i);
  temp_LONG_1.l *= (unsigned)100;       // Account for Math Truncation Error
  temp_LONG_1.l /= (unsigned)(V2_CAL - V1_CAL);
  
	  
  // Calculate the OFFSET
  temp_LONG_2.l  = (unsigned)temp_INT_1.i;
  temp_LONG_2.l -= (signed)(temp_LONG_1.l * V1_CAL/100);
 
  temp_LONG_1.l = 2050;                 // If no cal. use these
  temp_LONG_2.l = 0;                    //  as default values
    
  // Erased memory at page 0x1A00
  pwrite = (char xdata *)&(CHECK_BYTE.b[0]);

  PSCTL = 0x03;                         // MOVX writes target FLASH memory;
                                        // FLASH erase operations enabled

  FLKEY = 0xA5;                         // FLASH key sequence #1
  FLKEY = 0xF1;                         // FLASH key sequence #2
  *pwrite = 0x00;                       // initiate PAGE erase

  // Write the Volt SLOPE and OFFSET to Flash
  PSCTL = 1;                            // MOVX writes to Flash
   
  pwrite = (char xdata *)&(VOLT_SLOPE.b[0]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[0];
  pwrite = (char xdata *)&(VOLT_SLOPE.b[1]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[1];
  pwrite = (char xdata *)&(VOLT_SLOPE.b[2]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[2];
  pwrite = (char xdata *)&(VOLT_SLOPE.b[3]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[3];

  pwrite = (char xdata *)&(VOLT_OFFSET.b[0]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[0];
  pwrite = (char xdata *)&(VOLT_OFFSET.b[1]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[1];
  pwrite = (char xdata *)&(VOLT_OFFSET.b[2]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[2];
  pwrite = (char xdata *)&(VOLT_OFFSET.b[3]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[3];
  
  PSCTL = 0;                            // MOVX writes target XRAM

//-----------------------------------------------------------------------------
// Initialize CalibrateADCforCurrentMeasurement_NOAMP
//-----------------------------------------------------------------------------
// This function calibrates the current channel with no external amp
// and stores the calibration coefficients in the 
// parameters i_noamp_slope and i_noamp__offset.
//
// This calibration routine uses a 2 point cal.  
  // Wait until calibration voltage is ready for cal
  //while (SW0 == 1);                     // Wait until SW0 pushed
  //for (i=0;i<100000;i++);               // Wait for Switch Bounce

  // Once ready, Get the first calibration voltage
  AMX0SL = IBAT;                        // Select appropriate input for AMUX
  ADC0CF = (SYSCLK/5000000) << 3;       // ADC conversion clock = 5.0MHz
  ADC0CF &=0xF8;                        // Clear any Previous Gain Settings
  ADC0CF |= 0x03;                       // Set PGA gain = 4
  temp_INT_1.i = Measure();             // Acquire 16-bit Conversion
  temp_INT_1.i *= 2;                    // Account for Differential Mode
  
  // Wait until 2nd calibration voltage is ready for cal
  //while (SW0 == 1);                     // Wait until SW0 pushed
  //for (i=0;i<100000;i++);               // Wait for Switch Bounce

  // Once ready, Get the 2nd calibration voltage
  temp_INT_2.i = Measure();             // Acquire 16-bit Conversion   
  temp_INT_2.i *=2;                     // Account for Differential Mode
  
  // Calculate the SLOPE 
  temp_LONG_1.l =  (unsigned)(temp_INT_2.i - temp_INT_1.i);
  temp_LONG_1.l *= (unsigned)100;       // Account for Math Truncation Error
  temp_LONG_1.l /= (unsigned)(I2_CAL - I1_CAL);
  temp_LONG_1.l /= (unsigned)INT_CURRENT_GAIN;// Account for Gain

  // Calculate the OFFSET
  temp_LONG_2.l =  (signed)(temp_INT_1.i/INT_CURRENT_GAIN);
  temp_LONG_2.l -= (signed)(temp_LONG_1.l * V1_CAL/100);
   
  temp_LONG_1.l = 2050;                 // If no cal. use these
  temp_LONG_2.l = 0;                    //  as default values

  // Memory at 0x1A00 is already erased
  // Write the Volt SLOPE and OFFSET to Flash
  PSCTL = 1;                            // MOVX writes to Flash
   
  pwrite = (char xdata *)&(I_NOAMP_SLOPE.b[0]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[0];
  pwrite = (char xdata *)&(I_NOAMP_SLOPE.b[1]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[1];
  pwrite = (char xdata *)&(I_NOAMP_SLOPE.b[2]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[2];
  pwrite = (char xdata *)&(I_NOAMP_SLOPE.b[3]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_1.b[3];

  pwrite = (char xdata *)&(I_NOAMP_OFFSET.b[0]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[0];
  pwrite = (char xdata *)&(I_NOAMP_OFFSET.b[1]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[1];
  pwrite = (char xdata *)&(I_NOAMP_OFFSET.b[2]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[2];
  pwrite = (char xdata *)&(I_NOAMP_OFFSET.b[3]);
  FLKEY = 0xA5;
  FLKEY = 0xF1;                         // enable flash write
  *pwrite = temp_LONG_2.b[3];
  
  PSCTL = 0;                            // MOVX writes target XRAM
}

//-----------------------------------------------------------------------------
// Measure
//-----------------------------------------------------------------------------
//
// This routine averages 65536 ADC samples and returns a 16-bit unsigned 
// result.
// 
unsigned int Measure (void)
{
  unsigned i;                           // sample counter
  unsigned long accumulator=0L;         // here's where we integrate the
                                        // ADC samples

  // read the ADC value and add to running total
  i = 0;
  do {
    AD0INT = 0;                         // clear end-of-conversion indicator
    AD0BUSY = 1;                        // initiate conversion
    while(!AD0INT);                     // wait for conversion to complete
    accumulator += ADC0;                // read adc value and accumulate
    i++;                                // update counter
  } while (i != 0x0000);
   
  // the accumulator now contains 16 added bits of which 8 are usable 
  return (unsigned int) (accumulator >> 8);    
}

//-----------------------------------------------------------------------------
// Regulate_Current
//-----------------------------------------------------------------------------
// This routine monitors the battery's current and adjusts 
// the PWM (i.e. duty cycle) to keep the current at a known value
//
void Regulate_Current(int passed_current)
{ unsigned int temp = 0,delay_count = 0;
  do{
    temp = Monitor_Battery(CURRENT);    // Measure Current
    if (temp < passed_current)
    {     PCA0CPH1--;
	      for(delay_count = 0;delay_count<2500;delay_count++);
	}
    if (temp > passed_current)
    {  PCA0CPH1++;
       for(delay_count = 0;delay_count<2500;delay_count++);
    }
  
  }while ((temp < (passed_current - CURRENT_TOLERENCE)) || 
          (temp > (passed_current + CURRENT_TOLERENCE)));  
                                        // I_BULK or I_LOWCURRENT is set now

  temp = Monitor_Battery(VOLTAGE_PWM_OFF);
                                        // If VOLTAGE within range, 
                                        // change from constant CURRENT charge
                                        // mode to constant VOLTAGE charge mode
  if ((temp >= (VOLT_LOWCURRENT - VOLT_TOLERANCE*2)) &&
   (temp <= (VOLT_LOWCURRENT + VOLT_TOLERANCE*2)))
  {
    CONST_C = 0;
    CONST_V = 1;
  }

}

//-----------------------------------------------------------------------------
// Regulate_Voltage
//-----------------------------------------------------------------------------
// This routine monitors the battery's voltage and adjusts 
// the PWM (i.e. duty cycle) to keep the voltage at a known value
//
void Regulate_Voltage(void)
{ unsigned int temp = 0,delay_count = 0;
                                        // set VOLT_BULK (with "soft start")
  do{
    temp = Monitor_Battery(VOLTAGE);
    
	if (temp < VOLT_BULK)
    {      PCA0CPH1--;
           for(delay_count = 0;delay_count<2500;delay_count++);
    }
    if (temp > VOLT_BULK)
    {  PCA0CPH1++;
       for(delay_count = 0;delay_count<2500;delay_count++);
    }

  }while ((temp < (VOLT_BULK - VOLT_TOLERANCE)) || 
            (temp > (VOLT_BULK + VOLT_TOLERANCE)));
                                        // VOLTAGE is set now
}

//-----------------------------------------------------------------------------
// Turn_PWM_Off
//-----------------------------------------------------------------------------
// This routine peforms a soft charge turn off by taking the PWM's  
// duty cycle slowly to zero.
//
void Turn_PWM_Off(void)
{ 
  do{
    if (PCA0CPH1 < 0xF0)
      PCA0CPH1++;
                  
  }while (PCA0CPH1 < 0xF0);                       
  // Duty Cycle is now small and safe to turn off.

  PCA0CPM0 = 0x00;                      // Disable PWM
}

//-----------------------------------------------------------------------------
// Monitor_Battery
//-----------------------------------------------------------------------------
// This routine acts as a switch when gathering different conversion types.
// It adjusts the throughput, adjust the AMUX and returns the current in mA,
//  voltage in mV, and temperature in C, 2% accurate.
//
int Monitor_Battery(unsigned char value)
{
  char i;
  unsigned long av =0,delay_count=0;
  long signed result;

  ADC0CF = (SYSCLK/5000000) << 3;       // ADC conversion clock = 5.0MHz
  ADC0CF &= 0xF8;                       // Clear any Previous Gain Settings
  
  switch (value)
  {
    case TEMPERATURE:
      //Turn_PWM_Off();                   // Turn PWM Off   
      AMX0SL = TBAT;                    // Select appropriate input for AMUX
      ADC0CF |= 0x02;                   // Set PGA gain = 2
      break; 
	  
    case VOLTAGE: 
      AMX0SL = VBAT;                    // Select appropriate input for AMUX
      ADC0CF |= 0x01;                   // Set PGA gain = 1
      break;   

    case VOLTAGE_PWM_OFF: 
      //Turn_PWM_Off();                   // Turn PWM Off   
      AMX0SL = VBAT;                    // Select appropriate input for AMUX
      ADC0CF |= 0x01;                   // Set PGA gain = 1
      break;  

	case CURRENT: 
      AMX0SL = IBAT;                    // Select appropriate input for AMUX
      ADC0CF |= 0x01;                   // Set PGA gain = 1
      break;
	
  }  
  
  //Compute average of next 10 A/D conversions
  for(delay_count = 0;delay_count<2500;delay_count++);// Allow Settling Time
  for(av=0,i=10;i;--i){	
    AD0INT = 0;                         // clear end-of-conversion indicator
    AD0BUSY = 1;                        // initiate conversion
    while(!AD0INT);                     // wait for conversion to complete
    av = av+ADC0;
  }

  av = av/10;                           // Compute the average
  av = av<<8;                           // Convert to 16-bit conversion
                                        // ...to account for 16-bit cal.
                                        //    coefficients
  
  PCA0CPM0 = 0x46;                      // Turn on PWM

  switch (value)
  { case TEMPERATURE:
      result =  (long) av * 1000/TEMP_SLOPE; 
      break;

    case VOLTAGE:
    case VOLTAGE_PWM_OFF:
      result = (av - VOLT_OFFSET.l);    // Account for System Errors
      result *= 100;                    // Account for Math Truncation Error
      result *= RESAB;                  // Account for Divide Resistors
	  result /= VOLT_SLOPE.l;           // Convert to Voltage in Millivolts
      result /= RESB;                   
      result -= ((RSENSE*Current)/100); // Account for Sense Resistor Voltage Drop
	break;
    case CURRENT:
      result = (av - I_NOAMP_OFFSET.l); // Account for System Errors
      result *= 100;                    // Account for Math Truncation Error
      result *= 100;                    // Account for Sense Resistor
	  result /= I_NOAMP_SLOPE.l;        // Convert to Milliamps
      result /= RSENSE;                 // Account for Sense Resistor
      result /= EXT_CURRENT_GAIN;       // Account for external Amplifier
	  Current = (int) result;
	  break;
  }

  return (int) result;  	
}

//-----------------------------------------------------------------------------
// Bulk_Charge Function
//-----------------------------------------------------------------------------
void Bulk_Charge(void)
{
  unsigned int temp = 0;
  unsigned int bulk_finish_hour = 0;
  unsigned int bulk_finish_min = 0;
  unsigned int delay_hour = 0;  
  unsigned int delay_min = 0;
  unsigned int last_min = 0;
                                        	
  Reset_Time_Base();                    // Reset Time Base to zero

                                        // Calculate BULK charge finish time
  bulk_finish_min = (TIME.min + MAX_TIME_BULK);
  bulk_finish_hour = TIME.hour;
  while (bulk_finish_min > 60)
  {
    bulk_finish_min = bulk_finish_min - 60;
    bulk_finish_hour++;
  }

  CONST_C = 1;                          // Start in constant current charge mode
  DELAY   = 0;                          // Reset timer DELAY
    

  temp = Monitor_Battery(TEMPERATURE);  // Monitor Temperature
                                        // Is temperature within range? 
  
  if ((temp > MIN_TEMP_ABS) && (temp < MAX_TEMP_ABS))
  {
    temp = Monitor_Battery(VOLTAGE);    // Monitor Voltage    
	                                    // Is Voltage within range?  
    Voltage = temp;                     // for Debug

	if ((temp <= (MAX_VOLT_ABS + VOLT_TOLERANCE)) && (temp > MIN_VOLT_BULK))
    {
      PCA0CPM0 = 0x46;                  // Turn on PWM
	 
      // Enter main loop in Bulk_Charge()
      while ((BULK == 1) && (ERROR == 0))
      {
        if (CONST_C == 1)
          Regulate_Current(I_BULK);
		
        else if (CONST_V == 1)
        {  Current = Monitor_Battery(CURRENT);    // Measure Current
           if((Current < IMIN)||(Current > IMAX))
           { CONST_V = 0;                // Exit CONST_V
             CONST_C = 1;                // Prepare to enter CONST_C
             BULK = 0;                   // Prepare to exit BULK mode 
             LOWCURRENT = 1;             // Prepare to enter LOWCURRENT Mode
			 if (Current < IMIN)
                I_MIN = 1;              // Indicate Specific Error for Debug
              else
                I_MAX = 1;              // Indicate Specific Error for Debug
           }  
		  else if ((Current < IMAX) && (Current > IMIN))
		  {  I_MAX = 0;                 // Reset Error Flag
             I_MIN = 0;                 // Reset Error Flag
             Regulate_Voltage();        // Charge with Constant Voltage
          }
        }
		  
		
        // Now, Check for error and charge termination conditions
        // If above max charge time, flag error    
        // Test for BULK Charge Time Out
                   
                                        // Monitor Time 
        if ((TIME.hour == bulk_finish_hour) && (TIME.min == bulk_finish_min) 
           &&  (DELAY == 0))                               
        {
          Turn_PWM_Off();               // Turn Off PWM
          TIME_MAX = 1;                 // Set Time max error flag
          ERROR    = 1;                 // Set general error flag 
        }

                                        // Monitor Temperature  
        temp = Monitor_Battery(TEMPERATURE);
        if ((temp < MIN_TEMP_ABS) && (temp > MAX_TEMP_ABS))
		{
          Turn_PWM_Off();               // Turn Off PWM
          
		  if (temp < MIN_TEMP_ABS)          
            TEMP_MIN = 1;               // Set Temperature below minimum flag
          else
            TEMP_MAX = 1;               // Set Temperature exceeds maximum flag
                                            
          ERROR    = 1;                 // Set general error flag
        }
        
                                        // Minute elapsed?
                                        // Check for minimum current
                                        // if reached, enter last DELAY charge
        if (TIME.min != last_min)   
        {
          last_min = TIME.min;
          if ((CONST_V == 1) && (DELAY == 0) && (Monitor_Battery(CURRENT) 
             <= MIN_I_BULK))
          {
                                        // Calculate TOP OFF Battery Time finish time 
            delay_min = (TIME.min + BULK_TIME_DELAY);
            delay_hour = TIME.hour;
            while (delay_min > 60)
            {
              delay_min = delay_min - 60;
              delay_hour++;
            }            
            DELAY = 1;                  // Set Delay Flag
          }
 
                                        // Monitor Delay time, time up?
          if ((TIME.hour == delay_hour)&&(TIME.min == delay_min) &&
             (DELAY == 1))
          {
            Turn_PWM_Off();             // Turn Off PWM
            CONST_V = 0;                // Exit CONST_V
            CONST_C = 1;                // Prepare to enter CONST_C
            BULK = 0;                   // Prepare to exit BULK mode 
            LOWCURRENT = 1;             // Prepare to enter LOWCURRENT Mode
          }                         
        }                        
      }                                 // End Main While loop
    }
    
    else if(ERROR == 0)
    {
      if (temp > (MAX_VOLT_ABS + VOLT_TOLERANCE))
      { VOLT_MAX = 1;                   // Set Max Voltage error flag
        ERROR    = 1;                   // Set general error flag
      }
	  else if(temp < MIN_VOLT_BULK)
      { VOLT_MIN = 1;                   // Set Minimum bulk voltage error flag  
        LOWCURRENT = 1;                 // Switch to LOWCURRENT mode
        BULK = 0;                       // Exit Bulk Charge mode
      }                                 // battery's voltage very low
    }		 
  }		 

  else if(ERROR == 0)                   // Absolute temperature out of range?
  {   
    if (temp < MIN_TEMP_ABS)          
     TEMP_MIN = 1;                      // Set Temperature below minimum flag
    else
     TEMP_MAX = 1;                      // Set Temperature exceeds maximum flag
                                            
      ERROR = 1;                        // Set general error flag
  }
}

//-----------------------------------------------------------------------------
// Lowcurrent_Charge 
//-----------------------------------------------------------------------------

void Lowcurrent_Charge(void)
{
  unsigned int temp = 0;
  unsigned int lowcurrent_finish_min = 0;
  unsigned int lowcurrent_finish_hour = 0;

  Reset_Time_Base();                    // Reset Time base to zero

                                        // Calculate LOWCURRENT finish time
  lowcurrent_finish_min = (TIME.min + MAX_TIME_LOWCURRENT); 
  lowcurrent_finish_hour = TIME.hour;
  while (lowcurrent_finish_min > 60)
  {
    lowcurrent_finish_min = lowcurrent_finish_min - 60;
    lowcurrent_finish_hour++;
  }

  // Enter Main Lowcurrent Loop. 
  // Only exits are upon error and full charge
  while ((LOWCURRENT == 1) && (ERROR == 0))  
  {    
    temp = Monitor_Battery(TEMPERATURE);// Get Temperature Reading
                                        // Is TEMPERATURE within limits 
    if ((temp > MIN_TEMP_ABS) && (temp < MAX_TEMP_ABS))   
    {
      // Is Battery's Charge Voltage below max charge voltage
      temp = Monitor_Battery(VOLTAGE);  // Get Voltage Reading
      if (temp <= (VOLT_LOWCURRENT + VOLT_TOLERANCE))
      {
        if (CONST_C == 1)               // CONST_C ?, charge w/ constant current
          Regulate_Current(I_LOWCURRENT);

        if (CONST_V == 1)               // CONST_V?, charge w/ constant voltage
          Regulate_Voltage();       
      
        if ((temp >= MIN_VOLT_BULK) && (DELAY == 0))// Bulk Threshold voltage met?
        { LOWCURRENT = 0;               // Exit LOWCURRENT mode
          BULK = 1;                     // Switch to Bulk Charge mode
        }
                                        // Check elapsed time 
        if ((TIME.hour == lowcurrent_finish_hour) && 
        ( TIME.min == lowcurrent_finish_min))
        {
          TIME_MAX = 1;                 // Set Time MAX error flag
          ERROR    = 1;                 // Set general error flag    
        }
      }
      else if(ERROR == 0)               // Voltage to high?
      {
        VOLT_MAX = 1;                   // Set Max voltage error flag
        ERROR    = 1;                   // Set general error flag
      }		 
    }		 
    else if(ERROR == 0)                 // Absolute temperature out of range?
    {   
      if (temp < MIN_TEMP_ABS)          
        TEMP_MIN = 1;                   // Set Temperature below minimum flag
      else
        TEMP_MAX = 1;                   // Set Temperature exceeds maximum flag
      
      ERROR = 1;                        // Set general error flag
    }
  }
}

//-----------------------------------------------------------------------------
// Main Function
//-----------------------------------------------------------------------------
void main(void)
{ 
  EA = 0;                               // Disable All Interrupts
  Reset_Time_Base();
  Config_F300();                        // Config F300
//Turn_PWM_Off();                       // Turn PWM off before Calibration
//CalibrateADCforMeasurement();         // Calibrate F300

  EA = 1;                               // Enable All Active Interrupts	

  while(1)
  {
    LED0 = 0;                           // Turn LED0 off    
   
    TERMINATION = 0x00;                 // Reset Termination Flags
    CHARGE_STATUS = 0x00;               // Reset Charge Status Flags
	LOWCURRENT = 0;
	BULK = 1;                           // Start in LOWCURRENT Charge Mode
    CONST_C = 1;

    while (ERROR == 0)
    {
      if (BULK == 1)
      {
        Bulk_Charge();                  // Enter Bulk Charge Mode
	  }                   
      if (LOWCURRENT == 1)
        Lowcurrent_Charge();            // Enter Lowcurrent_Charge function 
                                        // Toggle LED0 at 1 Hz rate via ISR
    }

    if (ERROR == 1)
    {
      Turn_PWM_Off();;                  // Turn PWM Off
      EA = 0;                           // Disable All Interrupts
      while (1);                        // Enter a eternal loop 
                                        // No recovery except "reset-button"
    }   
  } 
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// PCA_ISR
//-----------------------------------------------------------------------------
// This routine counts the elapsed time in seconds, minutes, hours.
// It also toggles LED0 every second when in Lowcurrent Charge Mode.
// This routine interrupts every time the PCA counter overflows, every 256
//  SYSCLK cycles. After SYSCLK/256 interrupts, one second has elapsed.
// 
 void PCA_OVERFLOW_ISR (void) interrupt 9
{ int time_sec;
  PCA0CN = 0x40;                        // Reset all PCA Interrupt Flags

  PCA0H = 0x00;                         // Reset High Byte of PCA Counter 
                                        //   of 8-bit PWM we are using Module1 

  if (0x0000 == --TIME.t_count)
  {
    TIME.t_count = PWM_CLOCK;           // Reset 1 Second Clock
    if ( 60 == ++TIME.sec )             // Account for elapsed seconds
    {                                   // Reset second counter every minute
      TIME.sec = 0x00;                  
      if ( 60 == ++TIME.min )           // Account for elapsed minutes
      {                                 // Reset minute counter every hour
        TIME.min = 0x00;
        if ( 24 == ++TIME.hour )        // Account for elapsed hours
          TIME.hour = 0x00;             // Reset hour counter every day
      }
    }
   time_sec = TIME.sec;

    if ((LOWCURRENT == 1) && (ERROR == 0))
    {                                   // Blink LED0 at 1 Hz if in Lowcurrent 
      //if (TIME.sec % 2)
       // LED0 = 0;                       // Turn on LED every odd second
      //else 
       // LED0 = 1;                       // Turn on LED every even second
    }    		
  }
}


