//------------------------------------------------------------------------------------
// Blinky.c
//------------------------------------------------------------------------------------
// Copyright 2003 Cygnal Integrated Products, Inc.
//
// AUTH: JM
// DATE: 10 JUNE 03
//
// This program flashes the green LED on the C8051F020 target board about five times
// a second using the interrupt handler for Timer3.
// Target: C8051F02x
//
// Tool chain: HI-TECH Evaluation 'C'
//

//------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------
#include "8051.h"                                         // SFR declarations
#include "intrpt.h"                                           // Interrupt declarations

//------------------------------------------------------------------------------------
// Global CONSTANTS
//------------------------------------------------------------------------------------

#define SYSCLK 2000000                    // approximate SYSCLK frequency in Hz

static volatile bit LED @ 0x96;           // green LED: '1' = ON; '0' = OFF; Location 0x96 is P1^6

//------------------------------------------------------------------------------------
// Function PROTOTYPES
//------------------------------------------------------------------------------------
void PORT_Init (void);
void Timer3_Init (int counts);
interrupt void Timer3_ISR (void);


//------------------------------------------------------------------------------------
// MAIN Routine
//------------------------------------------------------------------------------------
void main (void) {
	
   ROM_VECTOR(TIMER3, Timer3_ISR);        //Interrupt vector TIMER3 (0x73) points to 
                                          //TIMER3_ISR for interrupt handling
   
   // disable watchdog timer
   WDTCN = 0xde;
   WDTCN = 0xad;

   PORT_Init ();
   Timer3_Init (SYSCLK / 12 / 10);        // Init Timer3 to generate interrupts
                                          // at a 10Hz rate.

   EA = 1;                                // Enable global interrupts
   
   while (1) {                            // spin forever
   }
}

//------------------------------------------------------------------------------------
// PORT_Init
//------------------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports
//
void PORT_Init (void)
{
   XBR2    = 0x40;                     // Enable crossbar and weak pull-ups
   P1MDOUT |= 0x40;                    // enable P1.6 (LED) as push-pull output
}

//------------------------------------------------------------------------------------
// Timer3_Init
//------------------------------------------------------------------------------------
//
// Configure Timer3 to auto-reload and generate an interrupt at interval
// specified by <counts> using SYSCLK/12 as its time base.
//
void Timer3_Init (int counts)
{
  

   TMR3CN = 0x00;                      // Stop Timer3; Clear TF3;
                                       // use SYSCLK/12 as timebase
   TMR3RLL = -counts;                  // Init reload values
   TMR3RLH = -counts >> 8;             // Init reload values
   
   TMR3L = 0xff;                       // set to reload immediately
   TMR3H = 0xff;                       // set to reload immediately

   EIE2   |= 0x01;                     // enable Timer3 interrupts
   TMR3CN |= 0x04;                     // start Timer3
}

//------------------------------------------------------------------------------------
// Interrupt Service Routines
//------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------
// Timer3_ISR
//------------------------------------------------------------------------------------
// This routine changes the state of the LED whenever Timer3 overflows.
//
interrupt void Timer3_ISR (void)
{
   TMR3CN &= ~(0x80);                     // clear TF3
   LED = !(LED);                          // change state of LED
}